/* $USAGI: sockaddr_utils.c,v 1.6 2002/10/04 11:31:28 mk Exp $ */
/*
 * Copyright (C)2001 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors:
 *   Kazunori MIYAZAWA <miyazawa@linux-ip.org>
 *   Mitsuru KANDA <mk@linux-ipv6.org>
 */
/*
 * sadb_utils.c include utility functions for SADB handling.
 */

#include <linux/config.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <net/ipv6.h>
#include <linux/inet.h>
#include <linux/ipsec.h>
#include <linux/pfkeyv2.h>
#include <net/sadb.h>

#define BUFSIZE 64

int
sockaddrtoa(struct sockaddr *addr, char *buf, size_t buflen)
{
	int ret = 0;

	switch (addr->sa_family) {
	case AF_INET:
		sprintf(buf, "%d.%d.%d.%d", NIPQUAD((((struct sockaddr_in *)addr)->sin_addr)));
		break;
	case AF_INET6:
		in6_ntop(&(((struct sockaddr_in6*)addr)->sin6_addr), buf);
		break;
	default:
		ret = -EAFNOSUPPORT;
		break;
	}

	return ret;
}

int
sockporttoa(struct sockaddr *addr, char *buf, size_t buflen)
{
	int ret = 0;

	switch (addr->sa_family) {
	case AF_INET:
		sprintf(buf, "%hd", ntohs(((struct sockaddr_in *)addr)->sin_port));
		break;
	case AF_INET6:
		sprintf(buf, "%hd", ntohs(((struct sockaddr_in6*)addr)->sin6_port));
		break;
	default:
		printk(KERN_WARNING "sockporttoa: unrecognized socket family: %d\n", addr->sa_family);
		ret = -EAFNOSUPPORT;
		break;
	}

	return ret;
}

/*
 * addr1, prefixlen1 : packet(must set 128 or 32 befor call this) 
 * addr2, prefixlen2 : sa/sp
 */
int
sockaddr_prefix_compare(struct sockaddr *addr1, __u8 prefixlen1,
			struct sockaddr *addr2, __u8 prefixlen2)
{
	__u8 prefixlen;

	if (!addr1 || !addr2) {
		SADB_DEBUG("addr1 or add2 is NULL\n");
		return -EINVAL;
	}

	if (addr1->sa_family != addr2->sa_family) {
		SADB_DEBUG("sa_family not match\n");
		return 1;
	}

	if (prefixlen1 < prefixlen2) 
		prefixlen = prefixlen1;
	else
		prefixlen = prefixlen2;
	SADB_DEBUG("prefixlen: %d, prefixlen1: %d, prefixlen2: %d\n", prefixlen, prefixlen1, prefixlen2);

	switch (addr1->sa_family) {
	case AF_INET:
		if (prefixlen > 32 )
			return 1;
		return (((struct sockaddr_in *)addr1)->sin_addr.s_addr ^
			  ((struct sockaddr_in *)addr2)->sin_addr.s_addr) &
			 htonl((0xffffffff << (32 - prefixlen)));
	case AF_INET6:
		if (prefixlen > 128)
			return 1;

		return ipv6_prefix_cmp(&((struct sockaddr_in6 *)addr1)->sin6_addr,
				       &((struct sockaddr_in6 *)addr2)->sin6_addr,
				       prefixlen);
	default:
		SADB_DEBUG("unknown sa_family\n");
		return 1;
	}
}

int 
sockaddr_compare_ports(struct sockaddr *addr1, struct sockaddr *addr2)
 {
	if (addr1->sa_family != addr2->sa_family)
		return -EINVAL;

	switch (addr1->sa_family) {
	case AF_INET:
		if (((struct sockaddr_in *)addr1)->sin_port && ((struct sockaddr_in *)addr2)->sin_port)
			return !( ((struct sockaddr_in *)addr1)->sin_port == ((struct sockaddr_in *)addr2)->sin_port);
		break;
	case AF_INET6:
		if (((struct sockaddr_in6 *)addr1)->sin6_port && ((struct sockaddr_in6 *)addr2)->sin6_port)
			return !( ((struct sockaddr_in6 *)addr1)->sin6_port == ((struct sockaddr_in6 *)addr2)->sin6_port);
		break;
	default:
		SADB_DEBUG("%s:%d: compare_ports_if_set: unsupported address family: %d\n", 
			  __FILE__, __LINE__, addr1->sa_family);
		return -EINVAL;
		break;
	}

	return 0; /* should never reach here */
}

