/*****************************************************************************
;
;   (C) Unpublished Work of ADMtek Incorporated.  All Rights Reserved.
;
;       THIS WORK IS AN UNPUBLISHED WORK AND CONTAINS CONFIDENTIAL,
;       PROPRIETARY AND TRADESECRET INFORMATION OF ADMTEK INCORPORATED.
;       ACCESS TO THIS WORK IS RESTRICTED TO (I) ADMTEK EMPLOYEES WHO HAVE A
;       NEED TO KNOW TO PERFORM TASKS WITHIN THE SCOPE OF THEIR ASSIGNMENTS
;       AND (II) ENTITIES OTHER THAN ADMTEK WHO HAVE ENTERED INTO APPROPRIATE
;       LICENSE AGREEMENTS.  NO PART OF THIS WORK MAY BE USED, PRACTICED,
;       PERFORMED, COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,
;       ABBRIDGED, CONDENSED, EXPANDED, COLLECTED, COMPILED, LINKED, RECAST,
;       TRANSFORMED OR ADAPTED WITHOUT THE PRIOR WRITTEN CONSENT OF ADMTEK.
;       ANY USE OR EXPLOITATION OF THIS WORK WITHOUT AUTHORIZATION COULD
;       SUBJECT THE PERPERTRATOR TO CRIMINAL AND CIVIL LIABILITY.
;
;------------------------------------------------------------------------------
;
;    Project : ADM5120
;    Creator : daniell@admtek.com.tw
;    File    : net/am5120sw/swdrv.c
;	 Date    : 2003.5.14
;    Abstract: ADM5120 Switch driver
;
;Modification History:
;
;*****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>

#include <linux/sched.h>
#include <linux/kernel.h>		/* printk() */
#include <linux/slab.h>			/* kmalloc() */
#include <linux/errno.h>		/* error codes */
#include <linux/types.h>		/* size_t */
#include <linux/interrupt.h>	/* mark_bh */

#include <linux/in.h>
#include <linux/netdevice.h>	/* struct device, and other headers */
#include <linux/etherdevice.h>	/* eth_type_trans */
#include <linux/ip.h>			/* struct iphdr */
#include <linux/tcp.h>			/* struct tcphdr */
#include <linux/skbuff.h>
#include <linux/in6.h>

#include <asm/checksum.h>

#include "if5120.h"

/*
 * Driver resources constants
 */
#define NUM_TX_H_DESC		16		/* Number of the Transmitting descriptors of high priority */
#define NUM_TX_L_DESC		128		/* Number of the Transmitting descriptors of low priority */
#define NUM_RX_H_DESC		16		/* Number of the Receiving descriptors of high priority */
#define NUM_RX_L_DESC		64		/* Number of the Receiving descriptors of low priority */


PSW_CONTEXT_T sw_context;

static long vlan[MAX_VLAN_GROUP] = {0x4f, 0x50, 0, 0, 0, 0};

extern struct net_device am5120sw_devs[AM5120SW_IFNUM];



/*
 * Set Mac address
 */
void macaddr_init(void)
{

/*
	BOARD_CFG_T boardCfg = {0};
	int i;
	
	// Read in the old configuration data.
    memcpy((char *)&boardCfg, (char *)PA2VA(AM5120SW_BOARD_CFG_ADDR), sizeof (boardCfg)); 

	if (boardCfg.macmagic != MAC_MAGIC)
	{
		boardCfg.macmagic = MAC_MAGIC;
		
		//set default mac address
		for (i = 0; i < AM5120SW_IFNUM; i++)
		{
			memcpy(&boardCfg.mac[i][0], AM5120SW_DEFAULT_MAC0, 6);
			boardCfg.mac[i][5] = i;
		}

		flash_erase((char *)PA2VA(AM5120SW_BOARD_CFG_ADDR), sizeof (boardCfg));
		flash_write((char *)PA2VA(AM5120SW_BOARD_CFG_ADDR), (char *)&boardCfg, sizeof (boardCfg));
	}
*/	
	return;
}


void macaddr_read(char *mac, int unit)
{
/*
	BOARD_CFG_T boardCfg={0};

	// Read in the old configuration data.
  	memcpy((char *)&boardCfg, (char *)PA2VA(AM5120SW_BOARD_CFG_ADDR), sizeof (boardCfg)); 
	memcpy(mac, boardCfg.mac[unit], 6);
*/
}


void macaddr_write(char *mac, int unit)
{
/*
	BOARD_CFG_T boardCfg={0};

	memcpy((char *)&boardCfg, (char *)PA2VA(AM5120SW_BOARD_CFG_ADDR), sizeof (boardCfg)); 

	memcpy(&boardCfg.mac[unit][0], mac, 6);

	flash_erase((char *)PA2VA(AM5120SW_BOARD_CFG_ADDR), sizeof (boardCfg));
	flash_write((char *)PA2VA(AM5120SW_BOARD_CFG_ADDR), (char *)&boardCfg, sizeof (boardCfg));
*/
}


/* Set Promisc mode
 * Daniel+ 2003.8.4
 */
void EnablePromisc(int unit, unsigned long rxmode)
{
	int port = sw_context->vlanGrp[unit];

//	if (rxmode == IF_PROMISC_MODE)
		ADM5120_SW_REG(CPUp_conf_REG) &= ~((port & 0x3f) << 9);
//	else
//		ADM5120_SW_REG(CPUp_conf_REG) |= (port & 0x3f) << 9;
}


/*
 * Enable VLAN Group
 */
void EnableVlanGroup(int unit, unsigned long vlanGrp)
{
	int i;
	int vlanId = 0x01 << unit;
	
	vlanGrp &= SW_DISABLE_PORT_MASK;
	ADM5120_SW_REG(Port_conf0_REG) &= ~vlanGrp; 

	/* Mark the enabled ports */
	for (i = 0; i < NUM_IF5120_PORTS; i++)
	{
		if (vlanGrp & (0x01 << i)) 
		{
			sw_context->port[i].status = PORT_ENABLED;
			sw_context->port[i].vlanId = vlanId;
			sw_context->port[i].ifUnit = unit;
		}
	}
}


/*
 * DisableVlanGroup
 */
void DisableVlanGroup(int unit, unsigned long vlanGrp)
{
	int i;
	unsigned long reg;
	
	vlanGrp &= SW_DISABLE_PORT_MASK;
	
	reg = ADM5120_SW_REG(Port_conf0_REG) | vlanGrp; 
	ADM5120_SW_REG(Port_conf0_REG) = reg;

	/* Mark the disabled ports */
	for (i = 0; i < NUM_IF5120_PORTS; i++)
		if (vlanGrp & (0x01<<i)) 
			sw_context->port[i].status = PORT_DISABLED;
}


/*
 * Program VLAN mac
 */
int ProgramVlanMac(int vlan, char *Mac, int clear)
{
	unsigned long Reg0, Reg1;
	
	if (vlan < 0 || vlan >= MAX_VLAN_GROUP) 
		return -1;
	
	Reg0 = (((unsigned long)Mac[1] << 24) + ((unsigned long)Mac[0] << 16)) | (vlan << SW_MAC_VLANID_SHIFT)
			| SW_MAC_WRITE | SW_MAC_VLANID_EN;

	if (!clear)
		Reg0 |= SW_MAC_AGE_VALID;
	
	Reg1 = ((unsigned long)Mac[5] << 24) + ((unsigned long)Mac[4] << 16) + 
		((unsigned long)Mac[3] << 8) + Mac[2];

	ADM5120_SW_REG(MAC_wt1_REG) = Reg1;
	ADM5120_SW_REG(MAC_wt0_REG) = Reg0;

	while (!(ADM5120_SW_REG(MAC_wt0_REG) & SW_MAC_WRITE_DONE));

	return 0;
}


/*
 * Setup VLAN
 */
static int SetupVLAN(int unit, unsigned long portmask)
{
	unsigned long reg, shiftcnt;
	
	if (unit < 0 || unit > 6) 
		return -1;
	
	if (unit <= 3)
	{
		shiftcnt = 8 * unit;
		reg = ADM5120_SW_REG(VLAN_G1_REG) & ~(VLAN_PORT_MASK << shiftcnt);
		reg |= (portmask & VLAN_PORT_MASK) << shiftcnt;
		ADM5120_SW_REG(VLAN_G1_REG) = reg;
	}
	else
	{
		shiftcnt = 8 * (unit - 4);
		reg = ADM5120_SW_REG(VLAN_G2_REG) & ~(VLAN_PORT_MASK << shiftcnt);
		reg |= (portmask & VLAN_PORT_MASK) << shiftcnt;
		ADM5120_SW_REG(VLAN_G2_REG) = reg;
	}

	return 0;
}


/*
 * ProcessRxInt
 */
void ProcessRxInt(PRX_ENG_T rxEng)
{
	struct net_device *rdev;
    PSW_PRIV_T priv = 0;
	PRXDESC_T rxDesc = 0;
	PRX_DRV_DESC_T drvDesc = 0;
	int unit;
	int srcPort;
	int idx;
	int len;

	idx = rxEng->idx;
	rxDesc = &rxEng->hwDesc[idx];
	while (!(rxDesc->buf1cntl & OWN_BIT))
	{ 
		drvDesc = &rxEng->drvDesc[idx];
		if (drvDesc->skb == 0)
			goto get_desc;

		srcPort = (rxDesc->status & RX_SRC_PORT_MASK) >> RX_SRC_PORT_SHIFT;
    	unit = sw_context->port[srcPort].ifUnit;
		rdev = &am5120sw_devs[unit];
		priv = (PSW_PRIV_T)rdev->priv;

		len = ((rxDesc->status & PKT_LEN_MASK) >> PKT_LEN_SHIFT) - ETH_CRC_LEN;
		if (len <= 0 || (rxDesc->status & RX_PKT_IPSUM_ERR))
		{
			priv->stats.rx_errors++;
			priv->stats.rx_length_errors++;
			goto next;
		}
		
		skb_put(drvDesc->skb, len);
			
		/* Write metadata, and then pass to the receive level */
		drvDesc->skb->dev = rdev;
		drvDesc->skb->protocol = eth_type_trans(drvDesc->skb, rdev);
		drvDesc->skb->ip_summed = CHECKSUM_UNNECESSARY; /* don't check it */

#ifdef GPIO_MEASURE
		GPIO_TOGGLE(1);
#endif
		if (netif_rx(drvDesc->skb) == NET_RX_DROP)
		{
			priv->stats.rx_dropped++;
		}
		else
		{
			rdev->last_rx = jiffies;
			priv->stats.rx_packets++;
			priv->stats.rx_bytes += drvDesc->skb->len;
		}

#ifdef GPIO_MEASURE
		GPIO_TOGGLE(1);
#endif

get_desc:

#ifdef GPIO_MEASURE
		GPIO_TOGGLE(1);
#endif
		drvDesc->skb = dev_alloc_skb(DEF_RX_BUF_SIZE+16);

#ifdef GPIO_MEASURE
		GPIO_TOGGLE(1);
#endif

		if (drvDesc->skb) 
		{
			skb_reserve(drvDesc->skb, 2); /* align IP on 16B boundary */
next:
			rxDesc->buf2cntl = rxDesc->status = 0;
			rxDesc->buf1len = DEF_RX_BUF_SIZE;
			rxDesc->buf1cntl = (rxDesc->buf1cntl & END_OF_RING) | OWN_BIT
							| (((unsigned long)drvDesc->skb->data)& BUF_ADDR_MASK);
		}
		else
		{
			printk("Init rx skb : low on mem\n");
		}
			
		if (++idx == rxEng->numDesc)
		{
			idx = 0;
		}

		rxDesc = &rxEng->hwDesc[idx];
	}

	rxEng->idx = idx;

    return;
}


/*
 * ProcessTxInt
 */
void ProcessTxInt(PTX_ENG_T txEng)
{
	PTX_DRV_DESC_T drvDesc;
	int idx;

	idx = txEng->idxTail;

	while (!(txEng->hwDesc[idx].buf1cntl & OWN_BIT) && (idx != txEng->idxHead))
	{
		drvDesc = &txEng->drvDesc[idx];
#ifdef GPIO_MEASURE
	GPIO_TOGGLE(3);
#endif
		dev_kfree_skb_irq(drvDesc->skb);

#ifdef GPIO_MEASURE
	GPIO_TOGGLE(3);
#endif	
		drvDesc->skb = 0;

		if (++idx == txEng->numDesc)
			idx = 0;
	}

	txEng->idxTail = idx;
}


/*
 * swdrv_ProcessInt
 */
void swdrv_ProcessInt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned long intReg;
	
	spin_lock(&sw_context->lock);

#ifdef GPIO_MEASURE
	GPIO_TOGGLE(0);
#endif

	/* disable switch interrupt */
	ADM5120_SW_REG(SW_Int_mask_REG) |= sw_context->intMask;

	/* recording the current interrupts in the context. */	
	intReg = ADM5120_SW_REG(SW_Int_st_REG);
	
	/* acknowledge  all interrupts */
	ADM5120_SW_REG(SW_Int_st_REG ) = intReg;
	
	/* receive one high priority packet to cpu */
	if (intReg & RX_H_INT)	
		ProcessRxInt(&sw_context->rxH);
		
	/* receive one nromal priority packet to cpu */
	if (intReg & RX_L_INT)	
		ProcessRxInt(&sw_context->rxL);

	/* transmit one high priority packet to cpu */
	if (intReg & TX_H_INT)	
		ProcessTxInt(&sw_context->txH);

	/* transmit one nromal priority packet to cpu */
	if (intReg & TX_L_INT)	
		ProcessTxInt(&sw_context->txL);

	ADM5120_SW_REG(SW_Int_mask_REG) &= ~sw_context->intMask;

#ifdef GPIO_MEASURE
	GPIO_TOGGLE(0);
#endif

	spin_unlock(&sw_context->lock);
}


/*
 * swdrv_HwTx
 */
int swdrv_HwTx(struct sk_buff *skb, PSW_PRIV_T priv)
{
	PTX_ENG_T txEng;
	PTXDESC_T hdesc;
	PTX_DRV_DESC_T drvDesc;
	unsigned long csum = 0;
	int len;

	/* get tx engine */
	if (priv->priority == IF_TX_PRIORITY_H)
		txEng = &sw_context->txH;
	else
		txEng = &sw_context->txL;
	
	/* get check sum flag */
	if (priv->csum_flags)
		csum = TX_ADD_IPSUM;

	/* get hardware descriptor */
	hdesc = &txEng->hwDesc[txEng->idxHead];
	if (hdesc->buf1cntl & OWN_BIT)
	{
		dev_kfree_skb(skb);
		priv->stats.tx_dropped++;
		return 0;
	}

	drvDesc = &txEng->drvDesc[txEng->idxHead];
	drvDesc->skb = skb;
	
	hdesc->buf1cntl = (hdesc->buf1cntl & END_OF_RING) | 
			 	(((unsigned long)skb->data & BUF_ADDR_MASK) | OWN_BIT);
		
	len = skb->len < ETH_ZLEN ? ETH_ZLEN : skb->len;
	hdesc->pktcntl = (len << PKT_LEN_SHIFT) | priv->vlanId | csum;
	hdesc->buf1len = len;

	priv->stats.tx_packets++;
	priv->stats.tx_bytes += len;

	if (++txEng->idxHead >= txEng->numDesc)
		txEng->idxHead = 0;

	ADM5120_SW_REG(Send_trig_REG) = txEng->txTrig;

	return 0;
}


/*
 * swdrv_IfOpen
 */
int swdrv_IfOpen(PSW_PRIV_T priv, char *mac)
{
	int unit = priv->unit;

	spin_lock(&sw_context->lock);

	/* setup vlan reg */
	SetupVLAN(unit, sw_context->vlanGrp[unit]);
	
	/* program vlan mac */
	ProgramVlanMac(unit, mac, SW_MAC_AGE_VALID);

	/* Enable vlan Group */
	EnableVlanGroup(unit, sw_context->vlanGrp[unit]);

	priv->vlanId = 1 << unit;
	
	if (sw_context->actIfCnt == 0)
	{
		/* Enable interrupt */
		ADM5120_SW_REG(SW_Int_mask_REG) = ~sw_context->intMask; 
	}
	sw_context->actIfCnt++;

	spin_unlock(&sw_context->lock);

	return 0;
}


/*
 * swdrv_IfClose
 */
int swdrv_IfClose(PSW_PRIV_T priv)
{
	int unit = priv->unit;

	spin_lock(&sw_context->lock);

	/* Enable vlan Group */
	DisableVlanGroup(unit, sw_context->vlanGrp[unit]);

	if (--sw_context->actIfCnt <= 0)
	{
		ADM5120_SW_REG(SW_Int_mask_REG) = SWITCH_INT_MASK;
	}

	spin_unlock(&sw_context->lock);

	return 0;
}


/*
 * switch engine init
 */
void engine_init(void)
{
	int i;

	/* disable cpu port, CRC padding from cpu and 
	   no send unknown packet from port0 to port5 to cpu */
	ADM5120_SW_REG(CPUp_conf_REG) = DEF_CPUPORT_CFG	;

	/* Disable all port, enable BP & MC */
	ADM5120_SW_REG(Port_conf0_REG) = DEF_PORTS_CFG;

   	/* Wait untile switch enter idle state */
  	for (i=0; i<500000; i++);
  	
	/* Put Phys to normal mode */
	ADM5120_SW_REG(PHY_cntl2_REG) |= SW_PHY_AN_MASK | SW_PHY_NORMAL_MASK;

	/* Disable Switch Interrupts */
	ADM5120_SW_REG(SW_Int_mask_REG) = SWITCH_INT_MASK;

	/* Clear the Interrupt status */
	ADM5120_SW_REG(SW_Int_st_REG) = SWITCH_INT_MASK;

	/* Initialize the adm5120 Desc */
	ADM5120_SW_REG(Send_HBaddr_REG) = (unsigned long)sw_context->txH.hwDesc;
	ADM5120_SW_REG(Send_LBaddr_REG) = (unsigned long)sw_context->txL.hwDesc;
	ADM5120_SW_REG(Recv_HBaddr_REG) = (unsigned long)sw_context->rxH.hwDesc;
	ADM5120_SW_REG(Recv_LBaddr_REG) = (unsigned long)sw_context->rxL.hwDesc;

	/* Clear all vlan setting */
	ADM5120_SW_REG(VLAN_G1_REG) = 0;
	ADM5120_SW_REG(VLAN_G2_REG) = 0;

	/* Update link status */
	sw_context->linkStatus = 0;
		
	sw_context->intMask = RX_H_INT | RX_L_INT | TX_H_INT | TX_L_INT;

	ADM5120_SW_REG(CPUp_conf_REG) &= ~SW_CPU_PORT_DISABLE;
}


/* 
 * InitTxDesc
 */															
static void InitTxDesc(PTX_ENG_T pTxEng)
{
	int num = pTxEng->numDesc;

	pTxEng->hwDesc[--num].buf1cntl |= END_OF_RING;
	pTxEng->idxHead = pTxEng->idxTail = 0;
}


/* 
 * InitRxDesc
 */									
static void InitRxDesc(PRX_ENG_T pRxEng)
{
	PRX_DRV_DESC_T drvDesc = pRxEng->drvDesc;
	int i;

	for (i = 0; i < pRxEng->numDesc; i++, drvDesc++)
	{
		drvDesc->skb = dev_alloc_skb(DEF_RX_BUF_SIZE+16);
		if (!drvDesc->skb) 
		{
			printk("Init rx skb : low on mem\n");
			return;
		}
		skb_reserve(drvDesc->skb, 2); /* align IP on 16B boundary */
	}

	drvDesc = pRxEng->drvDesc;

	for (i = 0; i < pRxEng->numDesc; i++)
	{
		pRxEng->hwDesc[i].buf2cntl = pRxEng->hwDesc[i].status = 0;
		pRxEng->hwDesc[i].buf1len = DEF_RX_BUF_SIZE;
		pRxEng->hwDesc[i].buf1cntl = 
			((unsigned long)pRxEng->drvDesc[i].skb->data & 
			BUF_ADDR_MASK) | OWN_BIT;
	}

	pRxEng->hwDesc[--i].buf1cntl |= END_OF_RING;
	pRxEng->idx = 0;
}


/*
 * Switch driver init
 */
int swdrv_init(void)
{
	int i;
	char *rxBufPool;

	/* Allocate the switch driver context */
	if ((sw_context = (PSW_CONTEXT_T)kmalloc(sizeof(SW_CONTEXT_T), GFP_KERNEL)) == NULL) 
		return (-1);

	memset((char *)sw_context, 0, sizeof(SW_CONTEXT_T));

	/* Allocate the Tx/Rx hardware descriptor pool */
	i = HWDESC_ALIGN + sizeof(RXDESC_T) * (NUM_RX_H_DESC + NUM_RX_L_DESC)
		+ sizeof(TXDESC_T) * (NUM_TX_H_DESC + NUM_TX_L_DESC);

	if ((sw_context->hwDescPool = (char *)kmalloc(i, GFP_KERNEL)) == NULL) 
		goto ErrRes;
	
	memset(sw_context->hwDescPool, 0, i); 

	/* Allocate the tx driver descriptor */
	i = sizeof(TX_DRV_DESC_T) * (NUM_TX_H_DESC + NUM_TX_L_DESC);

	if ((sw_context->txDrvDescPool = (char *)kmalloc(i, GFP_KERNEL)) == NULL) 
		goto ErrRes;
	
	memset(sw_context->txDrvDescPool, 0, i); 

	/* Allocate the rx driver descriptor */
	i = sizeof(RX_DRV_DESC_T) * (NUM_RX_H_DESC + NUM_RX_L_DESC);

	if ((sw_context->rxDrvDescPool = (char *)kmalloc(i, GFP_KERNEL)) == NULL) 
		goto ErrRes;
	
	memset(sw_context->rxDrvDescPool, 0, i); 

	/* 
	 *!! Note: The Hardware Tx/Rx descriptors should be allocated at non-cached memory and 
	 * 			aligned on 16 bytes boundry!!!!!!
	 */

	/* assign hardware descriptor to txH pool */
	if (((unsigned long)sw_context->hwDescPool) & (HWDESC_ALIGN - 1))
	{
		sw_context->txH.hwDesc = (PTXDESC_T)MIPS_KSEG1A((unsigned long)(sw_context->hwDescPool 
								+ HWDESC_ALIGN - 1) & ~(HWDESC_ALIGN - 1));
	}
	else
		sw_context->txH.hwDesc = (PTXDESC_T)MIPS_KSEG1A(sw_context->hwDescPool);
	
	sw_context->txH.numDesc = NUM_TX_H_DESC;
	sw_context->txH.txTrig = SEND_TRIG_HIGH;
	InitTxDesc(&sw_context->txH);
	sw_context->txH.drvDesc = (PTX_DRV_DESC_T)sw_context->txDrvDescPool;

	/* assign hardware descriptor to txL pool */
	sw_context->txL.hwDesc = &sw_context->txH.hwDesc[NUM_TX_H_DESC];
	sw_context->txL.numDesc = NUM_TX_L_DESC;
	sw_context->txL.txTrig = SEND_TRIG_LOW;
	InitTxDesc(&sw_context->txL);
	sw_context->txL.drvDesc = (PTX_DRV_DESC_T)(sw_context->txDrvDescPool + 
		(sizeof(TX_DRV_DESC_T) * NUM_TX_H_DESC));

	/* assign hardware descriptor to rxH pool */
	sw_context->rxH.hwDesc = (PRXDESC_T)&sw_context->txL.hwDesc[NUM_TX_L_DESC];
	sw_context->rxH.numDesc = NUM_RX_H_DESC;
	sw_context->rxH.drvDesc = (PRX_DRV_DESC_T)sw_context->rxDrvDescPool;
	InitRxDesc(&sw_context->rxH);

	/* assign hardware descriptor to rxL pool */
	sw_context->rxL.hwDesc = &sw_context->rxH.hwDesc[NUM_RX_H_DESC];
	sw_context->rxL.numDesc = NUM_RX_L_DESC;
	rxBufPool += NUM_RX_H_DESC * RX_BUF_SIZE;
	sw_context->rxL.drvDesc = (PRX_DRV_DESC_T)(sw_context->rxDrvDescPool + 
							(sizeof(RX_DRV_DESC_T) * NUM_RX_H_DESC));
	InitRxDesc(&sw_context->rxL);

	for (i = 0; i < MAX_VLAN_GROUP; i++)
		sw_context->vlanGrp[i] = vlan[i];

	/* init switch engine */
	engine_init();

	spin_lock_init(&sw_context->lock);

#ifdef GPIO_MEASURE
	GPIO_MEASURE_INIT();

	GPIO_SET_LOW(0);
	GPIO_SET_LOW(1);
	GPIO_SET_LOW(2);
	GPIO_SET_LOW(3);
#endif

	return (0);

ErrRes:
	/* Free all resources */ 
	if (sw_context->hwDescPool != NULL)
		kfree(sw_context->hwDescPool);

	if (sw_context->txDrvDescPool != NULL)
		kfree(sw_context->txDrvDescPool);

	if (sw_context->rxDrvDescPool != NULL)
		kfree(sw_context->rxDrvDescPool);

	kfree(sw_context);

	return (-1);
}
