/*
 *      Routines to handle MIB operation
 *
 *      Authors: David Hsu	<davidhsu@realtek.com.tw>
 *
 *      $Id: apmib.c,v 1.2 2004/02/04 08:32:59 erwin Exp $
 *
 */

// include file
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "apmib.h"
#include "mibtbl.h"

// MAC address filtering
typedef struct _filter {
	struct _filter *prev, *next;
	char val[1];
} FILTER_T, *FILTER_Tp;

typedef struct _linkChain {
	FILTER_Tp pUsedList, pFreeList;
	int size, num, usedNum, compareLen;
	char *buf;
} LINKCHAIN_T, *LINKCHAIN_Tp;


// macro to remove a link list entry
#define REMOVE_LINK_LIST(entry) { \
	if ( entry ) { \
		if ( entry->prev ) \
			entry->prev->next = entry->next; \
		if ( entry->next ) \
			entry->next->prev = entry->prev; \
	} \
}

// macro to add a link list entry
#define ADD_LINK_LIST(list, entry) { \
	if ( list == NULL ) { \
		list = entry; \
		list->prev = list->next = entry; \
	} \
	else { \
		entry->prev = list; \
		entry->next = list->next; \
		list->next = entry; \
		entry->next->prev = entry; \
	} \
}

// local routine declaration
static int flash_read(char *buf, int offset, int len);
static int flash_write(char *buf, int offset, int len);
static int init_linkchain(LINKCHAIN_Tp pLinkChain, int size, int num);
static int add_linkchain(LINKCHAIN_Tp pLinkChain, char *val);
static int delete_linkchain(LINKCHAIN_Tp pLinkChain, char *val);
static int edit_linkchain(LINKCHAIN_Tp pLinkChain, char *val);
static void delete_all_linkchain(LINKCHAIN_Tp pLinkChain);
static int get_linkchain(LINKCHAIN_Tp pLinkChain, char *val, int index);

// local & global variable declaration
APMIB_Tp pMib=NULL;
APMIB_Tp pMibDef;
PARAM_HEADER_T hsHeader, dsHeader, csHeader;
HW_SETTING_Tp pHwSetting;

static LINKCHAIN_T wlanMacChain;

#ifdef HOME_GATEWAY
static LINKCHAIN_T portFwChain, ipFilterChain, portFilterChain, macFilterChain, triggerPortChain, virtSvChain, dmzChain, URLBChain, ACPCChain, sroutChain, rserCltChain, rserUsrChain;
#else
static LINKCHAIN_T rserCltChain, rserUsrChain;
#endif

void dump_mem(void *ptr, int size)
{
    int i;
    printf("\n");
    for(i=0; i < size; i++)
	{
        printf("%4.2x", (unsigned )(((unsigned char*)ptr)[i]));
		if(i%16 == 0)
			printf("\n");
	}
	printf("\n");
}

////////////////////////////////////////////////////////////////////////////////
char *apmib_hwconf(void)
{
	int ver;
	char *buff;

	// Read hw setting
	if ( flash_read((char *)&hsHeader, HW_SETTING_OFFSET, sizeof(hsHeader))==0 ) {
		printf("Read hw setting header failed!, Offset=%x\n", HW_SETTING_OFFSET);
		return NULL;
	}

//	dump_mem(&hsHeader, 20);
	
	if ( sscanf(&hsHeader.signature[TAG_LEN], "%02d", &ver) != 1)
		ver = -1;

	if ( memcmp(hsHeader.signature, HW_SETTING_HEADER_TAG, TAG_LEN) || // invalid signatur
		(ver != HW_SETTING_VER) || // version is less than current
		(hsHeader.len < (sizeof(HW_SETTING_T)+1)) ) { // length is less than current
		printf("Invalid hw setting signature or version number [sig=%c%c, ver=%d, len=%d]!\n", hsHeader.signature[0],
			hsHeader.signature[1], ver, hsHeader.len);
		return NULL;
	}
//	if (ver > HW_SETTING_VER)
//		printf("HW setting version is greater than current [f:%d, c:%d]!\n", ver, HW_SETTING_VER);

	buff = calloc(1, hsHeader.len);
	if ( buff == 0 ) {
		printf("Allocate buffer failed!\n");
		return NULL;
	}
	if ( flash_read(buff, HW_SETTING_OFFSET+sizeof(hsHeader), hsHeader.len)==0 ) {
		printf("Read hw setting failed!\n");
		free(buff);
		return NULL;
	}
	if ( !CHECKSUM_OK(buff, hsHeader.len) ) {
		printf("Invalid checksum of hw setting!\n");
		free(buff);
		return NULL;
	}
	return buff;
}

////////////////////////////////////////////////////////////////////////////////
char *apmib_dsconf(void)
{
	int ver;
	char *buff;

	// Read default s/w mib
	if ( flash_read((char *)&dsHeader, DEFAULT_SETTING_OFFSET, sizeof(dsHeader))==0 ) {
		printf("Read default setting header failed!\n");
		return NULL;
	}

	if ( sscanf(&dsHeader.signature[TAG_LEN], "%02d", &ver) != 1)
		ver = -1;

	if ( memcmp(dsHeader.signature, DEFAULT_SETTING_HEADER_TAG, TAG_LEN) || // invalid signatur
		(ver != DEFAULT_SETTING_VER) || // version is less than current
		(dsHeader.len < (sizeof(APMIB_T))) ) { // length is less than current
		printf("Invalid default setting signature or version number [sig=%c%c, ver=%d, len=%d]!\n",
			dsHeader.signature[0], dsHeader.signature[1], ver, dsHeader.len);
		return NULL;
	}
//	if (ver > DEFAULT_SETTING_VER)
//		printf("Default setting version is greater than current [f:%d, c:%d]!\n", ver, DEFAULT_SETTING_VER);

	buff = calloc(1, dsHeader.len);
	if ( buff == 0 ) {
		printf("Allocate buffer failed!\n");
		return NULL;
	}

	if ( flash_read(buff, DEFAULT_SETTING_OFFSET+sizeof(dsHeader), dsHeader.len)==0 ) {
		printf("Read default setting failed!\n");
		free(buff);
		return NULL;
	}

	if ( !CHECKSUM_OK(buff, dsHeader.len) ) {
		printf("Invalid checksum of current setting!\n");
		free(buff);
		return NULL;
	}
	return buff;
}

////////////////////////////////////////////////////////////////////////////////
char *apmib_csconf(void)
{
	int ver;
	char *buff;

	// Read current s/w mib
	if ( flash_read((char *)&csHeader, CURRENT_SETTING_OFFSET, sizeof(csHeader))==0 ) {
//		printf("Read current setting header failed!\n");
		return NULL;
	}

	if ( sscanf(&csHeader.signature[TAG_LEN], "%02d", &ver) != 1)
		ver = -1;

	if ( memcmp(csHeader.signature, CURRENT_SETTING_HEADER_TAG, TAG_LEN) || // invalid signatur
		(ver != CURRENT_SETTING_VER) || // version is less than current
			(csHeader.len < (sizeof(APMIB_T)+1)) ) { // length is less than current
//		printf("Invalid current setting signature or version number [sig=%c%c, ver=%d, len=%d]!\n",
//			csHeader.signature[0], csHeader.signature[1], ver, csHeader.len);
		return NULL;
	}

//	if (ver > CURRENT_SETTING_VER)
//		printf("Current setting version is greater than current [f:%d, c:%d]!\n", ver, CURRENT_SETTING_VER);

	buff = calloc(1, csHeader.len);
	if ( buff == 0 ) {
//		printf("Allocate buffer failed!\n");
		return NULL;
	}

	if ( flash_read(buff, CURRENT_SETTING_OFFSET+sizeof(csHeader), csHeader.len)==0 ) {
//		printf("Read current setting failed!\n");
		free(buff);
		return NULL;
	}

	if ( !CHECKSUM_OK(buff, csHeader.len) ) {
//		printf("Invalid checksum of current setting!\n");
		free(buff);
		return NULL;
	}

	return buff;
}


////////////////////////////////////////////////////////////////////////////////
int apmib_init(void)
{
	int i;
	char *buff;

	if ( pMib != NULL )	// has been initialized
		return 1;

	if ((buff=apmib_hwconf()) == NULL)
		return 0;
	pHwSetting = (HW_SETTING_Tp)buff;

	if ((buff=apmib_dsconf()) == NULL) {
		free(pHwSetting);
		return 0;
	}
	pMibDef = (APMIB_Tp)buff;

	if ((buff=apmib_csconf()) == NULL) {
		free(pHwSetting);
		free(pMibDef);
		return 0;
	}
	pMib = (APMIB_Tp)buff;

#if 0
	// Copy mac address from hw setting if no clone mac address is set
	if ( !memcmp(pMib->elanMacAddr, "\x0\x0\x0\x0\x0\x0", 6) )
		memcpy(pMib->elanMacAddr, pHwSetting->nic0Addr, 6);

	if ( !memcmp(pMib->wlanMacAddr, "\x0\x0\x0\x0\x0\x0", 6) )
		memcpy(pMib->wlanMacAddr, pHwSetting->wlanAddr, 6);

#ifdef HOME_GATEWAY
	if ( !memcmp(pMib->wanMacAddr, "\x0\x0\x0\x0\x0\x0", 6) )
		memcpy(pMib->wanMacAddr, pHwSetting->nic1Addr, 6);
#endif
#endif

	// initialize MAC access control list
	if ( !init_linkchain(&wlanMacChain, sizeof(MACFILTER_T), MAX_WLAN_AC_NUM)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}

	for (i=0; i<pMib->acNum; i++) {
		if ( !add_linkchain(&wlanMacChain, (char *)&pMib->acAddrArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	wlanMacChain.compareLen = sizeof(MACFILTER_T) - COMMENT_LEN;

#ifdef HOME_GATEWAY
///////////////////////////////////////////////////////////////////////////////////////////
	// initialize port forwarding table
	if ( !init_linkchain(&virtSvChain, sizeof(PORTFW_T), MAX_FILTER_NUM)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->vserNum; i++) {
		if ( !add_linkchain(&virtSvChain, (char *)&pMib->vserArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	virtSvChain.compareLen = sizeof(PORTFW_T) - COMMENT_LEN;
///////////////////////////////////////////////////////////////////////////////////////////
	// initialize port forwarding table
	if ( !init_linkchain(&portFwChain, sizeof(PORTFW_T), MAX_FILTER_NUM)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->portFwNum; i++) {
		if ( !add_linkchain(&portFwChain, (char *)&pMib->portFwArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	portFwChain.compareLen = sizeof(PORTFW_T) - COMMENT_LEN;
/////////////////////////////////////////////////////////////////////////////////////////////
	// initialize DMZ table
	if ( !init_linkchain(&dmzChain, sizeof(DMZ_T), MAX_FILTER_NUM)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->DMZNum; i++) {
		if ( !add_linkchain(&dmzChain, (char *)&pMib->DMZArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	dmzChain.compareLen = sizeof(DMZ_T);
	// initialize Static Routing table
	if ( !init_linkchain(&sroutChain, sizeof(SROUT_T), MAX_FILTER_NUM)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->sroutNum; i++) {
		if ( !add_linkchain(&sroutChain, (char *)&pMib->sroutArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	sroutChain.compareLen = sizeof(SROUT_T);
	// initialize URLB table
	if ( !init_linkchain(&URLBChain, sizeof(URLB_T), MAX_FILTER_NUM)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->URLBNum; i++) {
		if ( !add_linkchain(&URLBChain, (char *)&pMib->URLBArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	URLBChain.compareLen = sizeof(URLB_T) - COMMENT_LEN;
	// initialize ACPC table
	if ( !init_linkchain(&ACPCChain, sizeof(ACPC_T), MAX_FILTER_NUM)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->ACPCNum; i++) {
		if ( !add_linkchain(&ACPCChain, (char *)&pMib->ACPCArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	ACPCChain.compareLen = sizeof(ACPC_T) - COMMENT_LEN;
/////////////////////////////////////////////////////////////////////////////////////////////
	// initialize ip-filter table
//	if ( !init_linkchain(&ipFilterChain, sizeof(IPFILTER_T), MAX_FILTER_NUM)) {
	if ( !init_linkchain(&ipFilterChain, sizeof(IPFILTER_T), 1)) {//Erwin Modify 02.09.2003
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->ipFilterNum; i++) {
		if ( !add_linkchain(&ipFilterChain, (char *)&pMib->ipFilterArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	ipFilterChain.compareLen = sizeof(IPFILTER_T) - COMMENT_LEN;
	// initialize port-filter table
//	if ( !init_linkchain(&portFilterChain, sizeof(PORTFILTER_T), MAX_FILTER_NUM)) {
	if ( !init_linkchain(&portFilterChain, sizeof(PORTFILTER_T), 1)) {//Erwin Modify 09.24
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->portFilterNum; i++) {
		if ( !add_linkchain(&portFilterChain, (char *)&pMib->vserArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	portFilterChain.compareLen = sizeof(PORTFILTER_T) - COMMENT_LEN;

	// initialize mac-filter table
	if ( !init_linkchain(&macFilterChain, sizeof(MACFILTER_T), MAX_FILTER_NUM)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->macFilterNum; i++) {
		if ( !add_linkchain(&macFilterChain, (char *)&pMib->macFilterArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	macFilterChain.compareLen = sizeof(MACFILTER_T) - COMMENT_LEN;

	// initialize trigger-port table
	if ( !init_linkchain(&triggerPortChain, sizeof(TRIGGERPORT_T), MAX_FILTER_NUM)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->triggerPortNum; i++) {
		if ( !add_linkchain(&triggerPortChain, (char *)&pMib->triggerPortArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	triggerPortChain.compareLen = 5;	// length of trigger port range + proto type
/////////////////////////////////////////////////////////////////////////////////////////////
#else	// Radius server
	// initialize Radius client table
	if ( !init_linkchain(&rserCltChain, sizeof(RSER_CLT_T), 16)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->rserCltNum; i++) {
		if ( !add_linkchain(&rserCltChain, (char *)&pMib->rserCltArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	rserCltChain.compareLen = sizeof(RSER_CLT_T);
	// initialize Radius user table
	if ( !init_linkchain(&rserUsrChain, sizeof(RSER_USR_T), 96)) {
		free(pMib);
		free(pMibDef);
		free(pHwSetting);
		return 0;
	}
	for (i=0; i<pMib->rserUsrNum; i++) {
		if ( !add_linkchain(&rserUsrChain, (char *)&pMib->rserUsrArray[i]) ) {
			free(pMib);
			free(pMibDef);
			free(pHwSetting);
			return 0;
		}
	}
	rserUsrChain.compareLen = sizeof(RSER_USR_T);
#endif

	return 1;
}


///////////////////////////////////////////////////////////////////////////////
int apmib_reinit(void)
{
	if (pMib == NULL)	// has not been initialized
		return 0;

	free(pMib);
	free(pMibDef);
	free(pHwSetting);
	free(wlanMacChain.buf);

#ifdef HOME_GATEWAY
	free(virtSvChain.buf);////////////////////////////////////////////////////////
	free(dmzChain.buf);///////////////////////////////////////////////////
	free(URLBChain.buf);///////////////////////////////////////////////////
	free(ACPCChain.buf);///////////////////////////////////////////////////
	free(sroutChain.buf);///////////////////////////////////////////////////
	free(portFwChain.buf);
	free(ipFilterChain.buf);
	free(portFilterChain.buf);
	free(macFilterChain.buf);
	free(triggerPortChain.buf);
#else
	free(rserUsrChain.buf);
	free(rserCltChain.buf);
#endif

	pMib=NULL;
	pMibDef=NULL;

	return apmib_init();
}


////////////////////////////////////////////////////////////////////////////////
int apmib_get(int id, void *value)
{
	int i, index;
	void *pMibTbl;
	mib_table_entry_T *pTbl;
	unsigned char ch;
	unsigned short wd;
	unsigned long dwd;


	// search current setting mib table
	for (i=0; mib_table[i].id; i++) {
//	printf("\n%s",mib_table[i].name);
//	printf("\t\t\toffset=%d",mib_table[i].offset);
//	printf("\tsize=%d",mib_table[i].size);
		if ( mib_table[i].id == id )
			break;
	}
	if ( mib_table[i].id != 0 ) {
		pMibTbl = (void *)pMib;
		pTbl = mib_table;
	}
	else {
		// if not found, search hw setting table
		for (i=0; hwmib_table[i].id; i++) {
			if ( hwmib_table[i].id == id )
				break;
		}
		if (hwmib_table[i].id == 0) // not found
			return 0;
		pMibTbl = (void *)pHwSetting;
		pTbl = hwmib_table;
	}

	switch (pTbl[i].type) {
	case BYTE_T:
//		*((int *)value) =(int)(*((unsigned char *)(((long)pMibTbl) + pTbl[i].offset)));
		memcpy((char *)&ch, ((char *)pMibTbl) + pTbl[i].offset, 1);
		*((int *)value) = (int)ch;
		break;

	case WORD_T:
//		*((int *)value) =(int)(*((unsigned short *)(((long)pMibTbl) + pTbl[i].offset)));
		memcpy((char *)&wd, ((char *)pMibTbl) + pTbl[i].offset, 2);
		*((int *)value) = (int)wd;
		break;

	case STRING_T:
		strcpy( (char *)value, (const char *)(((long)pMibTbl) + pTbl[i].offset) );
		break;

	case BYTE5_T:
		memcpy( (unsigned char *)value, (unsigned char *)(((long)pMibTbl) + pTbl[i].offset), 5);
		break;

	case BYTE6_T:
		memcpy( (unsigned char *)value, (unsigned char *)(((long)pMibTbl) + pTbl[i].offset), 6);
		break;

	case BYTE13_T:
		memcpy( (unsigned char *)value, (unsigned char *)(((long)pMibTbl) + pTbl[i].offset), 13);
		break;
	
	case BYTE16_T:
		memcpy( (unsigned char *)value, (unsigned char *)(((long)pMibTbl) + pTbl[i].offset), 16);
		break;
	
	case DWORD_T:
		memcpy((char *)&dwd, ((char *)pMibTbl) + pTbl[i].offset, 4);
		*((int *)value) = (int)dwd;
		break;

	case INDEX_T:
		memcpy((char *)&dwd, ((char *)pMibTbl) + pTbl[i].offset, 4);
		*((int *)value) = (int)dwd;
		break;

	case INDEXDOS_T:
		memcpy((char *)&dwd, ((char *)pMibTbl) + pTbl[i].offset, 4);
		*((int *)value) = (int)dwd;
		break;

#ifdef WLAN_WPA
#if 0
	case BYTE32_T:
		memcpy( (unsigned char *)value, (unsigned char *)(((long)pMibTbl) + pTbl[i].offset), 32);
		break;
#endif		
#endif

	case BYTE_ARRAY_T:
		memcpy( (unsigned char *)value, (unsigned char *)(((long)pMibTbl) + pTbl[i].offset), pTbl[i].size);
		break;

	case IA_T:
		memcpy( (unsigned char *)value, (unsigned char *)(((long)pMibTbl) + pTbl[i].offset), 4);
		break;

	case WLAC_ARRAY_T:
		index = (int)( *((unsigned char *)value));
		return get_linkchain(&wlanMacChain, (char *)value, index );

#ifdef HOME_GATEWAY
/////////////////////////////////////////////////////////////////////////////////////////	
	case VSER_ARRAY_T:
                index = (int)( *((unsigned char *)value));
 		return get_linkchain(&virtSvChain, (char *)value, index );

/////////////////////////////////////////////////////////////////////////////////////////	
	case PORTFW_ARRAY_T:
                index = (int)( *((unsigned char *)value));
 		return get_linkchain(&portFwChain, (char *)value, index );
////////////////////////////////////////////////////////////////////////////////////////////
	case DMZ_ARRAY_T:
		index = (int)( *((unsigned char *)value));
 		return get_linkchain(&dmzChain, (char *)value, index );
	case URLB_ARRAY_T:
		index = (int)( *((unsigned char *)value));
 		return get_linkchain(&URLBChain, (char *)value, index );
	case ACPC_ARRAY_T:
		index = (int)( *((unsigned char *)value));
 		return get_linkchain(&ACPCChain, (char *)value, index );
	case SROUT_ARRAY_T:
		index = (int)( *((unsigned char *)value));
 		return get_linkchain(&sroutChain, (char *)value, index );
////////////////////////////////////////////////////////////////////////////////////////////
	case IPFILTER_ARRAY_T:
		index = (int)( *((unsigned char *)value));
 		return get_linkchain(&ipFilterChain, (char *)value, index );

	case PORTFILTER_ARRAY_T:
                index = (int)( *((unsigned char *)value));
 		return get_linkchain(&portFilterChain, (char *)value, index );

	case MACFILTER_ARRAY_T:
                index = (int)( *((unsigned char *)value));
 		return get_linkchain(&macFilterChain, (char *)value, index );

	case TRIGGERPORT_ARRAY_T:
                index = (int)( *((unsigned char *)value));
 		return get_linkchain(&triggerPortChain, (char *)value, index );
///////////////////////////////////////////////////////////////////////////////////////////
#else
	case RSER_USR_ARRAY_T:
		index = (int)( *((unsigned char *)value));
 		return get_linkchain(&rserUsrChain, (char *)value, index );
	case RSER_CLT_ARRAY_T:
		index = (int)( *((unsigned char *)value));
 		return get_linkchain(&rserCltChain, (char *)value, index );
#endif
/////////////////////////////////////////////////////////////////////////////////////////
	}
	return 1;
}


////////////////////////////////////////////////////////////////////////////////
int apmib_getDef(int id, void *value)
{
	int ret;
	APMIB_Tp saveMib=pMib;

	pMib = pMibDef;
	ret = apmib_get(id, value);
	pMib = saveMib;
	return ret;
}


////////////////////////////////////////////////////////////////////////////////
int apmib_set(int id, void *value)
{
	int i, ret;
	void *pMibTbl;
	mib_table_entry_T *pTbl;
	unsigned char ch;
	unsigned short wd;
	unsigned long dwd;


	if (id == MIB_WLAN_AC_ADDR_ADD) {
		ret = add_linkchain(&wlanMacChain, (char *)value);
		if ( ret )
			pMib->acNum++;

		return ret;
	}
	if (id == MIB_WLAN_AC_ADDR_DEL) {
		ret = delete_linkchain(&wlanMacChain, (char *)value);
		if ( ret )
			pMib->acNum--;

		return ret;
	}
	if (id == MIB_WLAN_AC_ADDR_DELALL) {
		delete_all_linkchain(&wlanMacChain);
		pMib->acNum = 0;
		return 1;
	}

#ifdef HOME_GATEWAY
//////////////////////////////////////////////////////////////////////////////////////////////
	if (id == MIB_VSER_ADD) {
		ret = add_linkchain(&virtSvChain, (char *)value);
		if ( ret )
			pMib->vserNum++;
		return ret;
	}
	if (id == MIB_VSER_DEL) {
		ret = delete_linkchain(&virtSvChain, (char *)value);
		if ( ret )
			pMib->vserNum--;

		return ret;
	}
	if (id == MIB_VSER_DELALL) {
		delete_all_linkchain(&virtSvChain);
		pMib->vserNum = 0;
		return 1;
	}
//////////////////////////////////////////////////////////////////////////////////////////////
	if (id == MIB_PORTFW_ADD) {
		ret = add_linkchain(&portFwChain, (char *)value);
		if ( ret )
			pMib->portFwNum++;

		return ret;
	}
	if (id == MIB_PORTFW_DEL) {
		ret = delete_linkchain(&portFwChain, (char *)value);
		if ( ret )
			pMib->portFwNum--;

		return ret;
	}
	if (id == MIB_PORTFW_DELALL) {
		delete_all_linkchain(&portFwChain);
		pMib->portFwNum = 0;
		return 1;
	}
///////////////////////////////////////////////////////////////////////////////////////////
                                //DMZ
	if (id == MIB_DMZ_ADD) {

		ret = add_linkchain(&dmzChain, (char *)value);
		if ( ret )
			pMib->DMZNum++;

		return ret;
	}
	if (id == MIB_DMZ_DEL) {
		ret = delete_linkchain(&dmzChain, (char *)value);
		if ( ret )
			pMib->DMZNum--;

		return ret;
	}
	if (id == MIB_DMZ_EDIT) {
		ret = edit_linkchain(&dmzChain, (char *)value);

		return ret;
	}
	if (id == MIB_DMZ_DELALL) {
		delete_all_linkchain(&dmzChain);
		pMib->DMZNum = 0;
		return 1;
	}
                                //Static Routing
	if (id == MIB_SROUT_ADD) {
		ret = add_linkchain(&sroutChain, (char *)value);
		if ( ret )
			pMib->sroutNum++;

		return ret;
	}
	if (id == MIB_SROUT_DEL) {
		ret = delete_linkchain(&sroutChain, (char *)value);
		if ( ret )
			pMib->sroutNum--;

		return ret;
	}
	if (id == MIB_SROUT_DELALL) {
		delete_all_linkchain(&sroutChain);
		pMib->sroutNum = 0;
		return 1;
	}
                                //URLB
	if (id == MIB_URLB_ADD) {
		ret = add_linkchain(&URLBChain, (char *)value);
		if ( ret )
			pMib->URLBNum++;

		return ret;
	}
	if (id == MIB_URLB_DEL) {
		ret = delete_linkchain(&URLBChain, (char *)value);
		if ( ret )
			pMib->URLBNum--;

		return ret;
	}
	if (id == MIB_URLB_DELALL) {
		delete_all_linkchain(&URLBChain);
		pMib->URLBNum = 0;
		return 1;
	}
                                //ACPC
	if (id == MIB_ACPC_ADD) {
		ret = add_linkchain(&ACPCChain, (char *)value);
		if ( ret )
			pMib->ACPCNum++;

		return ret;
	}
	if (id == MIB_ACPC_DEL) {
		ret = delete_linkchain(&ACPCChain, (char *)value);
		if ( ret )
			pMib->ACPCNum--;

		return ret;
	}
	if (id == MIB_ACPC_DELALL) {
		delete_all_linkchain(&ACPCChain);
		pMib->ACPCNum = 0;
		return 1;
	}
///////////////////////////////////////////////////////////////////////////////////////////
	if (id == MIB_IPFILTER_ADD) {
		ret = add_linkchain(&ipFilterChain, (char *)value);
		if ( ret )
			pMib->ipFilterNum++;

		return ret;
	}
	if (id == MIB_IPFILTER_DEL) {
		ret = delete_linkchain(&ipFilterChain, (char *)value);
		if ( ret )
			pMib->ipFilterNum--;

		return ret;
	}
	if (id == MIB_IPFILTER_DELALL) {
		delete_all_linkchain(&ipFilterChain);
		pMib->ipFilterNum = 0;
		return 1;
	}

	if (id == MIB_PORTFILTER_ADD) {
		ret = add_linkchain(&portFilterChain, (char *)value);
		if ( ret )
			pMib->portFilterNum++;

		return ret;
	}
	if (id == MIB_PORTFILTER_DEL) {
		ret = delete_linkchain(&portFilterChain, (char *)value);
		if ( ret )
			pMib->portFilterNum--;

		return ret;
	}
	if (id == MIB_PORTFILTER_DELALL) {
		delete_all_linkchain(&portFilterChain);
		pMib->portFilterNum = 0;
		return 1;
	}

	if (id == MIB_MACFILTER_ADD) {
		ret = add_linkchain(&macFilterChain, (char *)value);
		if ( ret )
			pMib->macFilterNum++;

		return ret;
	}
	if (id == MIB_MACFILTER_DEL) {
		ret = delete_linkchain(&macFilterChain, (char *)value);
		if ( ret )
			pMib->macFilterNum--;

		return ret;
	}
	if (id == MIB_MACFILTER_DELALL) {
		delete_all_linkchain(&macFilterChain);
		pMib->macFilterNum = 0;
		return 1;
	}
	if (id == MIB_TRIGGERPORT_ADD) {
		ret = add_linkchain(&triggerPortChain, (char *)value);
		if ( ret )
			pMib->triggerPortNum++;

		return ret;
	}
	if (id == MIB_TRIGGERPORT_DEL) {
		ret = delete_linkchain(&triggerPortChain, (char *)value);
		if ( ret )
			pMib->triggerPortNum--;

		return ret;
	}
	if (id == MIB_TRIGGERPORT_DELALL) {
		delete_all_linkchain(&triggerPortChain);
		pMib->triggerPortNum = 0;
		return 1;
	}
///////////////////////////////////////////////////////////////////////////////////////
#else
                                //Radius User
	if (id == MIB_RSER_USR_ADD) {
		ret = add_linkchain(&rserUsrChain, (char *)value);
		if ( ret )
			pMib->rserUsrNum++;

		return ret;
	}
	if (id == MIB_RSER_USR_DEL) {
		ret = delete_linkchain(&rserUsrChain, (char *)value);
		if ( ret )
			pMib->rserUsrNum--;

		return ret;
	}
	if (id == MIB_RSER_USR_DELALL) {
		delete_all_linkchain(&rserUsrChain);
		pMib->rserUsrNum = 0;
		return 1;
	}
                                //Radius Client
	if (id == MIB_RSER_CLT_ADD) {
		ret = add_linkchain(&rserCltChain, (char *)value);
		if ( ret )
			pMib->rserCltNum++;

		return ret;
	}
	if (id == MIB_RSER_CLT_DEL) {
		ret = delete_linkchain(&rserCltChain, (char *)value);
		if ( ret )
			pMib->rserCltNum--;

		return ret;
	}
	if (id == MIB_RSER_CLT_DELALL) {
		delete_all_linkchain(&rserCltChain);
		pMib->rserCltNum = 0;
		return 1;
	}
///////////////////////////////////////////////////////////////////////////////////////////
#endif

	for (i=0; mib_table[i].id; i++) {
		if ( mib_table[i].id == id )
			break;
	}
	if ( mib_table[i].id != 0 ) {
		pMibTbl = (void *)pMib;
		pTbl = mib_table;
	}
	else {
		// if not found, search hw setting table
		for (i=0; hwmib_table[i].id; i++) {
			if ( hwmib_table[i].id == id )
				break;
		}
		if (hwmib_table[i].id == 0) // not found
			return 0;
		pMibTbl = (void *)pHwSetting;
		pTbl = hwmib_table;
	}

	switch (pTbl[i].type) {
	case BYTE_T:
//		*((unsigned char *)(((long)pMibTbl) + pTbl[i].offset)) = (unsigned char)(*((int *)value));
		ch = (unsigned char)(*((int *)value));
		memcpy( ((char *)pMibTbl) + pTbl[i].offset, &ch, 1);
		break;

	case WORD_T:
//		*((unsigned short *)(((long)pMibTbl) + pTbl[i].offset)) = (unsigned short)(*((int *)value));
		wd = (unsigned short)(*((int *)value));
		memcpy( ((char *)pMibTbl) + pTbl[i].offset, &wd, 2);
		break;

	case STRING_T:
		if ( strlen(value)+1 > pTbl[i].size )
			return 0;
		strcpy((char *)(((long)pMibTbl) + pTbl[i].offset), (char *)value);
		break;

	case BYTE5_T:
		memcpy((unsigned char *)(((long)pMibTbl) + pTbl[i].offset), (unsigned char *)value, 5);
		break;

	case BYTE6_T:
		memcpy((unsigned char *)(((long)pMibTbl) + pTbl[i].offset), (unsigned char *)value, 6);
		break;

	case BYTE13_T:
		memcpy((unsigned char *)(((long)pMibTbl) + pTbl[i].offset), (unsigned char *)value, 13);
		break;

	case BYTE16_T:
		memcpy((unsigned char *)(((long)pMibTbl) + pTbl[i].offset), (unsigned char *)value, 16);
		break;

	case DWORD_T:
		dwd = *(unsigned long *)value;
		memcpy( ((char *)pMibTbl) + pTbl[i].offset, &dwd, 4);
		break;

	case INDEX_T:
		dwd = *(unsigned long *)value;
		memcpy( ((char *)pMibTbl) + pTbl[i].offset, &dwd, 4);
		break;

	case INDEXDOS_T:
		dwd = *(unsigned long *)value;
		memcpy( ((char *)pMibTbl) + pTbl[i].offset, &dwd, 4);
		break;
#if 0
#ifdef WLAN_WPA
	case BYTE32_T:
		memcpy((unsigned char *)(((long)pMibTbl) + pTbl[i].offset), (unsigned char *)value, 32);
		break;
#endif
#endif

	case BYTE_ARRAY_T:
		memcpy((unsigned char *)(((long)pMibTbl) + pTbl[i].offset), (unsigned char *)value, pTbl[i].size);
		break;

	case IA_T:
		memcpy((unsigned char *)(((long)pMibTbl) + pTbl[i].offset), (unsigned char *)value,  4);
		break;

	case WLAC_ARRAY_T:
#ifdef HOME_GATEWAY
	case VSER_ARRAY_T://///////////////////////////////////////////////////////////////
	case DMZ_ARRAY_T://///////////////////////////////////////////////////////////////
	case URLB_ARRAY_T://///////////////////////////////////////////////////////////////
	case ACPC_ARRAY_T://///////////////////////////////////////////////////////////////
	case SROUT_ARRAY_T://///////////////////////////////////////////////////////////////
	case PORTFW_ARRAY_T:
	case IPFILTER_ARRAY_T:
	case PORTFILTER_ARRAY_T:
	case MACFILTER_ARRAY_T:
	case TRIGGERPORT_ARRAY_T:
#else
	case RSER_CLT_ARRAY_T:	//radius client
	case RSER_USR_ARRAY_T:	//radius user
#endif
		return 0;
	}
	return 1;
}


////////////////////////////////////////////////////////////////////////////////
int apmib_setDef(int id, void *value)
{
	int ret;
	APMIB_Tp saveMib=pMib;

	pMib = pMibDef;
	ret = apmib_set(id, value);
	pMib = saveMib;
	return ret;
}



////////////////////////////////////////////////////////////////////////////////
/* Update current used MIB into flash in current setting area
 */
int apmib_update(CONFIG_DATA_T type)
{
	int i, len;
	unsigned char checksum;
	unsigned char *data;

	if (type & HW_SETTING) {
		data = (char *)pHwSetting;
		checksum = CHECKSUM(data, hsHeader.len-1);
		data[hsHeader.len-1] = checksum;

		if ( flash_write((char *)data, HW_SETTING_OFFSET+sizeof(hsHeader), hsHeader.len)==0 ) {
			printf("write hs MIB failed!\n");
			return 0;
		}
	}

	if ((type & CURRENT_SETTING) || (type & DEFAULT_SETTING)) {
		memset( pMib->acAddrArray, '\0', MAX_WLAN_AC_NUM*6 );
		for (i=0; i<pMib->acNum; i++) {
			get_linkchain(&wlanMacChain, (void *)&pMib->acAddrArray[i], i+1);
		}

#ifdef HOME_GATEWAY
		memset( pMib->portFwArray, '\0', MAX_FILTER_NUM*sizeof(PORTFW_T) );
		for (i=0; i<pMib->portFwNum; i++) {
			get_linkchain(&portFwChain, (void *)&pMib->portFwArray[i], i+1);
		}
////////////////////////////////////////////////////////////////////////////////////////////
		memset( pMib->DMZArray, '\0', MAX_FILTER_NUM*sizeof(DMZ_T) );
		for (i=0; i<pMib->DMZNum; i++) {
			get_linkchain(&dmzChain, (void *)&pMib->DMZArray[i], i+1);
		}
		memset( pMib->URLBArray, '\0', MAX_FILTER_NUM*sizeof(URLB_T) );
		for (i=0; i<pMib->URLBNum; i++) {
			get_linkchain(&URLBChain, (void *)&pMib->URLBArray[i], i+1);
		}
		memset( pMib->ACPCArray, '\0', MAX_FILTER_NUM*sizeof(ACPC_T) );
		for (i=0; i<pMib->ACPCNum; i++) {
			get_linkchain(&ACPCChain, (void *)&pMib->ACPCArray[i], i+1);
		}
		memset( pMib->sroutArray, '\0', MAX_FILTER_NUM*sizeof(SROUT_T) );
		for (i=0; i<pMib->sroutNum; i++) {
			get_linkchain(&sroutChain, (void *)&pMib->sroutArray[i], i+1);
		}
////////////////////////////////////////////////////////////////////////////////////////////
//		memset( pMib->ipFilterArray, '\0', MAX_FILTER_NUM*sizeof(IPFILTER_T) );
		memset( pMib->ipFilterArray, '\0', sizeof(IPFILTER_T) );
		for (i=0; i<pMib->ipFilterNum; i++) {
			get_linkchain(&ipFilterChain, (void *)&pMib->ipFilterArray[i], i+1);
		}
		
//////////////////////////////////////////////////////////////////////////////////////////////
		memset( pMib->vserArray, '\0', MAX_FILTER_NUM*sizeof(PORTFW_T) );
		for (i=0; i<pMib->vserNum; i++) {
			get_linkchain(&virtSvChain, (void *)&pMib->vserArray[i], i+1);
		}
//////////////////////////////////////////////////////////////////////////////////////////////
//		memset( pMib->portFilterArray, '\0', MAX_FILTER_NUM*sizeof(PORTFILTER_T) );
		memset( pMib->portFilterArray, '\0', sizeof(PORTFILTER_T) );
		for (i=0; i<pMib->portFilterNum; i++) {
			get_linkchain(&portFilterChain, (void *)&pMib->portFilterArray[i], i+1);
		}
		memset( pMib->macFilterArray, '\0', MAX_FILTER_NUM*sizeof(MACFILTER_T) );
		for (i=0; i<pMib->macFilterNum; i++) {
			get_linkchain(&macFilterChain, (void *)&pMib->macFilterArray[i], i+1);
		}
		memset( pMib->triggerPortArray, '\0', MAX_FILTER_NUM*sizeof(TRIGGERPORT_T) );
		for (i=0; i<pMib->triggerPortNum; i++) {
			get_linkchain(&triggerPortChain, (void *)&pMib->triggerPortArray[i], i+1);
		}
////////////////////////////////////////////////////////////////////////////////////
#else
		memset( pMib->rserCltArray, '\0', 16*sizeof(RSER_CLT_T) );
		for (i=0; i<pMib->rserCltNum; i++) {
			get_linkchain(&rserCltChain, (void *)&pMib->rserCltArray[i], i+1);
		}
		memset( pMib->rserUsrArray, '\0', 96*sizeof(RSER_USR_T) );
		for (i=0; i<pMib->rserUsrNum; i++) {
			get_linkchain(&rserUsrChain, (void *)&pMib->rserUsrArray[i], i+1);
		}
////////////////////////////////////////////////////////////////////////////////////
#endif
		if (type & CURRENT_SETTING) {
			data = (unsigned char *)pMib;
			checksum = CHECKSUM(data, csHeader.len-1);
			*(data + csHeader.len - 1) = checksum;
			i = CURRENT_SETTING_OFFSET + sizeof(csHeader);
			len = csHeader.len;
		}
		else {
			data = (unsigned char *)pMibDef;
			checksum = CHECKSUM(data, dsHeader.len-1);
			*(data + dsHeader.len - 1) = checksum;
			i = DEFAULT_SETTING_OFFSET + sizeof(dsHeader);
			len = dsHeader.len;
		}

		if ( flash_write((char *)data, i, len)==0 ) {
			printf("Write flash current-setting failed!\n");
			return 0;
		}
	}

	return 1;
}


////////////////////////////////////////////////////////////////////////////////
/* Update default setting MIB into current setting area
 */
int apmib_updateDef(void)
{
	unsigned char *data, checksum;
	PARAM_HEADER_T header;
	int i;

	memcpy(header.signature, CURRENT_SETTING_HEADER_TAG, TAG_LEN);
	memcpy(&header.signature[TAG_LEN], &dsHeader.signature[TAG_LEN], SIGNATURE_LEN-TAG_LEN);

	header.len = dsHeader.len;
	data = (unsigned char *)pMibDef;
	checksum = CHECKSUM(data, header.len-1);
	*(data + header.len - 1) = checksum;

	i = CURRENT_SETTING_OFFSET;
	if ( flash_write((char *)&header, i, sizeof(header))==0 ) {
		printf("Write flash current-setting header failed!\n");
		return 0;
	}
	i += sizeof(header);

	if ( flash_write((char *)data, i, header.len)==0 ) {
		printf("Write flash current-setting failed!\n");
		return 0;
	}

	return 1;
}


////////////////////////////////////////////////////////////////////////////////
/* Update MIB into flash current setting area
 */
int apmib_updateFlash(CONFIG_DATA_T type, char *data, int len, int force, int ver)
{
	unsigned char checksum, checksum1, *ptr=NULL;
	int i, offset=0, curLen, curVer;
	unsigned char *pMibData, *pHdr, tmpBuf[20];

	if ( type == HW_SETTING ) {
		curLen = hsHeader.len - 1;
		pMibData = (unsigned char *)pHwSetting;
		pHdr = (unsigned char *)&hsHeader;
		i = HW_SETTING_OFFSET;
	}
	else if ( type == DEFAULT_SETTING ) {
		curLen = dsHeader.len - 1;
		pMibData = (unsigned char *)pMibDef;
		pHdr = (unsigned char *)&dsHeader;
		i = DEFAULT_SETTING_OFFSET;
	}
	else  {
		curLen = csHeader.len - 1;
		pMibData = (unsigned char *)pMib;
		pHdr = (unsigned char *)&csHeader;
		i = CURRENT_SETTING_OFFSET;
	}

	if (force==2) { // replace by input mib
		((PARAM_HEADER_Tp)pHdr)->len = len + 1;
		sprintf(tmpBuf, "%02d", ver);
		memcpy(&pHdr[TAG_LEN], tmpBuf, SIGNATURE_LEN-TAG_LEN);
		checksum = CHECKSUM(data, len);
		pMibData = data;
		curLen = len;
	}
	else if (force==1) { // update mib but keep not used mib
		sscanf(&((PARAM_HEADER_Tp)pHdr)->signature[TAG_LEN], "%02d", &curVer);
		if ( curVer < ver ) {
			sprintf(tmpBuf, "%02d", ver);
			memcpy(&((PARAM_HEADER_Tp)pHdr)->signature[TAG_LEN],
					tmpBuf, SIGNATURE_LEN-TAG_LEN);
		}
		checksum = CHECKSUM(data, len);
		if (curLen > len) {
			((PARAM_HEADER_Tp)pHdr)->len = curLen + 1;
			ptr = pMibData + len;
			offset = curLen - len;
			checksum1 = CHECKSUM(ptr, offset);
			checksum +=  checksum1;
		}
		else
			((PARAM_HEADER_Tp)pHdr)->len = len + 1;

		curLen = len;
		pMibData = data;
	}
	else { // keep old mib, only update new added portion
		sscanf(&((PARAM_HEADER_Tp)pHdr)->signature[TAG_LEN], "%02d", &curVer);
		if ( curVer < ver ) {
			sprintf(tmpBuf, "%02d", ver);
			memcpy(&((PARAM_HEADER_Tp)pHdr)->signature[TAG_LEN],
					tmpBuf, SIGNATURE_LEN-TAG_LEN);
		}
		if ( len > curLen ) {
			((PARAM_HEADER_Tp)pHdr)->len = len + 1;
			offset = len - curLen;
			checksum = CHECKSUM(pMibData, curLen);
			ptr = data + curLen;
			checksum1 = CHECKSUM(ptr, offset);
			checksum +=  checksum1;
		}
		else
			checksum = CHECKSUM(pMibData, curLen);
	}

	if ( flash_write((char *)pHdr, i, sizeof(PARAM_HEADER_T))==0 ) {
		printf("Write flash current-setting header failed!\n");
		return 0;
	}
	i += sizeof(PARAM_HEADER_T);

	if ( flash_write(pMibData, i, curLen)==0 ) {
		printf("Write flash current-setting failed!\n");
		return 0;
	}
	i += curLen;

	if (offset > 0) {
		if ( flash_write((char *)ptr, i, offset)==0 ) {
			printf("Write flash current-setting failed!\n");
			return 0;
		}
		i += offset;
	}

	if ( flash_write((char *)&checksum, i, sizeof(checksum))==0 ) {
		printf("Write flash current-setting checksum failed!\n");
		return 0;
	}

	return 1;
}

/////////////////////////////////////////////////////////////////////////////////
static int flash_read(char *buf, int offset, int len)
{
	int fh;
	int ok=1;

	fh = open(FLASH_DEVICE_NAME, O_RDWR);
	if ( fh == -1 )
		return 0;

	lseek(fh, offset, SEEK_SET);

	if ( read(fh, buf, len) != len)
		ok = 0;

	close(fh);

	return ok;
}


////////////////////////////////////////////////////////////////////////////////
static int flash_write(char *buf, int offset, int len)
{
	int fh;
	int ok=1;

	fh = open(FLASH_DEVICE_NAME, O_RDWR);

	if ( fh == -1 )
		return 0;

	lseek(fh, offset, SEEK_SET);

	if ( write(fh, buf, len) != len)
		ok = 0;

	close(fh);
	sync();

	return ok;
}


///////////////////////////////////////////////////////////////////////////////
static int init_linkchain(LINKCHAIN_Tp pLinkChain, int size, int num)
{
	FILTER_Tp entry;
	int offset=sizeof(FILTER_Tp)*2;
	char *pBuf;
	int i;

	if (size%4)
		size = (size/4+1)*4;

	pBuf = calloc(num, size+offset);
	if ( pBuf == NULL )
		return 0;

	pLinkChain->buf = pBuf;
	pLinkChain->pUsedList = NULL;
	pLinkChain->pFreeList = NULL;
	entry = (FILTER_Tp)pBuf;

	ADD_LINK_LIST(pLinkChain->pFreeList, entry);
	for (i=1; i<num; i++) {
		entry = (FILTER_Tp)&pBuf[i*(size+offset)];
		ADD_LINK_LIST(pLinkChain->pFreeList, entry);
	}

	pLinkChain->size = size;
	pLinkChain->num = num;
	pLinkChain->usedNum = 0;
	return 1;
}

///////////////////////////////////////////////////////////////////////////////
static int add_linkchain(LINKCHAIN_Tp pLinkChain, char *val)
{
	FILTER_Tp entry;

	// get a free entry
	entry = pLinkChain->pFreeList;
	if (entry == NULL)
		return 0;

	if (entry->next==pLinkChain->pFreeList)
		pLinkChain->pFreeList = NULL;
	else
		pLinkChain->pFreeList = entry->next;

	REMOVE_LINK_LIST(entry);

	// copy content
	memcpy(entry->val, val, pLinkChain->size);

	// add to used list
	if (pLinkChain->pUsedList == NULL) {
		ADD_LINK_LIST(pLinkChain->pUsedList, entry);
	}
	else {
		ADD_LINK_LIST(pLinkChain->pUsedList->prev, entry);
	}
	pLinkChain->usedNum++;
	return 1;
}

///////////////////////////////////////////////////////////////////////////////
static int edit_linkchain(LINKCHAIN_Tp pLinkChain, char *val)	//Erwin add
{
	FILTER_Tp curEntry=pLinkChain->pUsedList;
	
	while (curEntry != NULL) {
		if ( !memcmp(curEntry->val, (unsigned char *)(((EDIT_Tp)val)->selEntry), pLinkChain->compareLen) ) {
			memcpy(curEntry->val, ((EDIT_Tp)val)->editEntry, pLinkChain->size);
			return 1;
		}
		if ( pLinkChain->pUsedList->next == pLinkChain->pUsedList )
			return 0;
		curEntry = curEntry->next;
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
static int delete_linkchain(LINKCHAIN_Tp pLinkChain, char *val)
{
	FILTER_Tp curEntry=pLinkChain->pUsedList;

	while (curEntry != NULL) {
		if ( !memcmp(curEntry->val,(unsigned char *)val,pLinkChain->compareLen) ) {
				if (curEntry == pLinkChain->pUsedList) {
					if ( pLinkChain->pUsedList->next != pLinkChain->pUsedList )
						pLinkChain->pUsedList = pLinkChain->pUsedList->next;
					else
						pLinkChain->pUsedList = NULL;
				}
				REMOVE_LINK_LIST(curEntry);
				ADD_LINK_LIST(pLinkChain->pFreeList, curEntry);
				pLinkChain->usedNum--;
				return 1;
		}
		if ( curEntry->next == pLinkChain->pUsedList )
			return 0;
		curEntry = curEntry->next;
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
static void delete_all_linkchain(LINKCHAIN_Tp pLinkChain)
{
	FILTER_Tp curEntry;

	if (pLinkChain->pUsedList==NULL)
		return;

	// search for matched mac address
	while (pLinkChain->pUsedList) {
		curEntry = pLinkChain->pUsedList;
		if (pLinkChain->pUsedList->next != pLinkChain->pUsedList)
			pLinkChain->pUsedList = pLinkChain->pUsedList->next;
		else
			pLinkChain->pUsedList = NULL;

		REMOVE_LINK_LIST(curEntry);
		ADD_LINK_LIST(pLinkChain->pFreeList, curEntry);
		pLinkChain->usedNum--;
	}
}

///////////////////////////////////////////////////////////////////////////////
static int get_linkchain(LINKCHAIN_Tp pLinkChain, char *val, int index)
{
	FILTER_Tp curEntry=pLinkChain->pUsedList;

	if ( curEntry == NULL || index > pLinkChain->usedNum)
 		return 0;

	while (--index > 0)
        	curEntry = curEntry->next;
	
	memcpy( (unsigned char *)val, curEntry->val, pLinkChain->size);

	return 1;
}
