/*
 *      Operation routines for FLASH MIB access
 *
 *      Authors: David Hsu	<davidhsu@realtek.com.tw>
 *
 *      $Id: flash.c,v 1.4 2004/02/24 05:57:27 erwin Exp $
 *
 */


/* System include files */
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <ctype.h>

#define noPARSE_TXT_FILE

/* Local include files */
#include "apmib.h"
#include "mibtbl.h"

/* Constand definitions */
#define DEC_FORMAT	("%d")
#define LONGDEC_FORMAT  ("%lu")
#define BYTE5_FORMAT	("%02x%02x%02x%02x%02x")
#define BYTE6_FORMAT	("%02x%02x%02x%02x%02x%02x")
#define BYTE13_FORMAT	("%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x")
#define BYTE16_FORMAT	("%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x")

//#define STR_FORMAT	("%s")
//
#define STR_FORMAT	("'%s'")
#define HEX_FORMAT	("%02x")

#ifdef HOME_GATEWAY
#define PORTFW_FORMAT	("%s,%d,%d,%d")
#define PORTFILTER_FORMAT ("%d,%d,%d")
#define IPFILTER_FORMAT	("%s,%d")
//#define TRIGGERPORT_FORMAT ("%d,%d,%d,%d,%d,%d")
#define DMZ_FORMAT	("%s,%s")
#define SROUT_FORMAT	("%s,%s,%s,%d,%d")
#define URLB_FORMAT	("%s")
#define ACPC_FORMAT	 ("%s,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%lu,%d")//////

#define TRIGGERPORT_FORMAT ("%d-%d:%d:%d,%d,%d,%d,%d,%d,%d,%d,%d,%d:%d")
#endif
#define MACFILTER_FORMAT ("%02x%02x%02x%02x%02x%02x")
#define MACFILTER_COLON_FORMAT ("%02x:%02x:%02x:%02x:%02x:%02x")

#ifdef WLAN_WDS
#define WDS_FORMAT      MACFILTER_FORMAT
#endif

#define SPACE	(' ')
#define EOL	('\n')

static int config_area;

/* Macro definition */
static int _is_hex(char c)
{
    return (((c >= '0') && (c <= '9')) ||
            ((c >= 'A') && (c <= 'F')) ||
            ((c >= 'a') && (c <= 'f')));
}

static int string_to_hex(char *string, unsigned char *key, int len)
{
	char tmpBuf[4];
	int idx, ii=0;
	for (idx=0; idx<len; idx+=2) {
		tmpBuf[0] = string[idx];
		tmpBuf[1] = string[idx+1];
		tmpBuf[2] = 0;
		if ( !_is_hex(tmpBuf[0]) || !_is_hex(tmpBuf[1]))
			return 0;

		key[ii++] = (unsigned char) strtol(tmpBuf, (char**)NULL, 16);
	}
	return 1;
}

static void convert_lower(char *str)
{	int i;
	int len = strlen(str);
	for (i=0; i<len; i++)
		str[i] = tolower(str[i]);
}

static int APMIB_GET(int id, void *val)
{
	if (config_area == 2)
		return apmib_getDef(id, val);
	else
		return apmib_get(id, val);
}

static int APMIB_SET(int id, void *val)
{
	if (config_area == 2)
		return apmib_setDef(id, val);
	else
		return apmib_set(id, val);
}

/* Local declarations routines */
static int flash_read(char *buf, int offset, int len);
static int writeDefault(void);
static int searchMIB(char *token);
static void getMIB(char *name, int id, TYPE_T type, int num, int array_separate);
static void setMIB(char *name, int id, TYPE_T type, int len, int valnum, char **val);
static void dumpAll(void);
static void showHelp(void);
static void showAllMibName(void);
static void showSetACHelp(void);
static int read_flash_webpage(char *prefix, char *file);

#ifdef WLAN_WDS
static void showSetWdsHelp(void);
#endif

#ifdef HOME_GATEWAY
static void showSetPortFwHelp(void);
static void showSetPortFilterHelp(void);
static void showSetIpFilterHelp(void);
static void showSetMacFilterHelp(void);
static void showSetTriggerPortHelp(void);
#endif

#ifdef WLAN_WPA
static void generateWpaConf(char *outputFile);
#endif

#ifdef PARSE_TXT_FILE
static int parseTxtConfig(char *filename, APMIB_Tp pConfig);
static int getToken(char *line, char *value);
static int set_mib(APMIB_Tp pMib, int id, void *value);
static void getVal2(char *value, char **p1, char **p2);
#ifdef HOME_GATEWAY
static void getVal3(char *value, char **p1, char **p2, char **p3);
static void getVal4(char *value, char **p1, char **p2, char **p3, char **p4);
static void getVal5(char *value, char **p1, char **p2, char **p3, char **p4, char **p5);
#endif

static int acNum;

#ifdef WLAN_WDS
static int wdsNum;
#endif

#ifdef HOME_GATEWAY
static int macFilterNum, portFilterNum, ipFilterNum, portFwNum, triggerPortNum, vserNum, DMZNum, ACPCNum;
#else
static int reserUsrNum, reserCltNun;
#endif

/////////////////////////////////////////////////////////////////////////////////////////
static char __inline__ *getVal(char *value, char **p)
{
	int len=0;

	while (*value == ' ' ) value++;

	*p = value;

	while (*value && *value!=',') {
		value++;
		len++;
	}

	if ( !len ) {
		*p = NULL;
		return NULL;
	}

	if ( *value == 0)
		return NULL;

	*value = 0;
	value++;

	return value;
}
#endif  // PARSE_TXT_FILE


//////////////////////////////////////////////////////////////////////
static int resetHWDefault()
{
	PARAM_HEADER_T header;
	HW_SETTING_T mib;
	HW_SETTING_Tp pMib=&mib;
	unsigned char checksum;
	int fh, status;
	

	fh = open(DEFAULT_SETTING_FILE, O_RDWR);
	
	if ( fh == -1 ) {
		printf("open %s file failed!\n", DEFAULT_SETTING_FILE);
		return -1;
	}
	
	lseek(fh, HW_SETTING_OFFSET_IN_FILE, SEEK_SET);

	if (read(fh, &header, sizeof(header)) != sizeof(header)) {
		printf("read default setting header failed!\n");
		return -1;
	}
        
	if (read(fh, pMib, header.len) != header.len) {
		printf("read HW setting failed!, header.len = 0x%x\n", header.len);
		return -1;
	}
	close(fh);

//	dump_mem(pMib, 0x1ffa);
    
	checksum = CHECKSUM((void *)pMib, header.len-1);
	status = CHECKSUM_OK((void *)pMib, header.len);
	if ( !status) {
		printf("default setting Checksum error!, checksum = %x\n", checksum);
		return -1;
	}
	   
			
#ifdef __mips__
	fh = open(FLASH_DEVICE_NAME, O_RDWR);
#endif

#ifdef __i386__
	fh = open(FLASH_DEVICE_NAME, O_RDWR);
#endif
	if ( fh == -1 ) {
		printf("create file failed!\n");
		return -1;
	}

	lseek(fh, HW_SETTING_OFFSET, SEEK_SET);
    
	if ( write(fh, (const void *)&header, sizeof(header)) != sizeof(header)) {
        printf("write HW header failed!\n");
        return -1;
    }
	
	lseek(fh, HW_SETTING_OFFSET + sizeof(header), SEEK_SET);
    
	if ( write(fh, (const void *)pMib, header.len) !=  header.len) {
        printf("write hw MIB failed!\n");
        return -1;
    }
	
	close(fh);
	sync();
	return 0;
}

static int resetDFDefault()
{
	PARAM_HEADER_T header;
	APMIB_T mib;
	APMIB_Tp pMib=&mib;
	unsigned char checksum;
	int fh, status;
	
	
	fh = open(DEFAULT_SETTING_FILE, O_RDWR);
	
	if ( fh == -1 ) {
		printf("open %s file failed!\n", DEFAULT_SETTING_FILE);
		return -1;
	}
	
	lseek(fh, DF_SETTING_OFFSET_IN_FILE, SEEK_SET);

	if (read(fh, &header, sizeof(header)) != sizeof(header)) {
		printf("read default setting header failed!\n");
		return -1;
	}
        
	if (read(fh, pMib, header.len) != header.len) {
		printf("read default setting failed!, header.len = 0x%x\n", header.len);
		return -1;
	}
	close(fh);

//	dump_mem(pMib, 0x1ffa);
    
	checksum = CHECKSUM((void *)pMib, header.len-1);
	status = CHECKSUM_OK((void *)pMib, header.len);
	if ( !status) {
		printf("default setting Checksum error!, checksum = %x\n", checksum);
		return -1;
	}
	   
			
	fh = open(FLASH_DEVICE_NAME, O_RDWR);
	
	if ( fh == -1 ) {
		printf("create file failed!\n");
		return -1;
	}

	lseek(fh, DEFAULT_SETTING_OFFSET, SEEK_SET);
    
	if ( write(fh, (const void *)&header, sizeof(header)) != sizeof(header)) {
        printf("write DS header failed!\n");
        return -1;
    }
	
	lseek(fh, DEFAULT_SETTING_OFFSET + sizeof(header), SEEK_SET);
    
	if ( write(fh, (const void *)pMib, header.len) !=  header.len) {
        printf("write DS MIB failed!\n");
        return -1;
    }
	
	close(fh);
	sync();
	return 0;
}

static int resetDefault()
{
	PARAM_HEADER_T header;
	APMIB_T mib;
	APMIB_Tp pMib=&mib;
	unsigned char checksum;
	int fh, status;
#ifdef __i386__
	char	tmpBuf[80];
#endif
	unsigned int tmpLen;
	
	
	if( sizeof(HW_SETTING_T) != (HW_SETTING_LEN-sizeof(header)-sizeof(checksum)))
		printf("size of HW_SETTNG = %d, should be %d\n", sizeof(HW_SETTING_T), (HW_SETTING_LEN-sizeof(header)-sizeof(checksum)));
	
	if( sizeof(APMIB_T) != (DF_SETTING_LEN-sizeof(header)-sizeof(checksum)))
	{
		printf("size of APMIB = %d, should be %d\n", sizeof(APMIB_T), (DF_SETTING_LEN-sizeof(header)-sizeof(checksum)));
		printf("The size of APMIB_T->Reserved should be = %d\n", sizeof(pMib->Reserved) + (DF_SETTING_LEN-sizeof(header)-sizeof(checksum)) -  sizeof(APMIB_T));
	}

	fh = open(DEFAULT_SETTING_FILE, O_RDWR);
	
	if ( fh == -1 ) {
		printf("open %s file failed!\n", DEFAULT_SETTING_FILE);
		return -1;
	}
	
	lseek(fh, HW_SETTING_LEN, SEEK_SET);

	if (read(fh, &header, sizeof(header)) != sizeof(header)) {
		printf("read default setting header failed!\n");
		return -1;
	}
        
	tmpLen = header.len;
	
	if (read(fh, pMib, header.len) != header.len) {
		printf("read default setting failed!, header.len = 0x%x\n", header.len);
		return -1;
	}
	close(fh);

//	dump_mem(pMib, 0x1ffa);
    
	checksum = CHECKSUM((void *)pMib, header.len-1);
	status = CHECKSUM_OK((void *)pMib, header.len);
	if ( !status) {
		printf("default setting Checksum error!, checksum = %x\n", checksum);
		return -1;
	}
			
#ifdef __i386__
	sprintf(tmpBuf, "cp -f %s %s", DEFAULT_SETTING_FILE, FLASH_DEVICE_NAME);
	
	system(tmpBuf);
#endif
	
#ifdef __mips__
	fh = open(FLASH_DEVICE_NAME, O_RDWR);
#endif

#ifdef __i386__
	fh = open(FLASH_DEVICE_NAME, O_RDWR);
#endif
	if ( fh == -1 ) {
		printf("create file failed!\n");
		return -1;
	}

    sprintf(header.signature, "%s%02d", CURRENT_SETTING_HEADER_TAG, CURRENT_SETTING_VER); // This will corruptheader.len
	
	
	header.len = tmpLen;
	
	lseek(fh, CURRENT_SETTING_OFFSET, SEEK_SET);
    
	if ( write(fh, (const void *)&header, sizeof(header)) != sizeof(header)) {
        printf("write cs header failed!\n");
        return -1;
    }
	
	lseek(fh, CURRENT_SETTING_OFFSET + sizeof(header), SEEK_SET);
    
	if ( write(fh, (const void *)pMib, header.len) !=  header.len) {
        printf("write cs MIB failed!\n");
        return -1;
    }
	
	close(fh);
	sync();
	return 0;
}



//////////////////////////////////////////////////////////////////////
int main(int argc, char** argv)
{
	int argNum=1, action=0, idx, num, valNum=0;
	char mib[100]={0}, valueArray[20][100], *value[20], *ptr;
#ifdef PARSE_TXT_FILE
	char filename[100]={0};
	APMIB_T apmib;
#endif
	if ( argc > 1 ) {
#ifdef PARSE_TXT_FILE
		if ( !strcmp(argv[argNum], "-f") ) {
			if (++argNum < argc)
				sscanf(argv[argNum++], "%s", filename);
		}
#endif
		if ( !strcmp(argv[argNum], "get") ) {
			action = 1;
			if (++argNum < argc)
				sscanf(argv[argNum], "%s", mib);
		}
		else if ( !strcmp(argv[argNum], "set") ) {
			action = 2;
			if (++argNum < argc) {
				sscanf(argv[argNum], "%s", mib);
				while (++argNum < argc) {
					sscanf(argv[argNum], "%s", valueArray[valNum]);
					value[valNum] = valueArray[valNum];
					valNum++;
				}
			}
		}
		else if ( !strcmp(argv[argNum], "all") ) {
			action = 3;
		}
		else if ( !strcmp(argv[argNum], "default") ) {
			return writeDefault();
		}
		else if ( !strcmp(argv[argNum], "reset") ) {
			return resetDefault();
		}
		else if ( !strcmp(argv[argNum], "extr") ) {
			if (++argNum < argc) {
				char prefix[20], file[20]={0};
				int ret;
				sscanf(argv[argNum], "%s", prefix);
				if (++argNum < argc)
					sscanf(argv[argNum], "%s", file);
				ret  = read_flash_webpage(prefix, file);
				if (ret == 0) // success
					unlink(file);
				return ret;				
			}
			printf("Usage: %s web prefix\n", argv[0]);
			return -1;
		}
		else if ( !strcmp(argv[argNum], "test-hwconf") ) {
			if ((ptr=apmib_hwconf()) == NULL)
				return -1;
			free(ptr);
			return 0;
		}
		else if ( !strcmp(argv[argNum], "test-dsconf") ) {
			if ((ptr=apmib_dsconf()) == NULL)
				return -1;
			free(ptr);
			return 0;
		}
		else if ( !strcmp(argv[argNum], "test-csconf") ) {
			if ((ptr=apmib_csconf()) == NULL)
				return -1;
			free(ptr);
			return 0;
		}
#ifdef WLAN_WPA
                else if ( !strcmp(argv[argNum], "wpa") ) {
                        generateWpaConf(argv[argNum+1]);
                        return 0;
                }
#endif
				
	}

	if ( action == 0) {
		showHelp();
		return -1;
	}
	if ( (action==1 && !mib[0]) ||
	     (action==2 && !mib[0]) ) {
		showAllMibName();
		return -1;
	}

	if ( action==2 && (!strcmp(mib, "WLAN_MACAC_ADDR") || !strcmp(mib, "DEF_WLAN_MACAC_ADDR"))) {
		if ( strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall") ) {
			showSetACHelp();
			return -1;
		}
		if ( (!strcmp(value[0], "del") && !value[1]) || (!strcmp(value[0], "add") && !value[1]) ) {
			showSetACHelp();
			return -1;
		}
	}
#ifdef WLAN_WDS
        if ( action==2 && (!strcmp(mib, "WLAN_WDS") || !strcmp(mib, "DEF_WLAN_WDS"))) {
                if ( strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall") ) {
                        showSetWdsHelp();
                        return -1;
                }
                if ( (!strcmp(value[0], "del") && !value[1]) || (!strcmp(value[0], "add") && !value[1]) ) {
                        showSetWdsHelp();
                        return -1;
                }
        }
#endif
		
#ifdef HOME_GATEWAY
////////////////////////////////////////////////////////////////////////////////////////////////
	if ( action==2 && (!strcmp(mib, "VSER_TBL") || !strcmp(mib, "DEF_VSER_TBL"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetPortFwHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "DMZ_TBL") || !strcmp(mib, "DEF_DMZ_TBL"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "edit") && strcmp(value[0], "delall"))) {
			showSetPortFwHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "URLB_TBL") || !strcmp(mib, "DEF_URLB_TBL"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetPortFwHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "SROUT_TBL") || !strcmp(mib, "DEF_SROUT_TBL"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetPortFwHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "ACPC_TBL") || !strcmp(mib, "DEF_ACPC_TBL"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetPortFwHelp();
			return -1;
		}
	}
////////////////////////////////////////////////////////////////////////////////////////////////
	if ( action==2 && (!strcmp(mib, "PORTFW_TBL") || !strcmp(mib, "DEF_PORTFW_TBL"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetPortFwHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "PORTFILTER_TBL") || !strcmp(mib, "DEF_PORTFILTER_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetPortFilterHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "IPFILTER_TBL") || !strcmp(mib, "DEF_IPFILTER_TBL"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetIpFilterHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "MACFILTER_TBL") || !strcmp(mib, "DEF_MACFILTER_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetMacFilterHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "TRIGGERPORT_TBL") || !strcmp(mib, "DEF_TRIGGERPORT_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetTriggerPortHelp();
			return -1;
		}
	}
//////////////////////////////////////////////////////////////////////////////////////
#else
	if ( action==2 && (!strcmp(mib, "RSER_USR_TBL") || !strcmp(mib, "DEF_RSER_USR_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
//			showSetTriggerPortHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "RSER_CLT_TBL") || !strcmp(mib, "DEF_RSER_CLT_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
//			showSetTriggerPortHelp();
			return -1;
		}
	}
#endif

	switch (action) {
	case 1: // get

#ifdef PARSE_TXT_FILE
		if ( filename[0] ) {
			if ( parseTxtConfig(filename, &apmib) < 0) {
				printf("Parse text file error!\n");
				return -1;
			}

			if ( !apmib_init(&apmib)) {
				printf("Initialize AP MIB failed!\n");
				return -1;
			}
		}
		else

#endif
		if ( !apmib_init()) {
			printf("Initialize AP MIB failed!\n");
			return -1;
		}

		idx = searchMIB(mib);
		if ( idx == -1 ) {
			showHelp();
			showAllMibName();
			return -1;
		}

		if (!strcmp(mib, "WLAN_MACAC_ADDR"))
			APMIB_GET(MIB_WLAN_AC_NUM, (void *)&num);
#ifdef WLAN_WDS
                else if (!strcmp(mib, "WLAN_WDS"))
                        APMIB_GET(MIB_WLAN_WDS_NUM, (void *)&num);
#endif

#ifdef HOME_GATEWAY
///////////////////////////////////////////////////////////////////////////////////////////////
		else if (!strcmp(mib, "VSER_TBL"))
			APMIB_GET(MIB_VSER_NUM, (void *)&num);
		else if (!strcmp(mib, "DMZ_TBL"))
			APMIB_GET(MIB_DMZ_NUM, (void *)&num);
		else if (!strcmp(mib, "SROUT_TBL"))
			APMIB_GET(MIB_SROUT_NUM, (void *)&num);
		else if (!strcmp(mib, "URLB_TBL"))
			APMIB_GET(MIB_URLB_NUM, (void *)&num);
		else if (!strcmp(mib, "ACPC_TBL"))
			APMIB_GET(MIB_ACPC_NUM, (void *)&num);
///////////////////////////////////////////////////////////////////////////////////////////////
		else if (!strcmp(mib, "PORTFW_TBL"))
			APMIB_GET(MIB_PORTFW_NUM, (void *)&num);
		else if (!strcmp(mib, "PORTFILTER_TBL"))
			APMIB_GET(MIB_PORTFILTER_NUM, (void *)&num);
		else if (!strcmp(mib, "IPFILTER_TBL"))
			APMIB_GET(MIB_IPFILTER_NUM, (void *)&num);
		else if (!strcmp(mib, "MACFILTER_TBL"))
			APMIB_GET(MIB_MACFILTER_NUM, (void *)&num);
		else if (!strcmp(mib, "TRIGGERPORT_TBL"))
			APMIB_GET(MIB_TRIGGERPORT_NUM, (void *)&num);
///////////////////////////////////////////////////////////////////////////////////////////////
#else
		else if (!strcmp(mib, "RSER_CLT_TBL"))
			APMIB_GET(MIB_RSER_CLT_NUM, (void *)&num);
		else if (!strcmp(mib, "RSER_USR_TBL"))
			APMIB_GET(MIB_RSER_USR_NUM, (void *)&num);
#endif
		else
			num = 1;
		if (config_area == 1)
			getMIB(mib, hwmib_table[idx].id, hwmib_table[idx].type, num, 1);
		else
			getMIB(mib, mib_table[idx].id, mib_table[idx].type, num, 1);
		break;

	case 2: // set
		if ( !apmib_init()) {
			printf("Initialize AP MIB failed!\n");
			return -1;
		}
		idx = searchMIB(mib);
		if ( idx == -1 ) {
			showHelp();
			showAllMibName();
			return -1;
		}
		if ( valNum < 1) {
			showHelp();
			return -1;
		}
		if (config_area == 1)
			setMIB(mib, hwmib_table[idx].id, hwmib_table[idx].type, hwmib_table[idx].size, valNum, value);
		else
			setMIB(mib, mib_table[idx].id, mib_table[idx].type, mib_table[idx].size, valNum, value);
		break;

	case 3: // all
		dumpAll();
		break;
	}

	return 0;
}

//////////////////////////////////////////////////////////////////////////////////
static int writeDefault(void)
{	
#ifdef __mips__
	void *ptr;
	
	if ((ptr=apmib_hwconf()) == NULL)
	{
		resetHWDefault();
	}
	else	
		free(ptr);
	
	if ((ptr=apmib_dsconf()) == NULL)
	{
		resetDFDefault();
	}
	else	
		free(ptr);
#endif	
	
	resetDFDefault();
	resetDefault();
	return 0;
}

///////////////////////////////////////////////////////////////////////////////
static int flash_read(char *buf, int offset, int len)
{
	int fh;
	int ok=1;

#ifdef __mips__
	fh = open(FLASH_DEVICE_NAME, O_RDWR);
#endif

#ifdef __i386__
	fh = open(FLASH_DEVICE_NAME, O_RDONLY);
#endif
	if ( fh == -1 )
		return 0;

	lseek(fh, offset, SEEK_SET);

	if ( read(fh, buf, len) != len)
		ok = 0;

	close(fh);

	return ok;
}

///////////////////////////////////////////////////////////////////////////////
static int searchMIB(char *token)
{
	int idx = 0;
	char tmpBuf[100];

	if (!memcmp(token, "HW_", 3))
		config_area = 1;
	else if (!memcmp(token, "DEF_", 4)) {
		config_area = 2;
		strcpy(tmpBuf, &token[4]);
		strcpy(token, tmpBuf);
	}
	else
		config_area = 3;

	if ( config_area == 1 ) {
		while (hwmib_table[idx].id) {
			if ( !strcmp(hwmib_table[idx].name, token)) {
				return idx;
			}
			idx++;
		}
		return -1;
	}
	else {
		while (mib_table[idx].id) {
			if ( !strcmp(mib_table[idx].name, token)) {
				return idx;
			}
			idx++;
		}
		return -1;
	}
}

///////////////////////////////////////////////////////////////////////////////
static void getMIB(char *name, int id, TYPE_T type, int num, int array_separate)
{
	unsigned char array_val[200];
	unsigned long longVal;
	struct in_addr ia_val;
	void *value;
	char tmpBuf[1024]={0}, *format=NULL, *buf, tmp1[200];
	int int_val, i;
	int index=1, tbl=0;
	char mibName[100]={0};
#ifdef HOME_GATEWAY
	int j;
	char IpAddr[40];
	unsigned long indexVal;
	char Protocol[16][30] = {"80,3128,8000,8080,8081|TCP","25|TCP","119|TCP","110|TCP","443|TCP","21|TCP","1863|TCP","23|TCP","5190|TCP","389,522,1503,1720,1731|TCP","53|UDP","161,162|UDP","1723|TCP","1701|UDP","1:65535|TCP","1:65535|UDP"};
	char ALGArray[12][10]= {"amanda","egg","ftp","h323","irc","mms","quake3","talk","tftp","starcraft","msn","pptp"};
	char DOSArray[7][10]= {"nmap","xmastree","axmastree","nullscan","synrst","synfin","synouport"};
#endif
	if (num ==0)
		goto ret;

	if (config_area == 2)
		strcpy(mibName, "DEF_");
	strcat(mibName, name);

getval:
	buf = &tmpBuf[strlen(tmpBuf)];
	switch (type) {
	case BYTE_T:
		value = (void *)&int_val;
		format = DEC_FORMAT;
		break;
	case WORD_T:
		value = (void *)&int_val;
		format = DEC_FORMAT;
		break;
	case IA_T:
		value = (void *)&ia_val;
		format = STR_FORMAT;
		break;
	case BYTE5_T:
		value = (void *)array_val;
		format = BYTE5_FORMAT;
		break;
	case BYTE6_T:
		value = (void *)array_val;
		format = BYTE6_FORMAT;
		break;
	case BYTE13_T:
		value = (void *)array_val;
		format = BYTE13_FORMAT;
		break;
	case BYTE16_T:
		value = (void *)array_val;
		format = BYTE16_FORMAT;
		break;
	case STRING_T:
		value = (void *)array_val;
		format = STR_FORMAT;
		break;

	case BYTE_ARRAY_T:
		value = (void *)array_val;
		break;

	case DWORD_T:
		value = (void *)&longVal;
		format = LONGDEC_FORMAT;
		break;

#ifdef HOME_GATEWAY
	case INDEX_T://////////////////////////////////////////////////////////////
		value = (void *)&longVal;
		format = LONGDEC_FORMAT;
		break;
	case INDEXDOS_T://////////////////////////////////////////////////////////////
		value = (void *)&longVal;
		format = LONGDEC_FORMAT;
		break;
#endif
#ifdef WLAN_WPA
#if 0
	case BYTE32_T:
		value = (void *)array_val;
		format = HEX_FORMAT;
		break;
#endif
#endif

	case WLAC_ARRAY_T:
#ifdef WLAN_WDS
        case WDS_ARRAY_T:
#endif
#ifdef HOME_GATEWAY
	case VSER_ARRAY_T:///////////////////////////////////////////////////////////
	case DMZ_ARRAY_T:///////////////////////////////////////////////////////////
	case URLB_ARRAY_T:///////////////////////////////////////////////////////////
	case ACPC_ARRAY_T:///////////////////////////////////////////////////////////
	case SROUT_ARRAY_T:///////////////////////////////////////////////////////////
	case PORTFW_ARRAY_T:
	case IPFILTER_ARRAY_T:
	case PORTFILTER_ARRAY_T:
	case MACFILTER_ARRAY_T:
	case TRIGGERPORT_ARRAY_T:
#else
	case RSER_CLT_ARRAY_T:
	case RSER_USR_ARRAY_T:
#endif
		tbl = 1;
		value = (void *)array_val;
		array_val[0] = index++;
		break;
	default: printf("invalid mib!\n"); return;
	}

	if ( !APMIB_GET(id, value)) {
		printf("Get MIB failed!\n");
		return;
	}

	if ( type == IA_T )
		value = inet_ntoa(ia_val);

	if (type == BYTE_T || type == WORD_T)
		sprintf(buf, format, int_val);
	else if ( type == IA_T || type == STRING_T )
		sprintf(buf, format, value);
	else if (type == BYTE5_T) {
		sprintf(buf, format, array_val[0],array_val[1],array_val[2],
			array_val[3],array_val[4],array_val[5]);
		convert_lower(buf);
	}
	else if (type == BYTE6_T ) {
		sprintf(buf, format, array_val[0],array_val[1],array_val[2],
			array_val[3],array_val[4],array_val[5],array_val[6]);
		convert_lower(buf);
	}
	else if (type == BYTE13_T) {
		sprintf(buf, format, array_val[0],array_val[1],array_val[2],
			array_val[3],array_val[4],array_val[5],array_val[6],
			array_val[7],array_val[8],array_val[9],array_val[10],
			array_val[11],array_val[12]);
		convert_lower(buf);
	}
	else if (type == BYTE16_T) {
		sprintf(buf, format, array_val[0],array_val[1],array_val[2],
			array_val[3],array_val[4],array_val[5],array_val[6],
			array_val[7],array_val[8],array_val[9],array_val[10],
			array_val[11],array_val[12],array_val[13],array_val[14],array_val[15]);
		convert_lower(buf);
	}
	else if (type == DWORD_T){ 
		sprintf(buf, format, longVal);}
#ifdef HOME_GATEWAY
	else if (type == INDEX_T)
		{ // Lance 2003.08.27
			int first=1;
			indexVal=0x0001;
			for( j=0; j<12 ;j++ )
			{
				if (longVal & indexVal) 
				{
					if (first == 0) 
						strcat(buf,",");
					strcat(buf,ALGArray[j]);
					first = 0;
				}
				indexVal *=2;
			}
		}
	else if (type == INDEXDOS_T)
		{ // Lance 2003.08.27
			int first=1;
			indexVal=0x0001;
			for( j=0; j<=6 ;j++ )
			{
				if (longVal & indexVal) 
				{
					if (first == 0) 
						strcat(buf,",");
					strcat(buf,DOSArray[j]);
					first = 0;
				}
				indexVal *=2;
			}
		}
#endif
#ifdef WLAN_WPA
#if 0
	else if (type == BYTE32_T) {
		buf[0] = '\0';
		for (i=0; i<32; i++) {
			sprintf(tmp1, format, array_val[i]);
			strcat(buf, tmp1);
		}
		convert_lower(buf);
	}
#endif	
#endif

#ifdef HOME_GATEWAY
////////////////////////////////////////////////////////////////////////////////////////////
	else if (type == VSER_ARRAY_T) {
		PORTFW_Tp pEntry=(PORTFW_Tp)array_val;
		sprintf(buf, PORTFW_FORMAT, inet_ntoa(*((struct in_addr*)pEntry->ipAddr)),
			 pEntry->fromPort, pEntry->protoType, pEntry->toPort);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ",\"%s\"", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
	else if (type == SROUT_ARRAY_T) {
		SROUT_Tp pEntry=(SROUT_Tp)array_val;
		sprintf(buf, "%s,", inet_ntoa(*((struct in_addr*)pEntry->ipAddr)));
		strcat(buf, inet_ntoa(*((struct in_addr*)pEntry->maskAddr)));
		sprintf(tmp1, ",%s,%d,%d", inet_ntoa(*((struct in_addr*)pEntry->gatewayAddr)), pEntry->hopCount, pEntry->interface);
		strcat(buf, tmp1);
	}
	else if (type == DMZ_ARRAY_T) {
		DMZ_Tp pEntry=(DMZ_Tp)array_val;
		sprintf(buf, "%d,%d,%s,", pEntry->type, pEntry->session, inet_ntoa(*((struct in_addr*)pEntry->pipAddr)));
		sprintf(tmp1, "%s", inet_ntoa(*((struct in_addr*)pEntry->cipAddr)));
		strcat(buf, tmp1);
	}
	else if (type == URLB_ARRAY_T) {
		URLB_Tp pEntry=(URLB_Tp)array_val;
			sprintf(buf, "%s", pEntry->URLBAddr);
	}
	else if (type == ACPC_ARRAY_T) 
	{
		static int index=0;
		int first=1;
		char tmp_buf[20]={0};
		
		ACPC_Tp pEntry=(ACPC_Tp)array_val;
		indexVal=0x0001;
		sprintf(IpAddr, "'|%s:", inet_ntoa(*((struct in_addr*)pEntry->sipAddr)));	
		strcat(IpAddr, inet_ntoa(*((struct in_addr*)pEntry->eipAddr)));
		strcat(IpAddr,"|");
		index++;
		for (j=0; j<=16; j++) 
		{
			if (pEntry->serindex & indexVal ) 
			{
				if( first != 1 )
				{
					sprintf(tmp_buf,"\nACPC_TBL%d_%d=",index,j);
					strcat(buf,tmp_buf);
				}
				else
					first = 0;
				strcat(buf,IpAddr);
				strcat(buf,Protocol[j]); 
				strcat(buf,"|'");
			}
				indexVal *= 2;
		}
		memset(tmp1,0,sizeof(tmp1));
		if ((pEntry->Port[0][0])!=0) 
		{
			if( first != 1 )
			{
				sprintf(tmp_buf,"\nACPC_TBL%d_%d=",index,j);
				strcat(buf,tmp_buf);
			}
			strcat(buf, IpAddr);

			for (j=0;j<5;j++) 
			{
				if ((pEntry->Port[j][0])!=0) 
				{
					if (pEntry->Port[j][0]!=pEntry->Port[j][1])
						sprintf(tmp1, "%d:%d", pEntry->Port[j][0], pEntry->Port[j][1]);
					else
						sprintf(tmp1, "%d", pEntry->Port[j][0]);
					if(j != 0)  
						strcat(buf, ",");
					strcat(buf, tmp1);
				}
			}
			if ((pEntry->Port[0][0])!=0) 
			{
				if (pEntry->protoType==1) strcat(buf, "|TCP|'");
				if (pEntry->protoType==2) strcat(buf, "|UDP|'");
				if (pEntry->protoType==3) strcat(buf, "|ALL|'");
			}
		}
	}
////////////////////////////////////////////////////////////////////////////////////////////
	else if (type == PORTFW_ARRAY_T) {
		PORTFW_Tp pEntry=(PORTFW_Tp)array_val;
		sprintf(buf, PORTFW_FORMAT, inet_ntoa(*((struct in_addr*)pEntry->ipAddr)),
			 pEntry->fromPort, pEntry->toPort, pEntry->protoType);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ",\"%s\"", pEntry->comment);
			strcat(buf, tmp1);
		}
	}

	else if (type == PORTFILTER_ARRAY_T) {
		PORTFILTER_Tp pEntry=(PORTFILTER_Tp)array_val;
		sprintf(buf, PORTFILTER_FORMAT,
			 pEntry->fromPort, pEntry->toPort, pEntry->protoType);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ",\"%s\"", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
	else if (type == IPFILTER_ARRAY_T) {
		IPFILTER_Tp pEntry=(IPFILTER_Tp)array_val;
		sprintf(buf, IPFILTER_FORMAT, inet_ntoa(*((struct in_addr*)pEntry->ipAddr)),
			 pEntry->protoType);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ",\"%s\"", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
	else if (type == MACFILTER_ARRAY_T) {
		MACFILTER_Tp pEntry=(MACFILTER_Tp)array_val;
		sprintf(buf, MACFILTER_COLON_FORMAT, pEntry->macAddr[0],pEntry->macAddr[1],pEntry->macAddr[2],
			 pEntry->macAddr[3],pEntry->macAddr[4],pEntry->macAddr[5]);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ",\"%s\"", pEntry->comment);
			strcat(buf, tmp1);
		}
	}

	else if (type == TRIGGERPORT_ARRAY_T) {
		TRIGGERPORT_Tp pEntry=(TRIGGERPORT_Tp)array_val;
/*		sprintf(buf, TRIGGERPORT_FORMAT,
			 pEntry->tri_fromPort, pEntry->tri_toPort, pEntry->tri_protoType,
			 pEntry->inc_fromPort[0][0],pEntry->inc_fromPort[0][1],pEntry->inc_fromPort[1][0],
			 pEntry->inc_fromPort[1][1],pEntry->inc_fromPort[2][0],pEntry->inc_fromPort[2][1],
			 pEntry->inc_fromPort[3][0],pEntry->inc_fromPort[3][1],pEntry->inc_fromPort[4][0],
			 pEntry->inc_fromPort[4][1], pEntry->inc_protoType);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ", %s", pEntry->comment);
			strcat(buf, tmp1);
*/
//------------------------------------------------------------------------------------------
		sprintf(buf, "%d-%d:",pEntry->tri_fromPort, pEntry->tri_toPort);
		if (pEntry->tri_protoType==1) strcat(buf, "TCP:");
		if (pEntry->tri_protoType==2) strcat(buf, "UDP:");
		if (pEntry->tri_protoType==3) strcat(buf, "ALL:");
		for (j=0;j<=5;j++) {
			if ((pEntry->inc_fromPort[j][0])!=0) {
				sprintf(tmp1, "%d-%d", pEntry->inc_fromPort[j][0], pEntry->inc_fromPort[j][1]);
				if(j != 0)  strcat(buf, ",");
				strcat(buf, tmp1);
			}
		}
		if (pEntry->inc_protoType==1) strcat(buf, ":TCP");
		if (pEntry->inc_protoType==2) strcat(buf, ":UDP");
		if (pEntry->inc_protoType==3) strcat(buf, ":ALL");
	}
#else
///////////////////////////////////////////////////////////////////////////////////////
	else if (type == RSER_CLT_ARRAY_T) {
		RSER_CLT_Tp pEntry=(RSER_CLT_Tp)array_val;
		sprintf(buf, "'%s,%s'", inet_ntoa(*((struct in_addr*)pEntry->ipAddr)),
			 pEntry->password);
	}
	else if (type == RSER_USR_ARRAY_T) {
		RSER_USR_Tp pEntry=(RSER_USR_Tp)array_val;
		sprintf(buf, "'%s,%s'", pEntry->username, pEntry->password);
	}
#endif
	else if (type == WLAC_ARRAY_T) {
		MACFILTER_Tp pEntry=(MACFILTER_Tp)array_val;
		sprintf(buf, MACFILTER_FORMAT, pEntry->macAddr[0],pEntry->macAddr[1],pEntry->macAddr[2],
			 pEntry->macAddr[3],pEntry->macAddr[4],pEntry->macAddr[5]);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ",\"%s\"", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
#ifdef WLAN_WDS
        else if (type == WDS_ARRAY_T) {
                WDS_Tp pEntry=(WDS_Tp)array_val;
                sprintf(buf, WDS_FORMAT, pEntry->macAddr[0],pEntry->macAddr[1],pEntry->macAddr[2],
			pEntry->macAddr[3],pEntry->macAddr[4],pEntry->macAddr[5]);
                if ( strlen(pEntry->comment) ) {
                        sprintf(tmp1, ",\"%s\"", pEntry->comment);
                        strcat(buf, tmp1);
                }
        }
#endif
	else if ( type == BYTE_ARRAY_T ) {
		if ( id == MIB_HW_TX_POWER ) {
			buf[0] = '\0';
			for (i=0; i<MAX_CHAN_NUM; i++) {
				sprintf(tmp1, "%d", *(((unsigned char *)value)+ i) );
				if ( i+1 != MAX_CHAN_NUM)
					strcat(tmp1, ",");
				strcat(buf, tmp1);
			}
		}
	}

	if (--num > 0) {
		if (!array_separate)
			strcat(tmpBuf, " ");
		else 
		{
			if (tbl)
				printf("%s%d=%s\n", mibName, index-1, tmpBuf);
			else
				printf("%s=%s\n", mibName, tmpBuf);
			tmpBuf[0] = '\0';
		}
		goto getval;
	}
ret:
	if (tbl) 
	{
		printf("%s%d=%s\n", mibName, index-1, tmpBuf);
	}
	else
		printf("%s=%s\n", mibName, tmpBuf);
}

///////////////////////////////////////////////////////////////////////////////
static void setMIB(char *name, int id, TYPE_T type, int len, int valNum, char **val)
{
	unsigned char key[100];
	struct in_addr ia_val;
	void *value;
	int int_val, i;
	unsigned long longVal;
	MACFILTER_T wlAc;

#ifdef WLAN_WDS
        WDS_T wds;
#endif

#ifdef HOME_GATEWAY
	PORTFW_T portFw;
	PORTFILTER_T portFilter;
	IPFILTER_T ipFilter;
	DMZ_T DMZ;//////////////////////////////////////////////////////////
	URLB_T URLB;//////////////////////////////////////////////////////////
	ACPC_T ACPC;//////////////////////////////////////////////////////////
	SROUT_T SROUT;//////////////////////////////////////////////////////////
	MACFILTER_T macFilter;
	TRIGGERPORT_T triggerPort;
	EDIT_T Edit;
#else
	RSER_USR_T rserUsr;
	RSER_CLT_T rserClt;
#endif
	int entryNum;

	switch (type) {
	case BYTE_T:
	case WORD_T:
		int_val = atoi(val[0]);
		value = (void *)&int_val;
		break;

	case IA_T:
		if ( !inet_aton(val[0], &ia_val) ) {
			printf("invalid internet address!\n");
			return;
		}
		value = (void *)&ia_val;
		break;

	case BYTE5_T:
		if ( strlen(val[0])!=10 || !string_to_hex(val[0], key, 10)) {
			printf("invalid value!\n");
			return;
		}
		value = (void *)key;
		break;

	case BYTE6_T:
		if ( strlen(val[0])!=12 || !string_to_hex(val[0], key, 12)) {
			printf("invalid value!\n");
			return;
		}
		value = (void *)key;
		break;

	case BYTE_ARRAY_T:
		if ( id != MIB_HW_TX_POWER )  {
			printf("invalid mib!\n");
			return;
		}
		for (i=0; i<MAX_CHAN_NUM; i++) {
			if ( !sscanf(val[i], "%d", &int_val) ) {
				printf("invalid value!\n");
				return;
			}
			key[i] = (unsigned char)int_val;
		}
		value = (void *)key;
		break;
	
	case DWORD_T:
		longVal = strtoul(val[0], NULL, 0);
		value = (void *)&longVal;
		break;

#ifdef HOME_GATEWAY
	case INDEX_T:////////////////////////////////////////////////////////////////
		longVal = strtoul(val[0], NULL, 0);
		value = (void *)&longVal;
		break;
	case INDEXDOS_T:////////////////////////////////////////////////////////////////
		longVal = strtoul(val[0], NULL, 0);
		value = (void *)&longVal;
		break;
#endif
#ifdef WLAN_WPA
#if 0
	case BYTE32_T:
		if ( strlen(val[0])!=64 || !string_to_hex(val[0], key, 64)) {
			printf("invalid value!\n");
			return;
		}
		value = (void *)key;
		break;
#endif		
#endif

#ifdef HOME_GATEWAY
//////////////////////////////////////////////////////////////////////////////////////////////
	case VSER_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_VSER_ADD;
			if ( valNum < 5 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&portFw.ipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
			portFw.fromPort = atoi(val[2]);
			portFw.protoType = atoi(val[3]);
			portFw.toPort = atoi(val[4]);
			if ( valNum > 5)
				strcpy(portFw.comment, val[5]);
			else
				portFw.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_VSER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_VSER_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&portFw) = (char)int_val;
			if ( !APMIB_GET(MIB_VSER, (void *)&portFw)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_VSER_DELALL;

		value = (void *)&portFw;
		break;


	case SROUT_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_SROUT_ADD;
			if ( valNum < 5 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&SROUT.ipAddr)) {
				printf("invalid destination ip address!\n");
				return;
			}
			if ( !inet_aton(val[2], (struct in_addr *)&SROUT.maskAddr)) {
				printf("invalid mask address!\n");
				return;
			}
			if ( !inet_aton(val[3], (struct in_addr *)&SROUT.gatewayAddr)) {
				printf("invalid gateway address!\n");
				return;
			}
			SROUT.hopCount = atoi(val[4]);
			SROUT.interface = atoi(val[5]);

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_SROUT_DEL;
			if ( valNum < 5 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_SROUT_NUM, (void *)&entryNum)) {
				printf("Get static routing entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&SROUT) = (char)int_val;
			if ( !APMIB_GET(MIB_SROUT, (void *)&SROUT)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_SROUT_DELALL;
		
		value = (void *)&SROUT;
		break;


	case DMZ_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_DMZ_ADD;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&DMZ.pipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
			if ( !inet_aton(val[2], (struct in_addr *)&DMZ.cipAddr)) {
				printf("invalid internet address!\n");
				return;
			}

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_DMZ_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_DMZ_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&DMZ) = (char)int_val;
			if ( !APMIB_GET(MIB_DMZ, (void *)&DMZ)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "edit")) {
			DMZ_T selDMZ;
			id = MIB_DMZ_EDIT;
			if ( valNum < 5 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_DMZ_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&selDMZ) = (char)int_val;
			if ( !APMIB_GET(MIB_DMZ, (void *)&selDMZ)) {
				printf("Get table entry error!");
				return;
			}
			Edit.selEntry = (char *)&selDMZ;
			//*****************************************
			Edit.editEntry = (char *)&DMZ;
			DMZ.type = atoi(val[2]);
			DMZ.session = atoi(val[3]);
			if ( !inet_aton(val[4], (struct in_addr *)&DMZ.pipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
			if ( !inet_aton(val[5], (struct in_addr *)&DMZ.cipAddr)) {
				printf("invalid internet address!\n");
				return;
			}

			value = (void *)&Edit;
			break;
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_DMZ_DELALL;
		
		value = (void *)&DMZ;
		break;

	case URLB_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_URLB_ADD;
			if ( valNum < 1 ) {
				printf("input argument is not enough!\n");
				return;
			}
				strcpy(URLB.URLBAddr, val[1]);

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_URLB_DEL;
			if ( valNum < 1 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_URLB_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&URLB) = (char)int_val;
			if ( !APMIB_GET(MIB_URLB, (void *)&URLB)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_URLB_DELALL;
		value = (void *)&URLB;
		break;
//---------------------------------------------------------------------------------
	case ACPC_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_ACPC_ADD;
			if ( valNum < 15 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&ACPC.sipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
			if ( !inet_aton(val[2], (struct in_addr *)&ACPC.eipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
			ACPC.Port[0][0] = atoi(val[3]);		ACPC.Port[0][1] = atoi(val[4]);
			ACPC.Port[1][0] = atoi(val[5]);		ACPC.Port[1][1] = atoi(val[6]);
			ACPC.Port[2][0] = atoi(val[7]);		ACPC.Port[2][1] = atoi(val[8]);
			ACPC.Port[3][0] = atoi(val[9]);		ACPC.Port[3][1] = atoi(val[10]);
			ACPC.Port[4][0] = atoi(val[11]);	ACPC.Port[4][1] = atoi(val[12]);
			ACPC.serindex = atol(val[13]);
			ACPC.protoType = atoi(val[14]);
			if ( valNum > 15)
				strcpy(ACPC.comment, val[15]);
			else
				ACPC.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_IPFILTER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_ACPC_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&ACPC) = (char)int_val;
			if ( !APMIB_GET(MIB_ACPC, (void *)&ACPC)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_ACPC_DELALL;
		
		value = (void *)&ACPC;
		break;

///////////////////////////////////////////////////////////////////////////////////////////////
	case PORTFW_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_PORTFW_ADD;
			if ( valNum < 5 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&portFw.ipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
			portFw.fromPort = atoi(val[2]);
			portFw.toPort = atoi(val[3]);
			portFw.protoType = atoi(val[4]);
			if ( valNum > 5)
				strcpy(portFw.comment, val[5]);
			else
				portFw.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_PORTFW_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_PORTFW_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&portFw) = (char)int_val;
			if ( !APMIB_GET(MIB_PORTFW, (void *)&portFw)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_PORTFW_DELALL;

		value = (void *)&portFw;
		break;

	case PORTFILTER_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_PORTFILTER_ADD;
			if ( valNum < 4 ) {
				printf("input argument is not enough!\n");
				return;
			}
			portFilter.fromPort = atoi(val[1]);
			portFilter.toPort = atoi(val[2]);
			portFilter.protoType = atoi(val[3]);
			if ( valNum > 4)
				strcpy(portFilter.comment, val[4]);
			else
				portFilter.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_PORTFILTER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_PORTFILTER_NUM, (void *)&entryNum)) {
				printf("Get port filter entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&portFilter) = (char)int_val;
			if ( !APMIB_GET(MIB_PORTFILTER, (void *)&portFilter)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_PORTFILTER_DELALL;

		value = (void *)&portFilter;
		break;

	case IPFILTER_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_IPFILTER_ADD;
			if ( valNum < 3 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&ipFilter.ipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
			ipFilter.protoType = atoi(val[2]);
			if ( valNum > 3)
				strcpy(ipFilter.comment, val[3]);
			else
				ipFilter.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_IPFILTER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_IPFILTER_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&ipFilter) = (char)int_val;
			if ( !APMIB_GET(MIB_IPFILTER, (void *)&ipFilter)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_IPFILTER_DELALL;
		
		value = (void *)&ipFilter;
		break;

	case MACFILTER_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_MACFILTER_ADD;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( strlen(val[1])!=12 || !string_to_hex(val[1], wlAc.macAddr, 12)) {
				printf("invalid value!\n");
				return;
			}

			if ( valNum > 2)
				strcpy(macFilter.comment, val[2]);
			else
				macFilter.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_MACFILTER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_MACFILTER_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&macFilter) = (char)int_val;
			if ( !APMIB_GET(MIB_MACFILTER, (void *)&macFilter)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_MACFILTER_DELALL;
		value = (void *)&macFilter;
		break;

	case TRIGGERPORT_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_TRIGGERPORT_ADD;
			if ( valNum < 15 ) {
				printf("input argument is not enough!\n");
				return;
			}
			
			triggerPort.tri_fromPort = atoi(val[1]);
			triggerPort.tri_toPort = atoi(val[2]);
			triggerPort.tri_protoType = atoi(val[3]);
			triggerPort.inc_fromPort[0][0] = atoi(val[4]);
			triggerPort.inc_fromPort[0][1] = atoi(val[5]);
			triggerPort.inc_fromPort[1][0] = atoi(val[6]);
			triggerPort.inc_fromPort[1][1] = atoi(val[7]);
			triggerPort.inc_fromPort[2][0] = atoi(val[8]);
			triggerPort.inc_fromPort[2][1] = atoi(val[9]);
			triggerPort.inc_fromPort[3][0] = atoi(val[10]);
			triggerPort.inc_fromPort[3][1] = atoi(val[11]);
			triggerPort.inc_fromPort[4][0] = atoi(val[12]);
			triggerPort.inc_fromPort[4][1] = atoi(val[13]);
			triggerPort.inc_toPort = atoi(val[14]);
			triggerPort.inc_protoType = atoi(val[15]);
			
////////////////////////////////////////////////////////////////////////////////////////////    
/*          if ( strlen(val[4])!=10 || !string_to_dec(val[4], triggerPort.inc_fromPort, 10)) {
                printf("invalid value!\n");
                return;
            }   
            triggerPort.inc_fromPort=strtoul(val[4],NULL,0);
*/          
//////////////////////////////////////////////////////////////////////////////////////////// 
                                                            
			if ( valNum > 15)
				strcpy(triggerPort.comment, val[16]);
			else
				triggerPort.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_TRIGGERPORT_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_TRIGGERPORT_NUM, (void *)&entryNum)) {
				printf("Get trigger-port entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&triggerPort) = (char)int_val;
			if ( !APMIB_GET(MIB_TRIGGERPORT, (void *)&triggerPort)) {
				printf("Get trigger-port table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_TRIGGERPORT_DELALL;

		value = (void *)&triggerPort;
		break;
/////////////////////////////////////////////////////////////////////////////////////////////
#else
	case RSER_USR_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_RSER_USR_ADD;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
				strcpy(rserUsr.username, val[1]);
				strcpy(rserUsr.password, val[2]);

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_RSER_USR_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_RSER_USR_NUM, (void *)&entryNum)) {
				printf("Get Radius Server User entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&rserUsr) = (char)int_val;
			if ( !APMIB_GET(MIB_RSER_USR, (void *)&rserUsr)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_RSER_USR_DELALL;

		value = (void *)&rserUsr;
		break;

	case RSER_CLT_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_RSER_CLT_ADD;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&rserClt.ipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
				strcpy(rserClt.password, val[2]);

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_RSER_CLT_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_RSER_CLT_NUM, (void *)&entryNum)) {
				printf("Get Radius Server Client entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&rserClt) = (char)int_val;
			if ( !APMIB_GET(MIB_RSER_CLT, (void *)&rserClt)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_RSER_CLT_DELALL;

		value = (void *)&rserClt;
		break;

#endif

	case WLAC_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_WLAN_AC_ADDR_ADD;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( strlen(val[1])!=12 || !string_to_hex(val[1], wlAc.macAddr, 12)) {
				printf("invalid value!\n");
				return;
			}

			if ( valNum > 2)
				strcpy(wlAc.comment, val[2]);
			else
				wlAc.comment[0] = '\0';
		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_WLAN_AC_ADDR_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_WLAN_AC_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&wlAc) = (char)int_val;
			if ( !APMIB_GET(MIB_WLAN_AC_ADDR, (void *)&wlAc)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_WLAN_AC_ADDR_DELALL;
		value = (void *)&wlAc;
		break;
#ifdef WLAN_WDS
        case WDS_ARRAY_T:
                if ( !strcmp(val[0], "add")) {
                        id = MIB_WLAN_WDS_ADD;
                        if ( valNum < 2 ) {
                                printf("input argument is not enough!\n");
                                return;
                        }
                        if ( strlen(val[1])!=12 || !string_to_hex(val[1], wds.macAddr, 12)) {
                                printf("invalid value!\n");
                                return;
                        }
                        if ( valNum > 2)
                                strcpy(wds.comment, val[2]);
                        else
                                wds.comment[0] = '\0';
	        }
                else if ( !strcmp(val[0], "del")) {
                        id = MIB_WLAN_WDS_DEL;
                        if ( valNum < 2 ) {
                                printf("input argument is not enough!\n");
                                return;
                        }
                        int_val = atoi(val[1]);
                        if ( !APMIB_GET(MIB_WLAN_WDS_NUM, (void *)&entryNum)) {
                                printf("Get wds number error!");
                                return;
                        }
                        if ( int_val > entryNum ) {
                                printf("Element number is too large!\n");
                                return;
                        }
                        *((char *)&wds) = (char)int_val;
                        if ( !APMIB_GET(MIB_WLAN_WDS, (void *)&wds)) {
                                printf("Get table entry error!");
                                return;
                        }
                }
                else if ( !strcmp(val[0], "delall"))
                        id = MIB_WLAN_WDS_DELALL;
                value = (void *)&wds;
                break;
#endif // WLAN_WDS
		
	case BYTE13_T:
		if ( strlen(val[0])!=26 || !string_to_hex(val[0], key, 26)) {
			printf("invalid value!\n");
			return;
		}
		value = (void *)key;
		break;

	case BYTE16_T:
		if ( strlen(val[0])!=32 || !string_to_hex(val[0], key, 32)) {
			printf("invalid value!\n");
			return;
		}
		value = (void *)key;
		break;

	case STRING_T:
		if ( strlen(val[0]) > len) {
			printf("string value too long!\n");
			return;
		}
		value = (void *)val[0];
		break;
	default: printf("invalid mib!\n"); return;
	}

	if ( !APMIB_SET(id, value))
		printf("set MIB failed!\n");

	if (config_area == 1)
		apmib_update(HW_SETTING);
	else if (config_area == 2)
		apmib_update(DEFAULT_SETTING);
	else
		apmib_update(CURRENT_SETTING);
}


////////////////////////////////////////////////////////////////////////////////
static void dumpAll(void)
{
	int idx=0, num;
	mib_table_entry_T *pTbl;

	if ( !apmib_init()) {
		printf("Initialize AP MIB failed!\n");
		return;
	}

	 config_area=0;

next_tbl:
	if (++config_area > 3)
	 	return;
	if (config_area == 1)
		pTbl = hwmib_table;
	else
		pTbl = mib_table;

	while (pTbl[idx].id) {
		if ( pTbl[idx].id == MIB_WLAN_AC_ADDR)
			APMIB_GET(MIB_WLAN_AC_NUM, (void *)&num);
#ifdef WLAN_WDS
                else if ( pTbl[idx].id == MIB_WLAN_WDS)
                        APMIB_GET(MIB_WLAN_WDS_NUM, (void *)&num);
#endif

#ifdef HOME_GATEWAY
//////////////////////////////////////////////////////////////////////////////////////////////
		else if ( pTbl[idx].id == MIB_VSER)
			APMIB_GET(MIB_VSER_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_DMZ)
			APMIB_GET(MIB_DMZ_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_URLB)
			APMIB_GET(MIB_URLB_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_ACPC)
			APMIB_GET(MIB_ACPC_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_SROUT)
			APMIB_GET(MIB_SROUT_NUM, (void *)&num);
//////////////////////////////////////////////////////////////////////////////////////////////
		else if ( pTbl[idx].id == MIB_PORTFW)
			APMIB_GET(MIB_PORTFW_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_PORTFILTER)
			APMIB_GET(MIB_PORTFILTER_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_IPFILTER)
			APMIB_GET(MIB_IPFILTER_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_MACFILTER)
			APMIB_GET(MIB_MACFILTER_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_TRIGGERPORT)
			APMIB_GET(MIB_TRIGGERPORT_NUM, (void *)&num);
#else
		else if ( pTbl[idx].id == MIB_RSER_USR)
			APMIB_GET(MIB_RSER_USR_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_RSER_CLT)
			APMIB_GET(MIB_RSER_CLT_NUM, (void *)&num);

#endif
		else
			num = 1;
		if (num >0) {
			getMIB(pTbl[idx].name, pTbl[idx].id,
						pTbl[idx].type, num, 1);
		}
		idx++;
	}
	idx = 0;
	goto next_tbl;
}

//////////////////////////////////////////////////////////////////////////////////
static void showHelp(void)
{
	printf("Usage: flash cmd\n");
	printf("option:\n");
	printf("cmd:\n");
	printf("      default -- write flash parameters to default.\n");
	printf("      get mib-name -- get a specific mib from flash memory.\n");
	printf("      set mib-name mib-value -- set a specific mib into flash memory.\n");
	printf("      all -- dump all flash parameters.\n");
	printf("      reset -- reset current setting to default.\n\n");
}


//////////////////////////////////////////////////////////////////////////////////
static void showAllMibName(void)
{
	int idx=0;
	
	if (config_area == 1) {
		printf("H/W mib-name:\n");
		while (hwmib_table[idx].id) {
			printf("%s\n", mib_table[idx].name);
			idx++;
		}
	}
	else {
		printf("S/W mib-name:\n");
		while (mib_table[idx].id) {
			printf("%s\n", mib_table[idx].name);
			idx++;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////
static void showSetACHelp(void)
{
	printf("flash set WLAN_MACAC_ADDR cmd\n");
	printf("cmd:\n");
	printf("      add mac-addr, comment -- append a filter mac address.\n");
	printf("      del mac-addr -- delete a filter mac address.\n");
	printf("      delall -- delete all filter mac address.\n");
}

#ifdef WLAN_WDS
///////////////////////////////////////////////////////////////////////////////////
static void showSetWdsHelp(void)
{
        printf("flash set WLAN_WDS cmd\n");
        printf("cmd:\n");
        printf("      add mac-addr comment -- append a WDS mac address.\n");
        printf("      del entry-number -- delete a WDS entry.\n");
        printf("      delall -- delete all WDS mac address.\n");
}
#endif

#ifdef HOME_GATEWAY
///////////////////////////////////////////////////////////////////////////////////
static void showSetPortFwHelp(void)
{
	printf("flash set PORTFW_TBL cmd\n");
	printf("cmd:\n");
	printf("      add ip from-port to-port protocol comment -- add a filter.\n");
	printf("      del entry-number -- delete a filter.\n");
	printf("      delall -- delete all filter.\n");
}


///////////////////////////////////////////////////////////////////////////////////
static void showSetPortFilterHelp(void)
{
	printf("flash set PORTFILTER_TBL cmd\n");
	printf("cmd:\n");
	printf("      add from-port to-port protocol comment -- add a filter.\n");
	printf("      del entry-number -- delete a filter.\n");
	printf("      delall -- delete all filter.\n");
}

///////////////////////////////////////////////////////////////////////////////////
static void showSetIpFilterHelp(void)
{
	printf("flash set IPFILTER_TBL cmd\n");
	printf("cmd:\n");
	printf("      add ip protocol comment -- add a filter.\n");
	printf("      del entry-number -- delete a filter.\n");
	printf("      delall -- delete all filter.\n");
}

///////////////////////////////////////////////////////////////////////////////////
static void showSetMacFilterHelp(void)
{
	printf("flash set MACFILTER_TBL cmd\n");
	printf("cmd:\n");
	printf("      add mac-addr comment -- add a filter.\n");
	printf("      del entry-number -- delete a filter.\n");
	printf("      delall -- delete all filter.\n");
}
///////////////////////////////////////////////////////////////////////////////////
static void showSetTriggerPortHelp(void)
{
	printf("flash set TRIGGER_PORT cmd\n");
	printf("cmd:\n");
	printf("   add trigger-from trigger-to trigger-proto incoming-from incoming-to incoming-proto comment -- add a trigger-port.\n");
	printf("   del entry-number -- delete a trigger-port.\n");
	printf("   delall -- delete all trigger-port.\n");
}
#endif


#ifdef PARSE_TXT_FILE
////////////////////////////////////////////////////////////////////////////////
static int parseTxtConfig(char *filename, APMIB_Tp pConfig)
{
	char line[100], value[100];
	FILE *fp;
	int id;

	fp = fopen(filename, "r");
	if ( fp == NULL )
		return -1;

	acNum = 0;
#ifdef WLAN_WDS
        wdsNum = 0;
#endif

#ifdef HOME_GATEWAY
	portFilterNum = ipFilterNum = macFilterNum = portFwNum = vserNum = DMZNum = URLBNum = ACPCNum = sroutNum = 0;
#else
	rserUsrNum = rserCltNum = 0;
#endif

	while ( fgets(line, 100, fp) ) {
		id = getToken(line, value);
		if ( id == 0 )
			continue;
		if ( set_mib(pConfig, id, value) < 0) {
			printf("Parse MIB [%d] error!\n", id );
			fclose(fp);
			return -1;
		}
	}

	fclose(fp);
	return 0;
}

////////////////////////////////////////////////////////////////////////////////
static int getToken(char *line, char *value)
{
	char *ptr=line, *p1;
	char token[100]={0};
	int len=0, idx;

	if ( *ptr == ';' )	// comments
		return 0;

	// get token
	while (*ptr && *ptr!=EOL) {
		if ( *ptr == '=' ) {
			memcpy(token, line, len);

			// delete ending space
			for (idx=len-1; idx>=0; idx--) {
				if (token[idx]!= SPACE )
					break;
			}
			token[idx+1] = '\0';
			ptr++;
			break;
		}
		ptr++;
		len++;
	}
	if ( !token[0] )
		return 0;

	// get value
	len=0;
	while (*ptr == SPACE ) ptr++; // delete space

	p1 = ptr;
	while ( *ptr && *ptr!=EOL) {
		ptr++;
		len++;
	}
	memcpy(value, p1, len );
	value[len] = '\0';

	idx = 0;
	while (mib_table[idx].id) {
		if (!strcmp(mib_table[idx].name, token))
			return mib_table[idx].id;
		idx++;
	}
	return 0;
}


////////////////////////////////////////////////////////////////////////////////
static int set_mib(APMIB_Tp pMib, int id, void *value)
{
	unsigned char key[100];
	char *p1, *p2;
#ifdef HOME_GATEWAY
	char *p3, *p4, *p5;
#endif
	struct in_addr inAddr, inAddr1, inAddr2;
	int i;
	MACFILTER_Tp pWlAc;
#ifdef WLAN_WDS
        WDS_Tp pWds;
#endif

#ifdef HOME_GATEWAY
	PORTFW_Tp pPortFw;
	PORTFILTER_Tp pPortFilter;
	IPFILTER_Tp pIpFilter;
	DMZ_Tp pDMZ;//////////////////////////////////////////////////////
	ACPC_Tp pACPC;//////////////////////////////////////////////////////
	URLB_Tp pURLB;//////////////////////////////////////////////////////
	SROUT_Tp pSROUT;//////////////////////////////////////////////////////
	MACFILTER_Tp pMacFilter;
#else
	RSER_USR_T pRserUsr;
	RSER_CLT_T pRserClt;
#endif


	for (i=0; mib_table[i].id; i++) {
		if ( mib_table[i].id == id )
			break;
	}
	if ( mib_table[i].id == 0 )
		return -1;

	switch (mib_table[i].type) {
	case BYTE_T:
		*((unsigned char *)(((long)pMib) + mib_table[i].offset)) = (unsigned char)atoi(value);
		break;

	case WORD_T:
		*((unsigned short *)(((long)pMib) + mib_table[i].offset)) = (unsigned short)atoi(value);
		break;

	case STRING_T:
		if ( strlen(value)+1 > mib_table[i].size )
			return 0;
		strcpy((char *)(((long)pMib) + mib_table[i].offset), (char *)value);
		break;

	case BYTE5_T:
		if (strlen(value)!=10 || !string_to_hex(value, key, 10))
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), key, 5);
		break;

	case BYTE6_T:
		if (strlen(value)!=12 || !string_to_hex(value, key, 12))
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), key, 6);
		break;

	case BYTE13_T:
		if (strlen(value)!=26 || !string_to_hex(value, key, 26))
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), key, 13);
		break;
	
	case BYTE16_T:
		if (strlen(value)!=32 || !string_to_hex(value, key, 32))
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), key, 16);
		break;
	
	case DWORD_T:
		*((unsigned long *)(((long)pMib) + mib_table[i].offset)) = value;
		break;

#ifdef HOME_GATEWAY
	case INDEX_T://////////////////////////////////////////////////////////////////////////////
		*((unsigned long *)(((long)pMib) + mib_table[i].offset)) = value;
		break;
	case INDEXDOS_T://////////////////////////////////////////////////////////////////////////////
		*((unsigned long *)(((long)pMib) + mib_table[i].offset)) = value;
		break;
#endif
#ifdef WLAN_WPA
#if 0
	case BYTE32_T:
		if (strlen(value)!=64 || !string_to_hex(value, key, 64))
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), key, 32);
		break;
#endif		
#endif

	case IA_T:
		if ( !inet_aton(value, &inAddr) )
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), (unsigned char *)&inAddr,  4);
		break;

	case WLAC_ARRAY_T:
		getVal2((char *)value, &p1, &p2);
		if (p1 == NULL) {
			printf("Invalid WLAC in argument!\n");
			break;
		}
		if (strlen(p1)!=12 || !string_to_hex(p1, key, 12))
			return -1;

		pWlAc = (MACFILTER_Tp)(((long)pMib)+mib_table[i].offset+acNum*sizeof(MACFILTER_T));
		memcpy(pWlAc->macAddr, key, 6);
		if (p2 != NULL )
			strcpy(pWlAc->comment, p2);
		acNum++;
		break;
#ifdef WLAN_WDS
        case WDS_ARRAY_T:
                getVal2((char *)value, &p1, &p2);
                if (p1 == NULL) {
                        printf("Invalid WDS in argument!\n");
                        break;
                }
                if (strlen(p1)!=12 || !string_to_hex(p1, key, 12))
                        return -1;
                pWds = (WDS_Tp)(((long)pMib)+mib_table[i].offset+wdsNum*sizeof(WDS_T));
                memcpy(pWds->macAddr, key, 6);
                if (p2 != NULL )
                        strcpy(pWds->comment, p2);
                wdsNum++;
                break;
#endif

#ifdef HOME_GATEWAY
	case MACFILTER_ARRAY_T:
		getVal2((char *)value, &p1, &p2);
		if (p1 == NULL) {
			printf("Invalid MACFILTER in argument!\n");
			break;
		}
		if (strlen(p1)!=12 || !string_to_hex(p1, key, 12))
			return -1;

		pMacFilter = (MACFILTER_Tp)(((long)pMib)+mib_table[i].offset+macFilterNum*sizeof(MACFILTER_T));
		memcpy(pMacFilter->macAddr, key, 6);
		if (p2 != NULL )
			strcpy(pMacFilter->comment, p2);
		macFilterNum++;
		break;
	
//////////////////////////////////////////////////////////////////////////////////////////////
	case VSER_ARRAY_T:
		getVal5((char *)value, &p1, &p2, &p3, &p4, &p5);
		if (p1 == NULL || p2 == NULL || p3 == NULL || p4 == NULL ) {
			printf("Invalid PORTFW arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;

		pPortFw = (PORTFW_Tp)(((long)pMib)+mib_table[i].offset+vserNum*sizeof(PORTFW_T));
		memcpy(pPortFw->ipAddr, (unsigned char *)&inAddr, 4);
		pPortFw->fromPort = (unsigned short)atoi(p2);
		pPortFw->toPort = (unsigned short)atoi(p3);
		pPortFw->protoType = (unsigned char)atoi(p4);
		if ( p5 )
			strcpy( pPortFw->comment, p5 );
		vserNum++;
		break;


	case DMZ_ARRAY_T:
		getVal2((char *)value, &p1, &p2);
		if (p1 == NULL || p2 == NULL) {
			printf("Invalid IPFILTER arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;
		if ( !inet_aton(p2, &inAddr1) )
			return -1;
		pDMZ = (DMZ_Tp)(((long)pMib)+mib_table[i].offset+DMZNum*sizeof(DMZ_T));
		memcpy(pDMZ->pipAddr, (unsigned char *)&inAddr, 4);
		memcpy(pDMZ->cipAddr, (unsigned char *)&inAddr1, 4);
		DMZNum++;
		break;

	case SROUT_ARRAY_T:
		getVal5((char *)value, &p1, &p2, &p3, &p4, &p5);
		if (p1 == NULL || p2 == NULL || p3 == NULL || p4 == NULL || p5 == NULL) {
			printf("Invalid Static Routing arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;
		if ( !inet_aton(p2, &inAddr1) )
			return -1;
		if ( !inet_aton(p3, &inAddr2) )
			return -1;
		pSROUT = (SROUT_Tp)(((long)pMib)+mib_table[i].offset+sroutNum*sizeof(SROUT_T));
		memcpy(pDMZ->ipAddr, (unsigned char *)&inAddr, 4);
		memcpy(pDMZ->maskAddr, (unsigned char *)&inAddr1, 4);
		memcpy(pDMZ->gatewayAddr, (unsigned char *)&inAddr2, 4);
		pSROUT->hopCount = (unsigned short)atoi(p4);
		pSROUT->interface = (unsigned short)atoi(p5);
		sroutNum++;
		break;
//----------------------------------------------------------------------------------------
	case URLB_ARRAY_T:
		getVal1((char *)value, &p1);
		if (p1 == NULL) {
			printf("Invalid URLB in argument!\n");
			break;
		}

		pURLB = (URLB_Tp)(((long)pMib)+mib_table[i].offset+URLBNum*sizeof(URLB_T));
			strcpy(pURLB->URLBAddr, p1);
		URLBNum++;
		break;
//------------------------------------------------------------------------------------------
	case ACPC_ARRAY_T:
		getVal15((char *)value, &p1, &p2, &p3, &p4, &p5, &p6, &p7, &p8, &p9, &p10, &p11, &p12, &p13, &p14, &p15);
		if (p1 == NULL || p2 == NULL) {
			printf("Invalid ACPC arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;
		pACPC = (ACPC_Tp)(((long)pMib)+mib_table[i].offset+ACPCNum*sizeof(ACPC_T));
		memcpy(pACPC->sipAddr, (unsigned char *)&inAddr, 4);
		if ( !inet_aton(p2, &inAddr) )
			return -1;
		memcpy(pACPC->eipAddr, (unsigned char *)&inAddr, 4);
		
		
	pACPC->Port[0][0] = (unsigned long)atoi(p3);	pACPC->Port[0][1] = (unsigned long)atoi(p4);
	pACPC->Port[1][0] = (unsigned long)atoi(p5);	pACPC->Port[1][1] = (unsigned long)atoi(p6);
	pACPC->Port[2][0] = (unsigned long)atoi(p7);	pACPC->Port[2][1] = (unsigned long)atoi(p8);
	pACPC->Port[3][0] = (unsigned long)atoi(p9);	pACPC->Port[3][1] = (unsigned long)atoi(p10);
	pACPC->Port[4][0] = (unsigned long)atoi(p11);	pACPC->Port[4][1] = (unsigned long)atoi(p12);

		pACPC->serindex = (unsigned char)atol(p13);
		pACPC->protoType = (unsigned char)atoi(p14);
		if ( p15 )
			strcpy( pACPC->comment, p15 );
		ACPCNum++;
		break;
//////////////////////////////////////////////////////////////////////////////////////////////

	case PORTFW_ARRAY_T:
		getVal5((char *)value, &p1, &p2, &p3, &p4, &p5);
		if (p1 == NULL || p2 == NULL || p3 == NULL || p4 == NULL ) {
			printf("Invalid PORTFW arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;

		pPortFw = (PORTFW_Tp)(((long)pMib)+mib_table[i].offset+portFwNum*sizeof(PORTFW_T));
		memcpy(pPortFw->ipAddr, (unsigned char *)&inAddr, 4);
		pPortFw->fromPort = (unsigned short)atoi(p2);
		pPortFw->toPort = (unsigned short)atoi(p3);
		pPortFw->protoType = (unsigned char)atoi(p4);
		if ( p5 )
			strcpy( pPortFw->comment, p5 );
		portFwNum++;
		break;

	case IPFILTER_ARRAY_T:
		getVal3((char *)value, &p1, &p2, &p3);
		if (p1 == NULL || p2 == NULL) {
			printf("Invalid IPFILTER arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;
		pIpFilter = (IPFILTER_Tp)(((long)pMib)+mib_table[i].offset+ipFilterNum*sizeof(IPFILTER_T));
		memcpy(pIpFilter->ipAddr, (unsigned char *)&inAddr, 4);
		pIpFilter->protoType = (unsigned char)atoi(p2);
		if ( p3 )
			strcpy( pIpFilter->comment, p3 );
		ipFilterNum++;
		break;

	case PORTFILTER_ARRAY_T:
		getVal4((char *)value, &p1, &p2, &p3, &p4);
		if (p1 == NULL || p2 == NULL || p3 == NULL) {
			printf("Invalid PORTFILTER arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;
		pPortFilter = (PORTFILTER_Tp)(((long)pMib)+mib_table[i].offset+portFilterNum*sizeof(PORTFILTER_T));
		pPortFilter->fromPort = (unsigned short)atoi(p1);
		pPortFilter->toPort = (unsigned short)atoi(p2);
		pPortFilter->protoType = (unsigned char)atoi(p3);
		if ( p4 )
			strcpy( pPortFilter->comment, p4 );
		portFilterNum++;
		break;
///////////////////////////////////////////////////////////////////////////////////////////
#else
	case RSER_CLT_ARRAY_T:
		getVal2((char *)value, &p1, &p2);
		if (p1 == NULL || p2 == NULL) {
			printf("Invalid RADIUS Server Client arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;

		pRserClt = (RSER_CLT_Tp)(((long)pMib)+mib_table[i].offset+rserCltNum*sizeof(RSER_CLT_T));
		memcpy(pRserClt->ipAddr, (unsigned char *)&inAddr, 4);
		strcpy( pRserClt->password, p2 );
		rserCltNum++;
		break;
		
	case RSER_USR_ARRAY_T:
		getVal2((char *)value, &p1, &p2);
		if (p1 == NULL || p2 == NULL) {
			printf("Invalid RADIUS Server User arguments!\n");
			break;
		}
		pRserusr = (RSER_USR_Tp)(((long)pMib)+mib_table[i].offset+rserUsrNum*sizeof(RSER_USR_T));
		strcpy( pRserUsr->username, p1 );
		strcpy( pRserUsr->password, p2 );
		rserUsrNum++;
		break;
#endif

	default:
		return -1;
	}
	return 0;
}

////////////////////////////////////////////////////////////////////////////////
static void getVal1(char *value, char **p1)
{
	value = getVal(value, p1);
}
//////////////////////////////////////////////////////////////////////////////////
static void getVal2(char *value, char **p1, char **p2)
{
	value = getVal(value, p1);
	if ( value )
		getVal(value, p2);
	else
		*p2 = NULL;
}

#ifdef HOME_GATEWAY
////////////////////////////////////////////////////////////////////////////////
static void getVal3(char *value, char **p1, char **p2, char **p3)
{
	*p1 = *p2 = *p3 = NULL;

	value = getVal(value, p1);
	if ( !value )
		return;
	value = getVal(value, p2);
	if ( !value )
		return;
	getVal(value, p3);
}

////////////////////////////////////////////////////////////////////////////////
static void getVal4(char *value, char **p1, char **p2, char **p3, char **p4)
{
	*p1 = *p2 = *p3 = *p4 = NULL;

	value = getVal(value, p1);
	if ( !value )
		return;
	value = getVal(value, p2);
	if ( !value )
		return;
	value = getVal(value, p3);
	if ( !value )
		return;
	getVal(value, p4);
}
///////////////////////////////////////////////////////////////////////////////////
static void getVal5(char *value, char **p1, char **p2, char **p3, char **p4, char **p5)
{
	*p1 = *p2 = *p3 = *p4 = *p5 = NULL;

	value = getVal(value, p1);
	if ( !value )
		return;
	value = getVal(value, p2);
	if ( !value )
		return;
	value = getVal(value, p3);
	if ( !value )
		return;
	value = getVal(value, p4);
	if ( !value )
		return;
	getVal(value, p5);
}

////////////////////////////////////////////////////////////////////////////////
/*static void getVal5(char *value, char **p1, char **p2, char **p3, char **p4, char **p5)
{
	*p1 = *p2 = *p3 = *p4 = *p5 = NULL;

	value = getVal(value, p1);
	if ( !value )
		return;
	value = getVal(value, p2);
	if ( !value )
		return;
	value = getVal(value, p3);
	if ( !value )
		return;
	value = getVal(value, p4);
	if ( !value )
		return;
	getVal(value, p5);
}*/
////////////////////////////////////////////////////////////////////////////////
static void getVal15(char *value, char **p1, char **p2, char **p3, char **p4, char **p5, char **p6, char **p7, char **p8, char **p9, char **p10, char **p11, char **p12, char **p13, char **p14, char **p15)
{
	*p1 = *p2 = *p3 = *p4 = *p5 = *p6 = *p7 = *p8 = *p9 = *p10= *p11 = *p12 = *p13 = *p14 = *p15 = NULL;

	value = getVal(value, p1);
	if ( !value )
		return;
	value = getVal(value, p2);
	if ( !value )
		return;
	value = getVal(value, p3);
	if ( !value )
		return;
	value = getVal(value, p4);
	if ( !value )
		return;
	value = getVal(value, p5);
	if ( !value )
		return;
	value = getVal(value, p6);
	if ( !value )
		return;
	value = getVal(value, p7);
	if ( !value )
		return;
	value = getVal(value, p8);
	if ( !value )
		return;
	value = getVal(value, p9);
	if ( !value )
		return;
	value = getVal(value, p10);
	if ( !value )
		return;
	value = getVal(value, p11);
	if ( !value )
		return;
	value = getVal(value, p12);
	if ( !value )
		return;
	value = getVal(value, p13);
	if ( !value )
		return;
	value = getVal(value, p14);
	if ( !value )
		return;
	getVal(value, p15);
}
#endif // HOME_GATEWAY

#endif // PARSE_TXT_FILE

////////////////////////////////////////////////////////////////////////////////
static int getdir(char *fullname, char *path, int loop)
{
	char tmpBuf[100], *p, *p1;

	strcpy(tmpBuf, fullname);
	path[0] = '\0';

	p1 = tmpBuf;
	while (1) {
		if ((p=strchr(p1, '/'))) {
			if (--loop == 0) {
				*p = '\0';
				strcpy(path, tmpBuf);
				return 0;
			}
			p1 = ++p;
		}
		else
			break;
	}
	return -1;
}

////////////////////////////////////////////////////////////////////////////////
static int read_flash_webpage(char *prefix, char *webfile)
{
	WEB_HEADER_T header;
	char *buf, tmpFile[100], tmpFile1[100], tmpBuf[100];
	int fh=0, i, loop, size;
	FILE_ENTRY_Tp pEntry;
	struct stat sbuf;
	char *file;

	if (webfile[0])
		file = webfile;
	else
		file = NULL;

	if (!file) {
		if ( flash_read((char *)&header, WEB_PAGE_OFFSET, sizeof(header)) == 0) {
			printf("Read web header failed!\n");
			return -1;
		}
	}
	else {
		if ((fh = open(file, O_RDONLY)) < 0) {
			printf("Can't open file %s\n", file);
			return -1;
		}
		lseek(fh, 0L, SEEK_SET);
		if (read(fh, &header, sizeof(header)) != sizeof(header)) {
			printf("Read web header failed %s!\n", file);
			close(fh);
			return -1;
		}
	}
//#ifndef _LITTLE_ENDIAN_
	header.len = DWORD_SWAP(header.len);
//#endif

	if (memcmp(header.signature, WEB_HEADER, SIGNATURE_LEN)) {
		printf("Invalid web image!\n");
		return -1;
	}

	buf = malloc(header.len);
	if (buf == NULL) {
		sprintf(tmpBuf, "Allocate buffer failed %ld!\n", header.len);
		printf(tmpBuf);
		return -1;
	}

	if (!file) {
		if ( flash_read(buf, WEB_PAGE_OFFSET+sizeof(header), header.len) == 0) {
			printf("Read web image failed!\n");
			return -1;
		}
	}
	else {
		if (read(fh, buf, header.len) != header.len) {
			printf("Read web image failed!\n");
			return -1;
		}
		close(fh);
	}

	if ( !CHECKSUM_OK(buf, header.len) ) {
		printf("Web image invalid!\n");
		free(buf);
		return -1;
	}
	// save to a file
	strcpy(tmpFile, "flashweb.gz");
	fh = open(tmpFile, O_RDWR|O_CREAT|O_TRUNC);
	if (fh == -1) {
		printf("Create output file error %s!\n", tmpFile );
		return -1;
	}
	if ( write(fh, buf, header.len-1) != header.len -1) {
		printf("write file error %s!\n", tmpFile);
		return -1;
	}
	close(fh);
	sync();

	// decompress file
	sprintf(tmpFile1, "%sXXXXXX", tmpFile);
	mkstemp(tmpFile1);

	sprintf(tmpBuf, "gunzip -c %s > %s", tmpFile, tmpFile1);
	system(tmpBuf);

	unlink(tmpFile);
	sync();

	if (stat(tmpFile1, &sbuf) != 0) {
		printf("Stat file error %s!\n", tmpFile1);
		return -1;
	}
	if (sbuf.st_size < sizeof(FILE_ENTRY_T) ) {
		sprintf(tmpBuf, "Invalid decompress file size %ld!\n", sbuf.st_size);
		printf(tmpBuf);
		unlink(tmpFile1);
		return -1;
	}

	buf = malloc(sbuf.st_size);
	if (buf == NULL) {
		sprintf(tmpBuf,"Allocate buffer failed %ld!\n", sbuf.st_size);
		printf(tmpBuf);
		return -1;
	}
	if ((fh = open(tmpFile1, O_RDONLY)) < 0) {
		printf("Can't open file %s\n", tmpFile1);
		return -1;
	}
	lseek(fh, 0L, SEEK_SET);
	if ( read(fh, buf, sbuf.st_size) != sbuf.st_size) {
		printf("Read file error %ld!\n", sbuf.st_size);
		return -1;
	}
	close(fh);
	unlink(tmpFile1);
	sync();
	size = sbuf.st_size;
	for (i=0; i<size; ) {
		pEntry = (FILE_ENTRY_Tp)&buf[i];

//#ifndef _LITTLE_ENDIAN_
		pEntry->size = DWORD_SWAP(pEntry->size);
//#endif

		strcpy(tmpFile, prefix);
		strcat(tmpFile, "/");
		strcat(tmpFile, pEntry->name);

		loop = 0;
		while (1) {
			if (getdir(tmpFile, tmpBuf, ++loop) < 0)
				break;
			if (tmpBuf[0] && stat(tmpBuf, &sbuf) < 0) { // not exist
 				if ( mkdir(tmpBuf, S_IREAD|S_IWRITE) < 0) {
					printf("Create directory %s failed!\n", tmpBuf);
					return -1;
				}
			}
		}

		fh = open(tmpFile, O_RDWR|O_CREAT|O_TRUNC);
		if (fh == -1) {
			printf("Create output file error %s!\n", tmpFile );
			return -1;
		}

		if ( write(fh, &buf[i+sizeof(FILE_ENTRY_T)], pEntry->size) != pEntry->size ) {
			printf("Write file error %s, len=%ld!\n", tmpFile, pEntry->size);
			return -1;
		}
		close(fh);
		// always set execuatble for script file
		chmod(tmpFile,  S_IXUSR);

		i += (pEntry->size + sizeof(FILE_ENTRY_T));
	}

	return 0;
}

#ifdef WLAN_WPA
////////////////////////////////////////////////////////////////////////////////
static void __inline__ WRITE_WPA_FILE(int fh, unsigned char *buf)
{
	if ( write(fh, buf, strlen(buf)) != strlen(buf) ) {
	        printf("Write WPA config file error!\n");
                close(fh);
                exit(1);
        }
}

static void generateWpaConf(char *outputFile)
{
	int fh, intVal, encrypt, enable1x, wep;
        unsigned char buf1[1024], buf2[1024];

        if ( !apmib_init()) {
                printf("Initialize AP MIB failed!\n");
                return;
        }
	
        fh = open(outputFile, O_RDWR|O_CREAT|O_TRUNC);
        if (fh == -1) {
                printf("Create WPA config file error!\n");
                return;
        }
	
        apmib_get( MIB_WLAN_ENCRYPT, (void *)&encrypt);
        sprintf(buf2, "encryption = %d\n", encrypt);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_SSID,  (void *)buf1);
        sprintf(buf2, "ssid = \"%s\"\n", buf1);
        WRITE_WPA_FILE(fh, buf2);
	
        apmib_get( MIB_WLAN_ENABLE_1X, (void *)&enable1x);
        sprintf(buf2, "enable1x = %d\n", enable1x);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_ENABLE_MAC_AUTH, (void *)&intVal);
		sprintf(buf2, "enableMacAuth = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_ENABLE_SUPP_NONWPA, (void *)&intVal);
	if (intVal)
	        apmib_get( MIB_WLAN_SUPP_NONWPA, (void *)&intVal);
        sprintf(buf2, "supportNonWpaClient = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_WEP, (void *)&wep);
        sprintf(buf2, "wepKey = %d\n", wep);
        WRITE_WPA_FILE(fh, buf2);
	
        if ( encrypt==1 && enable1x ) {
	        if (wep == 1) {
		        apmib_get( MIB_WLAN_WEP64_KEY1, (void *)buf1);
                        sprintf(buf2, "wepGroupKey = \"%02x%02x%02x%02x%02x\"\n", buf1[0],buf1[1],buf1[2],buf1[3],buf1[4]);
                }
                else {
                        apmib_get( MIB_WLAN_WEP128_KEY1, (void *)buf1);
                        sprintf(buf2, "wepGroupKey = \"%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\"\n",
                                buf1[0],buf1[1],buf1[2],buf1[3],buf1[4],
                                buf1[5],buf1[6],buf1[7],buf1[8],buf1[9],
                                buf1[10],buf1[11],buf1[12]);
                }
        }
        else
                strcpy(buf2, "wepGroupKey = \"\"\n");
        
		WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_WPA_AUTH, (void *)&intVal);
        sprintf(buf2, "authentication = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_WPA_CIPHER_SUITE, (void *)&intVal);
        sprintf(buf2, "unicastCipher = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_WPA_PSK_FORMAT, (void *)&intVal);
        if (intVal==0)
	                sprintf(buf2, "usePassphrase = 1\n");
        else
	                sprintf(buf2, "usePassphrase = 0\n");
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_WPA_PSK, (void *)buf1);
        sprintf(buf2, "psk = \"%s\"\n", buf1);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_WPA_GROUP_REKEY_TIME, (void *)&intVal);
        sprintf(buf2, "groupRekeyTime = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_RS_PORT, (void *)&intVal);
        sprintf(buf2, "rsPort = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

#ifdef _7207APB_
		// { Rex 11/18/2003 Modify for Internal Radius Server
        apmib_get( MIB_1X_MODE, (void *)&intVal);
//		printf("\nintVal=%d---\n",intVal);
		
//		if( buf1[0] == '\x0' )
		if( intVal == 0 )
		{
//				printf("\nGGGG1\n");
        	apmib_get( MIB_IP_ADDR, (void *)buf1);
        	sprintf(buf2, "rsIP = %s\n", inet_ntoa(*((struct in_addr *)buf1)));
			WRITE_WPA_FILE(fh, buf2);
			sprintf(buf2, "rsPassword = \"AP1234\"\n");
			WRITE_WPA_FILE(fh, buf2);
		}
		else 
#endif
#ifdef WLAN_WPA
		{
//				printf("\nGGGG2\n");
        	apmib_get( MIB_WLAN_RS_IP, (void *)buf1);
        	sprintf(buf2, "rsIP = %s\n", inet_ntoa(*((struct in_addr *)buf1)));
        	WRITE_WPA_FILE(fh, buf2);

        	apmib_get( MIB_WLAN_RS_PASSWORD, (void *)buf1);
        	sprintf(buf2, "rsPassword = \"%s\"\n", buf1);
        	WRITE_WPA_FILE(fh, buf2);
		}
#endif		
		// } Rex 11/18/2003 Modify for Internal Radius Server
        apmib_get( MIB_WLAN_RS_RETRY, (void *)&intVal);
        sprintf(buf2, "rsMaxReq = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_RS_INTERVAL_TIME, (void *)&intVal);
        sprintf(buf2, "rsAWhile = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_ACCOUNT_RS_ENABLED, (void *)&intVal);
        sprintf(buf2, "accountRsEnabled = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_ACCOUNT_RS_PORT, (void *)&intVal);
        sprintf(buf2, "accountRsPort = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_ACCOUNT_RS_IP, (void *)buf1);
        sprintf(buf2, "accountRsIP = %s\n", inet_ntoa(*((struct in_addr *)buf1)));
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_ACCOUNT_RS_PASSWORD, (void *)buf1);
        sprintf(buf2, "accountRsPassword = \"%s\"\n", buf1);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_ACCOUNT_UPDATE_ENABLED, (void *)&intVal);
        sprintf(buf2, "accountRsUpdateEnabled = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);
	
        apmib_get( MIB_WLAN_ACCOUNT_UPDATE_DELAY, (void *)&intVal);
        sprintf(buf2, "accountRsUpdateTime = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_ACCOUNT_RS_RETRY, (void *)&intVal);
        sprintf(buf2, "accountRsMaxReq = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        apmib_get( MIB_WLAN_ACCOUNT_RS_INTERVAL_TIME, (void *)&intVal);
        sprintf(buf2, "accountRsAWhile = %d\n", intVal);
        WRITE_WPA_FILE(fh, buf2);

        close(fh);
}
#endif // WLAN_WPA

