/*
 *  drivers/mtd/nand/spia.c
 *
 *  Copyright (C) 2000 Steven J. Hill (sjhill@realitydiluted.com)
 *
 *
 *	10-29-2001 TG	change to support hardwarespecific access
 *			to controllines	(due to change in nand.c)
 *			page_cache added
 *
 * $Id: k9f6408u0c.c,v 1.1.1.1 2004/01/05 03:08:58 lance Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Overview:
 *   This is a device driver for the NAND flash device found on the
 *   SPIA board which utilizes the Toshiba TC58V64AFT part. This is
 *   a 64Mibit (8MiB x 8 bits) NAND flash device.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <asm/io.h>
#include <asm/am5120/adm5120.h>

/*
 * MTD structure for K9F6408 on 5120 board
 */
static struct mtd_info *k9f6408_mtd = NULL;

/* 
 * control signals 
 */
#define NAND_RW_REG	0x0	//data register
#define NAND_SET_CEn	0x1	//CE# low
#define NAND_CLR_CEn	0x2	//CE# high
#define NAND_CLR_CLE	0x3	//CLE low
#define NAND_SET_CLE	0x4	//CLE high
#define NAND_CLR_ALE	0x5	//ALE low
#define NAND_SET_ALE	0x6	//ALE high
#define NAND_SET_SPn	0x7	//SP# low (use spare area)
#define NAND_CLR_SPn	0x8	//SP# high (do not use spare area)
#define NAND_SET_WPn	0x9	//WP# low
#define NAND_CLR_WPn	0xA	//WP# high
#define NAND_STS_REG	0xB	//Status register


/*
 * Module stuff
 */

static int k9f6408_io_base = KSEG1ADDR(SMEM1_BASE);

MODULE_PARM(k9f6408_io_base, "i");

/*
 * Define partitions for flash device
 */
const static struct mtd_partition partition_info[] = {
	{
		.name	= "K9F6408 flash partition 1",
		.offset	= 0,
		.size	= 5*1024*1024
	},
	{
		.name	= "K9F6408 flash partition 2",
		.offset	= 5*1024*1024,
		.size	= 3*1024*1024
	}
};
#define NUM_PARTITIONS 2


/* 
 *	hardware specific access to control-lines
*/
static void k9f6408_hwcontrol(struct mtd_info *mtd, int cmd){

    switch(cmd){

	case NAND_CTL_SETCLE: 
		(*(volatile unsigned char *) (k9f6408_io_base + NAND_SET_CLE)) =  0x01; 
		break;
	case NAND_CTL_CLRCLE: 
		(*(volatile unsigned char *) (k9f6408_io_base + NAND_CLR_CLE)) = 0x01; 
		break;

	case NAND_CTL_SETALE: 
		(*(volatile unsigned char *) (k9f6408_io_base + NAND_SET_ALE)) =  0x01; 
		break;
	case NAND_CTL_CLRALE: 
		(*(volatile unsigned char *) (k9f6408_io_base + NAND_CLR_ALE)) = 0x01; 
		break;

	case NAND_CTL_SETNCE: 
		(*(volatile unsigned char *) (k9f6408_io_base + NAND_SET_CEn)) = 0x01; 
		break;
	case NAND_CTL_CLRNCE: 
		(*(volatile unsigned char *) (k9f6408_io_base + NAND_CLR_CEn)) = 0x01; 
		break;
    }
}

/*
 * Main initialization routine
 */
int __init k9f6408_init (void)
{
	struct nand_chip *this;

	/* Allocate memory for MTD device structure and private data */
	k9f6408_mtd = kmalloc (sizeof(struct mtd_info) + sizeof (struct nand_chip),
				GFP_KERNEL);
	if (!k9f6408_mtd) {
		printk ("Unable to allocate K9F6408 NAND MTD device structure.\n");
		return -ENOMEM;
	}
	
	/* Get pointer to private data */
	this = (struct nand_chip *) (&k9f6408_mtd[1]);

	/* Initialize structures */
	memset((char *) k9f6408_mtd, 0, sizeof(struct mtd_info));
	memset((char *) this, 0, sizeof(struct nand_chip));

	/* Link the private data with the MTD structure */
	k9f6408_mtd->priv = this;

	/*
	 * Set GPIO Port E control register so that the pins are configured
	 * to be outputs for controlling the NAND flash.
	 */
	(*(volatile unsigned int *) (0xB1000200)) = 0x80; 	/* MPMC register; 8bit for SMEM1 */
	(*(volatile unsigned int *) (0xB2000064)) = 0x100;	/* enable NAND flash */
	(*(volatile unsigned int *) (0xB2000008)) = 0x1;	/* boot done */
	
	/* Set SPn */ 
	(*(volatile unsigned char *) (k9f6408_io_base + NAND_SET_SPn)) =  0x01; 
	/* Clear WPn */
	(*(volatile unsigned char *) (k9f6408_io_base + NAND_CLR_WPn)) =  0x01; 
	
	/* Set address of NAND IO lines */
	this->IO_ADDR_R = k9f6408_io_base;
	this->IO_ADDR_W = k9f6408_io_base;
	/* Set address of hardware control function */
	this->hwcontrol = k9f6408_hwcontrol;
	this->dev_ready = NULL;
	/* 10 us command delay time */
	this->chip_delay = 10;
	this->eccmode = NAND_ECC_SOFT;

	/* Scan to find existence of the device */
	if (nand_scan (k9f6408_mtd, 1)) {
		printk("Nand scan ERROR\n");
		kfree (k9f6408_mtd);
		return -EIO;
	}
	printk("Nand scan okay\n");

	/* Allocate memory for internal data buffer */
	this->data_buf = kmalloc (sizeof(u_char) * (k9f6408_mtd->oobblock + k9f6408_mtd->oobsize), GFP_KERNEL);
	if (!this->data_buf) {
		printk ("Unable to allocate NAND data buffer for K9F6408.\n");
		kfree (k9f6408_mtd);
		return -ENOMEM;
	}

	/* Register the partitions */
	add_mtd_partitions(k9f6408_mtd, partition_info, NUM_PARTITIONS);

	/* Return happy */
	return 0;
}
module_init(k9f6408_init);

/*
 * Clean up routine
 */
#ifdef MODULE
static void __exit k9f6408_cleanup (void)
{
	struct nand_chip *this = (struct nand_chip *) &k9f6408_mtd[1];

	/* Unregister the device */
	del_mtd_device (k9f6408_mtd);

	/* Free internal data buffer */
	kfree (this->data_buf);
	
	/* Free the MTD device structure */
	kfree (k9f6408_mtd);
}
module_exit(k9f6408_cleanup);
#endif

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jeanson Hung <jeansonh@admtek.com.tw>");
MODULE_DESCRIPTION("Board-specific glue layer for NAND flash on K9F6408 board");
