#ifndef _IP_CONNTRACK_TUPLE_H
#define _IP_CONNTRACK_TUPLE_H

/* A `tuple' is a structure containing the information to uniquely
  identify a connection.  ie. if two packets have the same tuple, they
  are in the same connection; if not, they are not.

  We divide the structure along "manipulatable" and
  "non-manipulatable" lines, for the benefit of the NAT code.
*/

/* The protocol-specific manipulable parts of the tuple: always in
   network order! */
union ip_conntrack_manip_proto
{
	/* Add other protocols here. */
	//u_int32_t all;
	u_int16_t all;
	u_int32_t allex; // Lance 2003.09.25

	struct {
		u_int16_t port;
	} tcp;
	struct {
		u_int16_t port;
	} udp;
	struct {
		u_int16_t id;
	} icmp;
	struct {
		u_int32_t key;  // Lance 2003.09.25
		//u_int16_t key;
	} gre;
};

/* The manipulable part of the tuple. */
struct ip_conntrack_manip
{
	u_int32_t ip;
	union ip_conntrack_manip_proto u;
};

/* This contains the information to distinguish a connection. */
struct ip_conntrack_tuple
{
	struct ip_conntrack_manip src;

	/* These are the parts of the tuple which are fixed. */
	struct {
		u_int32_t ip;
		union {
			/* Add other protocols here. */
			//u_int64_t all;
			u_int16_t all;
			u_int64_t allex; // Lance 2003.09.25

			struct {
				u_int16_t port;
			} tcp;
			struct {
				u_int16_t port;
			} udp;
			struct {
				u_int8_t type, code;
			} icmp;
			struct {
				u_int16_t protocol;
				u_int8_t version;
				u_int32_t key;
			} gre;
		} u;

		/* The protocol. */
		u_int16_t protonum;
	} dst;

	u_int16_t src_range; // Lance 2003.07.17
	u_int16_t dst_range; // Lance 2003.07.17
};

enum ip_conntrack_dir
{
	IP_CT_DIR_ORIGINAL,
	IP_CT_DIR_REPLY,
	IP_CT_DIR_MAX
};

#ifdef __KERNEL__

#define DUMP_TUPLE(tp)						\
DEBUGP("tuple %p: %u %u.%u.%u.%u:%u -> %u.%u.%u.%u:%u\n",	\
       (tp), (tp)->dst.protonum,				\
       NIPQUAD((tp)->src.ip), ntohl((tp)->src.u.all),		\
       NIPQUAD((tp)->dst.ip), ntohl((tp)->dst.u.all))

#define DUMP_TUPLE_RAW(x) 						\
	DEBUGP("tuple %p: %u %u.%u.%u.%u:0x%08x -> %u.%u.%u.%u:0x%08x\n",\
	(x), (x)->dst.protonum,						\
	NIPQUAD((x)->src.ip), ntohl((x)->src.u.all), 			\
	NIPQUAD((x)->dst.ip), ntohl((x)->dst.u.all))

#define CTINFO2DIR(ctinfo) ((ctinfo) >= IP_CT_IS_REPLY ? IP_CT_DIR_REPLY : IP_CT_DIR_ORIGINAL)

/* If we're the first tuple, it's the original dir. */
#define DIRECTION(h) ((enum ip_conntrack_dir)(&(h)->ctrack->tuplehash[1] == (h)))

/* Connections have two entries in the hash table: one for each way */
struct ip_conntrack_tuple_hash
{
	struct list_head list;

	struct ip_conntrack_tuple tuple;

	/* this == &ctrack->tuplehash[DIRECTION(this)]. */
	struct ip_conntrack *ctrack;
};

#endif /* __KERNEL__ */

static inline int ip_ct_tuple_src_equal(const struct ip_conntrack_tuple *t1,
				        const struct ip_conntrack_tuple *t2)
{
	return t1->src.ip == t2->src.ip
		&& t1->src.u.all == t2->src.u.all;
}

static inline int ip_ct_tuple_dst_equal(const struct ip_conntrack_tuple *t1,
				        const struct ip_conntrack_tuple *t2)
{
	return t1->dst.ip == t2->dst.ip
		&& t1->dst.u.all == t2->dst.u.all
		&& t1->dst.protonum == t2->dst.protonum;
}

static inline int ip_ct_tuple_equal(const struct ip_conntrack_tuple *t1,
				    const struct ip_conntrack_tuple *t2)
{
	return ip_ct_tuple_src_equal(t1, t2) && ip_ct_tuple_dst_equal(t1, t2);
}

static inline int ip_ct_tuple_mask_cmp(const struct ip_conntrack_tuple *t,
				       const struct ip_conntrack_tuple *tuple,
				       const struct ip_conntrack_tuple *mask)
{
    // Lance 2003.07.17
//  printk("cmp- %d %d\n",tuple->src_range,tuple->dst_range);

	if( (tuple->src_range && mask->src.u.all) || (tuple->dst_range && mask->dst.u.all) )
    {
	    int sret=1,dret=1,ret;
                         
//      printk("range cmp\n");

        if( tuple->src_range )
        {
	        if( (ntohs(tuple->src.u.all) <= ntohs(t->src.u.all)) && (ntohs(tuple->src_range) >= ntohs(t->src.u.all)) )
	            sret = 0;
        }
		else
            sret = (t->src.u.all ^ tuple->src.u.all) & mask->src.u.all;

//      printk("%d (%d-%d) sret= %d\n",ntohs(t->src.u.all),ntohs(tuple->src.u.all),ntohs(tuple->src_range),sret);

        if( tuple->dst_range )
        {
	        if( (ntohs(tuple->dst.u.all) <= ntohs(t->dst.u.all)) && (ntohs(tuple->dst_range) >= ntohs(t->dst.u.all)) )
	            dret = 0;
        }
        else
			dret = (t->dst.u.all ^ tuple->dst.u.all) & mask->dst.u.all;
//      printk("%d (%d-%d) dret= %d\n",ntohs(t->dst.u.all),ntohs(tuple->dst.u.all),ntohs(tuple->dst_range),dret);
      
        ret =  !(((t->src.ip ^ tuple->src.ip) & mask->src.ip)
				|| ((t->dst.ip ^ tuple->dst.ip) & mask->dst.ip)
				|| ( sret )
				|| ( dret )
				|| ((t->dst.protonum ^ tuple->dst.protonum)
				& mask->dst.protonum));
//      printk("return %d\n",ret);

        return ret;
	}                  

	return !(((t->src.ip ^ tuple->src.ip) & mask->src.ip)
		 || ((t->dst.ip ^ tuple->dst.ip) & mask->dst.ip)
		 || ((t->src.u.all ^ tuple->src.u.all) & mask->src.u.all)
		 || ((t->dst.u.all ^ tuple->dst.u.all) & mask->dst.u.all)
		 || ((t->dst.protonum ^ tuple->dst.protonum)
		     & mask->dst.protonum));
}

#endif /* _IP_CONNTRACK_TUPLE_H */
