#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/netdevice.h>
#include <linux/proc_fs.h>
#include <linux/etherdevice.h>
#include <linux/if_vlan.h>

#if defined(CONFIG_RALINK_RT3052_MP) || defined(CONFIG_RALINK_RT3052_MP2)

#ifdef MULTICAST_WORKAROUND_TABLE
#define MAX_MCAST_ENTRY	    64
#define AGEING_TIME	    5  //Unit: Sec
#endif /*MULTICAST_WORKAROUND_TABLE*/

#define MAC_ARG(x) ((u8*)(x))[0],((u8*)(x))[1],((u8*)(x))[2], \
	((u8*)(x))[3],((u8*)(x))[4],((u8*)(x))[5]

//#define MCAST_DEBUG

#ifdef MCAST_DEBUG
#define MCAST_PRINT(fmt, args...) printk(KERN_INFO fmt, ## args)
#else
#define MCAST_PRINT(fmt, args...) { }
#endif

#ifdef MULTICAST_WORKAROUND_TABLE
typedef struct {
	uint8_t	src_mac[6];
	uint8_t	dst_mac[6];
	uint32_t	valid;
	uint32_t	use_count;
	unsigned long ageout;
} mcast_entry;

static mcast_entry mcast_tbl[MAX_MCAST_ENTRY];

static unsigned int mcast_entry_num=0;
static unsigned int mcast_tx_cnt = 0;
static unsigned int mcast_rx_cnt = 0;
static unsigned int mcast_rx_drop_cnt = 0;

static struct proc_dir_entry *proc_mtrx;

int32_t inline mcast_entry_get(uint8_t *src_mac, uint8_t *dst_mac) 
{
	int i=0;

	for(i=0;i<MAX_MCAST_ENTRY;i++) {
		if(memcmp(mcast_tbl[i].src_mac,src_mac, 6)==0 &&
				memcmp(mcast_tbl[i].dst_mac, dst_mac, 6)==0 &&
				mcast_tbl[i].valid == 1) {
			return i;
		}
	}
	return -1;
}

int inline __add_mcast_entry(uint8_t *src_mac, uint8_t *dst_mac)
{
	int i=0;

	// use empty or ageout entry
	for(i=0;i<MAX_MCAST_ENTRY;i++) {
		if(mcast_tbl[i].valid==0 ||
				time_after(jiffies, mcast_tbl[i].ageout)) {

			if(mcast_tbl[i].valid==0)
				mcast_entry_num++;

			memcpy(mcast_tbl[i].src_mac, src_mac, 6);
			memcpy(mcast_tbl[i].dst_mac, dst_mac, 6);
			mcast_tbl[i].valid=1;
			mcast_tbl[i].use_count=1;
			mcast_tbl[i].ageout=jiffies + AGEING_TIME * HZ;
			return 1;
		}
	}

	MCAST_PRINT("RAETH: Multicast Table is FULL!!\n");
	return 0;
}

int inline mcast_entry_ins(uint8_t *src_mac, uint8_t *dst_mac) 
{
	int entry_num=0;

	if((entry_num = mcast_entry_get(src_mac, dst_mac)) >= 0) {
		mcast_tbl[entry_num].use_count++;
		MCAST_PRINT("%s: Update %0X:%0X:%0X:%0X:%0X:%0X's use_count=%d\n" \
				,__FUNCTION__, MAC_ARG(dst_mac), mcast_tbl[entry_num].use_count);
		return 1;
	} else { //if entry not found, create new entry.
		MCAST_PRINT("%s: Create new entry %0X:%0X:%0X:%0X:%0X:%0X\n", \
				__FUNCTION__, MAC_ARG(dst_mac));
		return __add_mcast_entry(src_mac,dst_mac);
	}
}

/*
 * Return:
 *	    0: entry found
 *	    1: entry not found
 */
int inline mcast_entry_del(uint8_t *src_mac, uint8_t *dst_mac) 
{
	int entry_num;

	if((entry_num = mcast_entry_get(src_mac, dst_mac)) >=0) {
		if((--mcast_tbl[entry_num].use_count)==0) {
			MCAST_PRINT("%s: %0X:%0X:%0X:%0X:%0X:%0X (entry_num=%d)\n", \
					__FUNCTION__, MAC_ARG(dst_mac), entry_num);
			mcast_tbl[entry_num].valid=0;
			mcast_entry_num--;
		}
		return 0;
	} else { 
		/* this multicast packet was not sent by myself, just ignore it */
		return 1;
	}
}
#endif /*MULTICAST_WORKAROUND_TABLE*/

/*
 *	Return 1 is this is multicast packet
 */
uint32_t inline is_multicast_pkt(uint8_t *mac)
{
	if(is_broadcast_ether_addr(mac))
		return 0;
	return is_multicast_ether_addr(mac);
}

/* 
 * Return
 *	    0: drop packet
 *	    1: continue
 */
int32_t mcast_rx(struct sk_buff * skb)
{
	struct vlan_ethhdr *eth=(struct vlan_ethhdr *)(skb->data-ETH_HLEN);
	/* if we do not send multicast packet before, 
	 * we don't need to check re-inject multicast packet.
	 */
#ifdef MULTICAST_WORKAROUND_TABLE
	if(mcast_entry_num == 0) {
		return 1;
	}
#endif /*MULTICAST_WORKAROUND_TABLE*/

	if(is_multicast_pkt(eth->h_dest)) {
		/*
		 * Perry: If we receive an multicast packet which VLAN ID is larger and equal to 0xff1, drop it.
		 *        Since the VLAN ID >= 0xff1 should not appear in normal network.
		 */
		if(eth->h_vlan_proto == htons(ETH_P_8021Q) && ((ntohs(eth->h_vlan_TCI) & 0xfff) >= 0xff1) )
			return 0;
#ifdef MULTICAST_WORKAROUND_TABLE
		mcast_rx_cnt++;
		MCAST_PRINT("%s: %0X:%0X:%0X:%0X:%0X:%0X\n", __FUNCTION__, \
				MAC_ARG(eth->h_dest));
		return mcast_entry_del(eth->h_source, eth->h_dest);
#endif /*MULTICAST_WORKAROUND_TABLE*/
	}

	return 1;
}

int32_t mcast_tx(struct sk_buff *skb)
{
	struct vlan_ethhdr *eth = (struct vlan_ethhdr *) skb->data;
	if(is_multicast_pkt(eth->h_dest)) {
		/*
		 * Perry: If this is multicast packet, use VLAN >= 0xff1 to send this packet.
		 * NOTE: 1 - ff1, 2 - ff2, 3 - ff3 should have the same port settings in switch.
		 */
		eth->h_vlan_TCI = htons((ntohs(eth->h_vlan_TCI)) | 0xff0);
#ifdef MULTICAST_WORKAROUND_TABLE
		mcast_tx_cnt++;
		MCAST_PRINT("%s: %0X:%0X:%0X:%0X:%0X:%0X\n", __FUNCTION__,\
				MAC_ARG(eth->h_dest));
		mcast_entry_ins(eth->h_source, eth->h_dest);
#endif /*MULTICAST_WORKAROUND_TABLE*/
	}
	return 1;
}

static int proc_read_mtrx(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len = 0;
	int i;

#ifdef MULTICAST_WORKAROUND_TABLE
	len = sprintf(page, "##### Driver Multicast TX/RX Info #####\n");
	len += sprintf(page + len, "mcast_entry_num:%d\n", mcast_entry_num);
	len += sprintf(page + len, "mcast_tx_cnt:%d mcast_rx_cnt:%d mcast_rx_drop_cnt:%d (clear each read)\n", mcast_tx_cnt, mcast_rx_cnt, mcast_rx_drop_cnt);
	mcast_tx_cnt = mcast_rx_cnt = mcast_rx_drop_cnt = 0;
	for(i = 0; i < MAX_MCAST_ENTRY; i++) {
		len += sprintf(page + len, "%2dth SRC:%02x:%02x:%02x:%02x:%02x:%02x DST:%02x:%02x:%02x:%02x:%02x:%02x v:%d use:%d\n",
			   	i, MAC_ARG(mcast_tbl[i].src_mac), MAC_ARG(mcast_tbl[i].dst_mac), mcast_tbl[i].valid, 
				mcast_tbl[i].use_count);
		if(len >= 2000)
			break;
	}
#else
	len = sprintf(page, "##### Driver Multicast TX/RX is not enable now #####\n");
#endif /*MULTICAST_WORKAROUND_TABLE*/
	return len;
}

void mtrx_init()
{
#ifdef MULTICAST_WORKAROUND_TABLE
	if((proc_mtrx = create_proc_entry("mtrx", 0, NULL))) {
		proc_mtrx->read_proc = (read_proc_t*)&proc_read_mtrx;
	}
#endif /*MULTICAST_WORKAROUND_TABLE*/
}

void mtrx_deinit()
{
#ifdef MULTICAST_WORKAROUND_TABLE
	remove_proc_entry("mtrx", NULL);
#endif /*MULTICAST_WORKAROUND_TABLE*/
}
#endif
