#include <stdlib.h>             
#include <stdio.h>             
#include <string.h>           
#include <sys/ioctl.h>
#include <fcntl.h>
#include <getopt.h>
#include <sys/stat.h>
#include <nvram.h>

#include <sys/mman.h>
#include <errno.h>

#include "image.h"

void show_usage(void)
{
	printf("fw_tool -f filename -p [0/1] -s -r\n");
	printf("-f: write given file into flash(firmware)\n");
	printf("-b: write given file into flash(bootloader)\n");
	printf("-p 0/1: write to kernel 0/1 partition\n");
	printf("-s: silent\n");
	printf("-r: reboot after write\n\n");
}


int check_image(unsigned char *buf, unsigned int flen)
{
	unsigned int data, checksum, len;
	image_header_t hdr;
	unsigned int tmp;

	if(!buf)
		return -1;

	memcpy(&hdr, buf, sizeof(image_header_t));

	if(ntohl(hdr.ih_magic) != IH_MAGIC) {
		printf("Magic is wrong.\n");
		return -1;
	}

	data = (unsigned int)&hdr;
	len  = sizeof(image_header_t);

	checksum = ntohl(hdr.ih_hcrc);
	hdr.ih_hcrc = 0;

	if((tmp = crc32(0, (char *)data, len)) != checksum) {
		printf("Header checksum is wrong.tmp:%x checksum:%x\n", tmp, checksum);
		return -1;
	}

	data = (unsigned int)buf + sizeof(image_header_t);
	len  = ntohl(hdr.ih_size);
	
	if(len + sizeof(image_header_t) != flen) {
		printf("Firmware size is wrong. len:%d flen:%d\n", len, flen);
		return -1;
	}

	if((tmp = crc32(0, (char *)data, len)) != ntohl(hdr.ih_dcrc)) {
		printf("Firmware checksum is wrong. tmp:%x checksum:%x\n", tmp, ntohl(hdr.ih_dcrc));
		return -1;
	}

	return 0;
}

int main(int argc, char *argv[])
{
	struct stat file_stat;
	int fp, type = 0;
	int flen, len, i, pass = -1, reboot = 0;
	unsigned int start_addr;
	int opt, part = 0, silent = 0, auto_part = 1;
	char options[] = "f:p:srb:?c";
	unsigned char *name = NULL;
	unsigned char *src;
	int confirm = 1;
	char buf[256];

	if(argc < 2) {
		show_usage();
		return 0;
	}

	while((opt = getopt (argc, argv, options)) != -1) {
		switch (opt) {
			case 'f':
				name = optarg;
				if(!name) {
					printf("No given filename.\n");
					return 0;
				}
				type = 0;
				break;
			case 'p':
				part = atoi(optarg);
				if(part != 0 && part != 1) {
					printf("partition number is wrong %d.\n", part);
					return 0;
				}
				auto_part = 0;
				break;
			case 's':
				silent = 1;
				break;
			case 'r':
				reboot = 1;
				break;
			case 'b':
				name = optarg;
				if(!name) {
					printf("No given filename.\n");
					return 0;
				}
				type = 1;
				break;
			case 'c':
				confirm = 0;
				break;
			case '?':
				show_usage();
		}
	}

	if(auto_part) {
		char *boot_loc;
		if(!(boot_loc = nvcfg_get("boot_loc"))) {
			printf("Can't find boot_loc. Use boot_loc = 0 as default.\n");
			part = 0;
		} else {
			part = atoi(boot_loc);
			if(part > 2 || part < 0)
				part = 1;
			part = 1 - part;	// If boot_loc = 0, write image to 1.
		}
	}
#ifdef CONFIG_SINGLE_IMAGE
	part = 0;
#endif

	if(!name) {
		printf("Please input the filename.\n");
		show_usage();
		return 0;
	}

	if(!(fp = open(name, O_RDONLY))) {
		printf("Can't open file, name:%s err:%s\n", name, strerror(errno));
		goto ret;
	}

	if(stat(name, &file_stat)) {
		printf("Can't stat. name:%s  err:%s\n",name, strerror(errno));
		close(fp);
		goto ret;
	}

	flen = file_stat.st_size;

	if(type == 0) {
#ifdef CONFIG_SINGLE_IMAGE
		if(flen > (4*1024*1024 - 4*64*1024)) {
#else
		if(flen > (4*1024*1024 - 4*64*1024)/2) {
#endif
			printf("Firmware is too big to place into flash...\n");
			close(fp);
			goto ret;
		}
	} else {
		/* type == 1 bootloader*/
		if(flen > 2*64*1024) {
			printf("Bootloader is too big to place into flash...\n");
			close(fp);
			goto ret;
		}

		if(confirm) {
			char input;
			printf("Are you sure you want to upgrade bootloader\nfile:%s size:%d[y/n]:", name, flen);
			scanf("%c", &input);
			if(input != 'y')
				goto ret;
		}
	}

	/*
	 *	We don't have enough memeory for use, so we have to use mmap to map the firmware to write.
	 */
	if((src = mmap(0, file_stat.st_size, PROT_READ, MAP_SHARED, fp, 0)) == (caddr_t) -1) {
		printf("mmap error for firmware\n");
		return -1;
	}

	if(type == 0) {
		/*
		 *	check image file here. 
		 */
		if(check_image(src, flen)) {
			printf("The given file format is wrong and not recognized. Please check it.\n");
			return -1;
		}

		if(part == 0) {
			start_addr = 0x40000;
		} else {
			start_addr = 0x40000 + 0x001e0000;
		}

		for(i = 0; i < flen; i+=0x10000) {
			if((int)(flen - i) < (int)(0x10000))
				len = flen-i;
			else
				len = 0x10000;

			if(FlashWrite(src + i, start_addr + i, len)<0) {
			    printf("Upgrade firmware fail... wrong\n");
			}else {
				if(!silent)
				    printf("Upgrade addr:0x%x Remained:%d\n", start_addr+i, flen - i);
			}
		}

		pass = 0;
		printf("Upgrade firmware successfully. \n");

		if(part == 0) {
			nvcfg_set("boot_loc", "0");
		} else {
			nvcfg_set("boot_loc", "1");
		}
		sprintf(buf, "cs_client set sys/fw/%d/name %s", part, name);
		system(buf);
		sprintf(buf, "cs_client set sys/boot_loc %d", part);
		system(buf);
		system("cs_client commit");
	} else {
		start_addr = 0;
		for(i = 0; i < flen; i+=0x10000) {
			if((int)(flen - i) < (int)0x10000)
				len = flen-i;
			else
				len = 0x10000;

			if(FlashWrite(src + i, start_addr + i, len)<0) {
			    printf("Upgrade bootloader fail...\n");
			}else {
				if(!silent)
				    printf("Upgrade addr:0x%x Remained:%d\n", start_addr+i, flen - i);
			}
		}
		sprintf(buf, "cs_client set sys/bl_name %s", name);
		system(buf);
		system("cs_client commit");
	}

	if(reboot) {
		system("reboot");
	}

ret:
	if(fp)
		close(fp);
    return pass;
}
