/***********************************************************************\
*   Copyright (C) 1992-1998 by Michael K. Johnson, johnsonm@redhat.com *
*                                                                      *
*      This file is placed under the conditions of the GNU Library     *
*      General Public License, version 2, or any later version.        *
*      See file COPYING for information on distribution conditions.    *
\***********************************************************************/

/* File for parsing top-level /proc entities. */
#include "libconf.h"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <dirent.h>

#define BAD_OPEN_MESSAGE					\
"Error: /proc must be mounted\n"				\
"  To mount /proc at boot you need an /etc/fstab line like:\n"	\
"      /proc   /proc   proc    defaults\n"			\
"  In the meantime, mount /proc /proc -t proc\n"

#define STAT_FILE    "/proc/stat"
static int stat_fd = -1;
#define UPTIME_FILE  "/proc/uptime"
static int uptime_fd = -1;
#define LOADAVG_FILE "/proc/loadavg"
static int loadavg_fd = -1;


static char buf[2048];

/* This macro opens filename only if necessary and seeks to 0 so
 * that successive calls to the functions are more efficient.
 * It also reads the current contents of the file into the global buf.
 */
#define FILE_TO_BUF(filename, fd) do{				\
    static int n;						\
    if (fd == -1 && (fd = open(filename, O_RDONLY)) == -1) {	\
	fprintf(stderr, BAD_OPEN_MESSAGE);			\
	close(fd);						\
	_exit(1);						\
    }								\
    lseek(fd, 0L, SEEK_SET);					\
    if ((n = read(fd, buf, sizeof buf - 1)) < 0) {		\
	perror(filename);					\
	close(fd);						\
	fd = -1;						\
	return 0;						\
    }								\
    buf[n] = '\0';						\
}while(0)

#define SET_IF_DESIRED(x,y)  if(x) *(x) = (y)	/* evals 'x' twice */


/***********************************************************************/
int uptime(double *uptime_secs, double *idle_secs) {
    double up=0, idle=0;

    FILE_TO_BUF(UPTIME_FILE,uptime_fd);
    if (sscanf(buf, "%lf %lf", &up, &idle) < 2) {
	fprintf(stderr, "bad data in " UPTIME_FILE "\n");
	return 0;
    }
    SET_IF_DESIRED(uptime_secs, up);
    SET_IF_DESIRED(idle_secs, idle);
    return up;	/* assume never be zero seconds in practice */
}

/***********************************************************************
 * Some values in /proc are expressed in units of 1/HZ seconds, where HZ
 * is the kernel clock tick rate. One of these units is called a jiffy.
 * The HZ value used in the kernel may vary according to hacker desire.
 * According to Linus Torvalds, this is not true. He considers the values
 * in /proc as being in architecture-dependant units that have no relation
 * to the kernel clock tick rate. Examination of the kernel source code
 * reveals that opinion as wishful thinking.
 *
 * In any case, we need the HZ constant as used in /proc. (the real HZ value
 * may differ, but we don't care) There are several ways we could get HZ:
 *
 * 1. Include the kernel header file. If it changes, recompile this library.
 * 2. Use the sysconf() function. When HZ changes, recompile the C library!
 * 3. Ask the kernel. This is obviously correct...
 *
 * Linus Torvalds won't let us ask the kernel, because he thinks we should
 * not know the HZ value. Oh well, we don't have to listen to him.
 * Someone smuggled out the HZ value. :-)
 *
 * This code should work fine, even if Linus fixes the kernel to match his
 * stated behavior. The code only fails in case of a partial conversion.
 *
 * (Albert Cahalan, I think, wrote the rant above.)
 *
 * Unfortunately, this code does not always succeed.  Anyone who can do
 * better is welcome to do so...
 */
unsigned long Hertz;
static int init_Hertz_value(void) __attribute__((constructor));
static int init_Hertz_value(void){
  unsigned long user_j, nice_j, sys_j, other_j;  /* jiffies (clock ticks) */
  double up_1, up_2, seconds;
  unsigned long jiffies, h;
  int i = 0;
  do{
    FILE_TO_BUF(UPTIME_FILE,uptime_fd);  sscanf(buf, "%lf", &up_1);
    /* uptime(&up_1, NULL); */
    FILE_TO_BUF(STAT_FILE,stat_fd);
    /* If we are SMP, then the first line is the sum of jiffies by all CPUs */
    /* In that case, skip it and use the jiffies of the first CPU instead. */
    /* On a single-CPU machine, the 2nd sscanf should harmlessly fail. */
    sscanf(buf, "cpu %lu %lu %lu %lu\n%n",
	   &user_j, &nice_j, &sys_j, &other_j, &i);
    sscanf(buf+i, "cpu0 %lu %lu %lu %lu", &user_j, &nice_j, &sys_j, &other_j);
    FILE_TO_BUF(UPTIME_FILE,uptime_fd);  sscanf(buf, "%lf", &up_2);
    /* uptime(&up_2, NULL); */
  } while((long)( (up_2-up_1)*1000.0/up_1 )); /* want under 0.1% error */
  jiffies = user_j + nice_j + sys_j + other_j;
  seconds = (up_1 + up_2) / 2;
  h = (unsigned long)( (double)jiffies/seconds );
  switch(h){
  case   48 ...   52 :  Hertz =   50; break;
  case   58 ...   62 :  Hertz =   60; break;
  case   95 ...  105 :  Hertz =  100; break; /* normal Linux */
  case  124 ...  132 :  Hertz =  128; break;
  case  195 ...  204 :  Hertz =  200; break; /* normal << 1 */
  case  253 ...  260 :  Hertz =  256; break;
  case  393 ...  408 :  Hertz =  400; break; /* normal << 2 */
  case  790 ...  808 :  Hertz =  800; break; /* normal << 3 */
  case  990 ... 1010 :  Hertz = 1000; break;
  case 1015 ... 1035 :  Hertz = 1024; break; /* Alpha */
  default:
#ifdef HZ
    Hertz = (unsigned long)HZ;    /* <asm/param.h> */
#else
    Hertz = (sizeof(long)==sizeof(int)) ? 100UL : 1024UL;
#endif
#if 0 /* This ends up causing more harm than good.  :-( */
    fprintf(stderr, "Unknown HZ value! (%ld) Assume %ld.\n", h, Hertz);
#endif
  }
  return 0; /* useless, but FILE_TO_BUF has a return in it */
}

/***********************************************************************/
#define JT unsigned long
int four_cpu_numbers(JT *uret, JT *nret, JT *sret, JT *iret) {
    static JT u, n, s, i;
    JT user_j, nice_j, sys_j, idle_j;

    FILE_TO_BUF(STAT_FILE,stat_fd);
    sscanf(buf, "cpu %lu %lu %lu %lu", &user_j, &nice_j, &sys_j, &idle_j);
    SET_IF_DESIRED(uret, user_j-u);
    SET_IF_DESIRED(nret, nice_j-n);
    SET_IF_DESIRED(sret,  sys_j-s);
    SET_IF_DESIRED(iret, idle_j-i);
    u=user_j;
    n=nice_j;
    s=sys_j;
    i=idle_j;
    return 0;
}
#undef JT

/***********************************************************************/
int MCT_Get_CPU_Loadavg(double *av1, double *av5, double *av15) 
{
    double avg_1=0, avg_5=0, avg_15=0;
    
    FILE_TO_BUF(LOADAVG_FILE,loadavg_fd);
    if (sscanf(buf, "%lf %lf %lf", &avg_1, &avg_5, &avg_15) < 3) {
	fprintf(stderr, "bad data in " LOADAVG_FILE "\n");
	exit(1);
    }
    SET_IF_DESIRED(av1,  avg_1);
    SET_IF_DESIRED(av5,  avg_5);
    SET_IF_DESIRED(av15, avg_15);
    return 1;
}

unsigned long get_task_start_time(int pid)
{
	unsigned long seconds_since_boot;
	unsigned long seconds_since_1970;
	unsigned long time_of_boot;
	time_t t;
  	char filename[128];
	char seps[] = " \t\n";
	char *token;
	int cnt=0,fd=-1;
	char stime[20];
	
	sprintf(filename,"/proc/%d/stat",pid);
	do{				
    		static int n;						
    		if (fd == -1 && (fd = open(filename, O_RDONLY)) == -1) {
			fprintf(stderr, BAD_OPEN_MESSAGE);		
			close(fd);					
			_exit(1);					
    		}							
    		lseek(fd, 0L, SEEK_SET);				
    		if ((n = read(fd, buf, sizeof buf - 1)) < 0) {		
			perror(filename);					
			close(fd);						
			fd = -1;						
			return 0;						
    		}								
    		buf[n] = '\0';						
	}while(0);
	
	close(fd);
//	printf("%s\n",buf);
	
	token = strtok(buf, seps);
        while(cnt<21) {
        	if(token == NULL) continue;
		token = strtok(NULL, seps);
		cnt++;
		strcpy(stime,token);
	}
	
	
	init_Hertz_value();
	
	seconds_since_boot = uptime(0,0);
  	seconds_since_1970 = time(NULL);
  	time_of_boot = seconds_since_1970 - seconds_since_boot;
  	
  	
 // 	printf("%s Hertz=%ld %ld\n",ctime(&time_of_boot),Hertz,atoll(stime));
  	
  	t = (((unsigned long)time_of_boot) + ((unsigned long)atoll(stime))/ Hertz);
  	
 // 	printf("%s\n",ctime(&t));
  	
  	return t;

}



unsigned long MCT_Get_Process_Life_Time(int pid) 
{
	time_t current_time,task_time;
	
	task_time = get_task_start_time(pid);
	current_time = time((time_t *) 0 );

	//seconds
	//printf("diff time = %ld\n",current_time - task_time);
	return 	current_time - task_time;
}




/* Info about a process. */
typedef struct _proc_
{
  char *fullname;	/* Name as found out from argv[0] */
  char *basename;	/* Only the part after the last / */
  char *statname;	/* the statname without braces    */
  ino_t ino;		/* Inode number			  */
  dev_t dev;		/* Device it is on		  */
  pid_t pid;		/* Process ID.			  */
  int sid;		/* Session ID.			  */
  struct _proc_ *next;	/* Pointer to next struct. 	  */
} PROC;

/* pid queue */
typedef struct _pidq_ {
  struct _pidq_ *front;
  struct _pidq_ *next;
  struct _pidq_ *rear;
  PROC		*proc;
} PIDQ;

/* List of processes. */
PROC *plist;

/* Did we stop a number of processes? */
int stopped;

int scripts_too = 0;


/* Malloc space, barf if out of memory. */
void *xmalloc(bytes)
int bytes;
{
  void *p;

  if ((p = malloc(bytes)) == NULL) {
	if (stopped) kill(-1, SIGCONT);
	exit(1);
  }
  return(p);
}


/* Read the proc filesystem. */
int readproc()
{
  DIR *dir;
  struct dirent *d;
  char path[256];
  char buf[256];
  char *s, *q;
  FILE *fp;
  int pid, f;
  PROC *p, *n;
  struct stat st;
  int c;

  /* Open the /proc directory. */
  if ((dir = opendir("/proc")) == NULL) {
	return(-1);
  }

  /* Free the already existing process list. */
  n = plist;
  for(p = plist; n; p = n) {
	n = p->next;
	if (p->fullname) free(p->fullname);
	free(p);
  }
  plist = NULL;

  /* Walk through the directory. */
  while((d = readdir(dir)) != NULL) {

	/* See if this is a process */
	if ((pid = atoi(d->d_name)) == 0) continue;

	/* Get a PROC struct . */
	p = (PROC *)xmalloc(sizeof(PROC));
	memset(p, 0, sizeof(PROC));

	/* Open the statistics file. */
	sprintf(path, "/proc/%s/stat", d->d_name);

	/* Read SID & statname from it. */
 	if ((fp = fopen(path, "r")) != NULL) {
		buf[0] = 0;
		fgets(buf, 256, fp);

		/* See if name starts with '(' */
		s = buf;
		while(*s != ' ') s++;
		s++;
		if (*s == '(') {
			/* Read program name. */
			q = strrchr(buf, ')');
			if (q == NULL) {
				p->sid = 0;
				free(p);
				continue;
			}
			s++;
		} else {
			q = s;
			while(*q != ' ') q++;
		}
		*q++ = 0;
		while (*q == ' ') q++;
		p->statname = (char *)xmalloc(strlen(s)+1);
		strcpy(p->statname, s);

		/* This could be replaced by getsid(pid) */
		if (sscanf(q, "%*c %*d %*d %d", &p->sid) != 1) {
			p->sid = 0;
			free(p);
			continue;
		}
		fclose(fp);
	} else {
		/* Process disappeared.. */
		free(p);
		continue;
	}

	/* Now read argv[0] */
	sprintf(path, "/proc/%s/cmdline", d->d_name);
	if ((fp = fopen(path, "r")) != NULL) {
		f = 0;
		while(f < 127 && (c = fgetc(fp)) != EOF && c) buf[f++] = c;
		buf[f++] = 0;
		fclose(fp);

		/* Store the name into malloced memory. */
		p->fullname = (char *)xmalloc(f);
		strcpy(p->fullname, buf);

		/* Get a pointer to the basename. */
		if ((p->basename = strrchr(p->fullname, '/')) != NULL)
			p->basename++;
		else
			p->basename = p->fullname;
	} else {
		/* Process disappeared.. */
		free(p);
		continue;
	}

	
	/* Try to stat the executable. */
//	sprintf(path, "/proc/%s/exe", d->d_name);
//	if (stat(path, &st) == 0) {
//		p->dev = st.st_dev;
//		p->ino = st.st_ino;
//	}
	
	/* Link it into the list. */
	p->next = plist;
	plist = p;
	p->pid = pid;
  }
  closedir(dir);

  /* Done. */
  return(0);
}

PIDQ *init_pid_q(PIDQ *q)
{
  q->front =  q->next = q->rear = NULL;
  q->proc = NULL;
  return q;
}

int empty_q(PIDQ *q)
{
  return ( q->front == NULL );
}

int add_pid_to_q(PIDQ *q, PROC *p)
{
   PIDQ *tmp;

   tmp = (PIDQ *)xmalloc(sizeof(PIDQ));

   tmp->proc = p;
   tmp->next = NULL;

   if (empty_q(q)) {
      q->front = tmp;
      q->rear  = tmp;
   } else {
      q->rear->next = tmp;
      q->rear = tmp;
   }
   return 0;
}

PROC *get_next_from_pid_q(PIDQ *q)
{
    PROC *p;
    PIDQ *tmp = q->front;

    if (!empty_q(q)) {
       p = q->front->proc;
       q->front = tmp->next; 
       free(tmp);
       return p;
    } else
      return NULL;
}

/* Try to get the process ID of a given process. */
PIDQ *pidof(prog)
char *prog;
{
  struct stat st;
  int dostat = 0;
  PROC *p;
  PIDQ *q;
  char *s;
  int foundone = 0;

  /* Try to stat the executable. */
  if (prog[0] == '/' && stat(prog, &st) == 0) dostat++;

  /* Get basename of program. */
  if ((s = strrchr(prog, '/')) == NULL)
	s = prog;
  else
	s++;

  q = (PIDQ *)xmalloc(sizeof(PIDQ));
  q = init_pid_q(q);

  /* First try to find a match based on dev/ino pair. */
  if (dostat) {
	for(p = plist; p; p = p->next)
		if (p->dev == st.st_dev && p->ino == st.st_ino) {
			add_pid_to_q(q, p);
			foundone++;
		}
  }

  /* If we didn't find a match based on dev/ino, try the name. */
  if (!foundone) {
	for(p = plist; p; p = p->next) {
		if ((strcmp(p->fullname, prog) == 0) ||
        	    (strcmp(p->basename, s) == 0) ||
        	    ((p->fullname[0] == 0 || scripts_too) && strcmp(p->statname, s) == 0)) {
			add_pid_to_q(q, p);
			foundone++;	
		}
	}
  }
  
  return q;
}

int MCT_Getpid(char *cmdline)
{
	PIDQ *q;
	PROC *ans,*tmp;
	int m_pid = -1;
	
	readproc();
	if ((q = pidof(cmdline)) != NULL) {
		ans = get_next_from_pid_q(q);
		if(ans != NULL ) {
			m_pid = ans->pid;
			while(tmp = get_next_from_pid_q(q)) ;
			return m_pid;
		}
			
	}
	return -1;
}
/*

main()
{
	int i;
	i = MCT_Getpid("pptpd");
	printf("i=%d\n",i);
}
*/
