
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <time.h>
#include "sysrepd.h"

#include <errno.h>
#include <asm/unistd.h>
#include <asm/mconfig.h>

static const unsigned char null_byte = 0;
#ifndef HOST_TEST
_syscall2(int, read_config, int, type, void *, config)
#endif

// If the transmitted data contains special character, the following function should be called
// to avoid those characters to be interpreted by receiver.
void filter_write_byte(int fd, char ch)
{
	static const unsigned char escape = ESCAPE_CHAR;
	switch ( ch )
	{
	case ESCAPE_CHAR:
	case END_OF_COMMAND:
		write(fd, &escape, 1);
		break;
	}
	write(fd, &ch, 1);
}

void filter_write(int fd, char *buf, int size)
{
	int i;
	for (i = 0 ; i < size ; i++) {
		filter_write_byte(fd, buf[i]);
	}
}

// In order to avoid the server to block, we should read one byte at a time, and then
// determine the next state.  If one complete request is received, it's time to reply some
// message in response to client's request.
//
int process_command(int fd, int state)
{
	char		bytein;
	time_t		curtime;
	struct tm	*ptm;
	char		buf[128];
	unsigned char	percentage;

	read(fd, &bytein, 1);
	switch ( state )
	{
	case STATE_UNKNOWN:
		if ( bytein == CMD_SYNC )
		{
			return STATE_INIT;
		}
		return STATE_UNKNOWN;
	case STATE_INIT:
		switch ( bytein )
		{
		case CMD_SYNC:
			return STATE_INIT;
		case CMD_SYSTEM_TIME:
			return STATE_SYSTEM_TIME;
		case CMD_UPDATE_PERCENTAGE:
			return STATE_UPDATE_PERCENTAGE;
		default:
			return STATE_UNKNOWN;
		}
	case STATE_SYSTEM_TIME:
		if ( bytein == END_OF_COMMAND )
		{
			filter_write_byte(fd, CMD_SYSTEM_TIME);
			time(&curtime);
			ptm = localtime(&curtime);
			sprintf(buf, "%02d:%02d:%02d %d/%d/%d", 
				ptm->tm_hour, ptm->tm_min, ptm->tm_sec,
				ptm->tm_mon+1, ptm->tm_mday, ptm->tm_year+1900);
			write(fd, buf, strlen(buf));
			write(fd, &null_byte, 1);
			return STATE_INIT;
		}
		return STATE_UNKNOWN;
	case STATE_UPDATE_PERCENTAGE:
		if ( bytein == END_OF_COMMAND ) {
#ifndef HOST_TEST
			read_config(MC_READ_PERCENTAGE, &percentage);
#endif
			filter_write_byte(fd, CMD_UPDATE_PERCENTAGE);
			filter_write_byte(fd, percentage);
			write(fd, &null_byte, 1);
			return STATE_INIT;
		}
		return STATE_UNKNOWN;
	default:
		return STATE_UNKNOWN;
	}
}

static void handle_sigterm(int sig)
{
	printf("exit due to signal %d\n", sig);
	exit(1);
}

static void usage(char *progname)
{
	printf("Usage: %s [-f]", progname);
	exit(1);
}

int main(int argc, char **argv)
{
	int	server_socket, client_socket;
	struct sockaddr_in server_addr, client_addr;
	fd_set	fds_read, fds_test;
#ifdef M68000
	unsigned int nlen;
#else
	socklen_t nlen;
#endif
	int	fd, nread, ret, i;
	int	clientstate[FD_SETSIZE];
	int	argn, fg_exec;

	fg_exec = 0;
	for (argn = 1; argn < argc; argn++) {
		if ( strcmp(argv[argn], "-f") == 0 ) {
			fg_exec = 1;
		} else {
			usage(argv[0]);
		}
	}

	if ( !fg_exec ) {
		if ( daemon(0, 0) == -1 ) {
			printf("fail to execute in background\n");
		}
	}

	signal(SIGTERM, handle_sigterm);
	signal(SIGINT, handle_sigterm);
	signal(SIGHUP, SIG_IGN);
	
	for (i = 0 ; i < FD_SETSIZE ; i++) {
		clientstate[i] = STATE_INIT;
	}

	server_socket = socket(AF_INET, SOCK_STREAM, 0);
	server_addr.sin_family = AF_INET;
#ifdef M68000
	server_addr.sin_addr.s_addr = INADDR_ANY;
	server_addr.sin_port = SERVER_PORT;
#else
	server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	server_addr.sin_port = htons(SERVER_PORT);
#endif

	if ( bind(server_socket, (struct sockaddr*) &server_addr, sizeof(server_addr)) < 0 ) {
		perror("bind");
		exit(1);
	}

	listen(server_socket, 9);
	FD_ZERO(&fds_read);
	FD_SET(server_socket, &fds_read);

	// Wait for request from client
	while ( 1 ) {
		fds_test = fds_read;
		ret = select(FD_SETSIZE, &fds_test, (fd_set*) 0, (fd_set*) 0, (struct timeval*) 0);
		if ( ret < 0 ) {
			perror("sysrepd: server failure\n");
			exit(1);
		}

		for (fd = 0 ; fd < FD_SETSIZE ; fd++) {
			if ( FD_ISSET(fd, &fds_test) ) {
				if ( fd == server_socket ) {
					nlen = sizeof(client_addr);
					client_socket = accept(server_socket, (struct sockaddr*) &client_addr, &nlen);
					FD_SET(client_socket, &fds_read);
#ifdef DEBUG
					printf("adding client on fd %d\n", client_socket);
#endif
				} else {
					ioctl(fd, FIONREAD, &nread);
					if ( nread == 0 ) {
						close(fd);
						clientstate[fd] = STATE_INIT;
						FD_CLR(fd, &fds_read);
#ifdef DEBUG
						printf("removing client on fd %d\n", fd);
#endif
					} else {
						clientstate[fd] = process_command(fd, clientstate[fd]);
					}
				}
			}
		}
	}
}
