/********************************************************************************
*
*	Copyright (c) 2002  Magic Control Technology Co.  All Rights Reserved.
*
*	FILE:
*		dhcp_leases.c
*
*	Abstract:
*		maintain dhcp leases	
*		
*	History:
*		07/23/2002	Ethan Created
*********************************************************************************/
#include<linux/dhcp_leases.h>
#include<linux/init.h>
#include<linux/sched.h>
#include <linux/list.h>
#include <linux/spinlock.h>

static spinlock_t urllog_list_lock = SPIN_LOCK_UNLOCKED;
static LIST_HEAD(urllog_free);

// urllog_active is an ordered list.
// recently reference url location was placed in the list tail.
static LIST_HEAD(urllog_active);

static URLLOG urllog[URLLOG_MAX];	
static struct list_head urllog_hash_list[URLLOG_HASH_SIZE];

static inline int get_url_hash_value(unsigned long val)
{
	return ((val >> 16) ^ (val & 0xffff)) / URLLOG_HASH_SIZE;
}

void mct_add_urllog(unsigned long srcip, unsigned long destip)
{
	int	log_index;
	URLLOG	*url, *newurl;
	struct list_head *pfree, *p;

#if 0
	printk(KERN_INFO "add_urllog: srcip = %u.%u.%u.%u, destip = %u.%u.%u.%u\n",
		srcip & 0xff, (srcip>>8) & 0xff, (srcip>>16) & 0xff, (srcip>>24) & 0xff,
		destip & 0xff, (destip>>8) & 0xff, (destip>>16) & 0xff, (destip>>24) & 0xff);
#endif

	spin_lock_irq(&urllog_list_lock);
	log_index = get_url_hash_value(srcip);
	list_for_each(p, &urllog_hash_list[log_index]) {
		url = list_entry(p, URLLOG, hash_list);
		if ( srcip == url->srcip ) {
			url->destip = destip;
			list_del(&url->list);
			list_add_tail(&url->list, &urllog_active);
			spin_unlock_irq(&urllog_list_lock);
			return;
		}
	}

	// get free urllog node
	if ( !list_empty(&urllog_free) ) {
		pfree = urllog_free.next;
		list_del(pfree);
		newurl = list_entry(pfree, URLLOG, list);
	} else {
		int	index;
		// remove the oldest node in the urllog_active
		pfree = urllog_active.next;
		list_del(pfree);
		newurl = list_entry(pfree, URLLOG, list);

		// remove the correspoinging node in the urllog_hash_list
		index = get_url_hash_value(newurl->srcip);
		list_for_each(p, &urllog_hash_list[index]) {
			url = list_entry(p, URLLOG, hash_list);	
			if ( url == newurl ) {
				list_del(&url->hash_list);
				break;
			}
		}
	}

	newurl->time = jiffies;
	newurl->srcip = srcip;
	newurl->destip = destip;

	list_add_tail(&newurl->list, &urllog_active);
	list_add(&newurl->hash_list, &urllog_hash_list[log_index]);

	spin_unlock_irq(&urllog_list_lock);
}

asmlinkage int sys_get_urllog(int *size, URLLOG_INFO *urllog_info)
{
	int	i;
	struct list_head *p;
	URLLOG		*url;
	URLLOG_INFO	*pinfo;

	spin_lock_irq(&urllog_list_lock);
	i = 0;
	pinfo = &urllog_info[i];
	list_for_each(p, &urllog_active) {
		url = list_entry(p, URLLOG, list);
		pinfo->time = url->time;
		pinfo->srcip = url->srcip;
		pinfo->destip = url->destip;
		i++;
		pinfo++;
	}
	spin_unlock_irq(&urllog_list_lock);

	*size = i;

	return 0;
}

// called from net/ipv4/ip_fragment.c
void mct_putlog(char *msg)
{
}

int __init init_urllog(void)
{
	URLLOG *p;
	struct list_head *entry;
	
	printk(KERN_INFO "URL log initialization\n");

	// There is no need to initialize "list_head" member in the URLLOG
	// before we use list_add function. This is because __list_add function
	// will update both "prev" and "next" member of added entry.
	for (p = &urllog[0]; p < &urllog[URLLOG_MAX]; p++) {
		list_add_tail(&p->list, &urllog_free);
	}

	for (entry = &urllog_hash_list[0]; entry < &urllog_hash_list[URLLOG_HASH_SIZE]; entry++)
		INIT_LIST_HEAD(entry);

	return 0;
}

__initcall(init_urllog);

