/*
 *  linux/arch/mips/toshiba-boards/generic/dma.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2001 Toshiba Corporation
 *
 * $Id: dma.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/slab.h>
#include <asm/system.h>
#include <asm/mipsregs.h>
#include <asm/cpu.h>
#include <asm/bootinfo.h>
#include <asm/dma.h>
#include <asm/mct-boards/dma.h>
#include <asm/mct-boards/irq.h>

#if defined(CONFIG_MCT_SG100) || defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100)
#include <asm/tx3927.h>
#define HAVE_3927_DMA
#endif
#if defined(CONFIG_TOSHIBA_TX4927EVB) || defined(CONFIG_TOSHIBA_RBTX4927)
#include <asm/tx4927.h>
#define HAVE_4927_DMA
#endif
#if defined(CONFIG_MCT_SG4100) || defined(CONFIG_MCT_SA200)
#include <asm/tx4925.h>
#define HAVE_4925_DMA
#endif

#ifdef HAVE_3927_DMA
/* TX3927 stype DMA controller */

/* reset a specific DMA channel */
static void
init_tx3927_dma(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].ccr = TX3927_DMA_CCR_CHRST;
	tx3927_dmaptr->ch[ch].cha = 0;
	tx3927_dmaptr->ch[ch].sar = 0;
	tx3927_dmaptr->ch[ch].dar = 0;
	tx3927_dmaptr->ch[ch].cntr = 0;
	tx3927_dmaptr->ch[ch].sair = 0;
	tx3927_dmaptr->ch[ch].dair = 0;
	tx3927_dmaptr->ch[ch].ccr = 0;
}

/* enable/disable a specific DMA channel */
static void
enable_tx3927_dma(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].ccr |= TX3927_DMA_CCR_XFACT;
}

static void
disable_tx3927_dma(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].ccr &= ~TX3927_DMA_CCR_XFACT;
}

/* set mode for a specific DMA channel */
static void
set_tx3927_dma_mode(unsigned int dmanr, unsigned int mode)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].ccr = mode;
}

/* Set source/destination address for specific DMA channel.
 */
static void
set_tx3927_dma_addr(unsigned int dmanr,
		    unsigned int sa,
		    unsigned int da)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].sar = sa;
	tx3927_dmaptr->ch[ch].dar = da;
}
static void
set_tx3927_dma_addr64(unsigned int dmanr,
		      unsigned long long sa,
		      unsigned long long da)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].sar = (unsigned long)sa;
	tx3927_dmaptr->ch[ch].dar = (unsigned long)da;
}

/* Set transfer size for a specific DMA channel.
 */
static void
set_tx3927_dma_count(unsigned int dmanr, unsigned int count)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].cntr = count;
}

/* Set source/destination address increment size for a specific DMA channel.
 */
static void
set_tx3927_dma_inc(unsigned int dmanr, unsigned int sai, unsigned int dai)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].sair = sai;
	tx3927_dmaptr->ch[ch].dair = dai;
}


/* Get DMA residue count. After a DMA transfer, this
 * should return zero. Reading this while a DMA transfer is
 * still in progress will return unpredictable results.
 */
static int
get_tx3927_dma_residue(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	return tx3927_dmaptr->ch[ch].cntr;
}

/* Get DMA status.
 */
static unsigned int
get_tx3927_dma_status(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	return tx3927_dmaptr->ch[ch].csr;
}

/* clear DMA status.
 */
static void
clear_tx3927_dma_status(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].csr = 0xffffffff;
}

/* get DMA completion interrupt number.
 */
static int
get_tx3927_dma_irqno(unsigned int dmanr)
{
	if (dmanr < TXX927_DMA_CHANNEL_START ||
	    dmanr >= TXX927_DMA_CHANNEL_START + MAX_TXX927_DMA_CHANNELS)
		return -1;
	return tx3927_irq_to_irq(TX3927_IR_DMA);
}

/*
 * chain DMA operations.
 * txx927_chdma_desc_t is nocache virtual address of the command descriptor.
 */

static void
set_tx3927_dma_chain(unsigned int dmanr, txx927_chdma_desc_t cha)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx3927_dmaptr->ch[ch].cha = virt_to_phys(cha);
}

static txx927_chdma_desc_t
get_tx3927_dma_chain(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	unsigned int cha = cha = tx3927_dmaptr->ch[ch].cha;
	if (cha == 0)
		return 0;
	return (txx927_chdma_desc_t)KSEG1ADDR(cha);
}

#define tx3927_chdma_desc(desc)	((struct tx3927_dma_ch_reg*)desc)

static void
set_tx3927_chdma_mode(txx927_chdma_desc_t desc, unsigned int mode)
{
	tx3927_chdma_desc(desc)->ccr = mode;
}

static void
set_tx3927_chdma_addr(txx927_chdma_desc_t desc, unsigned int sa, unsigned int da)
{
	tx3927_chdma_desc(desc)->sar = sa;
	tx3927_chdma_desc(desc)->dar = da;
}

static void
get_tx3927_chdma_addr(txx927_chdma_desc_t desc, unsigned int *sa, unsigned int *da)
{
	*sa = tx3927_chdma_desc(desc)->sar;
	*da = tx3927_chdma_desc(desc)->dar;
}

static void
set_tx3927_chdma_addr64(txx927_chdma_desc_t desc, unsigned long long sa, unsigned long long da)
{
	tx3927_chdma_desc(desc)->sar = (unsigned long)sa;
	tx3927_chdma_desc(desc)->dar = (unsigned long)da;
}

static void
get_tx3927_chdma_addr64(txx927_chdma_desc_t desc, unsigned long long *sa, unsigned long long *da)
{
	*sa = tx3927_chdma_desc(desc)->sar;
	*da = tx3927_chdma_desc(desc)->dar;
}

static void
set_tx3927_chdma_count(txx927_chdma_desc_t desc, unsigned int count)
{
	tx3927_chdma_desc(desc)->cntr = count;
}

static void
get_tx3927_chdma_count(txx927_chdma_desc_t desc, unsigned int *count)
{
	*count = tx3927_chdma_desc(desc)->cntr;
}

static void
set_tx3927_chdma_inc(txx927_chdma_desc_t desc, unsigned int sai, unsigned int dai)
{
	tx3927_chdma_desc(desc)->sair = sai;
	tx3927_chdma_desc(desc)->dair = dai;
}

static void
set_tx3927_chdma_chain(txx927_chdma_desc_t desc, txx927_chdma_desc_t cha)
{
	tx3927_chdma_desc(desc)->cha = virt_to_phys(cha);
}

static txx927_chdma_desc_t
get_tx3927_chdma_chain(txx927_chdma_desc_t desc)
{
	unsigned int cha = tx3927_chdma_desc(desc)->cha;
	if (cha == 0)
		return 0;
	return (txx927_chdma_desc_t)KSEG1ADDR(cha);
}

static txx927_chdma_desc_t
alloc_tx3927_chdma_cmd(void)
{
	void *desc;
	int size = sizeof(struct tx3927_dma_ch_reg);
	/* kmalloc always return 32-byte aligned address... */
	desc = kmalloc(size, GFP_ATOMIC);
	if (desc) {
		if ((unsigned int)desc & 31)
			panic("dma: kmalloc returns unaligned address.");
		dma_cache_wback_inv((unsigned long)desc, size);
		desc = (void *)KSEG1ADDR(desc);
	}
	return (txx927_chdma_desc_t)desc;
}

static void
free_tx3927_chdma_cmd(txx927_chdma_desc_t desc)
{
	if (desc)
		kfree((void *)KSEG0ADDR(desc));
}

static struct txx927_dma_ops tx3927_dma_ops = {
	init_dma: init_tx3927_dma,
	enable_dma: enable_tx3927_dma,
	disable_dma: disable_tx3927_dma,
	set_dma_mode: set_tx3927_dma_mode,
	set_dma_addr: set_tx3927_dma_addr,
	set_dma_addr64: set_tx3927_dma_addr64,
	set_dma_count: set_tx3927_dma_count,
	set_dma_inc: set_tx3927_dma_inc,
	get_dma_residue: get_tx3927_dma_residue,
	get_dma_status: get_tx3927_dma_status,
	clear_dma_status: clear_tx3927_dma_status,
	get_dma_irqno: get_tx3927_dma_irqno,
	set_dma_chain: set_tx3927_dma_chain,
	get_dma_chain: get_tx3927_dma_chain,
	set_chdma_mode: set_tx3927_chdma_mode,
	set_chdma_addr: set_tx3927_chdma_addr,
	get_chdma_addr: get_tx3927_chdma_addr,
	set_chdma_addr64: set_tx3927_chdma_addr64,
	get_chdma_addr64: get_tx3927_chdma_addr64,
	set_chdma_count: set_tx3927_chdma_count,
	get_chdma_count: get_tx3927_chdma_count,
	set_chdma_inc: set_tx3927_chdma_inc,
	set_chdma_chain: set_tx3927_chdma_chain,
	get_chdma_chain: get_tx3927_chdma_chain,
	alloc_chdma_cmd: alloc_tx3927_chdma_cmd,
	free_chdma_cmd: free_tx3927_chdma_cmd
};

#endif	/* HAVE_3927_DMA */

#ifdef HAVE_4927_DMA
/* TX4927 stype DMA controller */

/* reset a specific DMA channel */
static void
init_tx4927_dma(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4927_dmaptr->ch[ch].ccr = TX4927_DMA_CCR_CHRST;
	tx4927_dmaptr->ch[ch].cha = 0;
	tx4927_dmaptr->ch[ch].sar = 0;
	tx4927_dmaptr->ch[ch].dar = 0;
	tx4927_dmaptr->ch[ch].cntr = 0;
	tx4927_dmaptr->ch[ch].sair = 0;
	tx4927_dmaptr->ch[ch].dair = 0;
	tx4927_dmaptr->ch[ch].ccr = 0;
}

/* enable/disable a specific DMA channel */
static void
enable_tx4927_dma(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4927_dmaptr->ch[ch].ccr |= TX4927_DMA_CCR_XFACT;
}

static void
disable_tx4927_dma(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4927_dmaptr->ch[ch].ccr &= ~TX4927_DMA_CCR_XFACT;
}

/* set mode for a specific DMA channel */
static void
set_tx4927_dma_mode(unsigned int dmanr, unsigned int mode)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
#ifdef __LITTLE_ENDIAN
	mode |= TX4927_DMA_CCR_LE;
#endif
	tx4927_dmaptr->ch[ch].ccr = mode;
}

/* Set source/destination address for specific DMA channel.
 */
static void
set_tx4927_dma_addr(unsigned int dmanr,
		    unsigned int sa,
		    unsigned int da)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4927_dmaptr->ch[ch].sar = sa;
	tx4927_dmaptr->ch[ch].dar = da;
}
static void
set_tx4927_dma_addr64(unsigned int dmanr,
		      unsigned long long sa,
		      unsigned long long da)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4927_dmaptr->ch[ch].sar = sa;
	tx4927_dmaptr->ch[ch].dar = da;
}

/* Set transfer size for a specific DMA channel.
 */
static void
set_tx4927_dma_count(unsigned int dmanr, unsigned int count)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4927_dmaptr->ch[ch].cntr = count;
}

/* Set source/destination address increment size for a specific DMA channel.
 */
static void
set_tx4927_dma_inc(unsigned int dmanr, unsigned int sai, unsigned int dai)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4927_dmaptr->ch[ch].sair = sai;
	tx4927_dmaptr->ch[ch].dair = dai;
}


/* Get DMA residue count. After a DMA transfer, this
 * should return zero. Reading this while a DMA transfer is
 * still in progress will return unpredictable results.
 */
static int
get_tx4927_dma_residue(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	return tx4927_dmaptr->ch[ch].cntr;
}

/* Get DMA status.
 */
static unsigned int
get_tx4927_dma_status(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	return tx4927_dmaptr->ch[ch].csr;
}

/* clear DMA status.
 */
static void
clear_tx4927_dma_status(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4927_dmaptr->ch[ch].csr = 0xffffffff;
}

/* get DMA completion interrupt number.
 */
static int
get_tx4927_dma_irqno(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;

	if (dmanr < TXX927_DMA_CHANNEL_START ||
	    dmanr >= TXX927_DMA_CHANNEL_START + MAX_TXX927_DMA_CHANNELS)
		return -1;
	return tx4927_irq_to_irq(TX4927_IR_DMA(ch));
}

/*
 * chain DMA operations.
 * txx927_chdma_desc_t is nocache virtual address of the command descriptor.
 */

static void
set_tx4927_dma_chain(unsigned int dmanr, txx927_chdma_desc_t cha)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4927_dmaptr->ch[ch].cha = virt_to_phys(cha);
}

static txx927_chdma_desc_t
get_tx4927_dma_chain(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	unsigned int cha = (unsigned int)tx4927_dmaptr->ch[ch].cha;
	if (cha == 0)
		return 0;
	return (txx927_chdma_desc_t)KSEG1ADDR(cha);
}

#define tx4927_chdma_desc(desc)	((struct tx4927_dma_ch_reg*)desc)

static void
set_tx4927_chdma_mode(txx927_chdma_desc_t desc, unsigned int mode)
{
#ifdef __LITTLE_ENDIAN
	mode |= TX4927_DMA_CCR_LE;
#endif
	tx4927_chdma_desc(desc)->ccr = mode;
}

static void
set_tx4927_chdma_addr(txx927_chdma_desc_t desc, unsigned int sa, unsigned int da)
{
	tx4927_chdma_desc(desc)->sar = sa;
	tx4927_chdma_desc(desc)->dar = da;
}

static void
get_tx4927_chdma_addr(txx927_chdma_desc_t desc, unsigned int *sa, unsigned int *da)
{
	*sa = tx4927_chdma_desc(desc)->sar;
	*da = tx4927_chdma_desc(desc)->dar;
}

static void
set_tx4927_chdma_addr64(txx927_chdma_desc_t desc, unsigned long long sa, unsigned long long da)
{
	tx4927_chdma_desc(desc)->sar = sa;
	tx4927_chdma_desc(desc)->dar = da;
}

static void
get_tx4927_chdma_addr64(txx927_chdma_desc_t desc, unsigned long long *sa, unsigned long long *da)
{
	*sa = tx4927_chdma_desc(desc)->sar;
	*da = tx4927_chdma_desc(desc)->dar;
}

static void
set_tx4927_chdma_count(txx927_chdma_desc_t desc, unsigned int count)
{
	tx4927_chdma_desc(desc)->cntr = count;
}

static void
get_tx4927_chdma_count(txx927_chdma_desc_t desc, unsigned int *count)
{
	*count = tx4927_chdma_desc(desc)->cntr;
}

static void
set_tx4927_chdma_inc(txx927_chdma_desc_t desc, unsigned int sai, unsigned int dai)
{
	tx4927_chdma_desc(desc)->sair = sai;
	tx4927_chdma_desc(desc)->dair = dai;
}

static void
set_tx4927_chdma_chain(txx927_chdma_desc_t desc, txx927_chdma_desc_t cha)
{
	tx4927_chdma_desc(desc)->cha = virt_to_phys(cha);
}

static txx927_chdma_desc_t
get_tx4927_chdma_chain(txx927_chdma_desc_t desc)
{
	unsigned int cha = (unsigned int)tx4927_chdma_desc(desc)->cha;
	if (cha == 0)
		return 0;
	return (txx927_chdma_desc_t)KSEG1ADDR(cha);
}

static txx927_chdma_desc_t
alloc_tx4927_chdma_cmd(void)
{
	int size = sizeof(struct tx4927_dma_ch_reg);
	void *desc;
	/* kmalloc always return 32-byte aligned address... */
	desc = kmalloc(size, GFP_ATOMIC);
	if (desc) {
		if ((unsigned int)desc & 31)
			panic("dma: kmalloc returns unaligned address.");
		dma_cache_wback_inv((unsigned long)desc, size);
		desc = (void *)KSEG1ADDR(desc);
	}
	return (txx927_chdma_desc_t)desc;
}

static void
free_tx4927_chdma_cmd(txx927_chdma_desc_t desc)
{
	if (desc)
		kfree((void *)KSEG0ADDR(desc));
}

static struct txx927_dma_ops tx4927_dma_ops = {
	init_dma: init_tx4927_dma,
	enable_dma: enable_tx4927_dma,
	disable_dma: disable_tx4927_dma,
	set_dma_mode: set_tx4927_dma_mode,
	set_dma_addr: set_tx4927_dma_addr,
	set_dma_addr64: set_tx4927_dma_addr64,
	set_dma_count: set_tx4927_dma_count,
	set_dma_inc: set_tx4927_dma_inc,
	get_dma_residue: get_tx4927_dma_residue,
	get_dma_status: get_tx4927_dma_status,
	clear_dma_status: clear_tx4927_dma_status,
	get_dma_irqno: get_tx4927_dma_irqno,
	set_dma_chain: set_tx4927_dma_chain,
	get_dma_chain: get_tx4927_dma_chain,
	set_chdma_mode: set_tx4927_chdma_mode,
	set_chdma_addr: set_tx4927_chdma_addr,
	get_chdma_addr: get_tx4927_chdma_addr,
	set_chdma_addr64: set_tx4927_chdma_addr64,
	get_chdma_addr64: get_tx4927_chdma_addr64,
	set_chdma_count: set_tx4927_chdma_count,
	get_chdma_count: get_tx4927_chdma_count,
	set_chdma_inc: set_tx4927_chdma_inc,
	set_chdma_chain: set_tx4927_chdma_chain,
	get_chdma_chain: get_tx4927_chdma_chain,
	alloc_chdma_cmd: alloc_tx4927_chdma_cmd,
	free_chdma_cmd: free_tx4927_chdma_cmd
};

#endif	/* HAVE_4927_DMA */

#ifdef HAVE_4925_DMA
/* TX4925 stype DMA controller */

/* reset a specific DMA channel */
static void
init_tx4925_dma(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4925_dmaptr->ch[ch].ccr = TX4925_DMA_CCR_CHRST;
	tx4925_dmaptr->ch[ch].cha = 0;
	tx4925_dmaptr->ch[ch].sar = 0;
	tx4925_dmaptr->ch[ch].dar = 0;
	tx4925_dmaptr->ch[ch].cntr = 0;
	tx4925_dmaptr->ch[ch].sair = 0;
	tx4925_dmaptr->ch[ch].dair = 0;
	tx4925_dmaptr->ch[ch].ccr = 0;
}

/* enable/disable a specific DMA channel */
static void
enable_tx4925_dma(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4925_dmaptr->ch[ch].ccr |= TX4925_DMA_CCR_XFACT;
}

static void
disable_tx4925_dma(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4925_dmaptr->ch[ch].ccr &= ~TX4925_DMA_CCR_XFACT;
}

/* set mode for a specific DMA channel */
static void
set_tx4925_dma_mode(unsigned int dmanr, unsigned int mode)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
#ifdef __LITTLE_ENDIAN
	mode |= TX4925_DMA_CCR_LE;
#endif
	tx4925_dmaptr->ch[ch].ccr = mode;
}

/* Set source/destination address for specific DMA channel.
 */
static void
set_tx4925_dma_addr(unsigned int dmanr,
		    unsigned int sa,
		    unsigned int da)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4925_dmaptr->ch[ch].sar = sa;
	tx4925_dmaptr->ch[ch].dar = da;
}
static void
set_tx4925_dma_addr64(unsigned int dmanr,
		      unsigned long long sa,
		      unsigned long long da)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4925_dmaptr->ch[ch].sar = (unsigned long)sa;
	tx4925_dmaptr->ch[ch].dar = (unsigned long)da;
}

/* Set transfer size for a specific DMA channel.
 */
static void
set_tx4925_dma_count(unsigned int dmanr, unsigned int count)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4925_dmaptr->ch[ch].cntr = count;
}

/* Set source/destination address increment size for a specific DMA channel.
 */
static void
set_tx4925_dma_inc(unsigned int dmanr, unsigned int sai, unsigned int dai)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4925_dmaptr->ch[ch].sair = sai;
	tx4925_dmaptr->ch[ch].dair = dai;
}


/* Get DMA residue count. After a DMA transfer, this
 * should return zero. Reading this while a DMA transfer is
 * still in progress will return unpredictable results.
 */
static int
get_tx4925_dma_residue(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	return tx4925_dmaptr->ch[ch].cntr;
}

/* Get DMA status.
 */
static unsigned int
get_tx4925_dma_status(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	return tx4925_dmaptr->ch[ch].csr;
}

/* clear DMA status.
 */
static void
clear_tx4925_dma_status(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4925_dmaptr->ch[ch].csr = 0xffffffff;
}

/* get DMA completion interrupt number.
 */
static int
get_tx4925_dma_irqno(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;

	if (dmanr < TXX927_DMA_CHANNEL_START ||
	    dmanr >= TXX927_DMA_CHANNEL_START + MAX_TXX927_DMA_CHANNELS)
		return -1;
	return tx4925_irq_to_irq(TX4925_IR_DMA(ch));
}

/*
 * chain DMA operations.
 * txx927_chdma_desc_t is nocache virtual address of the command descriptor.
 */

static void
set_tx4925_dma_chain(unsigned int dmanr, txx927_chdma_desc_t cha)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	tx4925_dmaptr->ch[ch].cha = virt_to_phys(cha);
}

static txx927_chdma_desc_t
get_tx4925_dma_chain(unsigned int dmanr)
{
	int ch = dmanr - TXX927_DMA_CHANNEL_START;
	unsigned int cha = tx4925_dmaptr->ch[ch].cha;
	if (cha == 0)
		return 0;
	return (txx927_chdma_desc_t)KSEG1ADDR(cha);
}

#define tx4925_chdma_desc(desc)	((struct tx4925_dma_ch_reg*)desc)

static void
set_tx4925_chdma_mode(txx927_chdma_desc_t desc, unsigned int mode)
{
#ifdef __LITTLE_ENDIAN
	mode |= TX4925_DMA_CCR_LE;
#endif
	tx4925_chdma_desc(desc)->ccr = mode;
}

static void
set_tx4925_chdma_addr(txx927_chdma_desc_t desc, unsigned int sa, unsigned int da)
{
	tx4925_chdma_desc(desc)->sar = sa;
	tx4925_chdma_desc(desc)->dar = da;
}

static void
get_tx4925_chdma_addr(txx927_chdma_desc_t desc, unsigned int *sa, unsigned int *da)
{
	*sa = tx4925_chdma_desc(desc)->sar;
	*da = tx4925_chdma_desc(desc)->dar;
}

static void
set_tx4925_chdma_addr64(txx927_chdma_desc_t desc, unsigned long long sa, unsigned long long da)
{
	tx4925_chdma_desc(desc)->sar = (unsigned long)sa;
	tx4925_chdma_desc(desc)->dar = (unsigned long)da;
}

static void
get_tx4925_chdma_addr64(txx927_chdma_desc_t desc, unsigned long long *sa, unsigned long long *da)
{
	*sa = tx4925_chdma_desc(desc)->sar;
	*da = tx4925_chdma_desc(desc)->dar;
}

static void
set_tx4925_chdma_count(txx927_chdma_desc_t desc, unsigned int count)
{
	tx4925_chdma_desc(desc)->cntr = count;
}

static void
get_tx4925_chdma_count(txx927_chdma_desc_t desc, unsigned int *count)
{
	*count = tx4925_chdma_desc(desc)->cntr;
}

static void
set_tx4925_chdma_inc(txx927_chdma_desc_t desc, unsigned int sai, unsigned int dai)
{
	tx4925_chdma_desc(desc)->sair = sai;
	tx4925_chdma_desc(desc)->dair = dai;
}

static void
set_tx4925_chdma_chain(txx927_chdma_desc_t desc, txx927_chdma_desc_t cha)
{
	tx4925_chdma_desc(desc)->cha = virt_to_phys(cha);
}

static txx927_chdma_desc_t
get_tx4925_chdma_chain(txx927_chdma_desc_t desc)
{
	unsigned int cha = tx4925_chdma_desc(desc)->cha;
	if (cha == 0)
		return 0;
	return (txx927_chdma_desc_t)KSEG1ADDR(cha);
}

static txx927_chdma_desc_t
alloc_tx4925_chdma_cmd(void)
{
	int size = sizeof(struct tx4925_dma_ch_reg);
	void *desc;
	/* kmalloc always return 32-byte aligned address... */
	desc = kmalloc(size, GFP_ATOMIC);
	if (desc) {
		if ((unsigned int)desc & 31)
			panic("dma: kmalloc returns unaligned address.");
		dma_cache_wback_inv((unsigned long)desc, size);
		desc = (txx927_chdma_desc_t)KSEG1ADDR(desc);
	}
	return (txx927_chdma_desc_t)desc;
}

static void
free_tx4925_chdma_cmd(txx927_chdma_desc_t desc)
{
	if (desc)
		kfree((void *)KSEG0ADDR(desc));
}

static struct txx927_dma_ops tx4925_dma_ops = {
	init_dma: init_tx4925_dma,
	enable_dma: enable_tx4925_dma,
	disable_dma: disable_tx4925_dma,
	set_dma_mode: set_tx4925_dma_mode,
	set_dma_addr: set_tx4925_dma_addr,
	set_dma_addr64: set_tx4925_dma_addr64,
	set_dma_count: set_tx4925_dma_count,
	set_dma_inc: set_tx4925_dma_inc,
	get_dma_residue: get_tx4925_dma_residue,
	get_dma_status: get_tx4925_dma_status,
	clear_dma_status: clear_tx4925_dma_status,
	get_dma_irqno: get_tx4925_dma_irqno,
	set_dma_chain: set_tx4925_dma_chain,
	get_dma_chain: get_tx4925_dma_chain,
	set_chdma_mode: set_tx4925_chdma_mode,
	set_chdma_addr: set_tx4925_chdma_addr,
	get_chdma_addr: get_tx4925_chdma_addr,
	set_chdma_addr64: set_tx4925_chdma_addr64,
	get_chdma_addr64: get_tx4925_chdma_addr64,
	set_chdma_count: set_tx4925_chdma_count,
	get_chdma_count: get_tx4925_chdma_count,
	set_chdma_inc: set_tx4925_chdma_inc,
	set_chdma_chain: set_tx4925_chdma_chain,
	get_chdma_chain: get_tx4925_chdma_chain,
	alloc_chdma_cmd: alloc_tx4925_chdma_cmd,
	free_chdma_cmd: free_tx4925_chdma_cmd
};

#endif	/* HAVE_4925_DMA */

struct txx927_dma_ops *txx927_dma_ops;	EXPORT_SYMBOL(txx927_dma_ops);

void __init toshibaboards_dma_setup(void)
{
	switch (mips_cpu.cputype) {
#ifdef HAVE_3927_DMA
	case CPU_TX3927:
		txx927_dma_ops = &tx3927_dma_ops;
		break;
#endif
	case CPU_TX49XX:
		switch (mips_machtype) {
		case MACH_TOSHIBA_RBTX4925:
#ifdef HAVE_4925_DMA
			if (TX4925_REV_PCODE() == 0x4925 ||
			    TX4925_REV_PCODE() == 0x4926)
				txx927_dma_ops = &tx4925_dma_ops;
#endif
			break;
		case MACH_TOSHIBA_TX4927EVB:
		case MACH_TOSHIBA_RBTX4927:
#ifdef HAVE_4927_DMA
			if (TX4927_REV_PCODE() == 0x4927)
				txx927_dma_ops = &tx4927_dma_ops;
#endif
			break;
		}
		break;
	}
}
