/*
 * linux/arch/mips/toshiba-boards/generic/irq_tx4927.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 *
 * $Id: irq_tx4927.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */

/*
 * TX4927 defines 32 IRQs.
 *
 * This file exports one function:
 *	tx4927_irq_init(int irq_base);
 */

#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/ptrace.h>

#include <asm/tx4927.h>

#define irc_dlevel	0
#define irc_elevel	1
static unsigned char irc_level[TX4927_NUM_IR] = {
	7, 7,			/* ECCERR, WTOERR */
	5, 5, 5, 5, 5, 5,	/* INT[5:0] */
	7, 7,			/* SIO */
	5, 5, 5, 5, 5, 5, 5,	/* DMA[3:0], PIO, PDMAC, PCIC */
	6, 6, 6, 0, 0,			/* TMR[2:0] */
	7, 7,			/* PCIERR, PCIPME */
	5, 5,			/* ACLC, ACLCPME */
	0, 0, 0, 0, 0, 0
};

static int tx4927_irq_base = -1;

static void tx4927_irq_enable(unsigned int irq)
{
	int irq_nr = irq - tx4927_irq_base;
	volatile unsigned long *ilrp = &tx4927_ircptr->ilr[(irq_nr % 16 ) / 2];
	int ofs = (irq_nr >= 16 ? 16 : 0) + ((irq_nr & 1) ? 8 : 0);
	*ilrp = (*ilrp & ~(0xff << ofs)) | (irc_level[irq_nr] << ofs);
	/* update IRCSR */
	tx4927_ircptr->imr = 0;
	tx4927_ircptr->imr = irc_elevel;
}

static void tx4927_irq_disable(unsigned int irq)
{
	int irq_nr = irq - tx4927_irq_base;
	volatile unsigned long *ilrp = &tx4927_ircptr->ilr[(irq_nr % 16) / 2];
	int ofs = (irq_nr >= 16 ? 16 : 0) + ((irq_nr & 1) ? 8 : 0);
	*ilrp = (*ilrp & ~(0xff << ofs)) | (irc_dlevel << ofs);
	/* update IRCSR */
	tx4927_ircptr->imr = 0;
	tx4927_ircptr->imr = irc_elevel;
}

static unsigned int tx4927_irq_startup(unsigned int irq)
{
	tx4927_irq_enable(irq);
	return 0;
}
#define	tx4927_irq_shutdown	tx4927_irq_disable

static void tx4927_irq_ack(unsigned int irq)
{
	tx4927_irq_disable(irq);
}
static void tx4927_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		tx4927_irq_enable(irq);
}

static hw_irq_controller tx4927_irq_controller = {
	typename:	"tx4927_irq",
	startup:	tx4927_irq_startup,
	shutdown:	tx4927_irq_shutdown,
	enable:		tx4927_irq_enable,
	disable:	tx4927_irq_disable,
	ack:		tx4927_irq_ack,
	end:		tx4927_irq_end,
	set_affinity:	NULL,
};

void __init
tx4927_irq_init(int irq_base)
{
	int i;

	for (i= irq_base; i< irq_base + TX4927_NUM_IR; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &tx4927_irq_controller;
	}

	tx4927_irq_base = irq_base;

	/* enable interrupt control */
	tx4927_ircptr->cer = TX4927_IRCER_ICE;
	tx4927_ircptr->imr = irc_elevel;
}

int tx4927_irq_to_irq(int irq)
{
	if (irq < 0 || irq > TX4927_NUM_IR)
		return -1;

	return irq + tx4927_irq_base;
}

EXPORT_SYMBOL(tx4927_irq_to_irq);
