/* $Id: pci.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 *
 * TX4925 specific PCI support.
 *
 * based on arch/mips/sni/pci.c
 * Copyright (C) 1997, 1998 Ralf Baechle
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2001 Toshiba Corporation
 */
#include <linux/config.h>
#include <linux/pci.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <asm/byteorder.h>
#include <asm/mct-boards/rbtx4925.h>

#ifdef CONFIG_PCI

int rbtx4925_pci_map_irq(struct pci_dev *dev, u8 slot, u8 pin)
{
#if 0
	int irq = pin;

	/* IRQ rotation */
	irq--;	/* 0-3 */
	if (dev->bus->parent == NULL &&
	    (slot == TX4925_PCIC_IDSEL_AD_TO_SLOT(23))) {
		/* PCI CardSlot (IDSEL=A23) */
		/* PCIA => PCIA */
		irq = (irq + 0 + slot) % 4;
	} else {
		/* PCI Backplane */
		irq = (irq + 3 + slot) % 4;
	}
	irq++;	/* 1-4 */

	switch (irq) {
	case 1:
		irq = RBTX4925_IRQ_IOC_PCIA;
		break;
	case 2:
		irq = RBTX4925_IRQ_IOC_PCIB;
		break;
	case 3:
		irq = RBTX4925_IRQ_IOC_PCIC;
		break;
	case 4:
		irq = RBTX4925_IRQ_IOC_PCID;
		break;
	}
	return irq;
#else
	int irq = 0;

	if ( dev->bus->parent == NULL ) {
		switch ( slot ) {
		case 12:
			irq = 26;
			break;
		case 13:
			irq = 27;
			break;
		case 14:
			irq = 28;
			break;
		case 15:
			irq = 29;
			break;
		}
	}

	return irq;
#endif
}


static int
mkaddr(struct pci_dev *dev, int where)
{
	if (dev->bus->number == 0 &&
	    dev->devfn >= PCI_DEVFN(TX4925_PCIC_MAX_DEVNU, 0))
		return -1;
	tx4925_pcicptr->g2pcfgadrs = ((dev->bus->number & 0xff) << 0x10) |
				     ((dev->devfn & 0xff) << 0x08) |
				     (where  & 0xfc);
	/* clear M_ABORT and Disable M_ABORT Int. */
	tx4925_pcicptr->pcistatus =
		(tx4925_pcicptr->pcistatus & 0x0000ffff) |
		(PCI_STATUS_REC_MASTER_ABORT << 16);
	return 0;
}

static int
check_abort(void)
{
	int code = PCIBIOS_SUCCESSFUL;
	if (tx4925_pcicptr->pcistatus & (PCI_STATUS_REC_MASTER_ABORT << 16)) {
		tx4925_pcicptr->pcistatus =
			(tx4925_pcicptr->pcistatus & 0x0000ffff) |
			(PCI_STATUS_REC_MASTER_ABORT << 16);
		code = PCIBIOS_DEVICE_NOT_FOUND;
	}
	return code;
}

static inline u8 icd_readb(int offset)
{
	return *(volatile u8 *)((ulong)&tx4925_pcicptr->g2pcfgdata | offset);
}
static inline u16 icd_readw(int offset)
{
	return *(volatile u16 *)((ulong)&tx4925_pcicptr->g2pcfgdata | offset);
}
static inline u32 icd_readl(void)
{
	return tx4925_pcicptr->g2pcfgdata;
}
static inline void icd_writeb(u8 val, int offset)
{
	*(volatile u8 *)((ulong)&tx4925_pcicptr->g2pcfgdata | offset) = val;
}
static inline void icd_writew(u16 val, int offset)
{
	*(volatile u16 *)((ulong)&tx4925_pcicptr->g2pcfgdata | offset) = val;
}
static inline void icd_writel(u32 val)
{
	tx4925_pcicptr->g2pcfgdata = val;
}

/*
 * We can't address 8 and 16 bit words directly.  Instead we have to
 * read/write a 32bit word and mask/modify the data we actually want.
 */
static int rbtx4925_pci_read_config_byte(struct pci_dev *dev, int where,
					  u8 *val)
{
	if (mkaddr(dev, where))
		return -1;
	*val = icd_readb(where & 3);
	return check_abort();
}

static int rbtx4925_pci_read_config_word(struct pci_dev *dev, int where,
					  u16 *val)
{
	if (mkaddr(dev, where))
		return -1;
	*val = le16_to_cpu(icd_readw(where & 3));
	return check_abort();
}

static int rbtx4925_pci_read_config_dword(struct pci_dev *dev, int where,
					   u32 *val)
{
	if (mkaddr(dev, where))
		return -1;
	*val = le32_to_cpu(icd_readl());
	return check_abort();
}

static int rbtx4925_pci_write_config_byte(struct pci_dev *dev, int where,
					   u8 val)
{
	if (mkaddr(dev, where))
		return -1;
	icd_writeb(val, where & 3);
	return check_abort();
}

static int rbtx4925_pci_write_config_word(struct pci_dev *dev, int where,
					   u16 val)
{
	if (mkaddr(dev, where))
		return -1;
	icd_writew(cpu_to_le16(val), where & 3);
	return check_abort();
}

static int rbtx4925_pci_write_config_dword(struct pci_dev *dev, int where,
					    u32 val)
{
	if (mkaddr(dev, where))
		return -1;
	icd_writel(cpu_to_le32(val));
	return check_abort();
}

struct pci_ops rbtx4925_pci_ops = {
	read_byte:      rbtx4925_pci_read_config_byte,
	read_word:      rbtx4925_pci_read_config_word,
	read_dword:     rbtx4925_pci_read_config_dword,
	write_byte:     rbtx4925_pci_write_config_byte,
	write_word:     rbtx4925_pci_write_config_word,
	write_dword:    rbtx4925_pci_write_config_dword
};

#endif /* CONFIG_PCI */
