/* $Id: pci.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 *
 * TOSHIBA SDB specific PCI support.
 *
 * based on arch/mips/sni/pci.c
 * Copyright (C) 1997, 1998 Ralf Baechle
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1999-2001 Toshiba Corporation
 */
#include <linux/config.h>
#include <linux/pci.h>
#include <linux/types.h>
#include <asm/byteorder.h>
#include <asm/toshiba-boards/tsdb.h>

#ifdef CONFIG_PCI

int tsdb_pci_map_irq(struct pci_dev *dev, u8 slot, u8 pin)
{
	int irq = pin;

	/* IRQ rotation (see CompactPCI Specification) */
	irq--;	/* 0-3 */
	irq = (irq + 32 - slot) % 4;
	irq++;	/* 1-4 */

	switch (irq) {
	case 1:
		irq = TSDB_IRQ_PCI_INTA;
		break;
	case 2:
		irq = TSDB_IRQ_PCI_INTB;
		break;
	case 3:
		irq = TSDB_IRQ_PCI_INTC;
		break;
	case 4:
		irq = TSDB_IRQ_PCI_INTD;
		break;
	}
	return irq;
}


#define	TSDB_PCI_CONFIG_BASE	KSEG1ADDR(TSDB_LB_PCI_APERTURE_0)

#ifdef __BIG_ENDIAN
#define DO_AUTOSWAP
#endif

static struct v320usc_reg * const usc_reg = (struct v320usc_reg *)TSDB_USC_BASE;

/* setup PCI Aperture 0 for Configuration Read/Write */
static unsigned long
config_begin(struct pci_dev *dev, int where,
	     unsigned long *basep)
{
	unsigned long new_base;
	unsigned long addr;

	*basep = usc_reg->lb_pci_base[0];
	new_base = USC_LB_PCI_BASE_BASE(TSDB_LB_PCI_APERTURE_0) |
		USC_LB_PCI_BASE_PCI_CMD_CONFIG |
#ifdef DO_AUTOSWAP
		USC_LB_PCI_BASE_SWAP(USC_AUTOSWAP) |
#else
		USC_LB_PCI_BASE_SWAP(USC_NOSWAP) |
#endif
		USC_LB_PCI_BASE_SIZE_16M |
		USC_LB_PCI_BASE_ERR_EN;
	if (dev->bus->number == 0) {
		unsigned long sel;
		/* IDSEL pin of each board is connected to AD31:25 */
		/* Additional PCI devices may use AD24:11 */
		/* (see CompactPCI Specification) */
		sel = PCI_SLOT(dev->devfn) ?
			0x80000000UL >> (PCI_SLOT(dev->devfn) - 1): 0;
		new_base |= USC_LB_PCI_BASE_MAP_ADR(sel) |
			0x0 /* Type 0 */;
		addr = TSDB_PCI_CONFIG_BASE +
			((sel & 0xfff800) | ((dev->devfn & 0x07) << 0x08) | where);
	} else {
		new_base |= USC_LB_PCI_BASE_MAP_ADR(0) |
			0x1 /* Type 1 */;
		addr = TSDB_PCI_CONFIG_BASE +
			((dev->bus->number << 0x10) | (dev->devfn << 0x08) | where);
	}
	usc_reg->lb_pci_base[0] = new_base;
	/* clear M_ABORT */
	usc_reg->pci_stat |= PCI_STATUS_REC_MASTER_ABORT;
	usc_reg->int_stat |= USC_INTF_PCI_M_ABORT;
	return addr;
}
static int
config_end(unsigned long base)
{
	int abort = (usc_reg->pci_stat & PCI_STATUS_REC_MASTER_ABORT);
	usc_reg->lb_pci_base[0] = base;
	if (abort) {
		usc_reg->pci_stat |= PCI_STATUS_REC_MASTER_ABORT;
		usc_reg->int_stat |= USC_INTF_PCI_M_ABORT;
		/* avoid spurious interrupt...(why?) */
		*tsdb_fpga_rev_ptr = *tsdb_fpga_rev_ptr;
		wbflush();
	}
	return abort;
}

static int tsdb_pci_read_config_byte(struct pci_dev *dev, int where,
				     u8 *val)
{
	unsigned long base;

	*val = *(volatile u8 *)config_begin(dev, where, &base);
	if (config_end(base))
		return PCIBIOS_DEVICE_NOT_FOUND;

	return PCIBIOS_SUCCESSFUL;
}

static int tsdb_pci_read_config_word(struct pci_dev *dev, int where,
				     u16 *val)
{
	u16 res;
	unsigned long base;

	res = *(volatile u16 *)config_begin(dev, where, &base);
#ifndef DO_AUTOSWAP
	res = le32_to_cpu(res);
#endif
	*val = res;
	if (config_end(base))
		return PCIBIOS_DEVICE_NOT_FOUND;

	return PCIBIOS_SUCCESSFUL;
}

static int tsdb_pci_read_config_dword(struct pci_dev *dev, int where,
				      u32 *val)
{
	u32 res;
	unsigned long base;

	res = *(volatile u32 *)config_begin(dev, where, &base);
#ifndef DO_AUTOSWAP
	res = le32_to_cpu(res);
#endif
	*val = res;
	if (config_end(base))
		return PCIBIOS_DEVICE_NOT_FOUND;

	return PCIBIOS_SUCCESSFUL;
}

static int tsdb_pci_write_config_byte(struct pci_dev *dev, int where,
				      u8 val)
{
	unsigned long base;

	*(volatile u8 *)config_begin(dev, where, &base) = val;
	if (config_end(base))
		return PCIBIOS_DEVICE_NOT_FOUND;

	return PCIBIOS_SUCCESSFUL;
}

static int tsdb_pci_write_config_word(struct pci_dev *dev, int where,
				      u16 val)
{
	unsigned long base;

#ifndef DO_AUTOSWAP
	val = cpu_to_le16(val);
#endif
	*(volatile u16 *)config_begin(dev, where, &base) = val;
	if (config_end(base))
		return PCIBIOS_DEVICE_NOT_FOUND;

	return PCIBIOS_SUCCESSFUL;
}

static int tsdb_pci_write_config_dword(struct pci_dev *dev, int where,
				       u32 val)
{
	unsigned long base;

#ifndef DO_AUTOSWAP
	val = cpu_to_le32(val);
#endif
	*(volatile u32 *)config_begin(dev, where, &base) = val;
	if (config_end(base))
		return PCIBIOS_DEVICE_NOT_FOUND;

	return PCIBIOS_SUCCESSFUL;
}

struct pci_ops tsdb_pci_ops = {
	read_byte:      tsdb_pci_read_config_byte,
	read_word:      tsdb_pci_read_config_word,
	read_dword:     tsdb_pci_read_config_dword,
	write_byte:     tsdb_pci_write_config_byte,
	write_word:     tsdb_pci_write_config_word,
	write_dword:    tsdb_pci_write_config_dword
};

#endif /* CONFIG_PCI */
