/* $Id: pci.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 *
 * TX3927 specific PCI support.
 *
 * based on arch/mips/sni/pci.c
 * Copyright (C) 1997, 1998 Ralf Baechle
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 */
#include <linux/config.h>
#include <linux/pci.h>
#include <linux/types.h>
#include <asm/byteorder.h>
#include <asm/toshiba-boards/jmr3927.h>

#ifdef CONFIG_PCI

int jmr3927_pci_map_irq(struct pci_dev *dev, u8 slot, u8 pin)
{
#if 1	/* Modified by Ethan on 04/07/2002 */

#define PCI_IDSEL_ADDR_TO_SLOT(addr)	(addr - 11)

	int	irq = 0;

	if ( dev->bus->parent == NULL )
	{
		switch ( slot )
		{
		case PCI_IDSEL_ADDR_TO_SLOT(23):
			irq = 16;
			break;
		case PCI_IDSEL_ADDR_TO_SLOT(24):
			irq = 17;
			break;
		}
	}

	return irq;
#else 
	int irq = pin;

	/* IRQ rotation (PICMG) */
	irq--;	/* 0-3 */
	if (dev->bus->parent == NULL &&
	    slot == TX3927_PCIC_IDSEL_AD_TO_SLOT(23)) {
		/* PCI CardSlot (IDSEL=A23, DevNu=12) */
		/* PCIA => PCIC (IDSEL=A23) */
		/* NOTE: JMR3927 JP1 must be set to OPEN */
		irq = (irq + 2) % 4;
	} else if (dev->bus->parent == NULL &&
		   slot == TX3927_PCIC_IDSEL_AD_TO_SLOT(22)) {
		/* PCI CardSlot (IDSEL=A22, DevNu=11) */
		/* PCIA => PCIA (IDSEL=A22) */
		/* NOTE: JMR3927 JP1 must be set to OPEN */
		irq = (irq + 0) % 4;
	} else {
		/* PCI Backplane */
		irq = (irq + 3 + slot) % 4;
	}
	irq++;	/* 1-4 */

	printk("PCI: %04x:%04x, slot = %d, pin = %d, irq = %d\n", dev->vendor, dev->device, slot, pin, irq);
	printk("PCI: PCIA = %d, PCIB = %d, PCIC = %d, PCID = %d\n",
		JMR3927_IRQ_IOC_PCIA, JMR3927_IRQ_IOC_PCIB,
		JMR3927_IRQ_IOC_PCIC, JMR3927_IRQ_IOC_PCID);

	switch (irq) {
	case 1:
		irq = JMR3927_IRQ_IOC_PCIA;
		break;
	case 2:
		irq = JMR3927_IRQ_IOC_PCIB;
		break;
	case 3:
		irq = JMR3927_IRQ_IOC_PCIC;
		break;
	case 4:
		irq = JMR3927_IRQ_IOC_PCID;
		break;
	}

	/* Check OnBoard Ethernet (IDSEL=A24, DevNu=13) */
	if (dev->bus->parent == NULL &&
	    slot == TX3927_PCIC_IDSEL_AD_TO_SLOT(24)) {
		extern int jmi_ether_irq;
		/* check this irq line was reserved for JMI39IO2 ether */
		if (jmi_ether_irq != JMR3927_IRQ_ETHER)
			irq = JMR3927_IRQ_ETHER;
		else
			irq = 0;	/* disable */
	}
	return irq;
#endif
}


static int
mkaddr(struct pci_dev *dev, int where)
{
	if (dev->bus->number == 0 &&
	    dev->devfn >= PCI_DEVFN(TX3927_PCIC_MAX_DEVNU, 0))
		return -1;
	tx3927_pcicptr->ica = ((dev->bus->number & 0xff) << 0x10) |
	                      ((dev->devfn & 0xff) << 0x08) |
	                      (where & 0xfc);
	/* clear M_ABORT and Disable M_ABORT Int. */
	tx3927_pcicptr->pcistat |= PCI_STATUS_REC_MASTER_ABORT;
	tx3927_pcicptr->pcistatim &= ~PCI_STATUS_REC_MASTER_ABORT;
	return 0;
}

static int
check_abort(void)
{
	int code = PCIBIOS_SUCCESSFUL;
	if (tx3927_pcicptr->pcistat & PCI_STATUS_REC_MASTER_ABORT) {
		tx3927_pcicptr->pcistat |= PCI_STATUS_REC_MASTER_ABORT;
		tx3927_pcicptr->pcistatim |= PCI_STATUS_REC_MASTER_ABORT;
		code =PCIBIOS_DEVICE_NOT_FOUND;
	}
	return code;
}

/*
 * We can't address 8 and 16 bit words directly.  Instead we have to
 * read/write a 32bit word and mask/modify the data we actually want.
 */
static int jmr3927_pci_read_config_byte(struct pci_dev *dev, int where,
					u8 *val)
{
	if (mkaddr(dev, where))
		return -1;
	*val = *(volatile u8 *)((ulong)&tx3927_pcicptr->icd | (where&3));
	return check_abort();
}

static int jmr3927_pci_read_config_word(struct pci_dev *dev, int where,
					u16 *val)
{
	if (mkaddr(dev, where))
		return -1;
	*val = le16_to_cpu(*(volatile u16 *)((ulong)&tx3927_pcicptr->icd | (where&3)));
	return check_abort();
}

static int jmr3927_pci_read_config_dword(struct pci_dev *dev, int where,
					 u32 *val)
{
	if (mkaddr(dev, where))
		return -1;
	*val = le32_to_cpu(tx3927_pcicptr->icd);
	return check_abort();
}

static int jmr3927_pci_write_config_byte(struct pci_dev *dev, int where,
					 u8 val)
{
	if (mkaddr(dev, where))
		return -1;
	*(volatile u8 *)((ulong)&tx3927_pcicptr->icd | (where&3)) = val;
	return check_abort();
}

static int jmr3927_pci_write_config_word(struct pci_dev *dev, int where,
					 u16 val)
{
	if (mkaddr(dev, where))
		return -1;
	*(volatile u16 *)((ulong)&tx3927_pcicptr->icd | (where&3)) = cpu_to_le16(val);
	return check_abort();
}

static int jmr3927_pci_write_config_dword(struct pci_dev *dev, int where,
					  u32 val)
{
	if (mkaddr(dev, where))
		return -1;
	tx3927_pcicptr->icd = cpu_to_le32(val);
	return check_abort();
}

struct pci_ops jmr3927_pci_ops = {
	read_byte:      jmr3927_pci_read_config_byte,
	read_word:      jmr3927_pci_read_config_word,
	read_dword:     jmr3927_pci_read_config_dword,
	write_byte:     jmr3927_pci_write_config_byte,
	write_word:     jmr3927_pci_write_config_word,
	write_dword:    jmr3927_pci_write_config_dword
};

#endif /* CONFIG_PCI */
