#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/rtc.h>		// for struct rtc_time
#include <asm/delay.h>
#include <asm/time.h>		// for rtc_set_time/rtc_get_time

//#define DEBUG_S3511A

#define S3511A_ADDR		0xb4001000
#define S3511A_SCK		0x04
#define S3511A_CS		0x02
#define S3511A_SIO		0x01

#define S3511A_CMD_READ			0x01
#define S3511A_CMD_WRITE		0x00

#define S3511A_CMD_RESET		0x60
#define S3511A_CMD_STATUS		0x62
#define S3511A_CMD_REAL_TIME_DATA1	0x64
#define S3511A_CMD_REAL_TIME_DATA2	0x66
#define S3511A_CMD_FREQ			0x68

#define S3511A_STATUS_POWER		0x80
#define S3511A_STATUS_24EXPR		0x40
#define S3511A_STATUS_INTAE		0x20
#define S3511A_STATUS_INTME		0x08
#define S3511A_STATUS_INTFE		0x02

#ifndef BCD_TO_BIN
#define BCD_TO_BIN(val) ((val)=((val)&15) + ((val)>>4)*10)
#endif

#ifndef BIN_TO_BCD
#define BIN_TO_BCD(val) ((val)=(((val)/10)<<4) + (val)%10)
#endif

#ifdef DEBUG_S3511A
struct timer_list s3511a_timer;
#endif

#define s3511a_delay()		udelay(2)	// clock pulse width at least 1us
#define s3511a_short_delay()	udelay(1)	// setup/hold time at least 0.2us

void s3511a_access_prolog(void)
{
	*(volatile unsigned char*) S3511A_ADDR = S3511A_SCK | S3511A_SIO;
	s3511a_short_delay();	// setup time before CS rising
	*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SCK | S3511A_SIO;	
	s3511a_short_delay();	// hold time after CS rising
}

void s3511a_access_epilog(void)
{
	*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SCK | S3511A_SIO;
	s3511a_short_delay();	// setup time before CS falling
	*(volatile unsigned char*) S3511A_ADDR = S3511A_SCK | S3511A_SIO;
	s3511a_short_delay();	// hold time after CS falling
}

unsigned char s3511a_read_byte(void)
{
	int	i;
	unsigned char	regval, val;

	val = 0;
	for (i = 0; i < 8; i++)
	{
		s3511a_delay();
		*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SIO;
		s3511a_delay();
		*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SCK | S3511A_SIO;
		s3511a_delay();
		regval = *(volatile unsigned char*) S3511A_ADDR;
		// Any read/write operation performed by the real-time data
		// access command sends or receives the data from LSB.
		if ( regval & S3511A_SIO )
			val |= (1 << i);
	}

	return val;
}

void s3511a_write_byte(unsigned char data)
{
	unsigned char	mask;

	for (mask = 0x01; mask != 0; mask <<= 1)
	{
		if ( mask & data )
		{
			s3511a_delay();
			*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SIO;
			s3511a_delay();
			*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SCK | S3511A_SIO;
		}
		else
		{
			s3511a_delay();
			*(volatile unsigned char *) S3511A_ADDR = S3511A_CS;
			s3511a_delay();
			*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SCK;
		}
	}
}

// command should be sent from MSB to LSB.
void s3511a_write_command(unsigned char cmd)
{
	unsigned char	mask;

	for (mask = 0x80; mask != 0; mask >>= 1)
	{
		if ( mask & cmd )
		{
			s3511a_delay();
			*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SIO;
			s3511a_delay();
			*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SCK | S3511A_SIO;
		}
		else
		{
			s3511a_delay();
			*(volatile unsigned char *) S3511A_ADDR = S3511A_CS;
			s3511a_delay();
			*(volatile unsigned char*) S3511A_ADDR = S3511A_CS | S3511A_SCK;
		}
	}
}

unsigned long s3511a_get_time(void)
{
	int	i;
	unsigned char	val, rtctime[7];
	unsigned char	year, month, mday, hour, minute, second;

	s3511a_access_prolog();
	s3511a_write_command(S3511A_CMD_REAL_TIME_DATA1 | S3511A_CMD_READ);
	for (i = 0; i < 7; i++)
	{
		val = s3511a_read_byte();
		rtctime[i] = val;
	}
	s3511a_access_epilog();

	// clear AM/PM bit
	rtctime[4] &= 0x7f;
	
	year = BCD_TO_BIN(rtctime[0]);
	month = BCD_TO_BIN(rtctime[1]);
	mday = BCD_TO_BIN(rtctime[2]);
	hour = BCD_TO_BIN(rtctime[4]);
	minute = BCD_TO_BIN(rtctime[5]);
	second = BCD_TO_BIN(rtctime[6]);

//#ifdef DEBUG_S3511A
#if 1
	printk("S3511A: %02x %02x %02x %02x %02x %02x %02x\n",
		rtctime[0], rtctime[1], rtctime[2],
		rtctime[3], rtctime[4], rtctime[5], rtctime[6]);
#endif

	return mktime(year+2000, month, mday, hour, minute, second);
}

int s3511a_set_time(unsigned long time)
{
	int		year;
	unsigned char	month, mday, wday, hour, minute, second;
	struct rtc_time tm;

	to_tm(time, &tm);

	year = tm.tm_year;
	month = tm.tm_mon + 1;
	mday = tm.tm_mday;
	wday = tm.tm_wday;
	hour = tm.tm_hour;
	minute = tm.tm_min;
	second = tm.tm_sec;

#ifdef DEBUG_S3511A
	printk("S3511A SET: %02d/%02d/%02d %02d:%02d:%02d\n", year, month, mday, hour, minute, second);
#endif

	BIN_TO_BCD(year);
	BIN_TO_BCD(month);
	BIN_TO_BCD(mday);
	BIN_TO_BCD(wday);
	BIN_TO_BCD(hour);
	BIN_TO_BCD(minute);
	BIN_TO_BCD(second);

	s3511a_access_prolog();
	s3511a_write_command(S3511A_CMD_REAL_TIME_DATA1 | S3511A_CMD_WRITE);

	s3511a_write_byte(year);
	s3511a_write_byte(month);
	s3511a_write_byte(mday);
	s3511a_write_byte(wday);
	s3511a_write_byte(hour);
	s3511a_write_byte(minute);
	s3511a_write_byte(second);

	s3511a_access_epilog();

	return 0;
}

void s3511a_reset(void)
{
	s3511a_access_prolog();
	s3511a_write_command(S3511A_CMD_RESET | S3511A_CMD_WRITE);
	s3511a_access_epilog();
}

unsigned char s3511a_get_status(void)
{
	unsigned char	val;

	s3511a_access_prolog();
	s3511a_write_command(S3511A_CMD_STATUS | S3511A_CMD_READ);
	val = s3511a_read_byte();
	s3511a_access_epilog();
	return val;
}

void s3511a_set_status(unsigned char status)
{
	s3511a_access_prolog();
	s3511a_write_command(S3511A_CMD_STATUS | S3511A_CMD_WRITE);
	s3511a_write_byte(status);
	s3511a_access_prolog();
}

void s3511a_set_frequency(unsigned short freq)
{
	s3511a_access_prolog();
	s3511a_write_command(S3511A_CMD_FREQ | S3511A_CMD_WRITE);
	s3511a_write_byte((unsigned char) (freq & 0xff));
	s3511a_write_byte((unsigned char) ((freq >> 8) & 0xff));
	s3511a_access_epilog();
}

#ifdef DEBUG_S3511A
void s3511a_callback(unsigned long dummy)
{
	s3511a_get_time();
	init_timer(&s3511a_timer);
	s3511a_timer.function = s3511a_callback;
	s3511a_timer.expires = jiffies + HZ;
	add_timer(&s3511a_timer);
}
#endif
			
void __init s3511a_init(void)
{
	unsigned char status;

	status = s3511a_get_status();
#ifdef DEBUG_S3511A
	printk("S3511A: status = %02x\n", status);
#endif
	if ( status & S3511A_STATUS_POWER )
	{
		s3511a_reset();
	}

	// Make the INT# signal output using selected frequency.
	// INTFE = 1
	s3511a_set_frequency(0x0010);	// 2kHz
	s3511a_set_status(S3511A_STATUS_24EXPR | S3511A_STATUS_INTFE);

	rtc_set_time = s3511a_set_time;
	rtc_get_time = s3511a_get_time;

#ifdef DEBUG_S3511A
	init_timer(&s3511a_timer);
	s3511a_timer.function = s3511a_callback;
	s3511a_timer.expires = jiffies + HZ;
	add_timer(&s3511a_timer);
#endif
}

