/*
 *  linux/arch/mips/toshiba-boards/rbtx4925/irq.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2001 Toshiba Corporation
 *
 * $Id: irq.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */
#include <linux/config.h>
#include <linux/init.h>

#include <linux/errno.h>
#include <linux/kernel_stat.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/timex.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#ifdef CONFIG_BLK_DEV_IDEPCI
#include <linux/pci.h>
#endif

#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/mipsregs.h>
#include <asm/system.h>

#include <asm/ptrace.h>
#include <asm/processor.h>
#include <asm/toshiba-boards/irq.h>
#include <asm/toshiba-boards/rbtx4925.h>
#include <asm/toshiba-boards/jmi39io2.h>

#if RBTX4925_IRQ_END > NR_IRQS
#error RBTX4925_IRQ_END > NR_IRQS
#endif

#define RBTX4925_IRQ_ISACINT	RBTX4925_IRQ_IRC_INT(JMI39IO2_INT_ISAC)

static int rbtx4925_gen_iack(void)
{
	/* generate ACK cycle */
#ifdef __BIG_ENDIAN
	return (tx4925_pcicptr->g2pintack >> 24) & 0xff;
#else
	return tx4925_pcicptr->g2pintack & 0xff;
#endif
}

/*
 * RBTX4925 IOC controller definition
 */

#define flush_wb()	\
	__asm__ __volatile__( \
		".set push\n\t" \
		".set mips2\n\t" \
		"sync\n\t" \
		".set pop\n\t")

static void rbtx4925_ioc_irq_enable(unsigned int irq)
{
	int irq_nr = irq - RBTX4925_IRQ_IOC;
	/* 0: mask */
	*rbtx4925_imask_ptr |= 1 << irq_nr;
	flush_wb();
}

static void rbtx4925_ioc_irq_disable(unsigned int irq)
{
	int irq_nr = irq - RBTX4925_IRQ_IOC;
	/* 0: mask */
	*rbtx4925_imask_ptr &= ~(1 << irq_nr);
	flush_wb();
}

static unsigned int rbtx4925_ioc_irq_startup(unsigned int irq)
{
	rbtx4925_ioc_irq_enable(irq);
	return 0;
}
#define	rbtx4925_ioc_irq_shutdown	rbtx4925_ioc_irq_disable

#define rbtx4925_ioc_irq_ack	rbtx4925_ioc_irq_disable
static void rbtx4925_ioc_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		rbtx4925_ioc_irq_enable(irq);
}

static hw_irq_controller rbtx4925_ioc_irq_controller = {
	typename:	"IOC",
	startup:	rbtx4925_ioc_irq_startup,
	shutdown:	rbtx4925_ioc_irq_shutdown,
	enable:		rbtx4925_ioc_irq_enable,
	disable:	rbtx4925_ioc_irq_disable,
	ack:		rbtx4925_ioc_irq_ack,
	end:		rbtx4925_ioc_irq_end,
	set_affinity:	NULL,
};

static void rbtx4925_ioc2_irq_enable(unsigned int irq)
{
	int irq_nr = irq - RBTX4925_IRQ_IOC2;
	/* 0: mask */
	*rbtx4925_imask2_ptr |= 1 << irq_nr;
	flush_wb();
}

static void rbtx4925_ioc2_irq_disable(unsigned int irq)
{
	int irq_nr = irq - RBTX4925_IRQ_IOC2;
	/* 0: mask */
	*rbtx4925_imask2_ptr &= ~(1 << irq_nr);
	flush_wb();
}

static unsigned int rbtx4925_ioc2_irq_startup(unsigned int irq)
{
	rbtx4925_ioc2_irq_enable(irq);
	return 0;
}
#define	rbtx4925_ioc2_irq_shutdown	rbtx4925_ioc2_irq_disable

#define rbtx4925_ioc2_irq_ack	rbtx4925_ioc2_irq_disable
static void rbtx4925_ioc2_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		rbtx4925_ioc2_irq_enable(irq);
}

static hw_irq_controller rbtx4925_ioc2_irq_controller = {
	typename:	"IOC2",
	startup:	rbtx4925_ioc2_irq_startup,
	shutdown:	rbtx4925_ioc2_irq_shutdown,
	enable:		rbtx4925_ioc2_irq_enable,
	disable:	rbtx4925_ioc2_irq_disable,
	ack:		rbtx4925_ioc2_irq_ack,
	end:		rbtx4925_ioc2_irq_end,
	set_affinity:	NULL,
};

static void rbtx4925_pcmcia_irq_enable(unsigned int irq)
{
	int irq_nr = irq - RBTX4925_IRQ_PCMCIA;
	L1121_outb(L1121_inb(L1121_IER) | (1 << irq_nr), L1121_IER);
	flush_wb();
}

static void rbtx4925_pcmcia_irq_disable(unsigned int irq)
{
	int irq_nr = irq - RBTX4925_IRQ_PCMCIA;
	L1121_outb(L1121_inb(L1121_IER) & ~(1 << irq_nr), L1121_IER);
	flush_wb();
}

static unsigned int rbtx4925_pcmcia_irq_startup(unsigned int irq)
{
	int irq_nr = irq - RBTX4925_IRQ_PCMCIA;
	/* setup edge sense based on current state */
	L1121_outb((L1121_inb(L1121_ESNR) & ~(1 << irq_nr)) |
		   (L1121_inb(L1121_SR) & (1 << irq_nr)),
		   L1121_ESNR);
	rbtx4925_pcmcia_irq_enable(irq);
	return 0;
}
#define	rbtx4925_pcmcia_irq_shutdown	rbtx4925_pcmcia_irq_disable

static void rbtx4925_pcmcia_irq_ack(unsigned int irq)
{
	int irq_nr = irq - RBTX4925_IRQ_PCMCIA;
	/* setup edge sense based on current state */
	L1121_outb((L1121_inb(L1121_ESNR) & ~(1 << irq_nr)) |
		   (L1121_inb(L1121_SR) & (1 << irq_nr)),
		   L1121_ESNR);
	/* clear edge interrupt */
	L1121_outb(1 << irq_nr, L1121_ECLR);
	rbtx4925_pcmcia_irq_disable(irq);
}

static void rbtx4925_pcmcia_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		rbtx4925_pcmcia_irq_enable(irq);
}

/* IOC2 of RBTX4925 has only one PCMCIA interrupt.  To distinguish CS
   interrupt from CARD interrupt, we install the irq_controller for
   L1121 PCMCIA controller */
static hw_irq_controller rbtx4925_pcmcia_irq_controller = {
	typename:	"PCMCIA",
	startup:	rbtx4925_pcmcia_irq_startup,
	shutdown:	rbtx4925_pcmcia_irq_shutdown,
	enable:		rbtx4925_pcmcia_irq_enable,
	disable:	rbtx4925_pcmcia_irq_disable,
	ack:		rbtx4925_pcmcia_irq_ack,
	end:		rbtx4925_pcmcia_irq_end,
	set_affinity:	NULL,
};

/* find first bit set */
static inline int ffs8(unsigned char x)
{
	int r = 1;

	if (!x)
		return 0;
	if (!(x & 0xf)) {
		x >>= 4;
		r += 4;
	}
	if (!(x & 3)) {
		x >>= 2;
		r += 2;
	}
	if (!(x & 1)) {
		x >>= 1;
		r += 1;
	}
	return r;
}

static int rbtx4925_ioc_irqroute(void)
{
	unsigned char istat = *rbtx4925_imstat_ptr;
	int irq = ffs8(istat);
	if (irq != 0)
		return RBTX4925_IRQ_IOC + (irq - 1);
	/* if no IOC interrupt, check IOC2 interrupt */
	istat = *rbtx4925_imstat2_ptr;
	irq = ffs8(istat);
	if (irq != 0)
		return RBTX4925_IRQ_IOC2 + (irq - 1);
	return -1;
}

extern int rbtx4925_use_r4k_timer;
static int rbtx4925_irc_irqdispatch(struct pt_regs *regs)
{
	int irq = 0;
	unsigned int csr = (unsigned int)-1;

#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
	tx_fpu_c_bug_fixup(regs);
#endif
	if (rbtx4925_use_r4k_timer && (regs->cp0_cause & CAUSEF_IP7)) {
		irq = RBTX4925_IRQ_LOCAL_TIMER;
	} else if ((regs->cp0_cause & CAUSEF_IP0)) {
		irq = RBTX4925_IRQ_LOCAL_SOFT0;
	} else if ((regs->cp0_cause & CAUSEF_IP1)) {
		irq = RBTX4925_IRQ_LOCAL_SOFT1;
	} else {
		csr = tx4925_ircptr->csr;
		if (csr & TX4925_IRCSR_IF)
			return -1;
		irq = RBTX4925_IRQ_IRC + (csr & 0x1f);

		/* redirect IOC interrupts */
		switch (irq) {
		case RBTX4925_IRQ_IOCINT:
			irq = rbtx4925_ioc_irqroute();
#if 1	/* someone forgot flushing writebuffer...? */
			if (irq < 0)
				return -1;
#endif
			/* redirect PCMCIA interrupts */
			if (irq == RBTX4925_IRQ_IOC_PCMCIA) {
				irq = ffs8(L1121_inb(L1121_ISR));
				if (irq != 0)
					irq = RBTX4925_IRQ_PCMCIA + (irq - 1);
				else
					irq = -1;
				break;
			}
			irq = toshibaboards_i8259_irqroute(irq);
			break;
		case RBTX4925_IRQ_ISACINT:
			irq = jmi39io2_isac_irqroute();
			break;
		}
		if (irq < 0)
			return -1;
	}
#if 1 /* XXX DEBUG */
	*rbtx4925_led_ptr = ~irq;
#endif
	do_IRQ(irq, regs);
	flush_wb();
	return 0;
}

static struct irqaction ioc_action = {
	no_action, 0, 0, "cascade(IOC)", NULL, NULL,
};
static struct irqaction pcmcia_action = {
	no_action, 0, 0, "cascade(PCMCIA)", NULL, NULL,
};
static struct irqaction isac_action = {
	no_action, 0, 0, "cascade(JMI39IO2 ISAC)", NULL, NULL,
};
static struct irqaction irc_action = {
	no_action, 0, 0, "cascade(IRC)", NULL, NULL,
};

static void rbtx4925_pcierr_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
	extern void tx4925_report_pcic_status(void);
	extern void tx4925_dump_pcic_settings(void);
	extern void toshibaboards_dump_pci_config(void);
	extern struct pci_ops *toshibaboards_pci_ops;
	if (!toshibaboards_pci_ops) {
		printk("PCI disabled. disable %s interrupt (%d)\n",
		       irq == RBTX4925_IRQ_IRC_PCIC ? "PCIC" : "PCIERR", irq);
		disable_irq_nosync(irq);
		return;
	}
#ifdef CONFIG_BLK_DEV_IDEPCI
	/* ignore MasterAbort for ide probing... */
	if (irq == RBTX4925_IRQ_IRC_PCIERR &&
	    ((tx4925_pcicptr->pcistatus >> 16) & 0xf900) == PCI_STATUS_REC_MASTER_ABORT) {
		tx4925_pcicptr->pcistatus =
			(tx4925_pcicptr->pcistatus & 0x0000ffff) |
			(PCI_STATUS_REC_MASTER_ABORT << 16);
		return;
	}
#endif
	if (irq == RBTX4925_IRQ_IRC_PCIC)
		printk("PCIC interrupt (irq 0x%x)", irq);
	else
		printk("PCIERR interrupt (irq 0x%x)", irq);
	printk(" at 0x%08lx.\n", regs->cp0_epc);
        printk("ccfg:%08lx, toea:%08lx\n",
	       tx4925_ccfgptr->ccfg, tx4925_ccfgptr->toea);
	tx4925_report_pcic_status();
	show_regs(regs);
	tx4925_dump_pcic_settings();
	toshibaboards_dump_pci_config();
	panic("PCI error.");
}
static struct irqaction pcierr_action = {
	rbtx4925_pcierr_interrupt, SA_INTERRUPT, 0, "PCI error", NULL, NULL,
};

static void rbtx4925_isaerr_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
#if 1
	printk("ISA error interrupt (irq 0x%x).\n", irq);
#endif
}
static struct irqaction isaerr_action = {
	rbtx4925_isaerr_interrupt, SA_INTERRUPT, 0, "ISA error", NULL, NULL,
};

static int tx4925_print_irc_settings = 0;
extern void rbtx4925_setup_be_board_handler(void);

void __init rbtx4925_irq_setup(void)
{
	int i;

	/* Now, interrupt control disabled, */
	/* all IRC interrupts are masked, */
	/* all IRC interrupt mode are Low Active. */

	/* mask all IOC interrupts */
	*rbtx4925_imask_ptr = 0;

	/* clear SoftInt interrupts */
	*rbtx4925_softint_ptr = 0;

	/* Onboard Ether: High Active */
	tx4925_ircptr->cr[(RBTX4925_IRQ_ETHER - RBTX4925_IRQ_IRC) / 8] |=
		TX4925_IRCR_HIGH << (((RBTX4925_IRQ_ETHER - RBTX4925_IRQ_IRC) % 8) * 2);
#ifdef CONFIG_JMI39IO2_NET
	if (have_jmi39io2()) {
		/* JMI90IO2 ETHER (NE2000 compatible 10M-Ether): High Active */
		tx4925_ircptr->cr[TX4925_IR_INT(JMI39IO2_INT_ETHER) / 8] |=
			TX4925_IRCR_HIGH << ((TX4925_IR_INT(JMI39IO2_INT_ETHER) % 8) * 2);
	}
#endif

	toshibaboards_irqdispatch = rbtx4925_irc_irqdispatch;

	/* setup irq descriptors */
	mips_cpu_irq_init(RBTX4925_IRQ_LOCAL);
	tx4925_irq_init(RBTX4925_IRQ_IRC);
	for (i = RBTX4925_IRQ_IOC; i < RBTX4925_IRQ_IOC + RBTX4925_NR_IRQ_IOC; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &rbtx4925_ioc_irq_controller;
	}
	for (i = RBTX4925_IRQ_IOC2; i < RBTX4925_IRQ_IOC2 + RBTX4925_NR_IRQ_IOC2; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &rbtx4925_ioc2_irq_controller;
	}
	for (i = RBTX4925_IRQ_PCMCIA; i < RBTX4925_IRQ_PCMCIA + RBTX4925_NR_IRQ_PCMCIA; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &rbtx4925_pcmcia_irq_controller;
	}

	setup_irq(RBTX4925_IRQ_LOCAL_IRC, &irc_action);
	if (*rbtx4925_fpga_rev_ptr != 0x1f) {
		/* PCI is not supported on FPGA Ver 1f */
		setup_irq(RBTX4925_IRQ_IRC_PCIC, &pcierr_action);
		setup_irq(RBTX4925_IRQ_IRC_PCIERR, &pcierr_action);
	}

	if (have_jmi39io2()) {
		/* JMI39IO2 pulls down INT0.  So we can not use IOC
                   interrupt (including PCI interrupt). */
		printk(KERN_INFO "IOC interrupts not available.\n");

		jmi39io2_isac_irq_init();
		setup_irq(RBTX4925_IRQ_ISACINT, &isac_action);
		setup_irq(RBTX4925_IRQ_ISAC + JMI39IO2_ISAC_INTB_ISAER, &isaerr_action);
	} else {
		setup_irq(RBTX4925_IRQ_IOCINT, &ioc_action);
		setup_irq(RBTX4925_IRQ_IOC_PCMCIA, &pcmcia_action);
	}

	toshibaboards_gen_iack = rbtx4925_gen_iack;

	if (tx4925_print_irc_settings) {
		extern void tx4925_dump_irc_settings(void);
		tx4925_dump_irc_settings();
	}

	rbtx4925_setup_be_board_handler();
}
