/*
 *  linux/arch/mips/toshiba-boards/tx4927evb/irq.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 *
 * $Id: irq.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */
#include <linux/config.h>
#include <linux/init.h>

#include <linux/errno.h>
#include <linux/kernel_stat.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/timex.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#ifdef CONFIG_BLK_DEV_IDEPCI
#include <linux/pci.h>
#endif
#ifdef CONFIG_MAGIC_SYSRQ
#include <linux/sysrq.h>
#endif

#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/mipsregs.h>
#include <asm/system.h>

#include <asm/ptrace.h>
#include <asm/processor.h>
#include <asm/toshiba-boards/irq.h>
#include <asm/toshiba-boards/tx4927evb.h>

#if TX4927EVB_IRQ_END > NR_IRQS
#error TX4927EVB_IRQ_END > NR_IRQS
#endif

static int tx4927evb_gen_iack(void)
{
	/* generate ACK cycle */
#if 1
	return tx4927_pcicptr->g2pintack & (NR_ISA_IRQS - 1);
#else
	return tx4927_pcicptr->g2pintack & 0xff;
#endif
}

/*
 * TX4927EVB IOC controller definition
 */

#define flush_wb()	\
	__asm__ __volatile__( \
		".set push\n\t" \
		".set mips2\n\t" \
		"sync\n\t" \
		".set pop\n\t")

static void tx4927evb_ioc_irq_enable(unsigned int irq)
{
	int ioc_no = (irq - TX4927EVB_IRQ_IOC(0)) / TX4927EVB_NR_IRQ_IOC;
	int ioc_bit = (irq - TX4927EVB_IRQ_IOC(0)) % TX4927EVB_NR_IRQ_IOC;
	/* 0: mask */
	*tx4927evb_imask_ptr(ioc_no) |= 1 << ioc_bit;
	flush_wb();
}

static void tx4927evb_ioc_irq_disable(unsigned int irq)
{
	int ioc_no = (irq - TX4927EVB_IRQ_IOC(0)) / TX4927EVB_NR_IRQ_IOC;
	int ioc_bit = (irq - TX4927EVB_IRQ_IOC(0)) % TX4927EVB_NR_IRQ_IOC;
	/* 0: mask */
	*tx4927evb_imask_ptr(ioc_no) &= ~(1 << ioc_bit);
	flush_wb();
}

static unsigned int tx4927evb_ioc_irq_startup(unsigned int irq)
{
	tx4927evb_ioc_irq_enable(irq);
	return 0;
}
#define	tx4927evb_ioc_irq_shutdown	tx4927evb_ioc_irq_disable

#define tx4927evb_ioc_irq_ack	tx4927evb_ioc_irq_disable
static void tx4927evb_ioc_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		tx4927evb_ioc_irq_enable(irq);
}

static hw_irq_controller tx4927evb_ioc_irq_controller = {
	typename:	"IOC",
	startup:	tx4927evb_ioc_irq_startup,
	shutdown:	tx4927evb_ioc_irq_shutdown,
	enable:		tx4927evb_ioc_irq_enable,
	disable:	tx4927evb_ioc_irq_disable,
	ack:		tx4927evb_ioc_irq_ack,
	end:		tx4927evb_ioc_irq_end,
	set_affinity:	NULL,
};


/* find first bit set */
static inline int ffs8(unsigned char x)
{
	int r = 1;

	if (!x)
		return 0;
	if (!(x & 0xf)) {
		x >>= 4;
		r += 4;
	}
	if (!(x & 3)) {
		x >>= 2;
		r += 2;
	}
	if (!(x & 1)) {
		x >>= 1;
		r += 1;
	}
	return r;
}

static int tx4927evb_ioc_irqroute(int irq)
{
	unsigned char istat;
	int iocint;
	iocint = irq - TX4927EVB_IRQ_IRC_INT(0);
	istat = *tx4927evb_imstat_ptr(iocint);
	irq = ffs8(istat);
	if (irq == 0)
		return -1;
	return TX4927EVB_IRQ_IOC(iocint) + (irq - 1);
}

extern int tx4927evb_use_r4k_timer;
static int tx4927evb_irc_irqdispatch(struct pt_regs *regs)
{
	int irq = 0;
	unsigned int csr = (unsigned int)-1;

#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
	tx_fpu_c_bug_fixup(regs);
#endif
	if (tx4927evb_use_r4k_timer && (regs->cp0_cause & CAUSEF_IP7)) {
		irq = TX4927EVB_IRQ_LOCAL_TIMER;
	} else if ((regs->cp0_cause & CAUSEF_IP0)) {
		irq = TX4927EVB_IRQ_LOCAL_SOFT0;
	} else if ((regs->cp0_cause & CAUSEF_IP1)) {
		irq = TX4927EVB_IRQ_LOCAL_SOFT1;
	} else {
		csr = tx4927_ircptr->csr;
		if (csr & TX4927_IRCSR_IF)
			return -1;
		irq = TX4927EVB_IRQ_IRC + (csr & 0x1f);

		/* redirect IOC interrupts */
		if (TX4927EVB_IRQ_IRC_INT(0) <= irq &&
		    irq < TX4927EVB_IRQ_IRC_INT(TX4927EVB_NUM_IOC_INTREGS)) {
			irq = tx4927evb_ioc_irqroute(irq);
#if 1	/* someone forgot flushing writebuffer...? */
			if (irq < 0)
				return -1;
#endif
			irq = toshibaboards_i8259_irqroute(irq);
		}
		if (irq < 0)
			return -1;
	}
#if 1 /* XXX DEBUG */
	*tx4927evb_led_ptr = ~irq;
#endif
	do_IRQ(irq, regs);
	flush_wb();
	return 0;
}

static struct irqaction ioc_action = {
	no_action, 0, 0, "cascade(IOC)", NULL, NULL,
};
static struct irqaction irc_action = {
	no_action, 0, 0, "cascade(IRC)", NULL, NULL,
};

#ifdef CONFIG_MAGIC_SYSRQ
static int pushsw_sysrq;
static int __init tx4927evb_pushsw_setup(char *str)
{
	pushsw_sysrq = *str;
	return 0;
}
__setup("pushsw=", tx4927evb_pushsw_setup);
#endif

static void tx4927evb_pushsw_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
#ifdef CONFIG_MAGIC_SYSRQ
	if (pushsw_sysrq)
		handle_sysrq(pushsw_sysrq, regs, NULL, NULL);
	else
#endif
		tx4927evb_leddisp_put(regs->cp0_epc);
	*tx4927evb_istat_ptr(2) = TX4927EVB_INT2F_PUSHSW;
}
static struct irqaction pushsw_action = {
	tx4927evb_pushsw_interrupt, SA_INTERRUPT, 0, "INT Switch", NULL, NULL,
};

static void tx4927evb_pcierr_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
	extern void tx4927_report_pcic_status(void);
	extern void tx4927_dump_pcic_settings(void);
	extern void toshibaboards_dump_pci_config(void);
	extern struct pci_ops *toshibaboards_pci_ops;
	if (!toshibaboards_pci_ops) {
		printk("PCI disabled. disable %s interrupt (%d)\n",
		       irq == TX4927EVB_IRQ_IRC_PCIC ? "PCIC" : "PCIERR", irq);
		disable_irq_nosync(irq);
		return;
	}
#ifdef CONFIG_BLK_DEV_IDEPCI
	/* ignore MasterAbort for ide probing... */
	if (irq == TX4927EVB_IRQ_IRC_PCIERR &&
	    ((tx4927_pcicptr->pcistatus >> 16) & 0xf900) == PCI_STATUS_REC_MASTER_ABORT) {
		tx4927_pcicptr->pcistatus =
			(tx4927_pcicptr->pcistatus & 0x0000ffff) |
			(PCI_STATUS_REC_MASTER_ABORT << 16);
		return;
	}
#endif
	if (irq == TX4927EVB_IRQ_IRC_PCIC)
		printk("PCIC interrupt (irq 0x%x)", irq);
	else
		printk("PCIERR interrupt (irq 0x%x)", irq);
	printk(" at 0x%08lx.\n", regs->cp0_epc);
        printk("ccfg:%08lx, tear:%02lx_%08lx\n",
               (unsigned long)tx4927_ccfgptr->ccfg,
	       (unsigned long)(tx4927_ccfgptr->tear >> 32),
	       (unsigned long)tx4927_ccfgptr->tear);
	tx4927_report_pcic_status();
	show_regs(regs);
	tx4927_dump_pcic_settings();
	toshibaboards_dump_pci_config();
	panic("PCI error.");
}
static struct irqaction pcierr_action = {
	tx4927evb_pcierr_interrupt, SA_INTERRUPT, 0, "PCI error", NULL, NULL,
};

static int tx4927_print_irc_settings = 0;

extern void tx4927evb_setup_be_board_handler(void);

void __init tx4927evb_irq_setup(void)
{
	int i;

	/* Now, interrupt control disabled, */
	/* all IRC interrupts are masked, */
	/* all IRC interrupt mode are Low Active. */

	/* mask all IOC interrupts */
	for (i = 0; i < TX4927EVB_NUM_IOC_INTREGS; i++)
		*tx4927evb_imask_ptr(i) = 0;

	/* clear PushSW/NMI interrupts */
#if 1 /* XXX DEBUG */
	*tx4927evb_istat_ptr(2) = TX4927EVB_INT2F_PUSHSW;
#else
	*tx4927evb_istat_ptr(2) = 0; /* TX4927EVB_INT2F_PUSHSW */
#endif
	*tx4927evb_nmi_stat_ptr = 0; /* TX4927EVB_INTF_NMI_PSW */

	/* clear SoftReset/SoftInt interrupts */
	*tx4927evb_softreset_ptr = 0;
#if 1 /* XXX DEBUG */
	*tx4927evb_softint_ptr = 1;
#else
	*tx4927evb_softint_ptr = 0;
#endif

	/* clear PCI Reset interrupts */
	*tx4927evb_pcireset_ptr = 0;

	toshibaboards_irqdispatch = tx4927evb_irc_irqdispatch;

	/* setup irq descriptors */
	mips_cpu_irq_init(TX4927EVB_IRQ_LOCAL);
	tx4927_irq_init(TX4927EVB_IRQ_IRC);
	for (i = 0; i < TX4927EVB_NUM_IOC_INTREGS; i++) {
		int j, irq_base = TX4927EVB_IRQ_IOC(i);
		for (j= irq_base; j < irq_base + TX4927EVB_NR_IRQ_IOC; j++) {
			irq_desc[j].status = IRQ_DISABLED;
			irq_desc[j].action = NULL;
			irq_desc[j].depth = 1;
			irq_desc[j].handler = &tx4927evb_ioc_irq_controller;
		}
	}

	setup_irq(TX4927EVB_IRQ_LOCAL_IRC, &irc_action);
	for (i = 0; i < TX4927EVB_NUM_IOC_INTREGS; i++) {
		setup_irq(TX4927EVB_IRQ_IRC_INT(i), &ioc_action);
	}
	setup_irq(TX4927EVB_IRQ_IRC_PCIC, &pcierr_action);
	setup_irq(TX4927EVB_IRQ_IRC_PCIERR, &pcierr_action);
	setup_irq(TX4927EVB_IRQ_IOC_PUSHSW, &pushsw_action);

	toshibaboards_gen_iack = tx4927evb_gen_iack;

	if (tx4927_print_irc_settings) {
		extern void tx4927_dump_irc_settings(void);
		tx4927_dump_irc_settings();
	}

	tx4927evb_setup_be_board_handler();
}
