/* $Id: hdsp253x.c,v 1.1.1.1 2004/04/07 08:36:52 louistsai Exp $
 *
 * HDSP-253X Alphanumeric LED Display Driver for Linux.
 *
 * This is a little driver that lets a user-level program access
 * the LED Display.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1999-2001 Toshiba Corporation
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/bootinfo.h>
#ifdef CONFIG_TOSHIBA_SDB
#include <asm/toshiba-boards/tsdb.h>
#endif
#ifdef CONFIG_TOSHIBA_TX4927EVB
#include <asm/toshiba-boards/tx4927evb.h>
#endif

#define LDISP_VERSION		"0.3"
#define LDISP_MINOR	153	/* Please check include/linux/miscdevice.h */

static int ldisp_open_cnt = 0;	/* #times opened */
static int ldisp_open_mode;		/* special open modes */
#define	LDISP_WRITE		1	/* opened for writing (exclusive) */
#define	LDISP_EXCL		2	/* opened with O_EXCL */

#define	LDISP_BYTES		8

static char ldisp_chars[LDISP_BYTES];

static __inline__ char ldisp_read_char(int i)
{
	return ldisp_chars[i];
}

static __inline__ void ldisp_write_char(char c, int i)
{
	ldisp_chars[i] = c;
#ifdef __mips__
	switch (mips_machtype) {
#ifdef CONFIG_TOSHIBA_SDB
	case MACH_TOSHIBA_SDB:
		tsdb_leddisp_putc(i, c);
		break;
#endif
#ifdef CONFIG_TOSHIBA_TX4927EVB
	case MACH_TOSHIBA_TX4927EVB:
		tx4927evb_leddisp_putc(i, c);
		break;
#endif
	}
#endif /* __mips__ */
}

/*
 * The are the file operation function for user access to /dev/leddisp
 */

static long long ldisp_llseek(struct file *file,loff_t offset, int origin)
{
	switch( origin ) {
	  case 0:
		/* nothing to do */
		break;
	  case 1:
		offset += file->f_pos;
		break;
	  case 2:
		offset += LDISP_BYTES;
		break;
	}
	return (offset >= 0) ? (file->f_pos = offset) : -EINVAL;
}

static ssize_t ldisp_read(struct file * file,
			  char * buf, size_t count, loff_t *ppos)
{
	unsigned i = *ppos;
	char *tmp = buf;

	if (i != *ppos)
		return -EINVAL;

	for( ; count-- > 0 && i < LDISP_BYTES; ++i, ++tmp )
		put_user( ldisp_read_char(i), tmp );
	*ppos = i;

	return( tmp - buf );
}

static ssize_t ldisp_write(struct file * file,
			   const char * buf, size_t count, loff_t *ppos)
{
	unsigned i = *ppos;
	const char *tmp = buf;
	char c;

	if (i != *ppos)
		return -EINVAL;

	if (i >= LDISP_BYTES)
		return -ENOSPC;

	for( ; count-- > 0 && i < LDISP_BYTES; ++i, ++tmp ) {
		get_user( c, tmp );
		ldisp_write_char( c, i );
	}
	*ppos = i;

	return( tmp - buf );
}

static int ldisp_open(struct inode *inode, struct file *file)
{
	if ((ldisp_open_cnt && (file->f_flags & O_EXCL)) ||
		(ldisp_open_mode & LDISP_EXCL) ||
		((file->f_mode & 2) && (ldisp_open_mode & LDISP_WRITE)))
		return( -EBUSY );

	if (file->f_flags & O_EXCL)
		ldisp_open_mode |= LDISP_EXCL;
	if (file->f_mode & 2)
		ldisp_open_mode |= LDISP_WRITE;
	memset(ldisp_chars, ' ', LDISP_BYTES);
	ldisp_open_cnt++;
	return( 0 );
}

static int ldisp_release(struct inode *inode, struct file *file)
{
	ldisp_open_cnt--;
	if (file->f_flags & O_EXCL)
		ldisp_open_mode &= ~LDISP_EXCL;
	if (file->f_mode & 2)
		ldisp_open_mode &= ~LDISP_WRITE;

	return( 0 );
}

static struct file_operations ldisp_fops = {
	owner:		THIS_MODULE,
	llseek:		ldisp_llseek,
	read:		ldisp_read,
	write:		ldisp_write,
	open:		ldisp_open,
	release:	ldisp_release,
};

static struct miscdevice ldisp_dev = {
	LDISP_MINOR,
	"leddisp",
	&ldisp_fops
};


static int __init ldisp_init(void)
{
	int error = -ENODEV;

#ifdef CONFIG_TOSHIBA_BOARDS
	switch (mips_machtype) {
#ifdef CONFIG_TOSHIBA_SDB
	case MACH_TOSHIBA_SDB:
		error = 0;
		break;
#endif
#ifdef CONFIG_TOSHIBA_TX4927EVB
	case MACH_TOSHIBA_TX4927EVB:
		error = 0;
		break;
#endif
	}
#endif /* CONFIG_TOSHIBA_BOARDS */
	if (error)
		return error;

	printk(KERN_INFO "HDSP-253X LED Display driver v%s\n", LDISP_VERSION);
	misc_register( &ldisp_dev );
	return( 0 );
}

static void __exit ldisp_exit(void)
{
	misc_deregister( &ldisp_dev );
}

module_init(ldisp_init);
module_exit(ldisp_exit);
