/**
 ** eth1394.c version 0.1.4 -- Ethernet driver for Linux IEEE-1394
 ** Subsystem
 ** 
 ** Guido Fiala, 09.03.2001:
 ** Invested 2 more hours to make it really working!
 ** Guido Fiala, 08.03.2001:
 ** Just a 2 hours hack to make it compile and load under 2.4 kernels
 **
 ** Copyright (C) April 2000 Bonin Franck <boninf@free.fr>. 
 **
 ** Maintly based on work by Emanuel Pirker and Andreas E. Bombe
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **
 ** THIS IS A HACK IMPLEMENATION ONLY !
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **/


/**                              OHCI1394 WARNING !!!
 **  
 **      x    
 **     / \      TO USE WITH OHCI1394 MODULE, I HIGHLY RECOMMAND
 **    / | \     TO SET FLAG "IEEE1394_USE_BOTTOM_HALVES"
 **   /  |  \    TO 1 in file ohci1394.h.
 **  /   .   \   
 ** +---------+
 **
 **/

/**                             PCILYNX WARNING !!!
 **
 **      x
 **     / \      I HAVE NOT TESTED ETHER 1394 WITH PCILYNX DRIVER.
 **    / | \     IF SOME PROBLEMS OCCUR, SET FLAG LOUD_DEBUG TO 1
 **   /  |  \    AND SEND ME YOUR KERNEL MESSAGES FOR DEBUG.
 **  /   .   \
 ** +---------+
 **
 **/
//setting this to 1 may help to debug but decreases performance!
#define  LOUD_DEBUG 0 

#define ETHER1394_TIMEOUT_RECOVERY

#ifdef ETHER1394_TIMEOUT_RECOVERY
#define ETHER1394_MAX_TIMEOUT_COUNT	3
#define ETHER1394_TIMEOUT (HZ)
#else
#define ETHER1394_TIMEOUT (HZ/10) //100ms should be enough
#endif

/**                              COMMENTS WARNING !!!
 **
 * <- 1 toile pour les commentaires en francais
 *
 ** <- 2 stars for the same comments in english
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/types.h>

#include <linux/netdevice.h>   
#include <linux/inetdevice.h>   
#include <linux/etherdevice.h> 
#include <linux/ip.h>          
#include <linux/tcp.h>         
#include <linux/skbuff.h>
#include <asm/delay.h>
#include <asm/semaphore.h>
#include <asm/bitops.h>

#include "ieee1394_types.h"
#include "ieee1394_core.h"
#include "ieee1394_transactions.h"
#include "ieee1394.h"
#include "highlevel.h"

/* Plage de reception des paquets ethernet : */
/** Register place for incoming ethernet packets : **/
#define ADRESSE_ETHER1394_REGION       0xfffff0200000ULL
#define ADRESSE_ETHER1394_REGION_FIN   0xfffff0200800ULL
#define TAILLE_ETHER1394_REGION        0x800 /** size : 2048 bytes **/

/* Pour notre resolution des numeros de noeud : ceci permet de
 * recuperer les adresses MAC des noeuds a l'emplacement de leur
 * numero :
 *
 ** This register place is for node number resolution, we
 ** should receive here HW address :
 */
#define ADRESSE_MAC_REGION              0xfffff0300000ULL
#define ADRESSE_MAC_REGION_FIN          0xfffff0300040ULL
#define TAILLE_MAC_REGION               0x40 /** size for 63 HW address MAX **/

/* Adresse CSR ou il est possible de demander le numero du noeud
 * correspondant a une adresse MAC proposee
 *
 ** This register address accepts incoming broadcasted HW address In
 ** the case the node recognize itself, it respond at
 ** ADRESSE_MAC_REGION+(its node number * 4) by sending its HW address
 */
#define ADRESSE_MAC_DEMANDE             0xfffff0300040ULL
#define ADRESSE_MAC_DEMANDE_FIN         0xfffff0300044ULL

/* Mes adresses MAC n'ont que 4 octets significatifs */
/** For the moment, HW address only have 4 significant bytes **/
#define TAILLE_ADRESSE_MAC              0x4 /* ceci peut evoluer si besoin */ /** may be changed **/


/**
 ** I have obtaint the best performances with a MTU of 896 bytes. But
 ** You could change the MTU with ifconfig command to see if you can
 ** reach better performances on your system.
 **/

/* (ETHER1394_MTU + 14(entete ethernet)) % 4 = 0
 *
 ** (ETHER1394_MTU + 14(ethernet header)) % 4 = 0 */
#define ETHER1394_MTU           /*186 982*/ 898 //2030
#define ETHER1394_MTU_LOW       /*186 982*/ 186 //2030
#define ETHER1394_MTU_MAX       /*186 982*/ 2030 //2030

/* Notre unique pilote reseau ethernet : */
/** Our single network ethernet driver : **/
char etherdev_name[8]="";
static struct net_device ether1394_dev = {
  "        ",    /* pointe vers une zone memoire qui a ete reservee */ /** name is in a staticly alocated memory **/
  0, 0, 0, 0,       // shmem adresse
  0x0000,           // IOport
  0,                // irq
  0, 0, 0,          // flags
  NULL,             // next
  NULL,             /* init = null, pour le moment */ /* init = null, should be filled by init_ether1394() */
};


/* Notre Pilote de haut niveau ieee1394 : */
/** Our ieee1394 highlevel driver : **/
#define NOM_ETHER1394_PILOTE "ether1394"
#define VERSION_ETHER1394_PILOTE "0.2.2+"
static struct hpsb_highlevel *hl_handle = NULL;

/* Stucture prive du pilote ethernet : */
/** Private structure for our ethernet driver : **/
struct ether1394_priv {
  struct  net_device_stats stats;
  struct hpsb_host *host; /* La carte IEEE 1394 correspondante */ /** The ieee1394 cards used **/
  u32 arptab[63]; /* Correspondances entre adresses MAC et numeros de noeud */ /** Corresponding table between HW address and node numbers **/
  spinlock_t lock;
#ifdef ETHER1394_TIMEOUT_RECOVERY
  int timeout_count;
#endif
};

/* Addresse MAC constatnte donne eventuellemnt en parametre. Si elle
 * reste  0x0 l'adresse MAC sera calcule en utilisant le numro
 * courant du noeud.
 *
 ** HW address may be given by a module parameter. If this parameter
 ** stay 0x0, HW address will be calculated with current node number.
 */
#ifndef MODULE	/* XXX hack for namespace collisions */
static
#endif
quadlet_t hwaddr = 0x0;

/****************************************************************************/
/*                         Notre primitive "WRITE"                          */
/**                        Our "WRITE" transaction                         **/
/****************************************************************************/

#if 0
/* Donne un label pour une transaction (conformement  la norme) */
/** Give a tlabel to each transaction **/
int donne_tlabel(struct hpsb_host *host){
  unsigned long flags;
  int tlabel;
//  int i = 0;
  //was 5000 but works as well with only 1 cycle!
//  while (i < 1) 
  { 
    spin_lock_irqsave(&host->tlabel_lock, flags);
    if (host->tlabel_count.count.counter) {
      down(&(host->tlabel_count));
      if (host->tlabel_pool[0] != ~0) {
	tlabel = ffz(host->tlabel_pool[0]);
	host->tlabel_pool[0] |= 1 << tlabel;
      } else {
	tlabel = ffz(host->tlabel_pool[1]);
	host->tlabel_pool[1] |= 1 << tlabel;
	tlabel += 32;
      }     
      spin_unlock_irqrestore(&host->tlabel_lock, flags);
      return tlabel;
    }  
    spin_unlock_irqrestore(&host->tlabel_lock, flags);
//    i++;
  }
  return -1;
}

/* Libere un label alloue pour une transaction normalement terminee */
/** free a tlabel previously allocated **/
void libere_tlabel(struct hpsb_host *host, int tlabel) {
  unsigned long flags;
  spin_lock_irqsave(&host->tlabel_lock, flags);
  if (tlabel < 32) {
    host->tlabel_pool[0] &= ~(1 << tlabel);
  } else {
    host->tlabel_pool[1] &= ~(1 << (tlabel-32));
  }
  up(&(host->tlabel_count));
  spin_unlock_irqrestore(&host->tlabel_lock, flags);
}
#endif

/* Ceci correspond a notre Interruption "Ecriture confirmee" */
/** The following says that our write packet has been sent  **/

/* Pour eviter les fuites de mmoire dans le noyau, enregistre toutes
 * les donnes utilises par notre callback 
 *
 ** We store all the data used by our callback to avoid memory leak
 */
struct donnee_tache{
  struct hpsb_packet *p; /* le paquet */ /** the packet **/
  struct tq_struct *tq; /* la tache */ /** the task **/
};

/* Callback appele quand la confirmation d'une ecriture est recue */
/** This callback is called when our write packet has been sent **/
void ecriture_confirme(struct donnee_tache *dtache){  
  struct ether1394_priv *priv = (struct ether1394_priv *)ether1394_dev.priv;
  unsigned long flags;
  spin_lock_irqsave(&priv->lock, flags);
#if LOUD_DEBUG
  printk("Write has succeeded, freeing tq packet: %p\n",dtache->tq);
#endif
#if 1
  free_tlabel(priv->host, LOCAL_BUS | dtache->p->node_id, dtache->p->tlabel);
#else
  libere_tlabel(priv->host, dtache->p->tlabel);
#endif
  kfree(dtache->tq);
  free_hpsb_packet(dtache->p);
  kfree(dtache);
  netif_wake_queue(&ether1394_dev);/* on est de nouveau pret! */ /** we are ready **/
  spin_unlock_irqrestore(&priv->lock, flags);
}

/* Cree les donnees fournient a notre callback */
/** We build the data for our callback **/
struct donnee_tache *cree_tache(void (*routine)(void *), struct hpsb_packet *paquet){
  int kmflags = in_interrupt() ? GFP_ATOMIC : GFP_KERNEL;
  struct tq_struct *tache = kmalloc(sizeof(struct tq_struct), kmflags);
  struct donnee_tache *dtache = kmalloc(sizeof(struct donnee_tache), kmflags);
  if ((tache == NULL ) || (dtache == NULL)) {
    return NULL;
  }
  /* Voici les donnees de la tache : */
  /** This are task data : **/
  dtache->p = paquet;
  dtache->tq = tache;
  /* Voici notre tache */
  /** This is our task : **/  
  tache->sync = 0;
  tache->routine = routine;
  tache->data = dtache;

  return dtache;
}

/* Cree un paquet de type write request sans bloc de donnee */
/** Build a write packet without payload data **/
struct hpsb_packet *write_quadlet_packet(struct hpsb_host *host, nodeid_t node, u64 addr, quadlet_t data){
  struct hpsb_packet *p;
  struct donnee_tache *dtache;
#if LOUD_DEBUG
  printk("Calling write_quadlet_packet function to node %x\n", (u16) node);
#endif
  p = alloc_hpsb_packet(0);
  if (!p) return NULL; 
  p->host = host; 
#if 1
  if ((p->tlabel = get_tlabel(host, node, 0)) == -1) return NULL;
  p->generation = get_hpsb_generation(host);
#else
  if ((p->tlabel = donne_tlabel(host)) == -1) return NULL;
#endif
  /* On programme la tache qui sera lancee lorsque le destinataire confirmera
   * la reception du paquet
   *
   ** We schedule our callback
   */
  dtache = cree_tache((void (*)(void *))ecriture_confirme, p);
  queue_task(dtache->tq, &p->complete_tq);

  p->node_id = node;
  fill_async_writequad(p, addr, data);
  return p;
}

/* Cree un paquet de type write request avec bloc de donnee */
/** Build a write packet with payload data **/
struct hpsb_packet *write_block_packet(struct hpsb_host *host, nodeid_t node, u64 addr, size_t length){
  struct hpsb_packet *p;
  struct donnee_tache *dtache;
#if LOUD_DEBUG
  printk("Calling write_block_packet function to node %x\n", (u16) node);
#endif
  p = alloc_hpsb_packet(length);
  if (!p) return NULL;
  p->host = host;
#if 1
  if ((p->tlabel = get_tlabel(host, node, 0)) == -1) return NULL;
  p->generation = get_hpsb_generation(host);
#else
  if ((p->tlabel = donne_tlabel(host)) == -1) return NULL;
#endif
  /* On programme la tache qui sera lancee lorsque le destinataire confirmera
   * la reception du paquet
   *
   ** We schedule our callback
   */
  dtache = cree_tache((void (*)(void *))ecriture_confirme, p);
  queue_task(dtache->tq, &p->complete_tq);

  p->node_id = node;
  fill_async_writeblock(p, addr, length);
  return p;
}

/* Debute une transaction de type write, retour = 0 => tous va bien */
/** Start a write transaction, return 0 => OK! **/
int write_no_wait(struct hpsb_host *host, nodeid_t node, u64 addr, quadlet_t *buffer, size_t length){
  /* Permet de copier uniquement les octets significatifs */
  /** This is for counting significant bytes **/
  size_t longueur_alignee = length;
  /* Le paquet que l'on va envoyer */ /** Our packet **/
  struct hpsb_packet *packet_envoye;
#if LOUD_DEBUG
  printk("Calling write_no_wait function to node %x\n", (u16) node);
#endif
  if (length == 0) {
    return -EINVAL;
  }

  if (length & 0x3) {
    /* Alignement d'un paquet dont la longueur n'est pas un multiple de 4*/
    /** This align packet size to a multiple of four**/
    longueur_alignee -= longueur_alignee%4;
    longueur_alignee += 4;
  }

  if (longueur_alignee == 4) {
    /* Transaction pour le rsolutions de numro de noeud. Ne devrait
     * pas survenir pour un paquet ethernet */
    /** This should not append for an ethernet packet, 4 bytes is not enough **/
    packet_envoye = write_quadlet_packet(host, node, addr, *buffer);
  } else {
    /* Transaction "ethernet", on alloue un paquet de la longueur "alignee"*/
    /** "Ethernet" transaction : we allocate an aligned packet size **/
    packet_envoye = write_block_packet(host, node, addr, longueur_alignee);
  }

  if (!packet_envoye) {
#if LOUD_DEBUG
     printk("Couldn't create a packet\n");
#endif
     return -ENOMEM;
  }
      
  if (length != 4) {
    /* On copie uniquement les octets existant */
    /** We copying only existing bytes **/
    memcpy(packet_envoye->data, buffer, length);
  }

  /* On envoie le paquet, puit on s'en vas */
  /** We send the packet end we go **/
  return !hpsb_send_packet(packet_envoye);
}


/****************************************************************************/
/*                            GESTION DES CARTES                            */ 
/**                             CARDS HANDLING                             **/
/****************************************************************************/
#ifndef MODULE	/* XXX hack for namespace collisions */
#define host_info_list	eth1394_host_info_list
#define host_info_lock	eth1394_host_info_lock
#endif
LIST_HEAD(host_info_list);
static int host_count = 0;
spinlock_t host_info_lock = SPIN_LOCK_UNLOCKED;

struct host_info {
  struct list_head list;
  struct hpsb_host *host;
};

/* Une carte vient d'etre trouvee. Cette fonction est appelee en
 * generale au chargement du module dans le noyau. Elle est
 * appelee autant de fois qu'il y a de carte 1394 reconnue.
 *
 ** This function is called every time a card is found. It is generally called
 ** when module is installed.
 */
static void add_host(struct hpsb_host *host) {
  struct host_info *hi;
#if LOUD_DEBUG
  printk("Calling add_host function\n");
#endif
  hi = (struct host_info *)kmalloc(sizeof(struct host_info), SLAB_KERNEL);
  if (hi != NULL) {
    INIT_LIST_HEAD(&hi->list);
    hi->host = host;
    
    spin_lock_irq(&host_info_lock);
    list_add_tail(&hi->list, &host_info_list);
    host_count++;
    spin_unlock_irq(&host_info_lock);
  }
}


static struct host_info *find_host_info(struct hpsb_host *host) {
  struct list_head *lh;
  struct host_info *hi;  
  lh = host_info_list.next;
  while (lh != &host_info_list) {
    hi = list_entry(lh, struct host_info, list);
    if (hi->host == host) {
      return hi;
    }
    lh = lh->next;
  }
  return NULL;
}

/* Une carte vient de se deconnecter*/
/** A card has disappeared **/
static void remove_host(struct hpsb_host *host) {
  struct host_info *hi; 
#if LOUD_DEBUG
  printk("Calling remove_host function\n");
#endif
  spin_lock_irq(&host_info_lock);
  hi = find_host_info(host); 
  if (hi != NULL) {
    list_del(&hi->list);
    host_count--;
  }
  spin_unlock_irq(&host_info_lock); 
  if (hi == NULL) {
    printk(KERN_ERR "Can't remove unknown host : 0x%p\n", host);
    return;
  }
  kfree(hi);
}

/* Un reset vient de survenir */
/** A reset has just arisen **/
static void host_reset(struct hpsb_host *host) {
  int i; 
  netif_stop_queue(&ether1394_dev);/** can't transmit now **/


#if LOUD_DEBUG
  printk("Calling host_reset function\n");
#endif  
  /* Les correspondances : numero de noeud <=> adresse MAC, ne sont plus valide */
  /** Our corresponding table between node numbers and HW address is now invalidates **/
  for (i=0; i<63; i++) ((struct ether1394_priv*)(ether1394_dev.priv))->arptab[i] = 0; 
  /* Faut-il redefinir l'adresse MAC ? Peut-etre que OUI si il y a
   * eu un peripherique d'insere qui a eventuellement obtenu une
   * adresse MAC deja attribuee. Ou alors il faut trouver une autre
   * facon de fournir des adresses MAC.
   *
   ** If our HW address is calculated from node numbers, Maybe we
   ** should rebuild it because if a node has been inserted during
   ** reset, our old HW address may has been given to another node
   ** This is working because ARP will ask later for the HW address of
   ** this node by giving its IP address.  */
  if (!hwaddr) {
    for (i=0; i < ETH_ALEN; i++)
      ether1394_dev.dev_addr[i] = "\0\0iee\0"[i];
    ether1394_dev.dev_addr[ETH_ALEN-1] = host->node_id & 0x3f;
  }

  netif_wake_queue(&ether1394_dev);/** can transmit now **/
}

/****************************************************************************/
/*   Resolution des numeros de noeud : !(adresse mac => numero de noeud)    */ 
/**  Node numbers resolution : HW address doesn't imply a particular node  **/
/**  number because HW address may have been allocated at "insmod time"    **/
/****************************************************************************/

/* Reception d'une addresse MAC a l'emplacement du numero de noeud que
 * l'on recherchait.
 *
 ** This callback recieve HW address in our CSR address space
 */
static int recieve_adresse_MAC(struct hpsb_host *host, int nodeid, int destid, quadlet_t *buffer, u64 adresse, unsigned int taille) {
  if ((adresse < ADRESSE_MAC_REGION) || (adresse >= ADRESSE_MAC_REGION_FIN) || (taille > TAILLE_ADRESSE_MAC )) {
    return RCODE_ADDRESS_ERROR;
  }
#if LOUD_DEBUG
  printk("Receive HW address of node %d : %x\n",(int)(adresse - ADRESSE_MAC_REGION) / 4, *buffer);
#endif  
  ((struct ether1394_priv*)(ether1394_dev.priv))->arptab[(adresse - ADRESSE_MAC_REGION) / 4] = *buffer;
  return RCODE_COMPLETE;
}



/* Reception d'une demande de numero de noeud, si on s'est reconnu, on
 * renvoie notre addresse MAC a l'endroit reserver  cela, vers tout
 * les noeuds du bus.
 *
 ** This callback recieve HW address resolution demands.
 ** if this node reconize its HW address, it send a response
 */
static int recieve_demande_numero_noeud(struct hpsb_host *host, int nodeid, int destid, quadlet_t *buffer, u64 adresse, unsigned int taille) {
  quadlet_t* buf;
  if ((adresse != ADRESSE_MAC_DEMANDE) || (adresse >= ADRESSE_MAC_DEMANDE_FIN) || (taille > TAILLE_ADRESSE_MAC )) {
    return RCODE_ADDRESS_ERROR;
  } 

  /* Recuperation de notre adresse MAC : */
  /** We give our HW address : **/
  buf = (quadlet_t*)&(ether1394_dev.dev_addr[2]) ;
  if (*buffer == *buf){
#if LOUD_DEBUG
    printk("Receive a demand for this node number : Broadcast HW address (%x) of local node (%d) to %d node(s)\n", *buf, host->node_id & 0x3f, host->node_count);
#endif    
    write_no_wait(host, 0xffff, ADRESSE_MAC_REGION+((host->node_id & 0x3f)*4), buf, 4);
  }
  return RCODE_COMPLETE;
}


/****************************************************************************/
/*             Les Fonctions specifiques au pilote ethernet                 */ 
/**                      Ethernet driver functions                         **/
/****************************************************************************/

/* Cette fonction est appel quant l'interface est active */
/** This is call after an "ifup". How can I build a MAC address ? **/
int ether1394_open(struct net_device *dev) {
  int i;
  struct ether1394_priv *priv = (struct ether1394_priv *)dev->priv;

  printk("Opening interface %s\n",etherdev_name);

  /* Set the spinlock before grabbing IRQ! */
  priv->lock = (spinlock_t)SPIN_LOCK_UNLOCKED;
#ifdef ETHER1394_TIMEOUT_RECOVERY
  priv->timeout_count = 0;
#endif

  /* Assignation de l'adresse physique(MAC) de la carte :"\0\0ieeX",
   * il serait bien de trouver une methode pour recuperer un UID de 6
   * octets au moins. Solution partielle : utiliser les numeros de
   * noeud attribues dynamiquement. Attention ! : Lors d'un bus reset
   * ou lors de l'insertion d'un nouveau noeud utilisant le pilote
   * ethernet, cela risque de ne plus fonctionner : faut il alors
   * redefinir l'adresse MAC ? On peut galement spcifier une
   * adresse MAC lors de l'installation du module
   *
   ** We give an HW address to our card. If we use node numbers,
   ** resests are dangerous, so we can use module parameter "hwaddr"
   ** but it is Binding
   */

  for (i=0; i < ETH_ALEN; i++)
    dev->dev_addr[i] = "\0\0iee\0"[i];
  if (hwaddr) {
    for (i=2; i < ETH_ALEN; i++)
      dev->dev_addr[i] = ((unsigned char*)(&hwaddr))[ETH_ALEN-1-i];
  } else {
    dev->dev_addr[ETH_ALEN-1] = priv->host->node_id & 0x3f;
  }

  netif_start_queue(dev);
  MOD_INC_USE_COUNT;
  return 0;
}

/** This is call after an "ifdown" **/
int ether1394_release(struct net_device *dev) {
  printk("Closing interface %s\n",etherdev_name);
  netif_stop_queue(dev);
  MOD_DEC_USE_COUNT;
  return 0;
}

/** Configuration changes (passed on by ifconfig) **/
int ether1394_config(struct net_device *dev, struct ifmap *map) {
#if LOUD_DEBUG
  printk("Calling ether1394_config function\n");
#endif  
  if (dev->flags & IFF_UP) /** can't act on a running interface **/
    return -EBUSY;
  
  if (map->base_addr != dev->base_addr) {
    return -EOPNOTSUPP;
  }
  
  if (map->irq != dev->irq) {
    return -EOPNOTSUPP;
  }

  return 0;
}

/** Receive a packet: retrieve, encapsulate and pass over to upper levels **/
void ether1394_rx(struct net_device *dev, int len, unsigned char *buf) {
  struct sk_buff *skb;
  struct ether1394_priv *priv = (struct ether1394_priv *)dev->priv;

  /** The packet has been retrieved from the transmission
   ** medium. Build an skb around it, so upper layers can handle it
   **/
#if LOUD_DEBUG
  printk("Calling ether1394_rx function\n");
#endif
  skb = dev_alloc_skb(len+2);
  if (!skb) {
    HPSB_PRINT(KERN_ERR, "ether1394 rx: low on mem\n");
    priv->stats.rx_dropped++;
    return;
  }
  skb_reserve(skb, 2); /** align IP on 16B boundary **/  
  memcpy(skb_put(skb, len), buf, len);
  
  /** Write metadata, and then pass to the receive level **/
  skb->dev = dev;
  skb->protocol = eth_type_trans(skb, dev);
  skb->ip_summed = CHECKSUM_UNNECESSARY; /** don't check it **/
  /* Statistiques :*/
  priv->stats.rx_packets++;
  priv->stats.rx_bytes+=skb->len;
  /* On passa tout ca au dessus */
  /** We give the packet to the upper level **/
  netif_rx(skb);
  return;
}

/** Transmit a packet (low level interface) **/
int ether1394_hw_tx(char *buf, int len, struct net_device *dev) {
  /** This function deals with hw details while all other procedures
   ** are rather device-independent 
   **/
  struct ethhdr *eh;
  unsigned char* saddr;
  unsigned char* daddr;
  u32* dest_addr;

  struct ether1394_priv *priv = (struct ether1394_priv *)dev->priv;
  u16 dest_physical_id, i;
  int found_dest = 1;
  int ret_write;

  /** Ethhdr is 14 bytes, but the kernel arranges for iphdr
   ** to be aligned (i.e., ethhdr is unaligned 
   **/
  eh = (struct ethhdr *)(buf);
  saddr = eh->h_source;
  daddr = eh->h_dest;
  dest_addr = (u32*)&(daddr[2]);
#if LOUD_DEBUG
  printk("Calling ether1394_HW_tx function for HW addr %x\n",*dest_addr);
#endif
  
  /* S'agit-il d'un broadcast ? */
  /** Is it a broadcast packet ? **/
  dest_physical_id = 0xffff; /* par defaut, OUI! */ /** Yes, for the moment **/
  for (i = 0; i < ETH_ALEN; i++) 
    found_dest = found_dest & (daddr[i] == 0xff); 

  /* Recherche du numero de noeud correspondant a cette adresse MAC */
  /** We search the node number for this HW address **/
  if (!found_dest) {
    for (i = 0; i < 63; i++)
      if (priv->arptab[i] == *dest_addr){
	dest_physical_id = i;
	found_dest = 1;
      }
  }


  if (found_dest) {
    ret_write = write_no_wait(priv->host, dest_physical_id | 0xffc0, ADRESSE_ETHER1394_REGION, (quadlet_t *)buf, len);

    /* Statistiques : */
    if (!ret_write){
      priv->stats.tx_bytes+=len;
      priv->stats.tx_packets++;
    } else {
#if LOUD_DEBUG
      printk("Write transaction problem\n");
#endif      
      priv->stats.tx_dropped++;
      priv->stats.tx_errors++;
      netif_stop_queue(&ether1394_dev);
    } 
     return ret_write;
  } else {
    /* On demande le numero du noeud, en attendant on oublit le paquet courant */
    /** We ask for the node number, but the packet is dropped **/
#if LOUD_DEBUG
    printk("Asking for node number which got HW address %x\n",*dest_addr);
#endif    
    write_no_wait(priv->host, 0xffff, ADRESSE_MAC_DEMANDE, (quadlet_t *)dest_addr, 4);
    netif_stop_queue(dev);
    priv->stats.tx_dropped++;
    return 0;
  } 
}

/**Transmit a packet (called by the kernel) **/
int ether1394_tx(struct sk_buff *skb, struct net_device *dev) {
  int len, retval=0;
  unsigned long flags;
  char *data;
  struct ether1394_priv *priv = (struct ether1394_priv *)dev->priv;
#if LOUD_DEBUG
  printk("Calling ether1394_tx function\n");
#endif
  if (test_bit(__LINK_STATE_XOFF,&dev->state)) 
  { /** shouldn't happen **/
#if 1 //LOUD_DEBUG
    printk("Interface ether1394 is busy (but shouldn't)\n");
#endif    
    priv->stats.tx_dropped++;
    priv->stats.tx_errors++;

    /* Sinon ca plante la machine au bout de quelques essais...**/
    /** This is safer **/
    retval = 0;
    /* pour voir... */
    /** and this too **/
    netif_wake_queue(&ether1394_dev);
    goto tx_done;
  }
  if (skb == NULL) {
    return 0;
  }

   
  spin_lock_irqsave(&priv->lock, flags);
  netif_stop_queue(dev);
  len = ETH_ZLEN < skb->len ? skb->len : ETH_ZLEN; /** minimum len **/
  data = skb->data;
  dev->trans_start = jiffies; /** save the timestamp **/
  
  if (len <= dev->mtu + 14) 
    retval = ether1394_hw_tx(data, len, dev);
  else {
#if LOUD_DEBUG
    printk("Ethernet packet to large (%d, shouldn't happen)\n", len);
#endif    
    netif_wake_queue(&ether1394_dev);/** transmission is not busy **/
    priv->stats.tx_dropped++;
    priv->stats.tx_errors++;
    retval = -EINVAL;
  }
  spin_unlock_irqrestore(&priv->lock, flags);
  
tx_done:
  dev_kfree_skb(skb); /** release it **/
  
#if 1
  /* we always free skb, so we MUST return 0 */
  return 0;
#else
  return retval; /** zero == done; nonzero == fail **/
#endif
}

/** Ioctl commands **/
int ether1394_ioctl(struct net_device *dev, struct ifreq *rq, int cmd){
#if LOUD_DEBUG
  printk("Calling ioctl function\n");
#endif
  return 0;
}

/** Return statistics to the caller **/
struct net_device_stats *ether1394_stats(struct net_device *dev) {
  struct ether1394_priv *priv = (struct ether1394_priv *)dev->priv;
#if LOUD_DEBUG
  printk("Calling stats function\n");
#endif
  return &priv->stats;
}

/**The "change_mtu" method is usually not needed.
 ** If you need it, it must be like this.
 **/
int ether1394_change_mtu(struct net_device *dev, int new_mtu) {
#if LOUD_DEBUG
  printk("Calling change_MTU function\n");
#endif
  /** chack ranges **/
  if ((new_mtu < ETHER1394_MTU_LOW) || (new_mtu > ETHER1394_MTU_MAX))
    return -EINVAL;
  if ((new_mtu+14) % 4 ) 
        return -EINVAL;
  /** Do anything you need, and then accept the value **/
  dev->mtu = new_mtu;
  return 0; /** success **/
}

/* new timeout functionality */

void ether1394_tx_timeout(struct net_device *dev) 
{
#ifdef ETHER1394_TIMEOUT_RECOVERY
  unsigned long flags;
  struct ether1394_priv *priv = (struct ether1394_priv *)dev->priv;
  int do_reset = 0;
  spin_lock_irqsave(&priv->lock, flags);
  if (priv->timeout_count++ >= ETHER1394_MAX_TIMEOUT_COUNT) {
	  priv->timeout_count = 0;
	  do_reset = 1;
  }
  spin_unlock_irqrestore(&priv->lock, flags);
  if (do_reset) {
    printk("eth1394: tx_timeout: generating IEEE-1394 bus reset\n");
    if (hpsb_reset_bus(priv->host, LONG_RESET)) {
	    printk("eth1394: tx_timeout: generating IEEE-1394 bus reset (force)\n");
	    priv->host->template->devctl(priv->host, RESET_BUS, LONG_RESET);
    }
    return;
  }
#endif
#if 1 //LOUD_DEBUG
  printk("ether1394_tx_timeout called\n");
#endif
  netif_wake_queue(dev);  //or has it to be by interrupt?
}

/** The init function (sometimes called probe).
 ** It is invoked by register_netdev()
 **/
int ether1394_init(struct net_device *dev) {  
  struct list_head *lh;
  struct host_info *hi;
  int i;
  
  /** Then, assign other fields in dev, using ether_setup() and some
   ** hand assignments
   **/
  ether_setup(dev); /** assign some of the fields **/
  
  dev->open            = ether1394_open;
  dev->stop            = ether1394_release;
  dev->set_config      = ether1394_config;
  dev->hard_start_xmit = ether1394_tx;
  dev->do_ioctl        = ether1394_ioctl;
  dev->get_stats       = ether1394_stats;
  dev->change_mtu      = ether1394_change_mtu;
  dev->tx_timeout      = ether1394_tx_timeout;
  dev->watchdog_timeo  = ETHER1394_TIMEOUT;
 
  dev->flags           &= !IFF_MULTICAST; /** No multicast yet **/

  dev->mtu             = ETHER1394_MTU;
  /** Then, allocate the priv field. This encloses the statistics
   ** and a few private fields.
   **/
  dev->priv = kmalloc(sizeof(struct ether1394_priv), GFP_KERNEL);
  if (dev->priv == NULL)
    return -ENOMEM;
  memset(dev->priv, 0, sizeof(struct ether1394_priv));

  /** Clear HW address **/
#if LOUD_DEBUG
  printk("Reset HW address %s\n",etherdev_name);
#endif  
  for (i=0; i<63; i++) ((struct ether1394_priv*)(dev->priv))->arptab[i] = 0; 

  /* On utilise la premiere carte disponible que l'on trouve : */
  /** We choose the first card found : **/
  if (host_count > 0) {
    spin_lock_irq(&host_info_lock);
    lh = host_info_list.next;
    hi = list_entry(lh, struct host_info, list);
    /* Enregistrement de la carte */
    /** Registering card **/
    ((struct ether1394_priv*)(dev->priv))->host = hi->host;
    spin_unlock_irq(&host_info_lock);
  } else {
    printk("eth1394: ");
    printk("Error : no card found !\n");
    return -ENODEV;
  }   
  return 0;
}

/****************************************************************************/
/*              Interruption paquet arrive (ou presque)                     */ 
/**          This function is called for an incomming packet               **/ 
/****************************************************************************/

static int write_ether1394(struct hpsb_host *host, int nodeid, int destid, quadlet_t *data, u64 adresse, unsigned int taille) {
  struct ether1394_priv *priv = (struct ether1394_priv *)ether1394_dev.priv;
  unsigned long flags;
  if ((adresse != ADRESSE_ETHER1394_REGION) || (taille > TAILLE_ETHER1394_REGION )) {
    return RCODE_ADDRESS_ERROR;
  }  
#if LOUD_DEBUG
  printk("Calling Write_ether1394 function from node %x with HW address %x\n", nodeid, priv->arptab[nodeid &0x3f]);
#endif
  spin_lock_irqsave(&priv->lock, flags);
  ether1394_rx(&ether1394_dev, taille , (char *)data);
  spin_unlock_irqrestore(&priv->lock, flags);

  return RCODE_COMPLETE;
}

/***************************************************************************/
/*                         INITIALISATIONS MODULE                          */
/**                             INIT MODULE                               **/
/***************************************************************************/

/* Operations effectuees quand l'ecriture d'un paquet ethernet arrive */
/** Functions for incoming ethernet packets **/
struct hpsb_address_ops addr_ops = {
  NULL,
  write_ether1394,
  NULL,
  NULL
};

/* Operation effectuee quant on recoit une adresse MAC correspondant a
 * un numero de noeud
 *
 ** Functions for incoming HW address packets from a particular node
 */
struct hpsb_address_ops rep_mac_ops = {
  NULL,
  recieve_adresse_MAC,
  NULL,
  NULL
};

/* Operations effectuee quand on recoit une demande de numero de noeud
 * correspondant a une adresse MAC
 *
 ** Functions for incoming inquery packets that search a node number
 */
struct hpsb_address_ops dem_mac_ops = {
  NULL,
  recieve_demande_numero_noeud,
  NULL,
  NULL
};

/* Operations du pilote de haut niveau ieee1394 */
/** Oeee1394 highlevel driver functions **/
struct hpsb_highlevel_ops hl_ops = {
  add_host,
  remove_host, 
  host_reset,
  NULL,
  NULL,
};

#ifndef MODULE
/* disable by default. use "eth1394=hwaddr:0x????" to enable. */
static int eth1394_disable = 1;
#endif
int init_ether1394(void) {
#ifndef MODULE
  if (eth1394_disable) {
    printk(KERN_DEBUG "eth1394: disabled\n");
    return -ENODEV;
  }
#endif
  /* Pilote de haut niveau 1394 */
  /** Ieee1394 highlevel driver registering **/
  hl_handle = hpsb_register_highlevel(NOM_ETHER1394_PILOTE, &hl_ops);
  if (hl_handle == NULL) {
    HPSB_ERR("No more memory for driver %s\n",NOM_ETHER1394_PILOTE);
    return -ENOMEM;
  }  

  /* Ecoute des reception de paquets ethernet */
  /** Listen incoming ethernet packets **/
#if LOUD_DEBUG
  printk("Driver %s use CSR address space %x - %x, for receiving ethernet packets\n", NOM_ETHER1394_PILOTE, (int)ADRESSE_ETHER1394_REGION, (int)ADRESSE_ETHER1394_REGION_FIN);
#endif  
  hpsb_register_addrspace(hl_handle, &addr_ops, ADRESSE_ETHER1394_REGION, ADRESSE_ETHER1394_REGION_FIN);
  
  /* Ecoute des reponses de demande d'adresse MAC */
  /** Listen incoming HW address packetss **/
#if LOUD_DEBUG
  printk("Driver %s use CSR address space %x - %x, for receiving ethernet HW address responses\n",NOM_ETHER1394_PILOTE, (int)ADRESSE_MAC_REGION, (int)ADRESSE_MAC_REGION_FIN);
#endif  
  hpsb_register_addrspace(hl_handle, &rep_mac_ops, ADRESSE_MAC_REGION, ADRESSE_MAC_REGION_FIN);

  /* Ecoute des demandes d'adresse MAC */
  /** Listen incoming HW address inquery **/
#if LOUD_DEBUG
  printk("Driver %s use CSR address space %x - %x, for receiving ethernet HW address demands\n", NOM_ETHER1394_PILOTE, (int)ADRESSE_MAC_DEMANDE, (int)ADRESSE_MAC_DEMANDE_FIN);
#endif  
  hpsb_register_addrspace(hl_handle, &dem_mac_ops, ADRESSE_MAC_DEMANDE, ADRESSE_MAC_DEMANDE_FIN);
 
  /* Enregistrement du pilote reseau et de sa primitive d'initialisation */
  /** Ethernet driver registering **/
  ether1394_dev.init = ether1394_init;
  /* Pour obtenir un nom d'interface du style "ethX" ou "X" et le
   * dernier numero disponible pour designer une interface ethernet
   *
   ** The following line allows to obtain a interface name like "ethX"
   ** where "X" is a number not yet allocated */
  ether1394_dev.name[0] = ' ';

  if ( register_netdev(&ether1394_dev) ) {
    printk("eth1394: ");
    printk("Error during network driver init %s on interface %s\n", NOM_ETHER1394_PILOTE, etherdev_name);
    return -ENODEV;
  }  
  printk("eth1394: ");
  printk("Succesfully load %s driver version %s on interface %s\n", NOM_ETHER1394_PILOTE, VERSION_ETHER1394_PILOTE, etherdev_name);

  return 0;
}

void cleanup_ether1394(void) {
  kfree(ether1394_dev.priv);
  unregister_netdev(&ether1394_dev);
  
  hpsb_unregister_highlevel(hl_handle);
}

MODULE_PARM(hwaddr, "i");

module_init(init_ether1394);
module_exit(cleanup_ether1394);

#ifndef MODULE
static int __init eth1394_setup(char *str)
{
	char *p;
	if (strcmp(str, "off") == 0) {
		eth1394_disable = 1;
		return 0;
	}
	eth1394_disable = 0; /* enable */
	p = str;
	while (p) {
		if (strncmp(p, "hwaddr:", 7) == 0) {
			hwaddr = simple_strtol(p + 7,NULL,0);
		}
		p = strchr(p, ',');
		if (p)
			p++;
	}
	return 0;
}
__setup("eth1394=", eth1394_setup);
#endif
