/*
* Configuration access system call
*/

#include <linux/init.h>
#include <linux/linkage.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <linux/string.h>
#include <linux/mtd/mtd.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <asm/semaphore.h>

#include <linux/urlfilter.h>
#include <asm/mconfig.h>
#include <asm/flash.h>

#if defined(CONFIG_MCT_SG100) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SA200)
extern void get_switch_status(SWITCH_STATUS *switch_status);
#endif

extern unsigned char	__fsysbuf, __esysbuf;
DECLARE_MUTEX(sysimage_sem);
static int	sysimage_state;
unsigned char	sysimage_percentage;

int	conf_reset_in_progress;

GENCONF_CKSUM_BLOCK		genconf_buf;

#if defined(CONFIG_MCT_SG100)
USERACCOUNT_CKSUM_BLOCK		useraccount_buf;
#else
EXTCONF_CKSUM_BLOCK		extconf_buf;
#endif

EVENTLOG_CKSUM_BLOCK		eventlog_buf;
DHCPLEASE_CKSUM_BLOCK		dhcplease_buf;
#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
USERSHARE_CKSUM_BLOCK		usershare_buf;
LA_USERS_CKSUM_BLOCK		la_user_buf;
LA_GRPSHARE_CKSUM_BLOCK		la_grpshare_buf;
static void restore_olduser_data(USERSHARE_BLOCK* old ,LA_USERS_BLOCK* new_usr, LA_GRPSHARE_BLOCK* new_grpshare);
#endif

REGION_INFO genconf_region = {
	policy: FL_IMMEDIATE,
	base: GENCONF_ADDR,
	size: sizeof(GENCONF_CKSUM_BLOCK),
	buf_base: &genconf_buf,
	sem: __MUTEX_INITIALIZER(genconf_region.sem)
};

#if defined(CONFIG_MCT_SG100) 
REGION_INFO useraccount_region = {
	policy: FL_IMMEDIATE,
	base: USERACCOUNT_ADDR,
	size: sizeof(USERACCOUNT_CKSUM_BLOCK),
	buf_base: &useraccount_buf,
	sem: __MUTEX_INITIALIZER(useraccount_region.sem)
};
#else
REGION_INFO extconf_region = {
	policy: FL_IMMEDIATE,
	base: EXTCONF_ADDR,
	size: sizeof(EXTCONF_CKSUM_BLOCK),
	buf_base: &extconf_buf,
	sem: __MUTEX_INITIALIZER(extconf_region.sem)
};
#endif

REGION_INFO eventlog_region = {
	policy: FL_IMMEDIATE,
	base: EVENTLOG_ADDR,
	size: sizeof(EVENTLOG_CKSUM_BLOCK),
	buf_base: &eventlog_buf,
	sem: __MUTEX_INITIALIZER(eventlog_region.sem)
};

REGION_INFO dhcplease_region = {
	policy: FL_INTERVAL,
	short_term_interval: 60 * 60 * 100, /* 1 hour */
	long_term_interval: 120 * 60 * 100, /* 2 hours */
	dirty: 0,
	write_force: 0,
	dirty_threshold: 200,
	base: DHCPLEASE_ADDR,
	size: sizeof(DHCPLEASE_CKSUM_BLOCK),
	buf_base: &dhcplease_buf,
	last_time_modified: 0,
	sem: __MUTEX_INITIALIZER(dhcplease_region.sem)
};

#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
REGION_INFO usershare_region = {
	policy: FL_IMMEDIATE,
	base: USERSHARE_ADDR,
	dirty: 1,
	size: 64*1024,	//sizeof(USERSHARE_CKSUM_BLOCK),
	buf_base: &usershare_buf,
	sem: __MUTEX_INITIALIZER(usershare_region.sem)
};
REGION_INFO la_user_region = {
	policy: FL_IMMEDIATE,
	base: LA_USER_ADDR,
	dirty: 1,
	size: 64*1024, //sizeof(LA_USERS_CKSUM_BLOCK),
	buf_base: &la_user_buf,
	sem: __MUTEX_INITIALIZER(la_user_region.sem)
};
REGION_INFO la_grpshare_region = {
	policy: FL_IMMEDIATE,
	base: LA_GRPSHARE_ADDR,
	dirty: 1,
	size: 64*1024,//sizeof(LA_GRPSHARE_CKSUM_BLOCK),	
	buf_base: &la_grpshare_buf,
	sem: __MUTEX_INITIALIZER(la_grpshare_region.sem)
};
#endif


// the region_list is used for initializing timer.
REGION_INFO *region_list[] = {
	&genconf_region,
#if defined(CONFIG_MCT_SG100)
	&useraccount_region,
#else
	&extconf_region,
#endif
	&eventlog_region,
	&dhcplease_region,
#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
//	&usershare_region,
	&la_user_region,
	&la_grpshare_region,
#endif
};

ACCESS_CONF access_conf[] = {
	{ MC_NET_CONFIG, offsetof(GENCONF_BLOCK, net_conf), sizeof(REG_NET_CONF), &genconf_region },
	{ MC_IPALIAS_CONFIG, offsetof(GENCONF_BLOCK, ipalias_conf), sizeof(REG_IPALIAS_CONF), &genconf_region },
	{ MC_MNAT_CONFIG, offsetof(GENCONF_BLOCK, mnat_conf), sizeof(REG_MNAT_CONF), &genconf_region },
	{ MC_DHCP_CONFIG, offsetof(GENCONF_BLOCK, dhcp_conf), sizeof(REG_DHCP_CONF), &genconf_region },
	{ MC_MACMAPPING_CONFIG, offsetof(GENCONF_BLOCK, macmapping_conf), sizeof(REG_MACMAPPING_CONF), &genconf_region },
	{ MC_HOSTMAPPING_CONFIG, offsetof(GENCONF_BLOCK, hostmapping_conf), sizeof(REG_HOSTMAPPING_CONF), &genconf_region },
	{ MC_ROUTE_CONFIG, offsetof(GENCONF_BLOCK, route_conf), sizeof(REG_ROUTE_CONF), &genconf_region },
	{ MC_IPSEC_CONFIG, offsetof(GENCONF_BLOCK, ipsec_conf), sizeof(REG_IPSEC_CONF), &genconf_region },
	{ MC_PPTP_CONFIG, offsetof(GENCONF_BLOCK, pptp_conf), sizeof(REG_PPTP_CONF), &genconf_region },
	{ MC_VSERVER_CONFIG, offsetof(GENCONF_BLOCK, vserver_conf), sizeof(REG_VSERVER_CONF), &genconf_region },
	{ MC_IP_ACCESS_CONFIG, offsetof(GENCONF_BLOCK, ip_access_conf), sizeof(REG_IP_ACCESS_CONF), &genconf_region },
	{ MC_URL_ACCESS_CONFIG, offsetof(GENCONF_BLOCK, url_access_conf), sizeof(REG_URL_ACCESS_CONF), &genconf_region },
	{ MC_ANTIHACK_CONFIG, offsetof(GENCONF_BLOCK, antihack_conf), sizeof(REG_ANTIHACK_CONF), &genconf_region },
	{ MC_SAMBA_CONFIG, offsetof(GENCONF_BLOCK, smbprn_conf), sizeof(REG_SAMBA_CONF), &genconf_region },
	{ MC_ALARM_CONFIG, offsetof(GENCONF_BLOCK, alarm_conf), sizeof(REG_ALARM_CONF), &genconf_region },
	{ MC_ADMIN_CONFIG, offsetof(GENCONF_BLOCK, admin_conf), sizeof(REG_ADMIN_CONF), &genconf_region },
	{ MC_DDNS_CONFIG, offsetof(GENCONF_BLOCK, ddns_conf), sizeof(REG_DDNS_CONF), &genconf_region },
	{ MC_SNMP_CONFIG, offsetof(GENCONF_BLOCK, snmp_conf), sizeof(REG_SNMP_CONF), &genconf_region },
#if defined(CONFIG_MCT_SG100)
	{ MC_USER_CONFIG, offsetof(USERACCOUNT_BLOCK, useraccount_conf), sizeof(REG_USER_CONF), &useraccount_region },
#endif
	{ MC_DHCP_LEASE, offsetof(DHCPLEASE_BLOCK, dhcplease_conf), sizeof(REG_DHCP_LEASE), &dhcplease_region },
	{ MC_EVENTLOG_CONFIG, offsetof(EVENTLOG_BLOCK, eventlog_conf), sizeof(REG_EVENTLOG_CONF), &eventlog_region },
#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
	{ MC_USER,   	 offsetof(LA_USERS_BLOCK, users),     sizeof(LA_USERS),		&la_user_region },
	{ MC_GROUP,  	 offsetof(LA_GRPSHARE_BLOCK, groups), sizeof(LA_GROUPS), 	&la_grpshare_region },
	{ MC_SHARE,  	 offsetof(LA_GRPSHARE_BLOCK, shares), sizeof(LA_SHARES), 	&la_grpshare_region },
//	{ MC_USER,   	 offsetof(USERSHARE_BLOCK, users),  sizeof(USERS), 		&usershare_region },
//	{ MC_GROUP,  	 offsetof(USERSHARE_BLOCK, groups), sizeof(GROUPS), 		&usershare_region },
//	{ MC_SHARE,  	 offsetof(USERSHARE_BLOCK, shares), sizeof(SHARES), 		&usershare_region },
//	{ MC_USERSHARE,  offsetof(USERSHARE_BLOCK, users), sizeof(USERSHARE_BLOCK), 	&usershare_region },
//	{ MC_NFS_CONFIG, offsetof(GENCONF_BLOCK, nfs_conf), sizeof(REG_NFS_CONF), 	&genconf_region },
	{ MC_NFS_CONFIG, offsetof(EXTCONF_BLOCK, nfs_conf),  sizeof(REG_NFS_CONF), 	&extconf_region },
	{ MC_MAIL_CONFIG,offsetof(EXTCONF_BLOCK, mail_conf), sizeof(REG_MAIL_CONF), 	&extconf_region },
	{ MC_W3S_CONFIG, offsetof(EXTCONF_BLOCK, w3s_conf),  sizeof(REG_W3S_CONF), 	&extconf_region },
	{ MC_BPA_CONFIG, offsetof(EXTCONF_BLOCK, bpa_conf),  sizeof(REG_BPA_CONF), 	&extconf_region },
	{ MC_PPPOE_MTU,  offsetof(EXTCONF_BLOCK, pppoe_mtu), sizeof(int), 		&extconf_region },
	{ MC_TIMEZONE_CONFIG, offsetof(EXTCONF_BLOCK, time_zone),  sizeof(int), 	&extconf_region },
 	{ MC_IDESMART_CONFIG, offsetof(GENCONF_BLOCK,idesmart_conf), sizeof(REG_IDESMART_CONF), 	&genconf_region },
 	{ MC_FTP_PORT, offsetof(EXTCONF_BLOCK, ftp_port_num),  sizeof(int), &extconf_region },
 	{ MC_UPNP_MMS_CONFIG, offsetof(EXTCONF_BLOCK, upnp_mms_enable),  sizeof(int), &extconf_region },
 	{ MC_UPNP_DEVICE_CONFIG, offsetof(EXTCONF_BLOCK, upnp_device_enable), sizeof(int), &extconf_region },
#endif
};

DECLARE_WAIT_QUEUE_HEAD(flash_wq);
static struct mtd_info		*mtd;

static void erase_callback(struct erase_info *done)
{
	wait_queue_head_t	*wait_q = (wait_queue_head_t *) done->priv;
	wake_up(wait_q);
}

#define ERASE_SIZE	0x2000
#define ERASE_MASK	~(ERASE_SIZE-1)
#define ERASE_ALIGN(x)	((x + ERASE_SIZE-1) & ERASE_MASK)

static int erase_write(struct mtd_info *mtd, unsigned long pos, int len, const char *buf)
{
	struct erase_info	erase;
	DECLARE_WAITQUEUE(wait, current);
	wait_queue_head_t	wq_head;
	size_t			retlen;
	int			ret;

	// erase block
	init_waitqueue_head(&wq_head);
	erase.mtd = mtd;
	erase.callback = erase_callback;
	erase.addr = pos;
	erase.len = ERASE_ALIGN(len);
	erase.priv = (u_long) &wq_head;
	set_current_state(TASK_INTERRUPTIBLE);
	add_wait_queue(&wq_head, &wait);

	ret = MTD_ERASE(mtd, &erase);
	if ( ret ) {
		set_current_state(TASK_RUNNING);
		remove_wait_queue(&wq_head, &wait);
		printk(KERN_WARNING "flash: erase of region (0x%08x, 0x%04x) on \"%s\" failed\n", (unsigned int) pos, (unsigned int) len, mtd->name);
		return ret;
	}

	schedule();
	remove_wait_queue(&wq_head, &wait);

	// write block
	ret = MTD_WRITE(mtd, pos, len, &retlen, buf);
	if ( ret )
		return ret;
	if ( retlen != len )
		return -EIO;

	return 0;
}
//added by louistsai
//To ensure the integrity of updated fw, we should remove the signature on the 
//flash first.
static int remove_fwsig(struct mtd_info *mtd)
{
	unsigned long 		sigaddr;
	struct erase_info	erase;
	DECLARE_WAITQUEUE(wait, current);
	wait_queue_head_t	wq_head;
	size_t			retlen;
	int			ret;

	sigaddr = *((unsigned long *) (0xbfc20000 + 0x18));
	// erase block
	init_waitqueue_head(&wq_head);
	erase.mtd = mtd;
	erase.callback = erase_callback;
	erase.addr = (sigaddr/(64*1024))*(64*1024) - 0xbfc00000;
	erase.len = ERASE_ALIGN(64*1024);
	erase.priv = (u_long) &wq_head;
	set_current_state(TASK_INTERRUPTIBLE);
	add_wait_queue(&wq_head, &wait);

	ret = MTD_ERASE(mtd, &erase);
	if ( ret ) {
		set_current_state(TASK_RUNNING);
		remove_wait_queue(&wq_head, &wait);
		printk(KERN_WARNING "flash: erase of region (0x%08x, 0x%04x) on \"%s\" failed\n", (unsigned int) erase.addr, (unsigned int) erase.len, mtd->name);
		return ret;
	}
	schedule();
	remove_wait_queue(&wq_head, &wait);
	
	return 0;
}




static void gen_cksum(unsigned int *buf, int size)
{
	int		i;
	unsigned int	cksum = 0xffffffff;

	size = (size >> 2) - 1;
	for (i = 0; i < size; i++) {
		cksum += *buf++;
	}

	*buf = cksum;
}

static void set_default_leaseconf(void)
{
	DHCPLEASE_BLOCK *dhcplease_block;

	dhcplease_block = (DHCPLEASE_BLOCK *) dhcplease_region.buf_base;
	memset(dhcplease_block, 0, dhcplease_region.size);
}

static void dhcp_clear_conflict(void)
{
	int	i;
	struct timeval	tv;
	REG_DHCP_LEASE	*pconf;
	REG_IP_LEASES	*ipleases;
	char	zeromac[6] = { 0, 0, 0, 0, 0, 0 };

	down(&dhcplease_region.sem);

	pconf = &dhcplease_buf.dhcplease_block.dhcplease_conf;
	ipleases = pconf->item;

	do_gettimeofday(&tv);
	
	/* clear conflict address if time is expired */
	for (i = 0; i < pconf->lease_count; i++) {
		if ( memcmp(ipleases[i].mac, zeromac, 6) == 0 && tv.tv_sec > ipleases[i].end ) {
			pconf->lease_count--;
			if ( i != pconf->lease_count ) {
				memcpy(&ipleases[i], &ipleases[pconf->lease_count], sizeof(REG_IP_LEASES));
			}
			memset(&ipleases[pconf->lease_count], 0, sizeof(REG_IP_LEASES));
		}
	}

	up(&dhcplease_region.sem);
}

static int dhcp_find_lease_by_chaddr(DHCP_OFFERED_ADDR *lease)
{
	int	i, retval;
	REG_DHCP_LEASE	*pconf;
	REG_IP_LEASES	*ipleases;

	/* Notice that we trust the user process */
	down(&dhcplease_region.sem);

	retval = -ENOENT;
	pconf = &dhcplease_buf.dhcplease_block.dhcplease_conf;
	ipleases = pconf->item;
	for (i = 0; i < pconf->lease_count; i++) {
		if ( memcmp(lease->chaddr, ipleases[i].mac, 6) == 0 ) {
			lease->yiaddr = ipleases[i].ip;
			lease->expires = ipleases[i].end;
			retval = 0;
			break;
		}
	}

	up(&dhcplease_region.sem);

	return retval;
}

static int dhcp_find_lease_by_yiaddr(DHCP_OFFERED_ADDR *lease)
{
	int	i, retval;
	REG_DHCP_LEASE	*pconf;
	REG_IP_LEASES	*ipleases;

	/* Notice that we trust the user process */
	down(&dhcplease_region.sem);

	retval = -ENOENT;
	pconf = &dhcplease_buf.dhcplease_block.dhcplease_conf;
	ipleases = pconf->item;
	for (i = 0; i < pconf->lease_count; i++) {
		if ( lease->yiaddr == ipleases[i].ip ) {
			memcpy(lease->chaddr, ipleases[i].mac, 6);
			lease->expires = ipleases[i].end;
			retval = 0;
			break;
		}
	}

	up(&dhcplease_region.sem);

	return retval;
}

static int dhcp_clear_lease(void)
{
	down(&dhcplease_region.sem);
	set_default_leaseconf();
	up(&dhcplease_region.sem);
	dhcplease_region.write_force = 1;

	return 0;
}

static int dhcp_add_lease(DHCP_OFFERED_ADDR *lease)
{
	int	i, retval;
	int	iszeromac;
	struct timeval	tv;
	unsigned int	oldest_lease;
	REG_DHCP_LEASE	*pconf;
	REG_IP_LEASES	*ipleases, *pslot;

	retval = -ENOENT;

	dhcp_clear_conflict();

	down(&dhcplease_region.sem);
	/* clear lease
	   if hardware address is zero, DONOT remove the lease because it represents address collision.
	*/
	for (i = 0; i < 6; i++) {
		if ( lease->chaddr[i] != 0 )
			break;
	}
	iszeromac = (i == 6) ? 1 : 0;

	pconf = &dhcplease_buf.dhcplease_block.dhcplease_conf;
	ipleases = pconf->item;

	/* replace lease address if either IP or MAC address match with the current lease adderss */
	for (i = 0; i < pconf->lease_count; i++) {
		if ( (!iszeromac && memcmp(lease->chaddr, ipleases->mac, 6) == 0) ||
			lease->yiaddr == ipleases[i].ip ) {
			pconf->lease_count--;
			if ( i != pconf->lease_count ) {
				memcpy(&ipleases[i], &ipleases[pconf->lease_count], sizeof(REG_IP_LEASES));
			}
			memset(&ipleases[pconf->lease_count], 0, sizeof(REG_IP_LEASES));
			break;
		}
	}
	
	do_gettimeofday(&tv);

	/* choose the promising "lease slot" */
	if ( pconf->lease_count < MAX_DHCP_LEASES_RECORDS ) {
		pslot = &ipleases[pconf->lease_count++];
	} else {
		pslot = NULL;
		oldest_lease = tv.tv_sec;
		for (i = 0; i < MAX_DHCP_LEASES_RECORDS; i++) {
			if ( oldest_lease > ipleases[i].end ) {
				pslot = &ipleases[i];
				oldest_lease = ipleases[i].end;
			}
		}
	}

	if ( pslot ) {
		pslot->ip = lease->yiaddr;
		pslot->start = tv.tv_sec;
		pslot->end = lease->expires;
		memcpy(pslot->mac, lease->chaddr, 6);
		strncpy(pslot->hostname, lease->hostname, 20);

		dhcplease_region.dirty++;
		retval = 0;
	}

	up(&dhcplease_region.sem);

	return retval;
}


unsigned long kernel_read_config(int type)
{
	int		i, ret, entry, len;
	ACCESS_CONF	*paccess;
	REGION_INFO	*pregion;
	
	for (i = 0; i < sizeof(access_conf)/sizeof(ACCESS_CONF); i++) {
		paccess = &access_conf[i];
		if ( type == paccess->type ) {
			pregion = paccess->reg_info;
//printk("pregion->base = %x paccess->offset=%x\n", pregion->base+0xbfc00000, paccess->offset);
			return (pregion->base+ 0xbfc00000 + paccess->offset);
		}
	}
	return 0;
	
	
}
// The "conf" type may be GENCONF_BLOCK, USERACCOUNT_BLOCK, DHCPLEASE_BLOCK
asmlinkage int sys_read_config(int type, void *conf)
{
	int		i, ret, entry, len;
	ACCESS_CONF	*paccess;
	REGION_INFO	*pregion;
	REG_EVENTLOG_CONF	*pconf;
	FIRMWARE_STATUS	fwsts;
	MEM_ACCESS	mem_data;
	
#if defined(CONFIG_MCT_SG100) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SA200)
	SWITCH_STATUS	switch_status;
#endif

	switch ( type ) {
	case MC_GET_FWVER:
		return put_user(*(unsigned short *) 0xbfc20004, (unsigned short *) conf);
	case MC_RWMEM:
		copy_from_user(&mem_data, (void *)conf, sizeof(MEM_ACCESS));
		memcpy(mem_data.buf, (unsigned char *)mem_data.dest_addr, mem_data.size);
		return 0;
	case MC_UPDATE_STATUS:
		fwsts.state = sysimage_state;
		fwsts.percentage = sysimage_percentage;
		ret = copy_to_user(conf, &fwsts, sizeof(FIRMWARE_STATUS));
		if ( ret > 0 )
			return -EFAULT;
		return 0;
	case MC_EVENTLOG_GET_CNT:
		down(&eventlog_region.sem);
		pconf = &eventlog_buf.eventlog_block.eventlog_conf;
		ret = (pconf->tail + MAX_LOGS_NUM - pconf->head) % MAX_LOGS_NUM;
		up(&eventlog_region.sem);
		return ret;
	case MC_EVENTLOG_GET:
		down(&eventlog_region.sem);
		ret = get_user(entry, (unsigned int *) conf);
		if ( ret != 0 ) goto eventlog_err;
		pconf = &eventlog_buf.eventlog_block.eventlog_conf;
		len = (pconf->tail + MAX_LOGS_NUM - pconf->head) % MAX_LOGS_NUM;
		if ( entry > len ) goto eventlog_err;
		entry = (pconf->head + entry) % MAX_LOGS_NUM;
		ret = copy_to_user((void *) conf+4, &pconf->eventlog[entry], sizeof(EVENTLOG_ENTRY));
		up(&eventlog_region.sem);
		if ( ret > 0 )
			return -EFAULT;

		return 0;
	case MC_DHCP_FIND_LEASE_BY_CHADDR:
//printk("MC_DHCP_FIND_LEASE_BY_CHADDR\n");
		return dhcp_find_lease_by_chaddr((DHCP_OFFERED_ADDR *) conf);
	case MC_DHCP_FIND_LEASE_BY_YIADDR:
//printk("MC_DHCP_FIND_LEASE_BY_YIADDR\n");
		return dhcp_find_lease_by_yiaddr((DHCP_OFFERED_ADDR *) conf);

	case MC_DHCP_LEASE:
		dhcp_clear_conflict();
		break;

#if defined(CONFIG_MCT_SG100) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SA200)
	case MC_SWITCH_STATUS:
		get_switch_status(&switch_status);
		ret = copy_to_user(conf, &switch_status, sizeof(SWITCH_STATUS));
		if ( ret > 0 )
			return -EFAULT;
		return 0;
#endif
	}

	for (i = 0; i < sizeof(access_conf)/sizeof(ACCESS_CONF); i++) {
		paccess = &access_conf[i];
		if ( type == paccess->type ) {
			pregion = paccess->reg_info;
			down(&pregion->sem);
			ret = copy_to_user((void *) conf, pregion->buf_base + paccess->offset, paccess->size);
//printk("flash=%d\n",ret);
			up(&pregion->sem);
			if ( ret > 0 )
				return -EFAULT;
			break;
		}
	}

	if ( i == sizeof(access_conf)/sizeof(ACCESS_CONF) )
		return -EIO;

	return 0;

eventlog_err:
	up(&eventlog_region.sem);
	return -EFAULT;
}

static unsigned char	*sysbuf_pos;
int set_factory_default(void);
#if defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SA200)
extern wait_queue_head_t shutdown_wq;
extern int shutdown_type;
#endif

static int mct_replace_str(char *input,int len,char *key,char *repstr)
{
	int i,start=0,item_len=0,rep_done=0,diff_len=0;
	char tmp_input[PAGE_SIZE];

	while(start<len) {
		if(strstr(input+start,key) != NULL) {
			diff_len = strlen(input+start) - strlen(repstr);
			strcpy(tmp_input+start,repstr);
			rep_done = 1;
			break;	
		}
		else {
			strcpy(tmp_input+start+diff_len,input+start);
			item_len = strlen(input+start);
			start += item_len+1;					
		}
	}	
	
	for(i=0;i<len+diff_len;i++) input[i] = tmp_input[i];

	if(rep_done) 
		return len+diff_len;
	else	
		return len;
}

asmlinkage int sys_write_config(int type, void *conf)
{
	int		i, ret;
	ACCESS_CONF	*paccess;
	REGION_INFO	*pregion;
	unsigned int	size;
	EVENTLOG_ENTRY	log;
	EVENTLOG_ENTRY	*pentry;
	REG_EVENTLOG_CONF	*pconf;
	struct timeval	tv;

	/* handle special case */
	switch ( type ) {
#if !defined(CONFIG_MCT_SG100)
	case MC_UPDATE_ALL_PID_TIMEZONE:
	{
		struct task_struct *p;
		struct mm_struct *mm;
		char buffer[PAGE_SIZE],mct_timezone_str[10];
		int res,mct_abs_timezone;
		EXTCONF_BLOCK *extconf_block;
        	extconf_block = (EXTCONF_BLOCK *) extconf_region.buf_base;

		
		if(extconf_block->time_zone - 13 > 0) 
			mct_abs_timezone = extconf_block->time_zone - 13;
		else
			mct_abs_timezone = -(extconf_block->time_zone - 13);

		sprintf(mct_timezone_str,"TZ=GMT%s%d",(extconf_block->time_zone-13) > 0 ? "-" : "+",mct_abs_timezone);
	
		printk("mct_timezone_str=%s\n",mct_timezone_str);

	        write_lock(&tasklist_lock);
	        for_each_task(p) {
			int len;
			if(current->pid == p->pid) continue;
			task_lock(p);
			mm = p->mm;
        		if (mm)
                		atomic_inc(&mm->mm_users);
			if (mm) {
         		       int len = mm->env_end - mm->env_start;
		               if (len > PAGE_SIZE)
					len = PAGE_SIZE;
			                res = access_process_vm(p, mm->env_start, buffer, len, 0);
					if(res>0) {
						res = mct_replace_str(buffer,res,"TZ=",mct_timezone_str);
						access_process_vm(p, mm->env_start, buffer, res, 1);
			                mmput(mm);
					}
        		}
		        task_unlock(p);
       		}
       		write_unlock(&tasklist_lock);
		return 0;
	}
#endif
#if defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SA200)
	case MC_POWER_DOWN:
		shutdown_type = LINUX_REBOOT_CMD_POWER_OFF;
		wake_up_interruptible(&shutdown_wq);
		return 0;
#endif
	case MC_RESET_CONFIG:
		if ( !conf_reset_in_progress ) {
			conf_reset_in_progress = 1;
#if defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SA200)
			shutdown_type = LINUX_REBOOT_CMD_RESTART;
			wake_up_interruptible(&shutdown_wq);
#else
			wake_up_interruptible(&flash_wq);
#endif
		}
		return 0;
	case MC_UPDATE_SYSTEM_START:
		down(&sysimage_sem);
		sysimage_state = SI_LOADING;
		sysbuf_pos = &__fsysbuf;
		up(&sysimage_sem);
		return 0;
	case MC_START_HEARTBEAT:
		hbled_init();
		return 0;
	case MC_UPDATE_SYSTEM_END:
		// check the magic number && product ID
		if ( *((unsigned short *) &__fsysbuf) == 0x0503
			&& *((unsigned short *) 0xbfc20002) == *((unsigned short *) (((void *)&__fsysbuf) + 2)) ) {
			if(*((unsigned short *) 0xbfc20004) < 0x0148 ||
			   *((unsigned short *)(((void *)&__fsysbuf)+4))<0x0148 ) {
				printk("magic number : OK\n");
				sysimage_state = SI_LOADED;
				sysimage_percentage = 0;
				wake_up_interruptible(&flash_wq);
			} 
			else {
				// to fix sa112 sa113 sa114 sa100 having the same
				// product id

				if(*((unsigned short *) 0xbfc20002) != 0x0004 || *((unsigned short *) 0xbfc20006) == *((unsigned short *) (((void *)&__fsysbuf) + 6)) ) {
					printk("magic number : OK\n");
	                                sysimage_state = SI_LOADED;
        	                        sysimage_percentage = 0;
                	                wake_up_interruptible(&flash_wq);
				}
				else {
					printk("produt id = %04x fixid = %04x\n",
	                                *((unsigned short *) (((void *)&__fsysbuf) + 2)),
        	                        *((unsigned short *) (((void *)&__fsysbuf) + 6)));
                		        sysimage_state = SI_ERROR;
				}

			}
	
		}
		else if ( *((unsigned short *) &__fsysbuf) == 0x9fc0) {
			printk("boot check number(0x9fc0) : OK\n");
			sysimage_state = SI_BOOT_LOADED;
			sysimage_percentage = 0;
			wake_up_interruptible(&flash_wq);
		} 
		else {
			printk("magic number = %04x, product ID = %04x\n",
				*((unsigned short *) &__fsysbuf),
				*((unsigned short *) (((void *) &__fsysbuf) + 2)));
			sysimage_state = SI_ERROR;
		}
		return 0;
	case MC_UPDATE_SYSTEM:
		down(&sysimage_sem);
		ret = get_user(size, (unsigned long *) conf);
#if 0
		printk("flash: size = %d\n", size);
#endif
		if ( ret != 0 ) goto update_sys_err;
		if ( sysbuf_pos + size > &__esysbuf ) goto update_sys_err;
		ret = copy_from_user(sysbuf_pos, (void *) conf+4, size);
		if ( ret > 0 ) goto update_sys_err;
		sysbuf_pos += size;
		up(&sysimage_sem);
		return 0;
	case MC_EVENTLOG_ADD:
		down(&eventlog_region.sem);
		ret = copy_from_user(&log, conf, sizeof(EVENTLOG_ENTRY));
		if ( ret > 0 ) {
			up(&eventlog_region.sem);
			return -EFAULT;
		}
		pconf = &eventlog_buf.eventlog_block.eventlog_conf;
		if ( ((pconf->tail+1)%MAX_LOGS_NUM) == pconf->head ) { // buffer full
			pconf->head = (pconf->head + 1) % MAX_LOGS_NUM;
		}
		pentry = &pconf->eventlog[pconf->tail];
		*pentry = log;
		do_gettimeofday(&tv);
		pentry->time = tv.tv_sec;
		pconf->tail = (pconf->tail + 1) % MAX_LOGS_NUM;
		up(&eventlog_region.sem);
		eventlog_region.dirty++;
		wake_up_interruptible(&flash_wq);
		return 0;
	case MC_EVENTLOG_CLEAR:
		down(&eventlog_region.sem);
		pconf = &eventlog_buf.eventlog_block.eventlog_conf;
		pconf->head = pconf->tail = 0;
		up(&eventlog_region.sem);
		eventlog_region.dirty++;
		wake_up_interruptible(&flash_wq);
		return 0;
	case MC_DHCP_ADD_LEASE:
//printk("MC_DHCP_ADD_LEASE\n");
		return dhcp_add_lease((DHCP_OFFERED_ADDR *) conf);
	case MC_DHCP_CLEAR_LEASE:
		return dhcp_clear_lease();
#if defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
	case MC_DEFAULT_USR_GRP_SHARE:
		down(&la_user_region.sem);
		set_default_users();
		up(&la_user_region.sem);
		la_user_region.dirty++;
		down(&la_grpshare_region .sem);
		set_default_grpshares();
		up(&la_grpshare_region .sem);
		la_grpshare_region.dirty++;
		break;
#endif
	}

	/* handle generic case */
	for (i = 0; i < sizeof(access_conf)/sizeof(ACCESS_CONF); i++) {
		paccess = &access_conf[i];
		if ( type == paccess->type ) {
			void *struct_base;

			pregion = paccess->reg_info;

			down(&pregion->sem);
			struct_base = paccess->reg_info->buf_base + paccess->offset;
			ret = copy_from_user(struct_base, (void *) conf, paccess->size);
			// handle special case
			switch ( type ) {
			case MC_URL_ACCESS_CONFIG:
				submit_url_records((REG_URL_ACCESS_CONF *) struct_base);
				break;
			}
			up(&pregion->sem);

			if ( ret > 0 )
				return -EFAULT;
				
			switch ( pregion->policy ) {
			case FL_IMMEDIATE:
				pregion->dirty++;
				wake_up_interruptible(&flash_wq);
				break;
			case FL_INTERVAL:
				pregion->dirty++;
				break;
			}
			break;
		}
	}

	return 0;

update_sys_err:
	up(&sysimage_sem);
	return -EFAULT;
}


#if defined(CONFIG_ATEN_LED) && defined(CONFIG_MCT_NAS)
#define HOST_NAME  "IOGEARNAS"
#elif defined(CONFIG_ATEN_LED)
#define HOST_NAME "BOSS"
#elif defined(CONFIG_MCT_SA200) && defined(CONFIG_MCT_NAS)
#define HOST_NAME	"nas200"
#elif defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SA200)
#define HOST_NAME	"nas"
#else
#define HOST_NAME	"router"
#endif

static void set_default_genconf(void)
{
	GENCONF_BLOCK	*genconf;
	REG_NET_CONF	*net_conf;
	REG_DHCP_CONF	*dhcp_conf;
	REG_ANTIHACK_CONF *antihack_conf;
	REG_ADMIN_CONF	*admin_conf;
	REG_PPTP_CONF	*pptp_conf;
#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
	REG_NFS_CONF	*nfs_conf;
	REG_IDESMART_CONF *idesmart_conf;
#endif
	genconf = (GENCONF_BLOCK *) genconf_region.buf_base;
	memset(genconf, 0, genconf_region.size);

	net_conf = &genconf->net_conf;
	strcpy(net_conf->hostname, HOST_NAME);
	net_conf->enable_nat = 1;
	
#if defined(CONFIG_ATEN_LED) // for iogear model
	net_conf->private_ip = IPADDR(192, 168, 2, 1);
	net_conf->private_netmask = IPADDR(255, 255, 255, 0);
#else
	net_conf->private_ip = IPADDR(172, 16, 1, 1);
	net_conf->private_netmask = IPADDR(255, 255, 0, 0);
#endif
	
	// set external interface
#if 0
	net_conf->public_ip = IPADDR(10, 0, 0, 1);
	net_conf->public_netmask = IPADDR(255, 0, 0, 0);
	net_conf->gateway = IPADDR(10, 0, 0, 2);
#endif
//marked by Louistsai 2003/7/21 , the shipping product don't use the taiwan dns server as default setting.
//	The router dns server will get from isp dhcp server or user keyin.
//	net_conf->dns0 = IPADDR(168, 95, 1, 1);
	net_conf->public_if = PUBIF_DHCPC;	// dhcp client

#if defined(CONFIG_MCT_NAS) && !defined(CONFIG_MCT_SA200) && 0
	net_conf->public_ip = IPADDR(192, 168, 1, 252);
        net_conf->public_netmask = IPADDR(255, 255, 255, 0);
        net_conf->gateway = IPADDR(192, 168, 1, 1);
	net_conf->public_if = PUBIF_FIXED;
#endif
	dhcp_conf = &genconf->dhcp_conf;
	dhcp_conf->enable = TRUE;
	dhcp_conf->lease_time = 360;
	dhcp_conf->range_start0 = IPADDR(172, 16, 1, 2);
	dhcp_conf->range_end0 = IPADDR(172, 16, 1, 250);

	antihack_conf = &genconf->antihack_conf;
	antihack_conf->enable = TRUE;
	antihack_conf->EN_SYNFLOOD = FALSE;
	antihack_conf->EN_ICMPFLOOD = TRUE;
	antihack_conf->EN_UDPFLOOD = TRUE;
	antihack_conf->EN_PING_DEATH = TRUE;
	antihack_conf->EN_IP_SPOOF = TRUE;
	antihack_conf->EN_PORT_SCAN = TRUE;
	antihack_conf->EN_TEAR_DROP = TRUE;
	antihack_conf->syn_limit = 1;
	antihack_conf->icmp_limit = 1;
	antihack_conf->udp_limit = 1;

	admin_conf = &genconf->admin_conf;
	strcpy(admin_conf->password, "admin");
	admin_conf->enable_ext = 0;
	admin_conf->ext_webport = 8080;
#if defined(CONFIG_MCT_NAS)
	admin_conf->ext_webport = 80;
#endif
	pptp_conf = &genconf->pptp_conf;
	pptp_conf->enable = FALSE;
	pptp_conf->serverip= 254;
	pptp_conf->startip = 200;
	pptp_conf->endip   = 230;
#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
	nfs_conf = &genconf->nfs_conf;
	nfs_conf->enable = FALSE;
	nfs_conf->count = 0;
	idesmart_conf = &genconf->idesmart_conf;
	idesmart_conf->stimer=60;  /* set default standby timer 5 min */
#endif	

}

#if defined(CONFIG_MCT_SG100)
static void set_default_userconf(void)
{
	USERACCOUNT_BLOCK *useraccount_block;
	useraccount_block = (USERACCOUNT_BLOCK *) useraccount_region.buf_base;
	memset(useraccount_block, 0, useraccount_region.size);
}
#else
static void set_default_extconf(void)
{
	EXTCONF_BLOCK *extconf_block;
	REG_W3S_CONF *www_conf;
	extconf_block = (EXTCONF_BLOCK *) extconf_region.buf_base;
	www_conf = &extconf_block->w3s_conf;
	memset(extconf_block, 0, extconf_region.size);
	www_conf->port_num = 80;
	extconf_block->time_zone = 21;
	extconf_block->ftp_port_num = 21;
	extconf_block->upnp_device_enable = 1;
	extconf_block->upnp_mms_enable	  = 0;
	extconf_block->pppoe_mtu = 1492;
}
#endif

static void set_default_eventlog(void)
{
#if 0
	EVENTLOG_BLOCK	*eventlog_block;
	EVENTLOG_ENTRY	*eventlog;
	REG_EVENTLOG_CONF *eventlog_conf;
	struct timeval	tv;

	eventlog_block = (EVENTLOG_BLOCK *) eventlog_region.buf_base;
	memset(eventlog_block, 0, eventlog_region.size);
	eventlog_conf = &eventlog_block->eventlog_conf;

	eventlog_conf->head = 0;
	eventlog_conf->tail = 1;
	eventlog = &eventlog_conf->eventlog[0];
	eventlog->level = LEVEL_NORMAL;
	strcpy(eventlog->msg, "Configuration reset to factory setting");
	do_gettimeofday(&tv);
	eventlog->time = tv.tv_sec;
#else
	REG_EVENTLOG_CONF *eventlog_conf;
	EVENTLOG_BLOCK	*eventlog_block;
	
	eventlog_block = (EVENTLOG_BLOCK *) eventlog_region.buf_base;
	memset(eventlog_block, 0, eventlog_region.size);
	
	eventlog_conf = &eventlog_buf.eventlog_block.eventlog_conf;
	eventlog_conf->head = eventlog_conf->tail = 0;
#endif
}

#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)

static void set_default_users(void)
{
	LA_USERS_BLOCK *la_users_block;
	la_users_block = (LA_USERS_BLOCK *) la_user_region.buf_base;
	memset(la_users_block, 0, sizeof(LA_USERS_BLOCK));
	
	strcpy(la_users_block->users.usrdata[0].name , "admin");
	strcpy(la_users_block->users.usrdata[0].password , "admin");
	la_users_block->users.usrdata[0].uid = 0;
	strcpy(la_users_block->users.usrdata[1].name , "guest");
	strcpy(la_users_block->users.usrdata[1].password , "");
	la_users_block->users.usrdata[1].uid = 1;
	la_users_block->users.bitstr[0] = 0x00000003;
	la_users_block->users.pptp_on[0] = 0x00000000;
	
	
}

static void set_default_grpshares(void)
{
	LA_GRPSHARE_BLOCK *la_grpshare_block;
	la_grpshare_block = (LA_GRPSHARE_BLOCK *) la_grpshare_region.buf_base;
	memset(la_grpshare_block, 0, sizeof(LA_GRPSHARE_BLOCK));
		
	strcpy(la_grpshare_block->shares.workgrp_name,"Workgroup");
	strcpy(la_grpshare_block->shares.computer_des,HOST_NAME);
#if defined(CONFIG_MCT_SG600)
	la_grpshare_block->shares.cifs_enable = 1;
#else
	la_grpshare_block->shares.cifs_enable = 0;
#endif
	la_grpshare_block->shares.mac_enable = 0;
	la_grpshare_block->shares.ftp_enable = 0;
	strcpy(la_grpshare_block->shares.applezone,"*");
	
	
	strcpy(la_grpshare_block->groups.grpdata[0].name , "everyone");
	la_grpshare_block->groups.grpdata[0].gid = 0;
	la_grpshare_block->groups.grpdata[0].members[0] = 0x3;
	
	la_grpshare_block->groups.bitstr[0] = 0x00000001;
	la_grpshare_block->groups.pptp_on[0] = 0x00000000;
	
}


/* static void set_default_usershare(void)
{

	USERSHARE_BLOCK *usershare_block;
	usershare_block = (USERSHARE_BLOCK *) usershare_region.buf_base;
	memset(usershare_block, 0,sizeof(USERSHARE_CKSUM_BLOCK));
		
	strcpy(usershare_block->shares.workgrp_name,"Workgroup");
	strcpy(usershare_block->shares.computer_des,HOST_NAME);
#if defined(CONFIG_MCT_SG600)
	usershare_block->shares.cifs_enable = 1;
#else
	usershare_block->shares.cifs_enable = 0;
#endif
	usershare_block->shares.mac_enable = 0;
	usershare_block->shares.ftp_enable = 0;
	strcpy(usershare_block->shares.applezone,"*");
	
	strcpy(usershare_block->users.usrdata[0].name , "admin");
	strcpy(usershare_block->users.usrdata[0].password , "admin");
	usershare_block->users.usrdata[0].uid = 0;
	strcpy(usershare_block->users.usrdata[1].name , "guest");
	strcpy(usershare_block->users.usrdata[1].password , "");
	usershare_block->users.usrdata[1].uid = 1;
	usershare_block->users.bitstr[0] = 0x00000003;
	usershare_block->users.pptp_on[0] = 0x00000000;
	
	strcpy(usershare_block->groups.grpdata[0].name , "everyone");
	usershare_block->groups.grpdata[0].gid = 0;
//	usershare_block->groups.grpdata[0].nr_members = 2;
	usershare_block->groups.grpdata[0].members[0] = 0x3;
	
	usershare_block->groups.bitstr[0] = 0x00000001;
	usershare_block->groups.pptp_on[0] = 0x00000000;
	
	
}
*/
#endif 


int set_factory_default(void)
{
	down(&genconf_region.sem);
	set_default_genconf();
	up(&genconf_region.sem);
	genconf_region.dirty++;

#if 0	/* Don't remove user account settings */
	down(&useraccount_region.sem);
	set_default_userconf();
	up(&useraccount_region.sem);
	useraccount_region.dirty++;
#endif

#if 1
	down(&dhcplease_region.sem);
	set_default_leaseconf();
	up(&dhcplease_region.sem);
	dhcplease_region.write_force = 1;
#endif

	down(&eventlog_region.sem);
	set_default_eventlog();
	up(&eventlog_region.sem);
	eventlog_region.dirty++;
	
#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
	down(&usershare_region.sem);
//	set_default_usershare();
	up(&usershare_region.sem);
	usershare_region.dirty++;
	
	down(&extconf_region.sem);
	set_default_extconf();
	up(&extconf_region.sem);
	extconf_region.dirty++;
	
#endif

	return 0;
}

static int cksum_ok(unsigned int *buf, int size)
{
	int		i;
	unsigned int	cksum = 0xffffffff;

	size = (size >> 2) - 1;
	for (i = 0; i < size; i++) {
		cksum += *buf++;
	}

	return (cksum == *buf) ? 1 : 0;
}

static int handle_flash_write(void)
{
	int		i;
	int		ret = 0;
	REGION_INFO	*pregion;

	for (i = 0; i < sizeof(region_list)/sizeof(void*); i++) {
		pregion = region_list[i];
		switch ( pregion->policy ) {
		case FL_IMMEDIATE:
			if ( pregion->dirty ) {
				int size;
				down(&pregion->sem);
				size = pregion->size;
#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
				if ( pregion == &usershare_region ) {
					size = sizeof(USERSHARE_CKSUM_BLOCK);
					//printk("user share block, size = %08x\n", size);
				}
				if ( pregion == &la_user_region ) {
					size = sizeof(LA_USERS_CKSUM_BLOCK);
//					printk("user block, size = %08x\n", size);
				}
				if ( pregion == &la_grpshare_region ) {
					size = sizeof(LA_GRPSHARE_CKSUM_BLOCK);
//					printk("grpshare block, size = %08x\n", size);
				}
//printk("flash write\n");				
#endif
				gen_cksum(pregion->buf_base, size);
				ret = erase_write(mtd, pregion->base, pregion->size, pregion->buf_base);
				pregion->dirty = 0;
				up(&pregion->sem);
			}
			break;
		case FL_INTERVAL:
			if ( pregion->write_force ||
				(pregion->dirty >= pregion->dirty_threshold && time_after_eq(jiffies, pregion->last_time_modified+pregion->short_term_interval)) ||
				(pregion->dirty && time_after_eq(jiffies, pregion->last_time_modified+pregion->long_term_interval)) ) {
				down(&pregion->sem);
				gen_cksum(pregion->buf_base, pregion->size);
				ret = erase_write(mtd, pregion->base, pregion->size, pregion->buf_base);
				pregion->write_force = 0;
				pregion->dirty = 0;
				pregion->last_time_modified = jiffies;
				up(&pregion->sem);
			}
			break;
		}

		if ( ret < 0 )
			break;
	}

	// update system image
	if ( sysimage_state == SI_LOADED ) {
		unsigned int	steps, total_steps;
		unsigned long	flash_addr;
		unsigned char	*bufpos;

		down(&sysimage_sem);
		sysimage_state = SI_WRITING;
		total_steps = ((unsigned int) (sysbuf_pos - &__fsysbuf) + SYSIMAGE_SECTOR_SIZE - 1) / SYSIMAGE_SECTOR_SIZE;
		bufpos = &__fsysbuf;
		flash_addr = SYSIMAGE_ADDR;
		steps = 0;
		//added by louistsai 2004/7/9
		remove_fwsig(mtd);
		while ( bufpos < sysbuf_pos ) {
			printk("flash: address = %08x\n", (unsigned int) flash_addr);
			ret = erase_write(mtd, flash_addr, SYSIMAGE_SECTOR_SIZE, bufpos);
			bufpos += SYSIMAGE_SECTOR_SIZE;
			flash_addr += SYSIMAGE_SECTOR_SIZE;
			sysimage_percentage = (unsigned char) (steps * 100 / total_steps);
			steps += 1;
		}
		sysimage_percentage = 100;
		sysimage_state = SI_COMPLETED;
		up(&sysimage_sem);
	} 
	else if( sysimage_state == SI_BOOT_LOADED ) {
		unsigned int	steps, total_steps;
		unsigned long	flash_addr;
		unsigned char	*bufpos;
		int sector_size = 0;

		down(&sysimage_sem);
		sysimage_state = SI_WRITING;
		total_steps = 9 ; // 8k*8 + 64k sectors
		bufpos = &__fsysbuf;
		flash_addr = 0x00000000; // boot code zone
		steps = 0;
		while ( bufpos < sysbuf_pos ) {
			printk("flash: address = %08x\n", (unsigned int) flash_addr);
			if(steps <8)
				sector_size = 8 * 1024;
			else
				sector_size = 64 * 1024;
			
			if(steps == 0 || steps == 8)
				ret = erase_write(mtd, flash_addr, sector_size, bufpos);
			bufpos += sector_size;
			flash_addr += sector_size;
			sysimage_percentage = (unsigned char) (steps * 100 / total_steps);
			steps += 1;
		}
		sysimage_percentage = 100;
		sysimage_state = SI_COMPLETED;
		up(&sysimage_sem);
	}

	return ret;
}

extern long sys_reboot(int magic1, int magic2, unsigned int cmd, void *arg);
static int flash_thread(void *dummy)
{
	struct task_struct *tsk = current;
	DECLARE_WAITQUEUE(wait, tsk);

	tsk->session = 1;
	tsk->pgrp = 1;
	tsk->flags |= PF_MEMALLOC;
	strcpy(tsk->comm, "flashd");
	tsk->tty = NULL;
	spin_lock_irq(&tsk->sigmask_lock);
	sigfillset(&tsk->blocked);
	recalc_sigpending(tsk);
	spin_unlock_irq(&tsk->sigmask_lock);
	exit_mm(tsk);
	exit_files(tsk);
	exit_sighand(tsk);
	exit_fs(tsk);

	while ( 1 ) {
		add_wait_queue(&flash_wq, &wait);
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(6 * HZ);
		if ( conf_reset_in_progress ) {
			printk(KERN_INFO "Flash Update: set configuration to factory default\n");
			set_factory_default();
			handle_flash_write();
			sys_reboot(LINUX_REBOOT_MAGIC1, LINUX_REBOOT_MAGIC2, LINUX_REBOOT_CMD_RESTART, 0);
			conf_reset_in_progress = 0;
		} else {
			handle_flash_write();
		}
		remove_wait_queue(&flash_wq, &wait);
	}

	return 0;
}

int flash_notifier_reboot(struct notifier_block *notifier, unsigned long code, void *x)
{
	int		i;
	REGION_INFO	*pregion;

	switch ( code ) {
	case SYS_RESTART:
	case SYS_HALT:
	case SYS_POWER_OFF:
		printk(KERN_INFO "flash: write back all configuration data\n");
		for (i = 0; i < sizeof(region_list)/sizeof(void*); i++) {
			pregion = region_list[i];
			//added by louistsai, if no, we nerver write lease file to flash before 
			//short term or long term interval.
			
			if ( !conf_reset_in_progress && (pregion == &dhcplease_region)) pregion->write_force = 1;
			
			while ( (pregion->policy == FL_IMMEDIATE && pregion->dirty) ||
				(pregion->policy == FL_INTERVAL && pregion->write_force) ) {
				schedule_timeout(1);
			}
		}

		while ( sysimage_state == SI_WRITING )
			schedule_timeout(5);
	}

	return NOTIFY_DONE;
}

struct notifier_block flash_notifier = {
	notifier_call: flash_notifier_reboot,
	next: NULL,
	priority: INT_MAX,
};

extern void conf_reset_init(void);

// The function should be called after MTD was initialized.
int __init sysconf_init(void)
{
	int	retlen;
	//EVENTLOG_ENTRY	log;

	printk(KERN_INFO "Loading system configuration data from flash\n");
	// load configuration data
	mtd = __get_mtd_device(NULL, 0);
	mtd->read(mtd, GENCONF_ADDR, sizeof(GENCONF_CKSUM_BLOCK), &retlen, (char *) &genconf_buf);
	if ( retlen != sizeof(GENCONF_CKSUM_BLOCK) ) {
		printk(KERN_NOTICE "Sysconf: read error during reading general config\n");
	} else {
		if ( !cksum_ok((unsigned int *) &genconf_buf, retlen) ) {
			printk(KERN_NOTICE "Sysconf: invalid general config checksum\n");
			set_default_genconf();
		}
	}

#if defined(CONFIG_MCT_SG100)
	mtd->read(mtd, USERACCOUNT_ADDR, sizeof(USERACCOUNT_CKSUM_BLOCK), &retlen, (char *) &useraccount_buf);
	if ( retlen != sizeof(USERACCOUNT_CKSUM_BLOCK) ) {
		printk(KERN_NOTICE "Sysconf: error during reading user account\n");
	} else {
		if ( !cksum_ok((unsigned int *) &useraccount_buf, retlen) ) {
			printk(KERN_NOTICE "Sysconf: invalid user account checksum\n");
			set_default_userconf();
		}
	}
#else
	mtd->read(mtd, EXTCONF_ADDR, sizeof(EXTCONF_CKSUM_BLOCK), &retlen, (char *) &extconf_buf);
	if ( retlen != sizeof(EXTCONF_CKSUM_BLOCK) ) {
		printk(KERN_NOTICE "Sysconf: error during reading user account\n");
	} 
	else {
		if ( !cksum_ok((unsigned int *) &extconf_buf, retlen) ) {
			printk(KERN_NOTICE "Sysconf: invalid extend config checksum\n");
			set_default_extconf();
		}
	}
#endif

	mtd->read(mtd, DHCPLEASE_ADDR, sizeof(DHCPLEASE_CKSUM_BLOCK), &retlen, (char *) &dhcplease_buf);
	if ( retlen != sizeof(DHCPLEASE_CKSUM_BLOCK) ) {
		printk(KERN_NOTICE "Sysconf: error during reading dhcp lease config\n");
	} else {
		if ( !cksum_ok((unsigned int *) &dhcplease_buf, retlen) ) {
			printk(KERN_NOTICE "Sysconf: invalid dhcp lease checksum\n");
			set_default_leaseconf();
		}
	}
	
	mtd->read(mtd, EVENTLOG_ADDR, sizeof(EVENTLOG_CKSUM_BLOCK), &retlen, (char *) &eventlog_buf);
	if ( retlen != sizeof(EVENTLOG_CKSUM_BLOCK) ) {
		printk(KERN_NOTICE "Sysconf: error during reading event log config\n");
	} else {
		if ( !cksum_ok((unsigned int *) &eventlog_buf, retlen) ) {
			printk(KERN_NOTICE "Sysconf: invalid event log checksum\n");
			set_default_eventlog();
		}
	}

#if defined(CONFIG_MCT_SG600) || defined(CONFIG_MCT_SA100) || defined(CONFIG_MCT_SG4100_EV) || defined(CONFIG_MCT_SA200)
	mtd->read(mtd, USERSHARE_ADDR, sizeof(USERSHARE_CKSUM_BLOCK), &retlen, (char *) &usershare_buf);
	if ( retlen != sizeof(USERSHARE_CKSUM_BLOCK) ) {
		printk(KERN_NOTICE "Sysconf: error during usershare config\n");
	} else {
		if ( cksum_ok((unsigned int *) &usershare_buf, retlen) ) {
			restore_olduser_data(&usershare_buf,&la_user_buf,&la_grpshare_buf);
			printk(KERN_NOTICE "Sysconf: restore old user data\n");
		}
		else {
			mtd->read(mtd, LA_USER_ADDR, sizeof(LA_USERS_CKSUM_BLOCK), &retlen, (char *) &la_user_buf);
			if ( retlen != sizeof(LA_USERS_CKSUM_BLOCK) ) {
				printk(KERN_NOTICE "Sysconf: error during reading users data\n");
			} else {
				if ( !cksum_ok((unsigned int *) &la_user_buf, retlen) ) {
					printk(KERN_NOTICE "Sysconf: invalid users data checksum\n");
					set_default_users();
				}
			}
			mtd->read(mtd, LA_GRPSHARE_ADDR, sizeof(LA_GRPSHARE_CKSUM_BLOCK), &retlen, (char *) &la_grpshare_buf);
			if ( retlen != sizeof(LA_GRPSHARE_CKSUM_BLOCK) ) {
				printk(KERN_NOTICE "Sysconf: error during reading group & share data\n");
			} else {
				if ( !cksum_ok((unsigned int *) &la_grpshare_buf, retlen) ) {
					printk(KERN_NOTICE "Sysconf: invalid group & shares data checksum\n");
					set_default_grpshares();
				}
			}
		}
	}
#endif	
	
	// start kernel thread
	kernel_thread(flash_thread, NULL, CLONE_FS|CLONE_FILES|CLONE_SIGHAND);

#if !defined(CONFIG_MCT_SG4100)
	// initialization routine for button of resetting config setting.
	conf_reset_init();
#endif

	// initialization for system image
	//printk("sysbuf: start = %08x, end = %08x\n", &__fsysbuf, &__esysbuf);
	sysimage_state = SI_NORMAL;
	sysimage_percentage = 0;

	// set URL filter patterns
	submit_url_records(&genconf_buf.genconf_block.url_access_conf);

	// register reboot notifier
	register_reboot_notifier(&flash_notifier);

	// log system startup time
	// Marked by Louistsai ,because all log must be reported on app level 
	//log.level = LEVEL_NORMAL;
	//log.msgid = IDS_SYSTEM_START;
	//sys_write_config(MC_EVENTLOG_ADD, &log);

	return 0;
}

static void restore_olduser_data(USERSHARE_BLOCK* old ,LA_USERS_BLOCK* new_user, LA_GRPSHARE_BLOCK* new_grpshare)
{
	int i,j;
	
	//update users
	memset(new_user,0,sizeof(LA_USERS_BLOCK));
	memset(new_grpshare,0,sizeof(LA_GRPSHARE_BLOCK));
	
	for(i=0;i<MAX_USERS;i++) {
		strcpy(new_user->users.usrdata[i].name,old->users.usrdata[i].name);
		strcpy(new_user->users.usrdata[i].password,old->users.usrdata[i].password);
		new_user->users.usrdata[i].uid = old->users.usrdata[i].uid; 
	}
	for(i=0;i<8;i++) {
		new_user->users.bitstr[i] = old->users.bitstr[i];
		new_user->users.pptp_on[i] = old->users.pptp_on[i];
	}
	//update groups
	for(i=0;i<MAX_GROUPS;i++) {
		strcpy(new_grpshare->groups.grpdata[i].name,old->groups.grpdata[i].name);
		new_grpshare->groups.grpdata[i].gid = old->groups.grpdata[i].gid ;
		for(j=0;j<8;j++) new_grpshare->groups.grpdata[i].members[j] = old->groups.grpdata[i].members[j];
	}
	for(i=0;i<8;i++) {
		new_grpshare->groups.bitstr[i] = old->groups.bitstr[i];
		new_grpshare->groups.pptp_on[i] = old->groups.pptp_on[i];
	}
	//update share
	for(i=0;i<MAX_SHARES;i++){
		strcpy(new_grpshare->shares.shadata[i].name,old->shares.shadata[i].name);
		new_grpshare->shares.shadata[i].attr = old->shares.shadata[i].attr;
		for(j=0;j<8;j++) {
			new_grpshare->shares.shadata[i].r_users[j] = old->shares.shadata[i].r_users[j];
			new_grpshare->shares.shadata[i].r_grps[j]  = old->shares.shadata[i].r_grps[j];
			new_grpshare->shares.shadata[i].w_users[j] = old->shares.shadata[i].w_users[j];
			new_grpshare->shares.shadata[i].w_grps[j]  = old->shares.shadata[i].w_grps[j];
		}
	}
	new_grpshare->shares.cifs_enable = old->shares.cifs_enable;
	new_grpshare->shares.ftp_enable  = old->shares.ftp_enable ;
	new_grpshare->shares.mac_enable  = old->shares.mac_enable ;
	strcpy(new_grpshare->shares.applezone,old->shares.applezone);
	new_grpshare->shares.count = old->shares.count;
	strcpy(new_grpshare->shares.workgrp_name,old->shares.workgrp_name);
	strcpy(new_grpshare->shares.computer_des,old->shares.computer_des);
}


module_init(sysconf_init);
