/* $Id: tx4925ndfmc.c,v 1.1.1.1 2004/04/07 08:36:54 louistsai Exp $
 *  drivers/mtd/nand/tx4925ndfmc.c
 *
 *  Copyright (C) 2001 Toshiba Corporation
 *  Based on spia.c by Steven J. Hill
 *  Copyright (C) 2000 Steven J. Hill (sjhill@cotw.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Overview:
 *   This is a device driver for the NAND flash device connected to
 *   TX4925 internal NAND Memory Controller.
 */

#include <linux/slab.h>
#include <linux/module.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <asm/io.h>
#include <asm/bootinfo.h>
#include <asm/tx4925.h>
#ifdef CONFIG_TOSHIBA_RBTX4925
#include <asm/toshiba-boards/rbtx4925.h>
#endif

/*
 * MTD structure for TX4925 NDFMC
 */
static struct mtd_info *tx4925ndfmc_mtd = NULL;

/*
 * Define partitions for flash device
 */
#ifdef CONFIG_MTD_PARTITIONS
static struct mtd_partition partition_info[] = {
	{ name: "TX4925 NAND flash partition",
	  offset: 0,
	  size: 0 },
};
#define NUM_PARTITIONS 1
#endif

static unsigned char tx4925ndfmc_ioread(struct nand_chip *ncp)
{
	// return readl(ncp->IO_ADDR);
#if 0
	if ((tx4925_ndfmcptr->mcr & TX4925_NDFMCR_WE)
		tx4925_ndfmcptr->mcr &= ~TX4925_NDFMCR_WE;
#endif
	return tx4925_ndfmcptr->dtr;
}
static void tx4925ndfmc_iowrite(struct nand_chip *ncp, unsigned char dat)
{
	u32 mcr = tx4925_ndfmcptr->mcr;
	int datamode = !(mcr & (TX4925_NDFMCR_ALE|TX4925_NDFMCR_CLE));
	// writel(dat, ncp->IO_ADDR);
#if 0
	if (!(tx4925_ndfmcptr->mcr & TX4925_NDFMCR_WE))
		tx4925_ndfmcptr->mcr |= TX4925_NDFMCR_WE;
#endif
	if (datamode)
		tx4925_ndfmcptr->mcr = mcr | TX4925_NDFMCR_WE;
	tx4925_ndfmcptr->dtr = dat;
	if (datamode)
		tx4925_ndfmcptr->mcr = mcr;
}
static void tx4925ndfmc_clr_ctl(struct nand_chip *ncp, unsigned int sig)
{
	// *(volatile unsigned char *)ncp->CTRL_ADDR &= ~sig;
	if (sig == TX4925_NDFMCR_CE)
		tx4925_ndfmcptr->mcr |= sig;
	else
		tx4925_ndfmcptr->mcr &= ~sig;
}
static void tx4925ndfmc_set_ctl(struct nand_chip *ncp, unsigned int sig)
{
	// *(volatile unsigned char *)ncp->CTRL_ADDR |= sig;
	if (sig == TX4925_NDFMCR_CE)
		tx4925_ndfmcptr->mcr &= ~sig;
	else
		tx4925_ndfmcptr->mcr |= sig;
}
static int tx4925ndfmc_chk_busy(struct nand_chip *ncp)
{
	return tx4925_ndfmcptr->sr & TX4925_NDFSR_BUSY;
}

/*
 * Main initialization routine
 */
int __init tx4925ndfmc_init (void)
{
	struct nand_chip *this;
	int bsprt = 0, hold = 0xf, spw = 0xf;

	switch (mips_machtype) {
#ifdef CONFIG_TOSHIBA_RBTX4925
	case MACH_TOSHIBA_RBTX4925:
		if (*rbtx4925_piosel_ptr & RBTX4925_PIOSEL_NOSMART) {
			printk(KERN_DEBUG "TX4925 NDFMC: disabled by PIOSEL\n");
			return -ENODEV;
		}
		if (!(*rbtx4925_ssfdc_wp_ptr & 1))
			printk(KERN_INFO "TX4925 NDFMC: write protected.\n");
		bsprt = 1;
		hold = 2;
		spw = 6;
		break;
#endif
	default:
		return -ENODEV;
	}

	if ((tx4925_ccfgptr->pcfg & (TX4925_PCFG_SELACLC|TX4925_PCFG_SELNAND))
	    != TX4925_PCFG_SELNAND) {
		printk(KERN_DEBUG "TX4925 NDFMC: disabled by PCFG.\n");
		return -ENODEV;
	}

	/* reset NDFMC */
	tx4925_ndfmcptr->rstr |= TX4925_NDFRSTR_RST;
	while (tx4925_ndfmcptr->rstr & TX4925_NDFRSTR_RST)
		;
	/* setup BusSeparete, Hold Time, Strobe Pulse Width */
	tx4925_ndfmcptr->mcr = bsprt ? TX4925_NDFMCR_BSPRT : 0;
	tx4925_ndfmcptr->spr = hold << 4 | spw;

	/* Allocate memory for MTD device structure and private data */
	tx4925ndfmc_mtd = kmalloc (sizeof(struct mtd_info) + sizeof (struct nand_chip),
				GFP_KERNEL);
	if (!tx4925ndfmc_mtd) {
		printk ("Unable to allocate TX4925 NAND MTD device structure.\n");
		return -ENOMEM;
	}

	/* Get pointer to private data */
	this = (struct nand_chip *) (&tx4925ndfmc_mtd[1]);

	/* Initialize structures */
	memset((char *) tx4925ndfmc_mtd, 0, sizeof(struct mtd_info));
	memset((char *) this, 0, sizeof(struct nand_chip));

	/* Link the private data with the MTD structure */
	tx4925ndfmc_mtd->priv = this;

	/* Set address of NAND IO lines */
	this->IO_ADDR = (unsigned long)&tx4925_ndfmcptr->dtr;
	this->CTRL_ADDR = (unsigned long)&tx4925_ndfmcptr->mcr;
	this->CLE = TX4925_NDFMCR_CLE;
	this->ALE = TX4925_NDFMCR_ALE;
	this->NCE = TX4925_NDFMCR_CE;	/* 0:high 1:low */
	this->ioread = tx4925ndfmc_ioread;
	this->iowrite = tx4925ndfmc_iowrite;
	this->clr_ctl = tx4925ndfmc_clr_ctl;
	this->set_ctl = tx4925ndfmc_set_ctl;
	this->chk_busy = tx4925ndfmc_chk_busy;

	/* Scan to find existance of the device */
	if (nand_scan (tx4925ndfmc_mtd)) {
		kfree (tx4925ndfmc_mtd);
		return -ENXIO;
	}

	/* Allocate memory for internal data buffer */
	this->data_buf = kmalloc (sizeof(u_char) * (tx4925ndfmc_mtd->oobblock + tx4925ndfmc_mtd->oobsize), GFP_KERNEL);
	if (!this->data_buf) {
		printk ("Unable to allocate NAND data buffer for TX4925.\n");
		kfree (tx4925ndfmc_mtd);
		return -ENOMEM;
	}

#ifdef CONFIG_MTD_PARTITIONS
	/* Register the partitions */
	partition_info.size = tx4925ndfmc_mtd.size;
	add_mtd_partitions(tx4925ndfmc_mtd, partition_info, NUM_PARTITIONS);
#else
	add_mtd_device(tx4925ndfmc_mtd);
#endif

	/* Return happy */
	return 0;
}
module_init(tx4925ndfmc_init);

/*
 * Clean up routine
 */
#ifdef MODULE
static void __exit tx4925ndfmc_cleanup (void)
{
	struct nand_chip *this = (struct nand_chip *) &tx4925ndfmc_mtd[1];

	/* Unregister the device */
#ifdef CONFIG_MTD_PARTITIONS
	del_mtd_partitioons (tx4925ndfmc_mtd);
#else
	del_mtd_device (tx4925ndfmc_mtd);
#endif

	/* Free internal data buffer */
	kfree (this->data_buf);

	/* Free the MTD device structure */
	kfree (tx4925ndfmc_mtd);
}
module_exit(tx4925ndfmc_cleanup);
#endif

MODULE_LICENSE("GPL");
MODULE_AUTHOR("TOSHIBA Corporation");
MODULE_DESCRIPTION("Board-specific glue layer for NAND flash on TX4925 NDFMC");
