/*
 *
 * tdfx_hw.c
 * 
 *	Framebuffer driver for video cards with 3dfx chipsets
 *		
 * Authors: 
 *		Hannu Mallat <hannu@firsthop.com>
 *   		Attila Kesmarki <danthe@aat.hu>
 * 
 * Copyright (C) 1999,2000 Hannu Mallat & Attila Kesmarki 
 * All rights reserved
 *
 */

#include "tdfx_hw.h"

int nohwcursor = 0;

/********************************************************************
 		       2D accelerated functions
 ********************************************************************/

/* 
 * Set the starting position of the visible screen to var->yoffset 
 */
void do_pan_var(struct fb_var_screeninfo *var, struct fb_info_tdfx *i)
{
	u32 addr;
	void *mmiobase = i->regbase_virt;

	addr = var->yoffset * i->fb_info.fix.line_length;
	do_make_room(1);
	tdfx_outl(VIDDESKSTART, addr);
}

/*
 * Invert the hardware cursor image (timerfunc)  
 */
void do_flashcursor(unsigned long ptr)
{
	struct fb_info_tdfx *i = (struct fb_info_tdfx *) ptr;
	void *mmiobase = i->regbase_virt;
	unsigned long flags;
	
	spin_lock_irqsave(&i->DAClock, flags);
	do_make_room(1);
	tdfx_outl(VIDPROCCFG, tdfx_inl(VIDPROCCFG) ^ VIDCFG_HWCURSOR_ENABLE);
	i->cursor.timer.expires = jiffies + HZ / 2;
	add_timer(&i->cursor.timer);
	spin_unlock_irqrestore(&i->DAClock, flags);
}

/*
 * FillRect 2D command (solidfill or invert (via ROP_XOR))   
 */
void
do_fillrect(void *mmiobase, u32 x, u32 y, u32 w, u32 h,
	    u32 color, u32 stride, u32 bpp, u32 rop)
{

	u32 fmt = stride | ((bpp + ((bpp == 8) ? 0 : 8)) << 13);

	do_make_room(7);
	tdfx_outl(DSTFORMAT, fmt);
	tdfx_outl(COLORFORE, color);
	tdfx_outl(COMMAND_2D, COMMAND_2D_FILLRECT | (rop << 24));
	tdfx_outl(DSTBASE, y * stride);
	tdfx_outl(DSTSIZE, w | (h << 16));
	tdfx_outl(LAUNCH_2D, x);
	tdfx_outl(DSTBASE, 0);
	do_wait_idle();
}

/*
 * Screen-to-Screen BitBlt 2D command (for the bmove fb op.) 
 */

void
do_bitblt(void *mmiobase,
	  u32 curx,
	  u32 cury,
	  u32 dstx, u32 dsty, u32 width, u32 height, u32 stride, u32 bpp)
{

	u32 blitcmd = COMMAND_2D_S2S_BITBLT | (ROP_COPY << 24);
	u32 fmt = stride | ((bpp + ((bpp == 8) ? 0 : 8)) << 13);

	do_make_room(8);
	tdfx_outl(DSTBASE, dsty * stride);
	tdfx_outl(SRCBASE, cury * stride);

	if (curx <= dstx) {
		//-X 
		blitcmd |= BIT(14);
		curx += width - 1;
		dstx += width - 1;
	}
	if (cury <= dsty) {
		//-Y  
		blitcmd |= BIT(15);
		cury = height - 1;
		dsty = height - 1;
	} else {
		cury = 0;
		dsty = 0;
	}

	tdfx_outl(SRCFORMAT, fmt);
	tdfx_outl(DSTFORMAT, fmt);
	tdfx_outl(COMMAND_2D, blitcmd);
	tdfx_outl(DSTSIZE, width | (height << 16));
	tdfx_outl(DSTXY, dstx | (dsty << 16));
	tdfx_outl(LAUNCH_2D, curx | (cury << 16));

	do_make_room(2);
	tdfx_outl(DSTBASE, 0);
	tdfx_outl(SRCBASE, 0);
	do_wait_idle();
}

void do_putc(u32 fgx, u32 bgx, struct display *p, int c, int yy, int xx)
{
	int i;
	struct fb_info_tdfx *info = (struct fb_info_tdfx *) p->fb_info;
	void *mmiobase = info->regbase_virt;

	int stride = info->fb_info.fix.line_length;
	u32 bpp = info->current_par.bpp;
	int fw = (fontwidth(p) + 7) >> 3;
	u8 *chardata = p->fontdata + (c & p->charmask) * fontheight(p) * fw;
	u32 fmt = stride | ((bpp + ((bpp == 8) ? 0 : 8)) << 13);

	xx *= fontwidth(p);
	yy *= fontheight(p);

	do_make_room(9 + ((fontheight(p) * fw + 3) >> 2));
	tdfx_outl(COLORFORE, fgx);
	tdfx_outl(COLORBACK, bgx);
	tdfx_outl(SRCXY, 0);
	tdfx_outl(DSTBASE, yy * stride);
	tdfx_outl(DSTXY, xx);
	tdfx_outl(COMMAND_2D, COMMAND_2D_H2S_BITBLT | (ROP_COPY << 24));
#ifdef __BIG_ENDIAN
        tdfx_outl(SRCFORMAT, SRCFORMAT_BYTEPACKED | SRCFORMAT_BYTESWIZZLE);
#else
        tdfx_outl(SRCFORMAT, SRCFORMAT_BYTEPACKED);   
#endif
	tdfx_outl(DSTFORMAT, fmt);
	tdfx_outl(DSTSIZE, fontwidth(p) | (fontheight(p) << 16));
	i = fontheight(p);
	switch (fw) {
	case 1:
		while (i >= 4) {
			tdfx_outl(LAUNCH_2D, *(u32 *) chardata);
			chardata += 4;
			i -= 4;
		}
		switch (i) {
		case 0:
			break;
		case 1:
			tdfx_outl(LAUNCH_2D, *chardata);
			break;
		case 2:
			tdfx_outl(LAUNCH_2D, *(u16 *) chardata);
			break;
		case 3:
			tdfx_outl(LAUNCH_2D,
				  *(u16 *) chardata | ((chardata[3]) << 24));
			break;
		}
		break;
	case 2:
		while (i >= 2) {
			tdfx_outl(LAUNCH_2D, *(u32 *) chardata);
			chardata += 4;
			i -= 2;
		}
		if (i)
			tdfx_outl(LAUNCH_2D, *(u16 *) chardata);
		break;
	default:
		// Is there a font with width more that 16 pixels ?
		for (i = fontheight(p); i > 0; i--) {
			tdfx_outl(LAUNCH_2D, *(u32 *) chardata);
			chardata += 4;
		}
		break;
	}
	do_make_room(1);
	tdfx_outl(DSTBASE, 0);
	do_wait_idle();
}

void do_putcs(u32 fgx, u32 bgx,
	      struct display *p,
	      const unsigned short *s, int count, int yy, int xx)
{
	int i;
	struct fb_info_tdfx *info = (struct fb_info_tdfx *) p->fb_info;
	void *mmiobase = info->regbase_virt;

	int stride = info->fb_info.fix.line_length;
	u32 bpp = info->current_par.bpp;
	int fw = (fontwidth(p) + 7) >> 3;
	int w = fontwidth(p);
	int h = fontheight(p);
	int regsneed = 1 + ((h * fw + 3) >> 2);
	u32 fmt = stride | ((bpp + ((bpp == 8) ? 0 : 8)) << 13);

	xx *= w;
	yy *= h;
	do_make_room(9);

	tdfx_outl(COMMAND_3D, COMMAND_3D_NOP);
	tdfx_outl(COLORFORE, fgx);
	tdfx_outl(COLORBACK, bgx);
#ifdef __BIG_ENDIAN
        tdfx_outl(SRCFORMAT, SRCFORMAT_BYTEPACKED | SRCFORMAT_BYTESWIZZLE);
#else
        tdfx_outl(SRCFORMAT, SRCFORMAT_BYTEPACKED);   
#endif
	tdfx_outl(DSTFORMAT, fmt);
	tdfx_outl(DSTSIZE, w | (h << 16));
	tdfx_outl(SRCXY, 0);
	tdfx_outl(DSTBASE, yy * stride);
	tdfx_outl(COMMAND_2D, COMMAND_2D_H2S_BITBLT | (ROP_COPY << 24));

	yy <<= 16;
	while (count--) {
		u8 *chardata =
			p->fontdata + (scr_readw(s++) & p->charmask) * h * fw;

		do_make_room(regsneed);
		tdfx_outl(DSTXY, xx);
		xx += w;

		i = h;
		switch (fw) {
		case 1:
			while (i >= 4) {
				tdfx_outl(LAUNCH_2D, *(u32 *) chardata);
				chardata += 4;
				i -= 4;
			}
			switch (i) {
			case 0:
				break;
			case 1:
				tdfx_outl(LAUNCH_2D, *chardata);
				break;
			case 2:
				tdfx_outl(LAUNCH_2D, *(u16 *) chardata);
				break;
			case 3:
				tdfx_outl(LAUNCH_2D,
					  *(u16 *) chardata |
					  ((chardata[3]) << 24));
				break;
			}
			break;
		case 2:
			while (i >= 2) {
				tdfx_outl(LAUNCH_2D, *(u32 *) chardata);
				chardata += 4;
				i -= 2;
			}
			if (i)
				tdfx_outl(LAUNCH_2D, *(u16 *) chardata);
			break;
		default:
			// Is there a font with width more that 16 pixels ?
			for (; i > 0; i--) {
				tdfx_outl(LAUNCH_2D, *(u32 *) chardata);
				chardata += 4;
			}
			break;
		}
	}
	do_make_room(1);
	tdfx_outl(DSTBASE, 0);

	do_wait_idle();
}

/********************************************************************
     Recalculate, and set all the important registers 
 		   from the tdfxfb_par struct.
 ********************************************************************/

void do_set_par(struct tdfxfb_par *par, struct fb_info *info)
{
	struct fb_info_tdfx *i = (struct fb_info_tdfx *) info;
	void *mmiobase = i->regbase_virt;

	struct banshee_reg reg;
	u32 cpp;
	u32 hd, hs, he, ht, hbs, hbe;
	u32 vd, vs, ve, vt, vbs, vbe;
	u32 wd;
	int fout;
	int dbl = 0;
	int freq;

//	DPRINTK("set_par called on fb: %s\n",info->card->name);
	memset(&reg, 0, sizeof(reg));
	cpp = (par->bpp + 7) / 8;

	/* PLL settings */
	freq = par->pixclock;
	reg.vidcfg =
		VIDCFG_VIDPROC_ENABLE |
		VIDCFG_DESK_ENABLE |
		VIDCFG_CURS_X11 | 
     		((cpp - 1) << VIDCFG_PIXFMT_SHIFT) |
		(cpp != 1 ? VIDCFG_CLUT_BYPASS : 0);

	reg.dacmode=0;
	reg.vidcfg &= ~VIDCFG_2X;
	if (freq > i->max_pixclock / 2) {
		freq = freq > i->max_pixclock ? i->max_pixclock : freq;
		reg.dacmode |= DACMODE_2X;
		reg.vidcfg |= VIDCFG_2X;
		dbl = 1;
		par->hdispend >>= 1;
		par->hsyncsta >>= 1;
		par->hsyncend >>= 1;
		par->htotal >>= 1;
	}

	wd = (par->hdispend >> 3) - 1;

	hd = (par->hdispend >> 3) - 1;
	hs = (par->hsyncsta >> 3) - 1;
	he = (par->hsyncend >> 3) - 1;
	ht = (par->htotal >> 3) - 1;
	hbs = hd;
	hbe = ht;

	vd = par->vdispend - 1;
	vs = par->vsyncsta - 1;
	ve = par->vsyncend - 1;
	vt = par->vtotal - 2;
	vbs = vd;
	vbe = vt;

	/* this is all pretty standard VGA register stuffing */
	reg.misc[0x00] =
		0x0f |
		(par->hdispend < 400 ? 0xa0 :
		 par->hdispend < 480 ? 0x60 :
		 par->hdispend < 768 ? 0xe0 : 0x20);

// gra, ok.
	reg.gra[0x00] = 0x00;
	reg.gra[0x01] = 0x00;
	reg.gra[0x02] = 0x00;
	reg.gra[0x03] = 0x00;
	reg.gra[0x04] = 0x00;
	reg.gra[0x05] = 0x40;
	reg.gra[0x06] = 0x05;
	reg.gra[0x07] = 0x0f;
	reg.gra[0x08] = 0xff;

// att, ok. 
	reg.att[0x00] = 0x00;
	reg.att[0x01] = 0x01;
	reg.att[0x02] = 0x02;
	reg.att[0x03] = 0x03;
	reg.att[0x04] = 0x04;
	reg.att[0x05] = 0x05;
	reg.att[0x06] = 0x06;
	reg.att[0x07] = 0x07;
	reg.att[0x08] = 0x08;
	reg.att[0x09] = 0x09;
	reg.att[0x0a] = 0x0a;
	reg.att[0x0b] = 0x0b;
	reg.att[0x0c] = 0x0c;
	reg.att[0x0d] = 0x0d;
	reg.att[0x0e] = 0x0e;
	reg.att[0x0f] = 0x0f;
	reg.att[0x10] = 0x41;
	reg.att[0x11] = 0x00;
	reg.att[0x12] = 0x0f;
	reg.att[0x13] = 0x00;
	reg.att[0x14] = 0x00;

	reg.seq[0x00] = 0x03;	
	reg.seq[0x01] = 0x01;	/* fixme: clkdiv2? */
	reg.seq[0x02] = 0x0f;
	reg.seq[0x03] = 0x00;
	reg.seq[0x04] = 0x0e;

	reg.crt[0x00] = ht - 4;
	reg.crt[0x01] = hd;
	reg.crt[0x02] = hbs;
	reg.crt[0x03] = 0x80 | (hbe & 0x1f);
	reg.crt[0x04] = hs;
	reg.crt[0x05] = ((hbe & 0x20) << 2) | (he & 0x1f);
	reg.crt[0x06] = vt;
	reg.crt[0x07] =
		((vs & 0x200) >> 2) |
		((vd & 0x200) >> 3) |
		((vt & 0x200) >> 4) |
		0x10 |
		((vbs & 0x100) >> 5) |
		((vs & 0x100) >> 6) |
		((vd & 0x100) >> 7) | ((vt & 0x100) >> 8);
	reg.crt[0x08] = 0x00;
	reg.crt[0x09] = 0x40 | ((vbs & 0x200) >> 4);
	reg.crt[0x0a] = 0x00;
	reg.crt[0x0b] = 0x00;
	reg.crt[0x0c] = 0x00;
	reg.crt[0x0d] = 0x00;
	reg.crt[0x0e] = 0x00;
	reg.crt[0x0f] = 0x00;
	reg.crt[0x10] = vs;
	reg.crt[0x11] = (ve & 0x0f) | 0x20;
	reg.crt[0x12] = vd;
	reg.crt[0x13] = wd;
	reg.crt[0x14] = 0x00;
	reg.crt[0x15] = vbs;
	reg.crt[0x16] = vbe + 1;
	reg.crt[0x17] = 0xc3;
	reg.crt[0x18] = 0xff;

	/* 3dfx's nonvga stuff (CRT extension)*/
	reg.ext[0x00] = (((ht & 0x100) >> 8) |
			 ((hd & 0x100) >> 6) |
			 ((hbs & 0x100) >> 4) |
			 ((hbe & 0x40) >> 1) |
			 ((hs & 0x100) >> 2) | ((he & 0x20) << 2));
	reg.ext[0x01] = (((vt & 0x400) >> 10) |
			 ((vd & 0x400) >> 8) |
			 ((vbs & 0x400) >> 6) | ((vbe & 0x400) >> 4));

	reg.vgainit0 =
		VGAINIT0_8BIT_DAC |
		VGAINIT0_EXT_ENABLE |
		VGAINIT0_WAKEUP_3C3 |
#ifdef __mips__
		/* enable access to LEGACY VGA registers (X server use this) */
#else
		VGAINIT0_LEGACY_DISABLE |
#endif
		VGAINIT0_ALT_READBACK | VGAINIT0_EXTSHIFTOUT;
	reg.vgainit1 = 0;

	reg.stride = par->width * cpp;
	reg.cursloc = 0;

	reg.cursc0 = 0;
	reg.cursc1 = 0xffffff;

	reg.curspataddr = i->cursor.cursorimage;

	reg.startaddr = par->baseline * reg.stride;
	reg.srcbase = reg.startaddr;
	reg.dstbase = reg.startaddr;

	reg.vidpll = do_calc_pll(freq, &fout);
#if 0
	reg.mempll = do_calc_pll(..., &fout);
	reg.gfxpll = do_calc_pll(..., &fout);
#endif

	reg.screensize = par->width | (par->height << 12);
	reg.vidcfg &= ~VIDCFG_HALF_MODE;

	reg.miscinit0 = tdfx_inl(MISCINIT0);
#ifdef __BIG_ENDIAN
	switch (par->bpp) {
	case 8:   
	    reg.miscinit0 &= ~BIT(30);
	    reg.miscinit0 &= ~BIT(31);
	    break;
	case 16:   
	    reg.miscinit0 |= BIT(30);
	    reg.miscinit0 |= BIT(31);
	    break;
	case 24:
	    reg.miscinit0 &= ~BIT(30);
	    reg.miscinit0 &= ~BIT(31);
	    break;
	case 32:
	    reg.miscinit0 |= BIT(30);
	    reg.miscinit0 &= ~BIT(31);
	    break;
	}
#endif

	i->cursor.enable = reg.vidcfg | VIDCFG_HWCURSOR_ENABLE;
	i->cursor.disable = reg.vidcfg;

	//DPRINTK("wrregs: vidcfg:%08lx,dacmode:%08lx,vidpll:%08lx,freq:%08x\n",
	//      reg.vidcfg, reg.dacmode, reg.vidpll, freq);

	do_write_regs(i, &reg);
	if (dbl) {
		par->hdispend <<= 1;
		par->hsyncsta <<= 1;
		par->hsyncend <<= 1;
		par->htotal <<= 1;
	}

	i->current_par = *par;

}

void do_write_regs(struct fb_info_tdfx *info, struct banshee_reg *reg)
{
	int i;
	unsigned long iobase = info->iobase;
	void *mmiobase = info->regbase_virt;
    	
	vga_disable_palette();
	tdfx_outl(VGAINIT0, reg->vgainit0);
	tdfx_outl(VGAINIT1, reg->vgainit1 & 0xfff00000);

#if 0
	tdfx_outl(PLLCTRL1, reg->mempll);
	tdfx_outl(PLLCTRL2, reg->gfxpll);
#endif
	tdfx_outl(PLLCTRL0, reg->vidpll);
	vga_outb(iobase, VGA_MIS_W, reg->misc[0x00] | 0x01);
	seq_outb(iobase, 0, 1);		        // reset seq	
	vga_outb(iobase, VGA_MIS_W, reg->misc[0x00] | 0x01);
	for (i = 1; i < 5; i++)
		seq_outb(iobase, i, reg->seq[i]);
	seq_outb(iobase, 0, reg->seq[0]);	// enable seq
	crt_outb(iobase, 0x11, 0x20);		// disable vert. retr. 
						// interrupt, and unprotect CRT 
	for (i = 0; i < 25; i++)
		crt_outb(iobase, i, reg->crt[i]);

	for (i = 0; i < 9; i++)
		gra_outb(iobase, i, reg->gra[i]);

	vga_outb(iobase,0x3da,0);
	seq_outb(iobase, 1, seq_inb(iobase,1) | 0x20);		// screen_off
	for (i = 0; i < 21; i++)
		att_outb(iobase, i, reg->att[i]);
	vga_outb(iobase,VGA_PEL_MSK,0xff);

#if 0
   	vga_outb(iobase,VGA_PEL_IW,0);
	for (i=0;i<768;i++) 
	    vga_outb(iobase,VGA_PEL_D,i /3);
#endif
	crt_outb(iobase, 0x1a, reg->ext[0]);
	crt_outb(iobase, 0x1b, reg->ext[1]);
	tdfx_outl(DACMODE, reg->dacmode);
	tdfx_outl(VIDDESKSTART, reg->startaddr);
	tdfx_outl(VIDSCREENSIZE, reg->screensize);
	tdfx_outl(VIDDESKSTRIDE, reg->stride);

	if (nohwcursor) {
		tdfx_outl(HWCURPATADDR, 0);
	} else {
		tdfx_outl(HWCURPATADDR, reg->curspataddr);
		tdfx_outl(HWCURC0, reg->cursc0);
		tdfx_outl(HWCURC1, reg->cursc1);
		tdfx_outl(HWCURLOC, reg->cursloc);
	}
	tdfx_outl(VIDPROCCFG, reg->vidcfg & ~VIDCFG_VIDPROC_ENABLE);
	tdfx_outl(DRAMINIT1, tdfx_inl(DRAMINIT1) | 0x01);	
			// refresh enable 
	tdfx_outl(MISCINIT1, tdfx_inl(MISCINIT1) | 0x01);	// invert_clut_address
	tdfx_outl(MISCINIT0, reg->miscinit0);
	vga_outb(iobase,0x3c3,1);

	do_make_room(8);
	tdfx_outl(SRCBASE, reg->srcbase);
	tdfx_outl(DSTBASE, reg->dstbase);
	tdfx_outl(COMMANDEXTRA_2D, 0);
	tdfx_outl(CLIP0MIN, 0);
	tdfx_outl(CLIP0MAX, 0x0fff0fff);
	tdfx_outl(CLIP1MIN, 0);
	tdfx_outl(CLIP1MAX, 0x0fff0fff);
	tdfx_outl(SRCXY, 0);

	seq_outb(iobase, 1, seq_inb(iobase,1) & ~0x20);		// screen_on
	vga_enable_palette();
	vga_outb(iobase,0x3da,0);
    
}

/********************************************************************
  		HWCursor handling (lower level)
 ********************************************************************/

void do_hwcursor_disable(struct fb_info_tdfx *info)
{
	void *mmiobase = info->regbase_virt;
	tdfx_outl(VIDPROCCFG, info->cursor.disable);
}

void do_hwcursor_enable(struct fb_info_tdfx *info)
{
	void *mmiobase = info->regbase_virt;
	tdfx_outl(VIDPROCCFG, info->cursor.enable);
}

void do_hwcursor_reset(struct fb_info_tdfx *i)
{

	void *mmiobase = i->regbase_virt;

	tdfx_outl(HWCURPATADDR, i->cursor.cursorimage);
	tdfx_outl(HWCURC0, 0);
	tdfx_outl(HWCURC1, 0xffffff);
	tdfx_outl(HWCURLOC, 0);	// this reg. will be refreshed later 
}

void do_hwcursor_pos(struct fb_info_tdfx *info, int x, int y)
{
	void *mmiobase = info->regbase_virt;
	tdfx_outl(HWCURLOC, (y << 16) + x);
}

/********************************************************************
 	      Misc: PLL, getting framebuffer size, blank
 ********************************************************************/

void do_blank(struct fb_info_tdfx *info, int state, int vgablank)
{
	u32 dacmode;
	void *mmiobase = info->regbase_virt;
	unsigned long iobase = info->iobase;

	dacmode = tdfx_inl(DACMODE);
	dacmode &= ~(BIT(1) | BIT(3));
	dacmode |= state;
	tdfx_outl(DACMODE, dacmode);
	if (vgablank)
		vga_disable_video();
	else
		vga_enable_video();
}

unsigned long do_lfb_size(struct fb_info_tdfx *info)
{
	int memsize = 0;
	int memtype = MEM_TYPE_SGRAM;
	void *mmiobase = info->regbase_virt;
	int draminit0_strap;
	int draminit1_strap;
	int draminit1;
	int miscinit1;
	int nchips;
	int partsize;
	int banks = 1;

	tdfx_outl(MISCINIT0,0x73);
	udelay(200);
	tdfx_outl(MISCINIT0,0);

	draminit1_strap = tdfx_inl(DRAMINIT1);
	draminit1_strap &= DRAMINIT1_MEM_SDRAM;
	if (draminit1_strap)
		memtype = MEM_TYPE_SDRAM;
	draminit1 = 2 << SST_SGRAM_OFLOP_DEL_ADJ_SHIFT;
	draminit1 |= SST_SGRAM_CLK_NODELAY;
	draminit1 |= SST_DRAM_REFRESH_EN;
	draminit1 |=
		(0x18 << SST_DRAM_REFRESH_VALUE_SHIFT) &
		SST_DRAM_REFRESH_VALUE;
	draminit1 &= ~SST_MCTL_TYPE_SDRAM;
	draminit1 |= draminit1_strap;
	tdfx_outl(DRAMINIT1, draminit1);

	draminit0_strap = tdfx_inl(DRAMINIT0);
	if (info->card->device <= PCI_DEVICE_ID_3DFX_VOODOO3) {
		if (memtype == MEM_TYPE_SDRAM) {
			memsize = 16;
		} else {
			nchips =
				((draminit0_strap & SST_SGRAM_NUM_CHIPSETS) ==
				 0) ? 4 : 8;
			switch (draminit0_strap & SST_SGRAM_TYPE) {
			case (SST_SGRAM_TYPE_8MBIT):
				partsize = 8;
				break;
			case (SST_SGRAM_TYPE_16MBIT):
				partsize = 16;
				break;
			default:
				printk("fb error: Invalid sgram type %lu\n",
				       draminit0_strap & SST_SGRAM_TYPE);
				return 0;
			}
			memsize = (nchips * partsize) >> 3;
		}
	} else {
		nchips =
			((draminit0_strap & SST_SGRAM_NUM_CHIPSETS) ==
			 0) ? 4 : 8;
		partsize = 1 << ((draminit0_strap & 0x38000000) >> 28);
		banks = ((draminit0_strap & BIT(30)) == 0) ? 2 : 4;
		memsize = nchips * partsize * banks;
	}

	miscinit1 = tdfx_inl(MISCINIT1);
	if (memtype == MEM_TYPE_SDRAM) {
		miscinit1 |= SST_DISABLE_2D_BLOCK_WRITE;
	}
	miscinit1 |= MISCINIT1_CLUT_INV;
	tdfx_outl(MISCINIT1, miscinit1);

	return memsize * 1024;	// # of KBytes
}

u32 do_calc_pll(int freq, int *freq_out)
{
	int m, n, k, best_m, best_n, best_k, f_cur, best_error;
	int minm, maxm;

	/* this really could be done with more intelligence --
	   255*63*4 = 64260 iterations is silly */
	freq *= 100;

	best_error = freq;
	best_n = best_m = best_k = 0;
	minm = 1;
	maxm = 64;

	for (n = 1; n < 256; n++) {
		f_cur = REFFREQ * (n + 2);
		if (f_cur < freq) {
			f_cur = f_cur / 3;
			if (freq - f_cur < best_error) {
				best_error = freq - f_cur;
				best_n = n;
				best_m = 1;
				best_k = 0;
				continue;
			}
		}
		for (m = minm; m < maxm; m++) {
			for (k = 0; k < 4; k++) {
				f_cur =
					REFFREQ * (n + 2) / (m +
							     2) / (1 << k);
				if (abs(f_cur - freq) < best_error) {
					best_error = abs(f_cur - freq);
					best_n = n;
					best_m = m;
					best_k = k;
				}
			}
		}
	}
	n = best_n;
	m = best_m;
	k = best_k;
	*freq_out = REFFREQ * (n + 2) / (m + 2) / (1 << k) / 100;

	return (n << 8) | (m << 2) | k;
}

/*
 * Set the color of a palette entry in 8bpp mode 
 */
inline void do_setpalentry(void *mmiobase, unsigned regno, u32 c)
{
	tdfx_outl(DACADDR, regno);
	tdfx_outl(DACDATA, c);
}

int do_card_preinit(struct fb_info_tdfx *info)
{
   	int i;
	unsigned long iobase = info->iobase;
   	void *mmiobase = info->regbase_virt;
   	u32 phys_addr,addr,len;
	char *virtaddr;
   	u16 romconftable,oemconftable;
	u32 pciinit0,miscinit0,miscinit1,draminit0,draminit1,agpinit0;
   	u32 pllctrl1,pllctrl2,sgrammode;
	u32 strapinfo;
   
   	if (info->devflags.noinit) 
     		return 0;	     // It was initialized from bios.
   
// Getting ROM address.
	pci_read_config_dword( info->pdev, PCI_ROM_ADDRESS, &addr);
// Enabling ROM address decoding
	pci_write_config_dword( info->pdev, PCI_ROM_ADDRESS, addr & ~1);
	pci_write_config_dword( info->pdev, PCI_ROM_ADDRESS, addr | 1);
	phys_addr=addr&~0x7ff;
   
// Getting ROM size   
   	len=(tdfx_inl(MISCINIT1) & MISCINIT1_ROM_SIZE) ? 64*1024 : 32*1024;

// Mapping ROM area   
   	request_mem_region(phys_addr, len," tdfxfb rom ");
   	virtaddr=ioremap_nocache(phys_addr,len);

// Get reg. init values
   	romconftable=do_safe_readw(virtaddr+0x50);
   	if (romconftable>len-4) goto err;
   	oemconftable=do_safe_readw(virtaddr+romconftable);
   	if (oemconftable>len-9*4) goto err;
   	pciinit0=do_safe_readl(virtaddr+oemconftable);
        miscinit0=do_safe_readl(virtaddr+oemconftable+4);
   	miscinit1=do_safe_readl(virtaddr+oemconftable+8);
   	draminit0=do_safe_readl(virtaddr+oemconftable+12);
   	draminit1=do_safe_readl(virtaddr+oemconftable+16);
   	agpinit0=do_safe_readl(virtaddr+oemconftable+20);
   	pllctrl1=do_safe_readl(virtaddr+oemconftable+24);
   	pllctrl2=do_safe_readl(virtaddr+oemconftable+28);
   	sgrammode=do_safe_readl(virtaddr+oemconftable+32);
	   
   	strapinfo=tdfx_inl(STRAPINFO);
#if 0
	DPRINTK("romconftable %x, oemconftable %x\n", romconftable,oemconftable);
	DPRINTK("pciinit0 %x\n", pciinit0);
	DPRINTK("miscinit0,1 %x, %x\n", miscinit0, miscinit1);
	DPRINTK("draminit0,1 %x, %x\n", draminit0, draminit1);
	DPRINTK("agpinit0 %x\n", agpinit0);
	DPRINTK("pllctrl1,2 %x, %x\n", pllctrl1, pllctrl2);
	DPRINTK("sgrammode %x\n", sgrammode);
	DPRINTK("strapinfo %x\n", strapinfo);
#endif
   
   // pciinit0, agp bus
	if (strapinfo & 8) pciinit0 |= 0x300; 

   // VMI_DATA 0-4	
   	miscinit1=((strapinfo & 0x1f) << 24)  |
     		  miscinit1;
   // VMI_DATA 5-6	
   	draminit0=((strapinfo & 0x60) << 26) |
     		  draminit0;

   // VMI_ADDR 1-2	
   	draminit1=((strapinfo & 0x200) << 16) |
     		  ((strapinfo & 0x400) << 20) |
     		  draminit1;

   	tdfx_outl(PCIINIT0,pciinit0);
#ifdef __mips__
	/* enable access to LEGACY VGA registers (X server use this) */
	tdfx_outl(VGAINIT0, VGAINIT0_EXT_ENABLE | 
		  	    VGAINIT0_WAKEUP_3C3 |
		  	    VGAINIT0_8BIT_DAC );
#else
	tdfx_outl(VGAINIT0, VGAINIT0_EXT_ENABLE | 
		  	    VGAINIT0_WAKEUP_3C3 |
		  	    VGAINIT0_LEGACY_DISABLE |
		  	    VGAINIT0_8BIT_DAC );
#endif

        if (info->card->cardid>=4) tdfx_outl(VIDINFORMAT,0);
   
//	vga_outb(iobase,0x3c3,1);
	vga_outb(iobase,0x3c2,0x67);
	vga_outb(iobase,0x3ce,0x6);		
	vga_outb(iobase,0x3cf,1);		// graphics mode

	crt_outb(iobase,0x1c,0);
	if (info->card->cardid>=4) {
	    crt_outb(iobase,0x1d,0);
	    crt_outb(iobase,0x1e,0);
	}	
		
	tdfx_outl(VIDSERPARPORT, (tdfx_inl(VIDSERPARPORT) & 0xf07fffff) | 0x80000000);
	tdfx_outl(VIDINFORMAT, 0);
	
   	tdfx_outl(DRAMINIT1,draminit1);
	tdfx_outl(TMUGBEINIT,0xf0ff0);
	tdfx_outl(DRAMINIT0,draminit0);
   	tdfx_outl(MISCINIT0,miscinit0);
   	tdfx_outl(DRAMDATA,sgrammode);
   	tdfx_outl(DRAMCOMMAND,0x10d);		// write sgram mode
   	tdfx_outl(MISCINIT1,miscinit1);
   	tdfx_outl(AGPINIT,agpinit0);
   	tdfx_outl(PLLCTRL1,pllctrl1);
//   	tdfx_outl(PLLCTRL2,pllctrl2);
	tdfx_outl(LFBMEMORYCONFIG,0xa2200 | 0x1fff);

   	tdfx_outl(SIPMONITOR,0x40000000);
//	tdfx_outl(VGAINIT1,0);

// initial vga core settings
	vga_outb(iobase,0x3c8,0);
	for (i=0;i<256;i++) {
		vga_outb(iobase,0x3c9,8);
		vga_outb(iobase,0x3c9,8);
		vga_outb(iobase,0x3c9,8);
	}

	pci_write_config_dword( info->pdev, PCI_ROM_ADDRESS, addr & ~1);
   	iounmap(virtaddr);		// we don't need it anymoure...
  	release_mem_region(phys_addr, len);
	
	return 0;
           
   err:
	pci_write_config_dword( info->pdev, PCI_ROM_ADDRESS, addr & ~1);
   	iounmap(virtaddr);		// we don't need it anymoure...
  	release_mem_region(addr, len);
   	return 1;
}
#ifdef TDFXFB_DEBUG
int do_get_monitor_sense(struct fb_info_tdfx *info)
{
	unsigned long iobase = info->iobase;

	return (vga_inb(iobase, VGA_IS0_R) & 0x10) >> 4;
}

int do_inb(struct fb_info_tdfx *info,int reg)
{
	unsigned long iobase = info->iobase;
 	return vga_inb(iobase,reg);  
}
void do_outb(struct fb_info_tdfx *info,int reg,int val)
{
	unsigned long iobase = info->iobase;
        vga_outb(iobase,reg,val);     
}
#endif
