
/*-------------------------------------------------------------*/
/*-- bzip-like compression of 32k blocks        lib_bzip_e.c --*/
/*-------------------------------------------------------------*/

/*--
  This file is part of lib_bzip, a block-sorting compression 
  library designed to compress 32kbyte blocks, for on-the-fly
  disk compression/decompression.

  Version 0.02, 4-Jan-1998

  Copyright (C) 1996, 1997, 1998 by Julian Seward.
     Guildford, Surrey, UK
     email: jseward@acm.org

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--*/


#include "lib_bzip_private.h"
#include "lib_bzip.h"


/*---------------------------------------------*/
/*--
   Compressed block format:

      UChar 0xa2      -- version 0.02 of this lib
      
      (a single bit indicating whether or not this
       block was randomised)

      UInt16 origPtr  -- rotation posn of original

      (table indicating which chars are in use)

      (Huffman code length tables, one for each
       of N_GROUPS)

      (selector table)

      (codes for the MTF values proper)

--*/

#ifdef __KERNEL__
# include <linux/kernel.h>
# include <linux/linkage.h>
# include <linux/sched.h>
# include <asm/param.h>
# define __NO_VERSION__
# include <linux/module.h>

static unsigned long next_brk = 0;
extern unsigned long volatile jiffies;

/* Time slice of a fifth of a second. */
# define MAYBE_SCHEDULE                 \
   if (jiffies > next_brk) {            \
      schedule();                       \
      next_brk = jiffies + HZ / 5;      \
   }

# define e_verbosity 0  /* EXT2_BZIP2_VERBOSITY */
# define do_status_printf(_v, _a...) printk(KERN_DEBUG "bzip2: " ##_a)
# define status_printf(_v, _a...) do { if (e_verbosity >= (_v)) printk(KERN_DEBUG "bzip2: " ##_a); } while(0)
#else

# define MAYBE_SCHEDULE /*--*/
# define e_verbosity est->verb
# define do_status_printf(_v, _a...) fprintf(stderr, "" ##_a)
# define status_printf(_v, _a...) do { if (e_verbosity >= (_v)) fprintf(stderr, "" ##_a); } while(0)
#endif

/*---------------------------------------------*/
/*-- 
   Bit stream writing stuff (boring but necessary).
--*/

static void 
bsInit_e ( Lib_Bzip_Encode_Storage_Ty *est )
{
   est->outctr = 0;
   est->bsLive = 0;
   est->bsBuff = 0;
}


#define bs_MAKE_SPACE                         \
do {                                          \
   while (est->bsLive >= 8) {                 \
      est->outbuff[est->outctr++]             \
         = (UChar)(est->bsBuff >> 24);        \
      est->bsBuff <<= 8;                      \
      est->bsLive -= 8;                       \
   }                                          \
} while(0)


static inline
void 
bsW ( Lib_Bzip_Encode_Storage_Ty *est, Int32 n, UInt32 v )
{
   bs_MAKE_SPACE;
   est->bsBuff |= (v << (32 - est->bsLive - n));
   est->bsLive += n;
}


static void 
bsFlush_e ( Lib_Bzip_Encode_Storage_Ty *est )
{
   while (est->bsLive > 0) {
      est->outbuff[est->outctr++]
         = (UChar)(est->bsBuff >> 24);
      est->bsBuff <<= 8;
      est->bsLive -= 8;
   }
}


/*---------------------------------------------*/
static inline
Bool 
fullGt ( Lib_Bzip_Encode_Storage_Ty *est, Int32 i1, Int32 i2 )
{
   UInt16 s1, s2;
   Int32 k;

   if (i1 == i2) return False;

   s1 = est->block[i1]; s2 = est->block[i2];
   if (s1 != s2) return (s1 > s2);
   i1 += 2; i2 += 2;

   s1 = est->block[i1]; s2 = est->block[i2];
   if (s1 != s2) return (s1 > s2);
   i1 += 2; i2 += 2;

   s1 = est->block[i1]; s2 = est->block[i2];
   if (s1 != s2) return (s1 > s2);
   i1 += 2; i2 += 2;

   k = est->nblock + 16;

   do {

      s1 = est->block[i1]; s2 = est->block[i2];
      if (s1 != s2) return (s1 > s2);
      i1 += 2; i2 += 2;

      s1 = est->block[i1]; s2 = est->block[i2];
      if (s1 != s2) return (s1 > s2);
      i1 += 2; i2 += 2;

      s1 = est->block[i1]; s2 = est->block[i2];
      if (s1 != s2) return (s1 > s2);
      i1 += 2; i2 += 2;

      if (i1 >= (Int32) est->nblock)
         i1 -= est->nblock;
      if (i2 >= (Int32) est->nblock)
         i2 -= est->nblock;

      k -= 6;
      est->workDone ++;
   }
      while (k >= 0);

   return False;
}


/*---------------------------------------------*/
/*  This qsort is derived from Weiss' book 
    "Data Structures and Algorithm Analysis in C",
    Section 7.7.
*/

#define ISORT_BELOW 10

#define SWAP(za,zb)                                               \
do { Int32 zl = (za); Int32 zr = (zb);                            \
     UInt16 zt = est->zptr[zl]; est->zptr[zl] = est->zptr[zr];    \
     est->zptr[zr] = zt;                                          \
   } while(0)

static void 
sortBucket ( Lib_Bzip_Encode_Storage_Ty *est, Int32 left, Int32 right )
{
   UInt16 v;
   Int32 pivot;
   Int32 i, j;
   Int32 wuC;

   UInt16 stackL[20];
   UInt16 stackR[20];

   Int32 sp = 0;
   Int32 wuL = left;
   Int32 wuR = right;

   if (left >= right) return;

   while (True) {

      /*-- 
         At the beginning of this loop, wuL and wuR hold the
         bounds of the next work-unit.  First, though, check
         that we are not out of CPU time.
      --*/
      MAYBE_SCHEDULE;

      if (wuR - wuL > ISORT_BELOW) {

         /*--
            a large Work Unit; partition-exchange 
         --*/
         wuC = (wuL + wuR) >> 1;
         if (fullGt ( est, 1+(Int32)est->zptr[wuL], 1+(Int32)est->zptr[wuC] ))
            SWAP ( wuL, wuC );
         if (fullGt ( est, 1+(Int32)est->zptr[wuL], 1+(Int32)est->zptr[wuR] ))
            SWAP ( wuL, wuR );
         if (fullGt ( est, 1+(Int32)est->zptr[wuC], 1+(Int32)est->zptr[wuR] ))
            SWAP ( wuC, wuR );

         SWAP ( wuC, wuR-1 );
         pivot = 1 + (Int32)est->zptr[wuR-1];

         i = wuL;
         j = wuR - 1;
         for (;;) {
            do i++; while (fullGt ( est, pivot, 1+(Int32)est->zptr[i] ));
            do j--; while (fullGt ( est, 1+(Int32)est->zptr[j], pivot ));
            if (i < j) SWAP ( i, j ); else break;
            if (!est->randomised && est->workDone > est->workLimit) return;
         }
         SWAP ( i, wuR-1 );

         if ((i - wuL) > (wuR - i)) {
            stackL[sp] = wuL; stackR[sp] = i-1; sp++; wuL = i+1;
         } else {
            stackL[sp] = i+1; stackR[sp] = wuR; sp++; wuR = i-1;
         }

      } else {

         /*--
            a small Work-Unit; insertion-sort it
         --*/                
         for (i = wuL + 1; i <= wuR; i++) {
            v = est->zptr[i];
            j = i;
            while ( fullGt ( est, 1+(Int32)est->zptr[j-1], 1+(Int32)v )) {
               est->zptr[j] = est->zptr[j-1];
               j = j - 1;
               if (j <= wuL) break;
            }
            est->zptr[j] = v;
            if (!est->randomised && est->workDone > est->workLimit) return;
         }
         if (sp == 0) return;
         sp--; wuL = stackL[sp]; wuR = stackR[sp];

      } /* if this is a small work-unit */
   }
}

#undef RC
#undef SWAP
#undef ISORT_BELOW




/*---------------------------------------------*/
#define BIGFREQ(zz) (est->ftab[(zz)+1] - est->ftab[(zz)])

#define FREQ(zz)    (est->ftab[(zz)*3 +3] - est->ftab[(zz)*3 +0])
#define FREQ_LT(zz) (est->ftab[(zz)*3 +1] - est->ftab[(zz)*3 +0])
#define FREQ_EQ(zz) (est->ftab[(zz)*3 +2] - est->ftab[(zz)*3 +1])
#define FREQ_GT(zz) (est->ftab[(zz)*3 +3] - est->ftab[(zz)*3 +2])

#define TIMES3(zz) ((zz)+(zz)+(zz))

static void 
sortBlock ( Lib_Bzip_Encode_Storage_Ty *est )
{
   Int32 j, k, totalOrdered, bucket;
   unsigned i;
   UInt16 s, sLo, sHi;
   Int32 ftabNo;

   MAYBE_SCHEDULE;

   status_printf(3, "      bucket sort ...\n");

   // stripe the block, and set up overshoot area

   if (est->nblock == 1) {
      est->block[0] &= 0xff00;
      est->block[0] = (est->block[0] >> 8) | est->block[0];
   } else {
      for (i = 0; i < est->nblock - 1; i++) {
         est->block[i] &= 0xff00;
         est->block[i] |= (est->block[i+1] >> 8);
      }
      est->block[est->nblock-1] &= 0xff00;
      est->block[est->nblock-1] |= (est->block[0] >> 8);
   }

   for (i = 0; i < N_OVERSHOOT; i++)
      est->block[i + est->nblock] = est->block[i % est->nblock];


   // set up freq table

   for (i = 0; i < 769; i++) est->ftab[i] = 0;

   for (i = 0; i < est->nblock; i++) {
      s = est->block[i];
      sHi = s >> 8;
      sLo = s & 0xFF;
      if (sHi > sLo) ftabNo = TIMES3(sHi) + 0; else
      if (sHi < sLo) ftabNo = TIMES3(sHi) + 2; else
                     ftabNo = TIMES3(sHi) + 1;
      est->ftab[ftabNo]++;
   }

   for (i = 1; i < 769; i++) est->ftab[i] += est->ftab[i-1];

   for (i = 0; i < est->nblock; i++) {
      s = est->block[i];
      sHi = s >> 8;
      sLo = s & 0xFF;
      if (sHi > sLo) ftabNo = TIMES3(sHi) + 0; else
      if (sHi < sLo) ftabNo = TIMES3(sHi) + 2; else
                     ftabNo = TIMES3(sHi) + 1;
      j = est->ftab[ftabNo] - 1;
      est->ftab[ftabNo] = j;
      est->zptr[j] = i;
   }

   for (i = 0; i < 256; i++)
      est->inUse[i] = (FREQ(i) > 0);
   for (i = 0; i < 256; i++)
      est->rorder[i] = i;

   /* */
   {
      Int32 vv;
      Int32 h = 1;
      do h = 3 * h + 1; while (h <= 256);
      do {
         h = h / 3;
         for (i = h; i <= 255; i++) {
            vv = est->rorder[i];
            j = i;
            while ( FREQ(est->rorder[j-h]) > FREQ(vv) ) {
               est->rorder[j] = est->rorder[j-h];
               j = j - h;
               if (j < (Int32) h) break;
            }
            est->rorder[j] = vv;
         }
      } while (h != 1);
   }

   totalOrdered = 0;
   for (i = 0; i < 256; i++) {
      MAYBE_SCHEDULE;

      bucket = est->rorder[i];

      if (FREQ_LT(bucket) > 1) {
	 status_printf ( 3, "      sort 0x%2x <  size %-4d   ", 
			 bucket, FREQ_LT(bucket) );
         sortBucket ( est,
                      est->ftab[TIMES3(bucket) +0], 
                      est->ftab[TIMES3(bucket) +1]-1 );
         totalOrdered += FREQ_LT(bucket);
         status_printf ( 3, "done %d\n", totalOrdered );
         if (!est->randomised && est->workDone > est->workLimit) return;
      }

      if (FREQ_GT(bucket) > 1) {
	 status_printf ( 3, "      sort 0x%2x >  size %-4d   ", 
			 bucket, FREQ_GT(bucket) );
         sortBucket ( est,
                      est->ftab[TIMES3(bucket) +2], 
                      est->ftab[TIMES3(bucket) +3]-1 );
         totalOrdered += FREQ_LT(bucket);
         status_printf ( 3, "done %d\n", totalOrdered );
         if (!est->randomised && est->workDone > est->workLimit) return;
      }

      if (FREQ_EQ(bucket) > 1) {
         Int32 put0, get0, put1, get1;
         UChar c1;
         Int32 sbn = TIMES3(bucket) + 1;
         Int32 lo = est->ftab[sbn];
         Int32 hi = est->ftab[sbn+1] - 1;
         UChar ssc = (UChar)bucket;

	 status_printf ( 3, "      copy 0x%2x =  size %-4d   ", 
			 bucket, FREQ_EQ(bucket) );

         put0 = lo;
         get0 = est->ftab[TIMES3(bucket)];
         put1 = hi;
         get1 = est->ftab[TIMES3(bucket)+3] - 1;
         while (get0 < put0) {
            j = est->zptr[get0]-1; if (j < 0) j += est->nblock;
            c1 = (UChar)(est->block[j] >> 8);
            if (c1 == ssc) { est->zptr[put0] = j; put0++; };
            get0++;
         }
         while (get1 > put1) {
            j = est->zptr[get1]-1; if (j < 0) j += est->nblock;
            c1 = (UChar)(est->block[j] >> 8);
            if (c1 == ssc) { est->zptr[put1] = j; put1--; };
            get1--;
         }
         totalOrdered += FREQ_EQ(bucket);
         status_printf ( 3, "done %d\n", totalOrdered );
      }

      if (i < 255 && FREQ(bucket) < 256) {
         Int32 loLim = est->ftab[TIMES3(bucket)];
         Int32 hiLim = est->ftab[TIMES3(bucket) + 3];
         for (j = 0, k = loLim;   k < hiLim;   j++, k++) {
            UInt16 a2upd = est->zptr[k];
            est->block[a2upd] = (est->block[a2upd] & 0xff00) | ((UInt16)j);
            if (a2upd < N_OVERSHOOT)
               est->block[a2upd + est->nblock] 
                  = (est->block[a2upd + est->nblock] & 0xff00) | ((UInt16)j);
         }
      }

   }
}

#undef FREQ
#undef FREQ_LT
#undef FREQ_EQ
#undef FREQ_GT
#undef TIMES3


/*---------------------------------------------*/
static void makeMaps_e ( Lib_Bzip_Encode_Storage_Ty *est )
{
   unsigned i;

   est->nInUse = 0;
   for (i = 0; i < 256; i++)
      if (est->inUse[i]) {
         est->unseqToSeq[i] = est->nInUse;
         est->nInUse++;
      }
}



/*---------------------------------------------*/
static void 
generateMTFValues ( Lib_Bzip_Encode_Storage_Ty *est )
{
   UChar   yy[256];
   unsigned i;
   Int32   j;
   UChar   tmp;
   UChar   tmp2;
   Int32   zPend;
   Int32   wr;
   Int32   EOB;

   MAYBE_SCHEDULE;

   makeMaps_e (est);
   EOB = est->nInUse + RUNBASE - 1;

   for (i = 0; (Int32) i <= EOB; i++) est->mtfFreq[i] = 0;

   wr = 0;
   zPend = 0;
   for (i = 0; (Int32) i < est->nInUse; i++)
      yy[i] = (UChar) i;

   for (i = 0; i < est->nblock; i++) {
      UChar ll_i;

      j = est->zptr[i] - 1; if (j < 0) j = est->nblock-1;      
      ll_i = est->unseqToSeq[est->block[j] >> 8];

      j = 0;
      tmp = yy[j];
      while ( ll_i != tmp ) {
         j++;
         tmp2 = tmp;
         tmp = yy[j];
         yy[j] = tmp2;
      };
      yy[0] = tmp;

      if (j == 0) {
         zPend++;
      } else {
         if (zPend > 0) {
            zPend--;
            while (True) {
               switch (zPend % RUNBASE) {
                  case 0: est->zptr[wr] = RUNA; wr++; est->mtfFreq[RUNA]++; break;
                  case 1: est->zptr[wr] = RUNB; wr++; est->mtfFreq[RUNB]++; break;
               };
               if (zPend < RUNBASE) break;
               zPend = (zPend - RUNBASE) / RUNBASE;
            };
            zPend = 0;
         }
         est->zptr[wr] = j+RUNBASE-1; wr++; est->mtfFreq[j+RUNBASE-1]++;
      }
   }

   if (zPend > 0) {
      zPend--;
      while (True) {
         switch (zPend % RUNBASE) {
            case 0:  est->zptr[wr] = RUNA; wr++; est->mtfFreq[RUNA]++; break;
            case 1:  est->zptr[wr] = RUNB; wr++; est->mtfFreq[RUNB]++; break;
         };
         if (zPend < RUNBASE) break;
         zPend = (zPend - RUNBASE) / RUNBASE;
      };
   }

   est->zptr[wr] = EOB; wr++; est->mtfFreq[EOB]++;

   est->nMTF = wr;
}


/*---------------------------------------------*/
#define LESSER_ICOST  0
#define GREATER_ICOST 15

static Bool
sendMTFValues ( Lib_Bzip_Encode_Storage_Ty *est )
{
   Int32 v, t, i, j, gs, ge, totc, bt, bc, iter;
   Int32 alphaSize, selCtr;
   UInt31 nSelectors;
   Int32 nBytes;

   /*--
   UChar  len [N_GROUPS][MAX_ALPHA_SIZE];
   is a global since the decoder also needs it.

   UInt32 code[N_GROUPS][MAX_ALPHA_SIZE];
   Int32  rfreq[N_GROUPS][MAX_ALPHA_SIZE];
   are also globals only used in this proc.
   Made global to keep stack frame size small.
   --*/

   UInt16 cost[N_GROUPS];
   Int32  fave[N_GROUPS];

   MAYBE_SCHEDULE;

   status_printf ( 3, 
                "      %u in block, %d after MTF & 1-2 coding, %u+2 syms in use\n", 
                est->nblock, est->nMTF, est->nInUse );

   alphaSize = est->nInUse + RUNBASE;
   for (t = 0; t < N_GROUPS; t++)
      for (v = 0; v < alphaSize; v++)
         est->len[t][v] = GREATER_ICOST;

   /*--- Generate an initial set of coding tables ---*/
   { 
      Int32 nPart, remF, tFreq, aFreq;

      nPart = N_GROUPS;
      remF  = est->nMTF;
      gs = 0;
      while (nPart > 0) {
         tFreq = remF / nPart;
         ge = gs-1;
         aFreq = 0;
         while (aFreq < tFreq && ge < alphaSize-1) {
            ge++;
            aFreq += est->mtfFreq[ge];
         }

         if (ge > gs 
             && nPart != N_GROUPS && nPart != 1 
             && ((N_GROUPS-nPart) % 2 == 1)) {
            aFreq -= est->mtfFreq[ge];
            ge--;
         }
#ifndef __KERNEL__  /* floating point (todo) */
         status_printf ( 3, "      initial group %d, [%d .. %d], has %d syms (%4.1f%%)\n",
			 nPart, gs, ge, aFreq, 
			 (100.0 * (float)aFreq) / (float)est->nMTF );
#endif
         for (v = 0; v < alphaSize; v++)
            if (v >= gs && v <= ge) 
               est->len[nPart-1][v] = LESSER_ICOST; else
               est->len[nPart-1][v] = GREATER_ICOST;
 
         nPart--;
         gs = ge+1;
         remF -= aFreq;
      }
   }

   /*--- 
      Iterate up to N_ITERS times to improve the tables.
   ---*/
   for (iter = 0; iter < N_ITERS; iter++) {

      MAYBE_SCHEDULE;

      for (t = 0; t < N_GROUPS; t++) fave[t] = 0;

      for (t = 0; t < N_GROUPS; t++)
         for (v = 0; v < alphaSize; v++)
            est->rfreq[t][v] = 0;

      nSelectors = 0;
      totc = 0;
      gs = 0;
      while (True) {

         /*--- Set group start & end marks. --*/
         if (gs >= est->nMTF) break;
         ge = gs + G_SIZE - 1; 
         if (ge >= est->nMTF) ge = est->nMTF-1;

         /*-- 
            Calculate the cost of this group as coded
            by each of the coding tables.
         --*/
         for (t = 0; t < N_GROUPS; t++)
	    cost[t] = 0;

	 /* */
         {
            register UInt16 cost0, cost1;

#if N_GROUPS != 2
# error "bad assumption here"
#endif
            cost0 = cost1 = 0;
            for (i = gs; i <= ge; i++) { 
               UInt16 icv = est->zptr[i];
               cost0 += est->len[0][icv];
               cost1 += est->len[1][icv];
            }
            cost[0] = cost0; cost[1] = cost1;
         }
 
         /*-- 
            Find the coding table which is best for this group,
            and record its identity in the selector table.
         --*/
         bc = 999999999; bt = -1;
         for (t = 0; t < N_GROUPS; t++)
            if (cost[t] < bc) { bc = cost[t]; bt = t; };
         totc += bc;
         fave[bt]++;
         est->selector[nSelectors] = bt;
         nSelectors++;

         /*-- 
            Increment the symbol frequencies for the selected table.
          --*/
         for (i = gs; i <= ge; i++)
            est->rfreq[bt][ est->zptr[i] ]++;

         gs = ge+1;
      }

#ifndef __KERNEL__  /* too hard... todo */
      if (e_verbosity >= 3) {
         status_printf ( 3, "      pass %d: size is %d, grp uses are ", 
			 iter+1, totc/8 );
         for (t = 0; t < N_GROUPS; t++)
            status_printf ( 3, "%d ", fave[t] );
         fprintf ( stderr, "\n" );
      }
#endif

      /*--
        Recompute the tables based on the accumulated frequencies.
      --*/
      for (t = 0; t < N_GROUPS; t++)
         hbMakeCodeLengths ( 
            &(est->len[t][0]), &(est->rfreq[t][0]), alphaSize, 20,
            &(est->heap_tmp[0]), &(est->weight_tmp[0]), &(est->parent_tmp[0])
         );
   }

   /*--- Assign actual codes for the tables. --*/
   for (t = 0; t < N_GROUPS; t++) {
      unsigned minLen = 32;
      unsigned maxLen = 0;

      for (i = 0; i < alphaSize; i++) {
         if (est->len[t][i] > maxLen)
	    maxLen = est->len[t][i];
         if (est->len[t][i] < minLen)
	    minLen = est->len[t][i];
      }
      hbAssignCodes ( &(est->code[t][0]), &(est->len[t][0]), 
                      minLen, maxLen, alphaSize );
   }

   /*--- Transmit the mapping table. ---*/
   { 
      Bool inUse16[16];

      for (i = 0; i < 16; i++) {
          inUse16[i] = 0;
          for (j = 0; j < 16; j++)
             if (est->inUse[i * 16 + j])
		inUse16[i] = 1;
      }
     
      nBytes = est->outctr;
      for (i = 0; i < 16; i++)
	 bsW(est, 1, inUse16[i]);

      for (i = 0; i < 16; i++)
         if (inUse16[i]) {
            for (j = 0; j < 16; j++)
               if (est->inUse[i * 16 + j]) 
                  { bsW(est, 1,1); }
	       else 
                  { bsW(est, 1,0); };
         }
      status_printf ( 3, " bytes: mapping %d, ", est->outctr-nBytes );
   }

   /*--- Now the selectors. ---*/
   nBytes = est->outctr;
   bsW ( est, 10, nSelectors );
   for (i = 0; i < (Int32) nSelectors; i++)
      bsW(est, 1, (est->selector[i] != 0));

   status_printf ( 3, "selectors %d, ", est->outctr-nBytes );

   /*--- Now the coding tables. ---*/
   nBytes = est->outctr;

   for (t = 0; t < N_GROUPS; t++) {
      unsigned curr = est->len[t][0];
      bsW ( est, 5, curr );
      for (i = 0; i < alphaSize; i++) {
         while (curr < est->len[t][i]) { bsW(est, 2,2); curr++; /* 2 -> binary 10 */ };
         while (curr > est->len[t][i]) { bsW(est, 2,3); curr--; /* 3 -> binary 11 */ };
         bsW ( est, 1, 0 );
      }
   }

   status_printf ( 3, "code lengths %d, ", est->outctr - nBytes );


   /*--
      Compute the total size of output, and check whether 
      it will fit into the caller's output buffer, and, 
      if not, give up now.
   --*/
   {
      Int32 totalBits = 8 * est->outctr;

      selCtr = 0;
      gs = 0;
      while (True) {
         UChar* ltab;
         if (gs >= est->nMTF) break;
         ge = gs + G_SIZE - 1; 
         if (ge >= est->nMTF) ge = est->nMTF-1;
         ltab = &(est->len[est->selector[selCtr]][0]);
         for (i = gs; i <= ge; i++)
            totalBits += ltab[est->zptr[i]];
         gs = ge+1;
         selCtr++;
      }
      /*-- allow for worst-case flushing of bit buffer --*/
      totalBits += 32;
      /*-- convert to bytes --*/
      totalBits = (totalBits + 7) / 8;
      est->predictedSize = totalBits;
      if (totalBits > est->totalSizeAllowed) return False;
   }


   /*--- And finally, the block data proper ---*/
   nBytes = est->outctr;
   selCtr = 0;
   gs = 0;
   while (True) {
      UChar*  ltab;
      UInt32* ctab;

      if (gs >= est->nMTF) break;
      ge = gs + G_SIZE - 1; 
      if (ge >= est->nMTF) ge = est->nMTF-1;
      ltab = &(est->len [est->selector[selCtr]][0]);
      ctab = &(est->code[est->selector[selCtr]][0]);
      for (i = gs; i <= ge; i++) {
         UInt16 zptr_i = est->zptr[i];
         bsW ( est, ltab[zptr_i], ctab[zptr_i] );
      }
      gs = ge+1;
      selCtr++;
   }

   status_printf ( 3, "codes %d\n", est->outctr-nBytes );

   return True;
}


/*---------------------------------------------*/
Int16 const rNums[64] = {    
   619, 720, 127, 481, 931, 816, 813, 233, 566, 247, 
   985, 724, 205, 454, 863, 491, 741, 242, 949, 214, 
   733, 859, 335, 708, 621, 574, 73, 654, 730, 472, 
   419, 436, 278, 496, 867, 210, 399, 680, 480, 51, 
   878, 465, 811, 169, 869, 675, 611, 697, 867, 561, 
   862, 687, 507, 283, 482, 129, 807, 591, 733, 623, 
   150, 238, 59, 379
};


static void 
randomiseBlock ( Lib_Bzip_Encode_Storage_Ty *est )
{
   unsigned i;
   RAND_DECLS;

   for (i = 0; i < est->nblock; i++) {
      RAND_UPD_MASK;
      est->block[i] ^= RAND_MASK << 8;
   }
}


/*---------------------------------------------*/
size_t 
bzip_compressBlock ( unsigned char *in_buf, 
		     unsigned char *out_buf,
		     void *heap,
		     size_t num_in_buf,
		     size_t num_out_buf,
		     int param /* unused */ )
{
   size_t i;
   Bool fitsOk;
   Lib_Bzip_Encode_Storage_Ty *est;

   if (num_in_buf == 0)
      return 0; /* Some functions (e.g. sortBlock() don't like nblock to equal 0. */

   MOD_INC_USE_COUNT;

#if 0
   est = (Lib_Bzip_Encode_Storage_Ty *) bzip2_work_area_C;
#else
   est = (Lib_Bzip_Encode_Storage_Ty *) heap;
#endif

   est->totalSizeAllowed = num_out_buf;
   est->verb             = 0;
   est->nblock           = num_in_buf;

   // make est->block a copy of in_buf <<\'d 8

   for (i = 0; i < num_in_buf; i++)
      est->block[i] = ((UInt16)in_buf[i]) << 8;

   // try to sort it
   est->randomised = False;
   est->workLimit  = 250 * est->nblock;
   est->workDone   = 0;
   sortBlock (est);
#ifndef __KERNEL__  /* floating point (todo) */
   status_printf ( 2, "      %d work, %u nblock, %5.2f ratio\n",
		   est->workDone, est->nblock,
		   (float)est->workDone /
		   (float)(est->nblock==0 ? 1 : est->nblock) );
#endif
   // if sorting failed because of too much work, 
   // randomise and try again (this can\'t fail).

   if (est->workDone > est->workLimit) {
      status_printf ( 2, "      sorting aborted; randomising block\n" );
      randomiseBlock (est);
      est->randomised = True;
      est->workLimit = 0;
      est->workDone = 0;
      sortBlock (est);
#ifndef __KERNEL__  /* floating point (todo) */
      status_printf ( 3, "      %d work, %d nblock, %5.2f ratio\n",
		      est->workDone, est->nblock,
		      (float)est->workDone /
		      (float)(est->nblock==0 ? 1 : est->nblock) );
#endif
   }

   // find the posn of the original string

   i = 0;
   while (i < est->nblock && est->zptr[i] != 0)
      i++;

   est->outbuff = out_buf;
   bsInit_e (est);
   bsW ( est, 8, 0xa2 );  /* version number */

   if (est->randomised) bsW(est, 1,1); else bsW(est, 1,0);

   bsW ( est, 16, i );

   generateMTFValues (est);

   fitsOk = sendMTFValues (est);
   if (!fitsOk) {
      MOD_DEC_USE_COUNT;
      return 0;
   }

   if (est->outctr > est->predictedSize) {
#ifdef __KERNEL__
      printk ( KERN_ERR
	       "bzip2 panic: predicted size (%d) less than actual (%d)\n",
	       est->predictedSize, est->outctr );
      MOD_DEC_USE_COUNT;
      return 0;
#else
      fprintf ( stderr, "panic: predicted size is too small\n" );
      fprintf ( stderr, "       predicted %d, actual %d\n", 
                       est->predictedSize, est->outctr );
      exit(1);
#endif
   }

   bsFlush_e (est);

   MOD_DEC_USE_COUNT;
   return est->outctr;
}


/*-------------------------------------------------------------*/
/*-- end                                        lib_bzip_e.c --*/
/*-------------------------------------------------------------*/
