/* dhcpd.c
 *
 * Moreton Bay DHCP Server
 * Copyright (C) 1999 Matthew Ramsay <matthewr@moreton.com.au>
 *			Chris Trew <ctrew@moreton.com.au>
 *
 * Rewrite by Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <time.h>
#include <sys/time.h>

#include "debug.h"
#include "dhcpd.h"
#include "arpping.h"
#include "socket.h"
#include "options.h"
#include "packet.h"
#include "serverpacket.h"

/*Added by louistsai 2002/10/22, for adding log function*/
#include "libconf.h"
/*end*/

#ifdef HOST_TEST
#include "leases.h"
#else
#include "setting.h"
#endif

#define DEBUGGING

#ifdef HOST_TEST
/* globals */
DHCP_OFFERED_ADDR *leases;
#endif

/* globals */
int conf_conflict = 0 ;


struct server_config_t server_config = {
	start: IPADDR(192, 168, 1, 2),
	end: IPADDR(192, 168, 1, 254),
#ifdef HOST_TEST
	interface: "eth0",
#else
	interface: "eth1",
#endif
	max_leases: 254,
	decline_time: 3600,
	conflict_time: 3600,
	offer_time: 60,
	min_lease: 60,
	lease_file: "/var/log/dhcpd.leases",
	siaddr: IPADDR(0, 0, 0, 0),
};


/* Exit and cleanup */
static void exit_server(int retval)
{
	CLOSE_LOG();
	exit(retval);
}


/* SIGTERM handler */
static void dhcpd_killed(int sig)
{
	sig = 0;
	LOG(LOG_INFO, "Received SIGTERM");
	exit_server(0);
}


static unsigned char dhcp_dns_option[18] = {
	DHCP_DNS_SERVER, 4,
	168, 95, 1, 2,
};

static unsigned char dhcp_router_option[6] = {
	DHCP_ROUTER, 4,
	192, 168, 1, 254
};

static unsigned char dhcp_wins_option[6] = {
	DHCP_WINS_SERVER, 4,
};

static void add_option(unsigned char *data)
{
	struct option_set	*option, **curr;

	option = malloc(sizeof(struct option_set));
	option->data = data;
	
	curr = &server_config.options;
	while ( *curr && (*curr)->data[OPT_CODE] < data[OPT_CODE] ) {
		curr = &(*curr)->next;
	}

	option->next = *curr;
	*curr = option;
}


int main(int argc, char *argv[])
{	
	fd_set rfds;
	int	foreground = 0;
	int server_socket = -1;
	int bytes, retval;
	struct dhcpMessage packet;
	unsigned char *state;
	unsigned char *server_id, *requested;
	FILE *fp;
	u_int32_t server_id_align, requested_align;
	DHCP_OFFERED_ADDR *lease;
#ifndef HOST_TEST
	int	dns_count, dns_index;
	DHCP_OFFERED_ADDR lease_slot;
	REG_NET_CONF	netconf;
#endif
	
	if ( argc >= 2 && strcmp(argv[1], "-f") == 0 ) {
		foreground = 1;
	}

	OPEN_LOG("dhcpd");
	LOG(LOG_INFO, "dhcp server started");

#ifndef HOST_TEST
	read_config(MC_NET_CONFIG, &netconf);

	dns_count = 0;
	dns_index = 2;
	if ( netconf.dns0 != 0 ) {
		dns_count++;
		*((u_int32_t*) &dhcp_dns_option[dns_index]) = netconf.dns0;
		dns_index += 4;
	}
	if ( netconf.dns1 != 0 ) {
		dns_count++;
		*((u_int32_t*) &dhcp_dns_option[dns_index]) = netconf.dns1;
		dns_index += 4;
	}
	if ( netconf.dns2 != 0 ) {
		dns_count++;
		*((u_int32_t*) &dhcp_dns_option[dns_index]) = netconf.dns2;
		dns_index += 4;
	}

//2003/7/8 marked by louistsai, becuase we didn't have proxydns setup, if set private gateway as dns server,
//         user maybe confuse it!

	if ( dns_count == 0 && netconf.private_ip != 0 ) {
		dns_count++;
		*((u_int32_t*) &dhcp_dns_option[dns_index]) = netconf.private_ip;
		//dns_index += 4;
	}

//2003/7/8,added by louistsai, pass the dns server ipaddress that get from pppoe/dhcp server to the dhcp client!
/*
	switch(netconf.public_if) {
	case PUBIF_PPPOE:
	{
		fp=fopen("/etc/ppp/resolv.conf","r");
		if(fp) {
			char line[80];
			while(fgets(line,80,fp)) {
				if(!strstr(line,"nameserver")) continue;
				dns_count++;
				*((u_int32_t*) &dhcp_dns_option[dns_index]) = inet_addr(strchr(line,' ')+1);
				dns_index += 4;
			}	
			fclose(fp);
		}
		
	}	
	break;
	case PUBIF_DHCPC:
	{
		fp=fopen("/etc/dhcpc.resolv","r");
		if(fp) {
			char line[80];
			while(fgets(line,80,fp)) {
				if(!strstr(line,"nameserver")) continue;
				dns_count++;
				*((u_int32_t*) &dhcp_dns_option[dns_index]) = inet_addr(strchr(line,' ')+1);
				dns_index += 4;
			}	
			fclose(fp);
		}
	}
	break;
	}
*/
	if ( dns_count != 0 ) {
		dhcp_dns_option[1] = dns_count * 4;
		add_option(dhcp_dns_option);
	}

	if ( netconf.private_ip != 0 ) {
	//	*((u_int32_t*) &dhcp_wins_option[2]) = netconf.private_ip;
	//	add_option(dhcp_wins_option);
	}
	
	if(!check_dhcp_conf()) {
		REG_DHCP_CONF		dhcpconf;
		unsigned long 		req_ip,tmp_s,tmp_e;

		conf_conflict = 1;	
        	
        	/*Added one appropriate ip leased range*/
        	read_config(MC_DHCP_CONFIG, &dhcpconf);
  		
		req_ip = netconf.private_ip & netconf.private_netmask ;	
		
		tmp_s = dhcpconf.range_start0 ;
		tmp_e = dhcpconf.range_end0;

		printf("req_ip =%x \n", req_ip);
		if(req_ip == (dhcpconf.range_start1 & netconf.private_netmask) ) {
			dhcpconf.range_start0 = dhcpconf.range_start1;
			dhcpconf.range_end0   =	dhcpconf.range_end1;
			dhcpconf.range_start1 = tmp_s;
			dhcpconf.range_end1   =	tmp_e;
		}
		else
		if(req_ip == (dhcpconf.range_start2 & netconf.private_netmask) ) {
			dhcpconf.range_start0 = dhcpconf.range_start2;
			dhcpconf.range_end0   =	dhcpconf.range_end2;
			dhcpconf.range_start2 = tmp_s;
			dhcpconf.range_end2   =	tmp_e;
		}
		else
		if(req_ip == (dhcpconf.range_start3 & netconf.private_netmask) ) {
			dhcpconf.range_start0 = dhcpconf.range_start3;
			dhcpconf.range_end0   =	dhcpconf.range_end3;
			dhcpconf.range_start3 = tmp_s;
			dhcpconf.range_end3   =	tmp_e;
		}
		else {
			/*
			EVENTLOG_ENTRY  	log;
			
			log.level = LEVEL_NORMAL;
			log.msgid = IDS_DHCP_AUTO_ADD_NEW_RANGE;
			//strcpy(, "The dhcpd configuration was in conflict with internal subnet range, add a leased rang to make user can get legal ip");
        		write_config(MC_EVENTLOG_ADD, &log);
	        	
	        	*/
	        	Add_Logs(LEVEL_NORMAL,IDS_DHCP_AUTO_ADD_NEW_RANGE,NULL);
	        	
	        	
			//down the priolity of original ip leased range
        		dhcpconf.range_start3 = dhcpconf.range_start2;
        		dhcpconf.range_end3   = dhcpconf.range_end2;
        	
	        	dhcpconf.range_start2 = dhcpconf.range_start1;	
       			dhcpconf.range_end2   = dhcpconf.range_end1;
       		
       			dhcpconf.range_start1 = dhcpconf.range_start0;
	        	dhcpconf.range_end1   = dhcpconf.range_end0;
        	
        		//insert new ip leased range to first priolity to assure
        		//dhcp client can get legal ip at each time
	        	dhcpconf.range_start0 = (netconf.private_ip & netconf.private_netmask ) | inet_addr("0.0.0.2");
        		dhcpconf.range_end0 = (netconf.private_ip & netconf.private_netmask ) | inet_addr("0.0.0.254");
        	}
        	
		write_config(MC_DHCP_CONFIG, &dhcpconf);
	}
	
#else
	add_option(dhcp_dns_option);
	add_option(dhcp_router_option);
#endif

#if 0
	if ((option = find_option(server_config.options, DHCP_LEASE_TIME))) {
		memcpy(&server_config.lease, option->data + 2, 4);
		server_config.lease = ntohl(server_config.lease);
	}
	else server_config.lease = LEASE_TIME;
#endif
	
#ifdef HOST_TEST
	leases = malloc(sizeof(DHCP_OFFERED_ADDR) * server_config.max_leases);
	memset(leases, 0, sizeof(DHCP_OFFERED_ADDR) * server_config.max_leases);
#endif

	if (read_interface(server_config.interface, &server_config.ifindex,
			   &server_config.server, server_config.arp) < 0)
		exit_server(1);

	if ( !foreground && daemon(0, 0) == -1) {
		perror("fork");
		exit_server(1);
	}

	signal(SIGTERM, dhcpd_killed);

	while(1) { /* loop until universe collapses */

		if (server_socket < 0)
			if ((server_socket = listen_socket(INADDR_ANY, SERVER_PORT, server_config.interface)) < 0) {
				LOG(LOG_ERR, "FATAL: couldn't create server socket, %s", sys_errlist[errno]);
				exit_server(0);
			}			

		FD_ZERO(&rfds);
		FD_SET(server_socket, &rfds);
		retval = select(server_socket + 1, &rfds, NULL, NULL, NULL);

		if (retval < 0) {
			DEBUG(LOG_INFO, "error on select");
			continue;
		}
		
		if ((bytes = get_packet(&packet, server_socket)) < 0) { /* this waits for a packet - idle */
			if (bytes == -1 && errno != EINTR) {
				DEBUG(LOG_INFO, "error on read, %s, reopening socket", sys_errlist[errno]);
				close(server_socket);
				server_socket = -1;
			}
			continue;
		}

		if ((state = get_option(&packet, DHCP_MESSAGE_TYPE)) == NULL) {
			DEBUG(LOG_ERR, "couldn't get option from packet, ignoring");
			continue;
		}
		
		/* ADDME: look for a static lease */
#ifdef HOST_TEST
		lease = find_lease_by_chaddr(packet.chaddr);
#else
		lease = NULL;
		if ( find_lease_by_chaddr(packet.chaddr, &lease_slot) )
			lease = &lease_slot;
#endif
		switch (state[0]) {
		case DHCPDISCOVER:
			DEBUG(LOG_INFO,"received DISCOVER");
			
			if (sendOffer(&packet) < 0) {
				LOG(LOG_ERR, "send OFFER failed");
			}
			break;			
 		case DHCPREQUEST:
			DEBUG(LOG_INFO, "received REQUEST");

			requested = get_option(&packet, DHCP_REQUESTED_IP);
			server_id = get_option(&packet, DHCP_SERVER_ID);

#ifdef DEBUG
			if ( requested )
				printf("requested ip = %d.%d.%d.%d\n", 
					requested[0], requested[1], requested[2], requested[3]);
			if ( server_id )
				printf("server id = %d.%d.%d.%d\n", 
					server_id[0], server_id[1], server_id[2], server_id[3]);
#endif

			if (requested) memcpy(&requested_align, requested, 4);
			if (server_id) memcpy(&server_id_align, server_id, 4);
		
			if (lease) { /*ADDME: or static lease */
#ifdef DEBUG 
				printf("enter here\n");
#endif
				if (server_id) {
					/* SELECTING State */
					DEBUG(LOG_INFO, "server_id = %08x", ntohl(server_id_align));
#ifdef DEBUG
					printf("%x %x %x %x\n",server_id_align,server_config.server,requested_align,lease->yiaddr);

#endif
					if (server_id_align == server_config.server && requested && 
					    requested_align == lease->yiaddr) {
						sendACK(&packet, lease->yiaddr);
					}
				} else {
					if (requested) {
						/* INIT-REBOOT State */
						if (lease->yiaddr == requested_align)
							sendACK(&packet, lease->yiaddr);
						else sendNAK(&packet);
					} else {
						/* RENEWING or REBINDING State */
						if (lease->yiaddr == packet.ciaddr)
							sendACK(&packet, lease->yiaddr);
						else {
							/* don't know what to do!!!! */
							sendNAK(&packet);
						}
					}						
				}
			} /* else remain silent */				
#if 0
			else {
				unsigned long vip = IPADDR(192, 168, 1, 249);
				sendACK(&packet, vip);
			}
#endif
			break;
		case DHCPDECLINE:
			DEBUG(LOG_INFO,"received DECLINE");
			if (lease) {
				memset(lease->chaddr, 0, 16);
				lease->expires = time(0) + server_config.decline_time;
			}			
			break;
		case DHCPRELEASE:
			DEBUG(LOG_INFO,"received RELEASE");
			if (lease) lease->expires = time(0);
			break;
		case DHCPINFORM:
			DEBUG(LOG_INFO,"received INFORM");
			send_inform(&packet);
			break;	
		default:
			LOG(LOG_WARNING, "unsupported DHCP message (%02x) -- ignoring", state[0]);
		}
	}

	return 0;
}

