/* serverpacket.c
 *
 * Constuct and send DHCP server packets
 *
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <time.h>

/*Added by louistsai 2002/10/22, for adding log function*/
#include "libconf.h"
/*end*/

#include "packet.h"
#include "debug.h"
#include "dhcpd.h"
#include "options.h"
#include "leases.h"

/* send a packet to giaddr using the kernel ip stack */
static int send_packet_to_relay(struct dhcpMessage *payload)
{
	DEBUG(LOG_INFO, "Forwarding packet to relay");

	return kernel_packet(payload, server_config.server, SERVER_PORT,
			payload->giaddr, SERVER_PORT);
}


/* send a packet to a specific arp address and ip address by creating our own ip packet */
static int send_packet_to_client(struct dhcpMessage *payload, int force_broadcast)
{
	unsigned char *chaddr;
	u_int32_t ciaddr;
	
	if (force_broadcast) {
		DEBUG(LOG_INFO, "broadcasting packet to client (NAK)");
		ciaddr = INADDR_BROADCAST;
		chaddr = MAC_BCAST_ADDR;
	} else if (payload->ciaddr) {
		DEBUG(LOG_INFO, "unicasting packet to client ciaddr");
		ciaddr = payload->ciaddr;
		chaddr = payload->chaddr;
	} else if (ntohs(payload->flags) & BROADCAST_FLAG) {
		DEBUG(LOG_INFO, "broadcasting packet to client (requested)");
		ciaddr = INADDR_BROADCAST;
		chaddr = MAC_BCAST_ADDR;
	} else {
		DEBUG(LOG_INFO, "unicasting packet to client yiaddr");
		ciaddr = payload->yiaddr;
		chaddr = payload->chaddr;
	}
	return raw_packet(payload, server_config.server, SERVER_PORT, 
			ciaddr, CLIENT_PORT, chaddr, server_config.ifindex);
}


/* send a dhcp packet, if force broadcast is set, the packet will be broadcast to the client */
static int send_packet(struct dhcpMessage *payload, int force_broadcast)
{
	int ret;

	if (payload->giaddr)
		ret = send_packet_to_relay(payload);
	else ret = send_packet_to_client(payload, force_broadcast);
	return ret;
}


static void init_packet(struct dhcpMessage *packet, struct dhcpMessage *oldpacket, char type)
{
	init_header(packet, type);
	packet->xid = oldpacket->xid;
	memcpy(packet->chaddr, oldpacket->chaddr, 16);
	packet->flags = oldpacket->flags;
	packet->giaddr = oldpacket->giaddr;
	packet->ciaddr = oldpacket->ciaddr;
	add_simple_option(packet->options, DHCP_SERVER_ID, server_config.server);
}


/* add in the bootp options */
static void add_bootp_options(struct dhcpMessage *packet)
{
	packet->siaddr = server_config.siaddr;
	if (server_config.sname)
		strncpy(packet->sname, server_config.sname, sizeof(packet->sname) - 1);
	if (server_config.boot_file)
		strncpy(packet->file, server_config.boot_file, sizeof(packet->file) - 1);
}
	

#ifdef HOST_TEST
static unsigned char hwaddr[] = {
	0x00, 0xd0, 0x59, 0x7a, 0x90, 0x53
};
#endif

static int check_macaddr(struct dhcpMessage *packet, struct dhcpMessage *oldpacket)
{

#ifdef HOST_TEST
	if ( memcmp(oldpacket->chaddr, hwaddr, 6) == 0 ) {
		packet->yiaddr = IPADDR(192, 168, 1, 9);
		return 1;
	}
#else
	int	i;
	REG_MACMAPPING_CONF macconf;
	REG_MACMAPPING_TABLE *mactable;

	read_config(MC_MACMAPPING_CONFIG, &macconf);
	mactable = macconf.table;
	for (i = 0; i < macconf.count; i++) {
		if ( memcmp(oldpacket->chaddr, mactable[i].hwaddr, 6) == 0 ) {
			packet->yiaddr = mactable[i].ipaddr;
			return 1;
		}
	}
#endif

	return 0;
}

static char *get_client_host(struct dhcpMessage *packet)
{
	static char host_buf[128];
	char *client_host;

	client_host = get_option(packet, DHCP_HOST_NAME);

	if ( client_host != NULL ) {
		strncpy(host_buf, client_host, client_host[-1]);
		host_buf[(int) client_host[-1]] = 0;
		printf("client host = %s\n", host_buf);
	} else {
		printf("client host is unknown\n");
		return NULL;
	}
	
	return host_buf;
}

static int check_clienthost(char *client_host, struct dhcpMessage *packet)
{
#ifndef HOST_TEST
	int	i;
	REG_HOSTMAPPING_CONF hostconf;
	REG_HOSTMAPPING_TABLE *hosttable;

	if ( client_host == NULL )
		return 0;

	read_config(MC_HOSTMAPPING_CONFIG, &hostconf);
	hosttable = hostconf.table;
	for (i = 0; i < hostconf.count; i++) {
		if ( strcasecmp(client_host, hosttable[i].hostname) == 0 ) {
			packet->yiaddr = hosttable[i].ipaddr;
			return 1;
		}
	}
#endif

	return 0;
}

int check_pptp_ip_range(u_int32_t req_ip)
{
	unsigned long pptp_start,pptp_end,pptp_server;
	char tmpipstr[20];
	unsigned char ip0,ip1,ip2;
	
	REG_NET_CONF	net;
	REG_PPTP_CONF 	pconf;

LOG(LOG_INFO, "enter check pptp\n");	
	Config_Read(MC_PPTP_CONFIG, &pconf);
	Config_Read(MC_NET_CONFIG,(void *)&net);
	
	ip0 = (net.private_ip & 0xff);
	ip1 = (net.private_ip >> 8 ) & 0xff;
	ip2 = (net.private_ip >>16 ) & 0xff;
	
	sprintf(tmpipstr,"%d.%d.%d.%d",ip0,ip1,ip2,pconf.serverip);
	pptp_server = inet_addr(tmpipstr);
	sprintf(tmpipstr,"%d.%d.%d.%d",ip0,ip1,ip2,pconf.startip);
	pptp_start = inet_addr(tmpipstr);
	sprintf(tmpipstr,"%d.%d.%d.%d",ip0,ip1,ip2,pconf.endip);
	pptp_end = inet_addr(tmpipstr);
	
printf("start=%x end=%x server=%x addr=%x\n",pptp_start,pptp_end,pptp_server,req_ip);
	
	
	if((req_ip == pptp_server) || (req_ip >= pptp_start && req_ip <= pptp_end))
		return 0;
	else
		return 1;
}



/* send a DHCP OFFER to a DHCP DISCOVER */
int sendOffer(struct dhcpMessage *oldpacket)
{
	struct dhcpMessage packet;
#ifdef HOST_TEST
	DHCP_OFFERED_ADDR *lease = NULL;
#else
	DHCP_OFFERED_ADDR lease;
#endif
	u_int32_t req_align, lease_time_align;
	unsigned char *req, *lease_time;
	struct option_set *curr;
	struct in_addr addr;
	char	*client_host;
	u_int32_t conf_lease_time, conf_min_lease_time;

	get_lease_time(&conf_lease_time, &conf_min_lease_time);
	lease_time_align = conf_lease_time;

	init_packet(&packet, oldpacket, DHCPOFFER);
	
	client_host = get_client_host(oldpacket);	

	if ( check_macaddr(&packet, oldpacket) || check_clienthost(client_host, &packet) ) {
		/* ip was assigned */
	} 
#ifdef HOST_TEST
	/* the client is in our lease/offered table */
	else if ((lease = find_lease_by_chaddr(oldpacket->chaddr))) {
		if (!lease_expired(lease)) 
			lease_time_align = lease->expires - time(0);
		packet.yiaddr = lease->yiaddr;
#else
	else if ( find_lease_by_chaddr(oldpacket->chaddr, &lease) ) {
		if ( !lease_expired(&lease) )
			lease_time_align = lease.expires - time(0);
		packet.yiaddr = lease.yiaddr;
	LOG(LOG_INFO, "the client is in our lease/offered table\n");
#endif
		
	/* Or the client has a requested ip */
	} else if ((req = get_option(oldpacket, DHCP_REQUESTED_IP)) &&
		/* Don't look here (ugly hackish thing to do) */
		memcpy(&req_align, req, 4) &&
#ifdef HOST_TEST
		/* and the ip is in the lease range */
		ntohl(req_align) >= ntohl(server_config.start) &&
		ntohl(req_align) <= ntohl(server_config.end) &&
		   
		/* and its not already taken/offered */ /* ADDME: check that its not a static lease */
		((!(lease = find_lease_by_yiaddr(req_align)) ||
		   
	   	/* or its taken, but expired */ /* ADDME: or maybe in here */
	   	lease_expired(lease)))) {
#else
		//marked by louistsai 2003/1/3
		//ip_in_lease_range(req_align)  
		ip_in_lease_range(req_align) && check_pptp_ip_range(req_align) &&
		(!find_lease_by_yiaddr(req_align, &lease) || lease_expired(&lease)) ) {
#endif
		packet.yiaddr = req_align; /* FIXME: oh my, is there a host using this IP? */

		/* otherwise, find a free IP */ /*ADDME: is it a static lease? */
	} else {
		packet.yiaddr = find_address(0);
		
		/* try for an expired lease */
		if (!packet.yiaddr) packet.yiaddr = find_address(1);
	}
	
	if(!packet.yiaddr) {
		LOG(LOG_WARNING, "no IP addresses to give -- OFFER abandoned");
		return -1;
	}
	
	if (!add_lease(client_host, packet.chaddr, packet.yiaddr, server_config.offer_time)) {
		LOG(LOG_WARNING, "lease pool is full -- OFFER abandoned");
		return -1;
	}		

	if ((lease_time = get_option(oldpacket, DHCP_LEASE_TIME))) {
		memcpy(&lease_time_align, lease_time, 4);
		lease_time_align = ntohl(lease_time_align);
		if (lease_time_align > conf_lease_time) 
			lease_time_align = conf_lease_time;
	}

	/* Make sure we aren't just using the lease time from the previous offer */
	if (lease_time_align < conf_min_lease_time)
		lease_time_align = conf_lease_time;
	/* ADDME: end of short circuit */		
	add_simple_option(packet.options, DHCP_LEASE_TIME, htonl(lease_time_align));
#ifndef HOST_TEST
	{
		u_int32_t	router, netmask;
		get_router(packet.yiaddr, &router, &netmask);
		add_simple_option(packet.options, DHCP_ROUTER, router);	
		add_simple_option(packet.options, DHCP_SUBNET, netmask);
	}
#endif

	curr = server_config.options;
	while (curr) {
		if (curr->data[OPT_CODE] != DHCP_LEASE_TIME)
			add_option_string(packet.options, curr->data);
		curr = curr->next;
	}

	add_bootp_options(&packet);
	
	addr.s_addr = packet.yiaddr;
	LOG(LOG_INFO, "sending OFFER of %s", inet_ntoa(addr));
	return send_packet(&packet, 0);
}


int sendNAK(struct dhcpMessage *oldpacket)
{
	struct dhcpMessage packet;

	init_packet(&packet, oldpacket, DHCPNAK);
	
	DEBUG(LOG_INFO, "sending NAK");
	return send_packet(&packet, 1);
}


int sendACK(struct dhcpMessage *oldpacket, u_int32_t yiaddr)
{
	struct dhcpMessage packet;
	struct option_set *curr;
	unsigned char *lease_time;
	u_int32_t lease_time_align;
	struct in_addr addr;
	char	*client_host;
	u_int32_t	conf_lease_time, conf_min_lease_time;

#ifdef DEBUG
	printf("enter sendACK\n");
#endif

	get_lease_time(&conf_lease_time, &conf_min_lease_time);
	lease_time_align = conf_lease_time;

	init_packet(&packet, oldpacket, DHCPACK);
	packet.yiaddr = yiaddr;
	client_host = get_client_host(oldpacket);
	
	if ((lease_time = get_option(oldpacket, DHCP_LEASE_TIME))) {
		memcpy(&lease_time_align, lease_time, 4);
		lease_time_align = ntohl(lease_time_align);
		if (lease_time_align > conf_lease_time) 
			lease_time_align = conf_lease_time;
		else if (lease_time_align < conf_min_lease_time) 
			lease_time_align = conf_lease_time;
	}
	
	add_simple_option(packet.options, DHCP_LEASE_TIME, htonl(lease_time_align));
#ifndef HOST_TEST
	{
		u_int32_t	router, netmask;
		get_router(packet.yiaddr, &router, &netmask);
		add_simple_option(packet.options, DHCP_ROUTER, router);	
		add_simple_option(packet.options, DHCP_SUBNET, netmask);
	}
#endif
	
	curr = server_config.options;
	while (curr) {
		if (curr->data[OPT_CODE] != DHCP_LEASE_TIME)
			add_option_string(packet.options, curr->data);
		curr = curr->next;
	}

	add_bootp_options(&packet);

	addr.s_addr = packet.yiaddr;
	LOG(LOG_INFO, "sending ACK to %s", inet_ntoa(addr));

	if (send_packet(&packet, 0) < 0) 
		return -1;

	add_lease(client_host, packet.chaddr, packet.yiaddr, lease_time_align);

	return 0;
}


int send_inform(struct dhcpMessage *oldpacket)
{
	struct dhcpMessage packet;
	struct option_set *curr;

	init_packet(&packet, oldpacket, DHCPACK);
	
	curr = server_config.options;
	while (curr) {
		if (curr->data[OPT_CODE] != DHCP_LEASE_TIME)
			add_option_string(packet.options, curr->data);
		curr = curr->next;
	}

	add_bootp_options(&packet);

	return send_packet(&packet, 0);
}

/*To check range of lease ip is suitable for interal interface( include ip alias) */
static int valid_req_ip(u_int32_t req_ip)
{
	u_int32_t	dummy1, dummy2;

	return get_router(req_ip, &dummy1, &dummy2);
}

int check_dhcp_conf()
{
	REG_DHCP_CONF	dhcpconf;
	
	read_config(MC_DHCP_CONFIG, &dhcpconf);
	
	if( dhcpconf.range_start0 != 0 && !valid_req_ip(dhcpconf.range_start0) )
		return 0;
/*
	if( dhcpconf.range_start1 != 0 && !valid_req_ip(dhcpconf.range_start1) )
		return 0;
	if( dhcpconf.range_start2 != 0 && !valid_req_ip(dhcpconf.range_start2) )
		return 0;
	if( dhcpconf.range_start3 != 0 && !valid_req_ip(dhcpconf.range_start3) )
		return 0;
	
*/	
	return 1;
}
