/********************************************************************************
*
*	Copyright (c) 2002  Magic Control Technology Co.  All Rights Reserved.
*
*	FILE:
*		hd_ctl.c
*
*	Abstract:
*		Hard Drive Management Function
*		
*	History:
*		2002/9/12	Louis Created
*********************************************************************************/
#include <stdio.h>
#include <linux/types.h>
#include <getopt.h>
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <mntent.h>
#include <sys/vfs.h>
#include <fcntl.h>
#include <ctype.h>
#include <setjmp.h>
#include <errno.h>
#include <getopt.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <linux/hdreg.h>
#include <getopt.h>
#include <errno.h>
#include <mntent.h>
#include <linux/unistd.h>

_syscall5(int,  _llseek,  uint, fd, ulong, hi, ulong, lo, loff_t *,res, uint, wh);

#include "hd_ctl.h"
#include "libconf.h"

#define KILOBYTE 1024



struct { char *devname;
	 char *parname;
	 char *mp_string;
	 char *data_path;
	 char *spool_path;
	 char *http_path;
	 char *home_path;
} HD_Table[] = {
	{ "/dev/hda","/dev/hda1","/mnt/HDA","/mnt/HDA/data","/mnt/HDA/spool","/mnt/HDA/http","/mnt/HDA/home"},
	{ "/dev/hdb","/dev/hdb1","/mnt/HDB","/mnt/HDB/data","/mnt/HDB/spool","/mnt/HDB/http","/mnt/HDB/home"},
	{ "/dev/hdc","/dev/hdc1","/mnt/HDC","/mnt/HDC/data","/mnt/HDC/spool","/mnt/HDC/http","/mnt/HDC/home"},
	{ "/dev/hdd","/dev/hdd1","/mnt/HDD","/mnt/HDD/data","/mnt/HDD/spool","/mnt/HDD/http","/mnt/HDD/home"},
};




#define initial_mark	"112233445566"

#define hex_val(c)	({ char _c = (c); isdigit(_c) ? _c - '0' : tolower(_c) + 10 - 'a'; })


#define LINE_LENGTH	80
#ifndef HD_ENCRYPT
#define pt_offset(b, n)	((struct partition *)((b) + 0x1be + (n) * sizeof(struct partition)))
#else
#define pt_offset(b, n)	((struct partition *)((b) + 0x1b0 + (n) * sizeof(struct partition)))
#endif
#define sector(s)	((s) & 0x3f)
#define cylinder(s, c)	((c) | (((s) & 0xc0) << 2))

#define hsc2sector(h,s,c) (sector(s) - 1 + sectors * ((h) + heads * cylinder(s,c)))
#define set_hsc(h,s,c,sector) { s = sector % sectors + 1; sector /= sectors; h = sector % heads; sector /= heads;c = sector & 0xff;s |= (sector >> 2) & 0xc0;}


char MBRbuffer[MAX_SECTOR_SIZE];
uint	heads,
	sectors,
	cylinders,
	sector_offset = 1,
	sector_size = DEFAULT_SECTOR_SIZE;

char *cfg_str[] =
{	"", " HardSect", " SoftSect", " NotMFM", " HdSw>15uSec", " SpinMotCtl",
	" Fixed", " Removeable", " DTR<=5Mbs", " DTR>5Mbs", " DTR>10Mbs",
	" RotSpdTol>.5%", " dStbOff", " TrkOff", " FmtGapReq", " nonMagnetic"
};

char *SlowMedFast[]	= {"slow", "medium", "fast", "eide"};
char *BuffType[]	= {"unknown", "1Sect", "DualPort", "DualPortCache"};
#define YN(b)	(((b)==0)?"no":"yes")

static void dmpstr (char *prefix, unsigned int i, char *s[], unsigned int maxi)
{
	if (i > maxi)
		printf("%s%d", prefix, i);
	else
		printf("%s%s", prefix, s[i]);
}

/* A valid partition table sector ends in 0x55 0xaa */
static unsigned int
part_table_flag(char *b) {
	return ((uint) b[510]) + (((uint) b[511]) << 8);
}

int
valid_part_table_flag(unsigned char *b) {
	return (b[510] == 0x55 && b[511] == 0xaa);
}

static void
write_part_table_flag(char *b) {
	b[510] = 0x55;
	b[511] = 0xaa;
}

/* start_sect and nr_sects are stored little endian on all machines */
/* moreover, they are not aligned correctly */
static void
store4_little_endian(unsigned char *cp, unsigned int val) {
	cp[0] = (val & 0xff);
	cp[1] = ((val >> 8) & 0xff);
	cp[2] = ((val >> 16) & 0xff);
	cp[3] = ((val >> 24) & 0xff);
}

static unsigned int
read4_little_endian(unsigned char *cp) {
	return (uint)(cp[0]) + ((uint)(cp[1]) << 8)
		+ ((uint)(cp[2]) << 16) + ((uint)(cp[3]) << 24);
}

static void
set_start_sect(struct partition *p, unsigned int start_sect) {
	store4_little_endian(p->start4, start_sect);
}

unsigned int
get_start_sect(struct partition *p) {
	return read4_little_endian(p->start4);
}

static void
set_nr_sects(struct partition *p, unsigned int nr_sects) {
	store4_little_endian(p->size4, nr_sects);
}

unsigned int
get_nr_sects(struct partition *p) {
	return read4_little_endian(p->size4);
}

static void
get_sectorsize(int fd) {
#if defined(BLKSSZGET)
	int arg;
	if (ioctl(fd, BLKSSZGET, &arg) == 0) {
		sector_size = arg;
		if (sector_size != DEFAULT_SECTOR_SIZE)
			printf("Note: sector size is %d (not %d)\n",
			       sector_size, DEFAULT_SECTOR_SIZE);
	}
#else
	/* maybe the user specified it; and otherwise we still
	   have the DEFAULT_SECTOR_SIZE default */
#endif
}

/*
 * Raw disk label. For DOS-type partition tables the MBR,
 * with descriptions of the primary partitions.
 */



int read_mbr(int fd)
{
	lseek(fd,0,SEEK_SET);
	if (sector_size == read(fd, MBRbuffer, sector_size)) 
		return 0;
	else 
		return -1;
}

void dump_mbr(int fd)
{
	int i;
	read_mbr(fd);
	lseek(fd,0,SEEK_SET);
	for(i=0;i<512;i++) {
		printf("%.2x ",MBRbuffer[i]);
		if(i%16 == 0) printf("\n");	
	}	
}

int write_mbr(int fd)
{
	lseek(fd,0,SEEK_SET);
	if (sector_size == write(fd, MBRbuffer, sector_size)) 
		return 0;
	else 
		return -1;
}

static void set_partition(int i, uint start, uint stop,int sysid) 
{
	
	struct partition *p;
	
	p = pt_offset(MBRbuffer,i);
	p->boot_ind = 0;
	p->sys_ind = sysid;
	set_start_sect(p, start);
	set_nr_sects(p, stop - start + 1);
	if ( start/(sectors*heads)  > 1023)
		start = heads*sectors*1024 - 1;
	set_hsc(p->head, p->sector, p->cyl, start);
	if (stop/(sectors*heads) > 1023)
		stop = heads*sectors*1024 - 1;
	set_hsc(p->end_head, p->end_sector, p->end_cyl, stop);
}

static void dump_identity (struct hd_driveid *id)
{
	int i;
	char pmodes[64] = {0,}, dmodes[128]={0,};

	printf("\n Model=%.40s, FwRev=%.8s, SerialNo=%.20s", 
		id->model, id->fw_rev, id->serial_no);
	printf("\n Config={");
	for (i=0; i<=15; i++) {
		if (id->config & (1<<i))
			printf("%s", cfg_str[i]);
	}
	printf(" }\n");
	printf(" RawCHS=%d/%d/%d, TrkSize=%d, SectSize=%d, ECCbytes=%d\n",
		id->cyls, id->heads, id->sectors,
		id->track_bytes, id->sector_bytes, id->ecc_bytes);
	dmpstr (" BuffType=",id->buf_type,BuffType,3);
	printf(", BuffSize=%dkB, MaxMultSect=%d", id->buf_size/2, id->max_multsect);
	if (id->max_multsect) {
		printf(", MultSect=");
		if (!(id->multsect_valid&1))
			printf("?%d?", id->multsect);
		else if (id->multsect)
			printf("%d", id->multsect);
		else
			printf("off");
	}
	putchar('\n');
	if (id->tPIO <= 5) {
		strcat(pmodes, "pio0 ");
		if (id->tPIO >= 1) strcat(pmodes, "pio1 ");
		if (id->tPIO >= 2) strcat(pmodes, "pio2 ");
	}
	if (!(id->field_valid&1))
		printf(" (maybe):");
	printf(" CurCHS=%d/%d/%d, CurSects=%d",
		id->cur_cyls, id->cur_heads, id->cur_sectors, *(int *)&id->cur_capacity0);
	printf(", LBA=%s", YN(id->capability&2));
	if (id->capability&2)
 		printf(", LBAsects=%d", id->lba_capacity);

	if (id->capability&1) {
		if (id->dma_1word | id->dma_mword) {
			if (id->dma_1word & 0x100)	strcat(dmodes,"*");
			if (id->dma_1word & 1)		strcat(dmodes,"sdma0 ");
			if (id->dma_1word & 0x200)	strcat(dmodes,"*");
			if (id->dma_1word & 2)		strcat(dmodes,"sdma1 ");
			if (id->dma_1word & 0x400)	strcat(dmodes,"*");
			if (id->dma_1word & 4)		strcat(dmodes,"sdma2 ");
			if (id->dma_1word & 0xf800)	strcat(dmodes,"*");
			if (id->dma_1word & 0xf8)	strcat(dmodes,"sdma? ");
			if (id->dma_mword & 0x100)	strcat(dmodes,"*");
			if (id->dma_mword & 1)		strcat(dmodes,"mdma0 ");
			if (id->dma_mword & 0x200)	strcat(dmodes,"*");
			if (id->dma_mword & 2)		strcat(dmodes,"mdma1 ");
			if (id->dma_mword & 0x400)	strcat(dmodes,"*");
			if (id->dma_mword & 4)		strcat(dmodes,"mdma2 ");
			if (id->dma_mword & 0xf800)	strcat(dmodes,"*");
			if (id->dma_mword & 0xf8)	strcat(dmodes,"mdma? ");
		}
	}
	printf("\n IORDY=");
	if (id->capability&8)
		printf((id->capability&4) ? "on/off" : "yes");
	else
		printf("no");
	if ((id->capability&8) || (id->field_valid&2)) {
		if (id->field_valid&2) {
			printf(", tPIO={min:%d,w/IORDY:%d}", id->eide_pio, id->eide_pio_iordy);
			if (id->eide_pio_modes & 1)	strcat(pmodes, "pio3 ");
			if (id->eide_pio_modes & 2)	strcat(pmodes, "pio4 ");
			if (id->eide_pio_modes &~3)	strcat(pmodes, "pio? ");
		}

	}
	if ((id->capability&1) && (id->field_valid&2))
		printf(", tDMA={min:%d,rec:%d}", id->eide_dma_min, id->eide_dma_time);
	printf("\n PIO modes: %s", pmodes);
	printf("\n DMA modes: %s\n", dmodes);
}

static void reread_partition_table(int fd,int leave) {
	int error = 0;
	int i;

	printf("Calling ioctl() to re-read partition table.\n");
	sync();
	sleep(2);
	if ((i = ioctl(fd, BLKRRPART)) != 0) {
                error = errno;
        } else {
                /* some kernel versions (1.2.x) seem to have trouble
                   rereading the partition table, but if asked to do it
		   twice, the second time works. - biro@yggdrasil.com */
//                sync();
//                sleep(2);
                if((i = ioctl(fd, BLKRRPART)) != 0)
                        error = errno;
        }

	if (i < 0)
		printf("Re-read table failed with error %d: %s.\nReboot your "
			"system to ensure the partition table is updated.\n",
			error, strerror(error));

	if (leave) {
		close(fd);

		printf("Syncing disks.\n");
		sync();
		sleep(4);		/* for sync() */
	}
}


int HD_Check_Mount(char *mount_point)
{
	FILE *mount_table;
	struct mntent *mount_entry;

	mount_table = setmntent("/proc/mounts", "r");
	if (mount_table == 0) return 0;

	while ((mount_entry = getmntent(mount_table))) {
		if ( strcasecmp(mount_point,mount_entry->mnt_dir)== 0 ) {
			return 1;
			break;
		}
	}
	endmntent(mount_table);	
	return 0;
}

int do_df(char *mount_point , HD_INFO *info)
{
	struct statfs s;
	long blocks_used;
	long blocks_percent_used;
	FILE *mount_table;
	struct mntent *mount_entry;
	int found=0;

	mount_table = setmntent("/proc/mounts", "r");
	if (mount_table == 0) return -1;

	while ((mount_entry = getmntent(mount_table))) {
		if ( strcasecmp(mount_point,mount_entry->mnt_dir)== 0 ) {
			found = 1;
			break;
		}
	}
	endmntent(mount_table);

	if(!found) return -1;
	
	if (statfs(mount_point, &s) != 0) {
		return -1;
	}

	if (s.f_blocks > 0) {
		blocks_used = s.f_blocks - s.f_bfree;
		if(blocks_used == 0)
			blocks_percent_used = 0;
		else {
			blocks_percent_used = (long)
			  (blocks_used * 100.0 / (blocks_used + s.f_bavail) + 0.5);
		}
		
		info->capacity 		= s.f_blocks * (s.f_bsize / (double)KILOBYTE );
		info->used_percent	= blocks_percent_used;
//		info->available		= s.f_bavail * (s.f_bsize / (double)KILOBYTE);
		info->available		= s.f_bfree * (s.f_bsize / (double)KILOBYTE);
		strncpy(info->mount_point, mount_point, 30);
		
		
/*
		printf("%9ld %9ld %9ld %3ld%% %s\n",
				(long) (s.f_blocks * (s.f_bsize / (double)KILOBYTE)),
				(long) ((s.f_blocks - s.f_bfree)*(s.f_bsize/(double)KILOBYTE)),
				(long) (s.f_bavail * (s.f_bsize / (double)KILOBYTE)),
				blocks_percent_used, mount_point);

*/	
	}

	return 0;
}


/* =================================== *
 * Start to Define Export Function     *
 * =================================== */


/*Tag our label string "112233445566" on begin of hd*/
int HD_Add_Label(int hd_no)
{
	int fd;
	char* devname;
	
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDWR)) < 0) {
		printf("You will not be able to read the partition table.\n");
		return -1;
	}
	
	read_mbr(fd);
	strncpy(MBRbuffer,initial_mark,12);
	write_part_table_flag(MBRbuffer);
	
	write_mbr(fd);
	close(fd);
	return 0;
}

int HD_Remove_Label(int hd_no)
{
	int fd,i;
	char* devname;
	
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDWR)) < 0) {
		printf("You will not be able to read the partition table.\n");
		return -1;
	}
	
	read_mbr(fd);
	for(i=0;i<12;i++) MBRbuffer[i]=0;
	write_part_table_flag(MBRbuffer);
	
	write_mbr(fd);
	close(fd);
	return 0;
}

/*Check the first 12 bytes string of HD is equal to "112233445566" */
int HD_Is_Initial(int hd_no)
{	
	int fd,ret=0;
	char cmp_str[12];
	char* devname;
	
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDWR)) < 0) {
		printf("You will not be able to read the partition table.\n");
		return 0;
	}
	
	if(read_mbr(fd)<0) printf("read mbr failed");
//dump_mbr(fd);
	strncpy(cmp_str,MBRbuffer,12);
	if(strncmp(cmp_str,initial_mark,12) == 0)
		ret = 1;

	close(fd);
	
	return ret;
	
}


int HD_Clear_MBR(int hd_no)
{
	int fd;
	char* devname;
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDWR)) < 0) {
		printf("You will not be able to read the partition table.\n");
		return -1;
	}
	
	memset(MBRbuffer,0,sector_size);
	write_part_table_flag(MBRbuffer);
	write_mbr(fd);
	
	reread_partition_table(fd,0);
	close(fd);
	
	return 0;
}


/* --------------------------------------------------------------------	*							
 * Ask kernel about geometry. Invent something reasonable		*
 * in case the kernel has no opinion.					*
 * -------------------------------------------------------------------- */
 
int HD_Get_Geometry(int hd_no, struct geom *g) 
{
	
	int sec_fac;
	long longsectors;
	struct hd_geometry geometry;
	int res1, res2;
	int fd;
	char* devname;
	
	
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDONLY)) < 0) {
		printf("You will not be able to read the partition table.\n");
		return -1;
	}
	
	get_sectorsize(fd);
	sec_fac = sector_size / 512;

	res1 = ioctl(fd, BLKGETSIZE, &longsectors);
#ifdef HDIO_REQ
	res2 = ioctl(fd, HDIO_REQ, &geometry);
#else
	res2 = ioctl(fd, HDIO_GETGEO, &geometry);
#endif

	/* never use geometry.cylinders - it is truncated */
	heads = cylinders = sectors = 0;
	sector_offset = 1;
	if (res2 == 0) {
		heads = geometry.heads;
		sectors = geometry.sectors;
		if (heads * sectors == 0)
			res2 = -1;
		else
			sector_offset = sectors;
	}
	if (res1 == 0 && res2 == 0) { 	/* normal case */
		cylinders = longsectors / (heads * sectors);
		cylinders /= sec_fac;   /* do not round up */
	} else if (res1 == 0) {		/* size but no geometry */
		heads = cylinders = 1;
		sectors = longsectors / sec_fac;
	}

	if (g) {
		g->heads = heads;
		g->sectors = sectors;
		g->cylinders = cylinders;
	}
	
	close(fd);
	return 0;
}

/*----------------------------------------------------------------------*
 * Using HDIO_GET_IDENTITY ioctl code to get HD identification, the     *
 * stucture hd_driveid have been defined in linux/hdreg.h		*
 *----------------------------------------------------------------------*/
 
int HD_Get_Identification(int hd_no, struct hd_driveid *id)
{
	
	int fd;
	char* devname;
	
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDONLY)) < 0) {
		printf("Cannot open %s.\n",devname);
		return -1;
	}
	
	if (!ioctl(fd, HDIO_GET_IDENTITY, id)){
		//dump_identity(id);
	}
	else 
	if (errno == -ENOMSG)
		printf(" no identification info available\n");
	
	return 0;
}

/*----------------------------------------------------------------------*
 * Mounting HD on specific path		  			        *
 *----------------------------------------------------------------------*/
int HD_Mounting(int hd_no)
{
	char cmdline[80];
	int ret=0;
	
	HD_Set_Action_Flag(hd_no,HD_MOUNTING);
	sprintf(cmdline,"%s -p %s",MKDIR,HD_Table[hd_no].mp_string);
	ret = system(cmdline);
	sprintf(cmdline,"%s -t ext3 %s %s",MOUNT,HD_Table[hd_no].parname,HD_Table[hd_no].mp_string);
	ret = system(cmdline);
	
	HD_Set_Action_Flag(hd_no,HD_ACTION_COMPLETE);
	
	return ret;
	
}
/*----------------------------------------------------------------------*
 * Unmount HD		  					        *
 *----------------------------------------------------------------------*/
int HD_Unmounting(int hd_no)
{
	char cmdline[80];
	int ret=0;
	
	HD_Set_Action_Flag(hd_no,HD_UNMOUNTING);
	
	sprintf(cmdline,"%s %s",UMOUNT,HD_Table[hd_no].parname);
	ret = system(cmdline);
	
	HD_Set_Action_Flag(hd_no,HD_ACTION_COMPLETE);
	
	return ret;
	
}


int HD_Crate_Default_Dir(int hd_no)
{
	char cmdline[80];
	int ret;
	
	sprintf(cmdline,"%s -p %s",MKDIR,HD_Table[hd_no].data_path);
	ret = system(cmdline);
	sprintf(cmdline,"%s 777 %s",CHMOD,HD_Table[hd_no].data_path);
	ret = system(cmdline);
	
	sprintf(cmdline,"%s -p %s",MKDIR,HD_Table[hd_no].spool_path);
	ret = system(cmdline);
	sprintf(cmdline,"%s 777 %s",CHMOD,HD_Table[hd_no].spool_path);
	ret = system(cmdline);
	
	sprintf(cmdline,"%s -p %s",MKDIR,HD_Table[hd_no].http_path);
	ret = system(cmdline);
	sprintf(cmdline,"%s 777 %s",CHMOD,HD_Table[hd_no].http_path);
	ret = system(cmdline);
#ifdef QMAIL
	sprintf(cmdline,"%s -p %s",MKDIR,HD_Table[hd_no].home_path);
	ret = system(cmdline);
	sprintf(cmdline,"%s 777 %s",CHMOD,HD_Table[hd_no].home_path);
	ret = system(cmdline);
#endif
	
	return 0;
}




/*----------------------------------------------------------------------*
 * Check the HD status			  			        *
 * Return:	1 : HD is ok now					*
 * 		-1: HD is not exist					*
 *		-2: HD maybe crash or been removed suddently		*
 *----------------------------------------------------------------------*/
 
int HD_Is_Active(int hd_no)
{
	struct stat stat_buf;
	int fd;
	char* devname;
	
	devname = HD_Table[hd_no].devname;
	
	if (stat(devname,&stat_buf)) {
		perror(devname);
		return -1;
	}
	
	fd = open (devname, O_RDONLY|O_NONBLOCK);
	if (fd < 0) {
		perror(devname);
		return -2;
	}
	
	close(fd);
	return 1;
}

int HD_Set_Action_Flag(int hd_no, int sts)
{
	int fd;
	char* devname;
	
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDWR)) < 0) {
		printf("You will not be able to read the partition table.\n");
		return -1;
	}
	
	read_mbr(fd);
	
	MBRbuffer[12] = sts;
	write_mbr(fd);
	close(fd);	
	return 0;
}

int HD_Get_Action_Flag(int hd_no)
{
	int fd;
	char* devname;
	unsigned char stat;
	
	
	if( HD_Is_Initial(hd_no) != 1) 
		return HD_UNINITIAL;
	
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDWR)) < 0) {
		printf("You will not be able to read the partition table.\n");
		return -1;
	}
	
	read_mbr(fd);
	close(fd);	
	
	stat =(unsigned char) MBRbuffer[12];
	
	return stat;
}



/*********************************************************************************/

/*----------------------------------------------------------------------*
 * HD Initialize				  			*
 *----------------------------------------------------------------------*/



int HD_Initialization(int hd_no)
{
	
	char* devname;
	int ret;
	
	
	devname = HD_Table[hd_no].devname;
	
	if((ret = HD_Is_Active(hd_no)) != 1 )
		return ret; 
		
	
	
	if(!HD_Is_Initial(hd_no)) {
		int fd;
		unsigned long start,max;

		sync();	
		HD_Unmounting(hd_no);
		HD_Clear_MBR(hd_no);
		HD_Add_Label(hd_no);
		HD_Get_Geometry(hd_no,NULL);
		
		if ((fd = open(devname, O_RDWR)) < 0) {
			printf("You will not be able to read the partition table.\n");
			return -1;
		}	
		
		start = sector_offset;
		max = heads * sectors * cylinders - 1;
		
		read_mbr(fd);
//dump_mbr(fd);
		set_partition(0,start,max,LINUX_NATIVE);
		//set_partition(0,start,max-204800,LINUX_NATIVE);
		//set_partition(1,max-204799,max,LINUX_SWAP);
		write_mbr(fd);
//dump_mbr(fd);
		reread_partition_table(fd,0);
		close(fd);
		
		HD_Format_Ext3(hd_no);
		
		HD_Mounting(hd_no);
		HD_Set_Action_Flag(hd_no,HD_READY);
		HD_Crate_Default_Dir(hd_no);

		
	}
	else
		printf("The %s HD have been initialized\n",devname);
	
	return 0;
}


/*----------------------------------------------------------------------*
 * Make ext3 File system			  			*
 *----------------------------------------------------------------------*/

int HD_Format_Ext3(int hd_no)
{
	char cmdline[80];
	int ret=0;
	
	
	if((ret = HD_Is_Active(hd_no)) != 1 )
		return ret;
	
	
	HD_Set_Action_Flag(hd_no,HD_FORMATTING);
	
	sprintf(cmdline,"%s -j -F %s",MKEXT3,HD_Table[hd_no].parname);
	ret = system(cmdline);
	sprintf(cmdline,"%s -c0 -i0 %s",TUNE2FS,HD_Table[hd_no].parname);
	system(cmdline);
	
	HD_Set_Action_Flag(hd_no,HD_ACTION_COMPLETE);
	
	return ret;
}
/*----------------------------------------------------------------------*
 * Check file system on HD		  			        *
 *----------------------------------------------------------------------*/
int HD_Examine(int hd_no)
{
	char cmdline[80];
	int ret=0;
	
	
	if((ret = HD_Is_Active(hd_no)) != 1 )
		return ret;
	
	HD_Set_Action_Flag(hd_no,HD_SCANNING);
	
	sprintf(cmdline,"%s -C0 -fy %s",E2FSCK,HD_Table[hd_no].parname);
	ret = system(cmdline);
	
	HD_Set_Action_Flag(hd_no,HD_ACTION_COMPLETE);
	
	
	return ret;
}

/*----------------------------------------------------------------------*
 * HD Get Information			  			        *
 *----------------------------------------------------------------------*/
int HD_Get_Information(int hd_no,HD_INFO *info)
{
	int ret=0;
	struct hd_driveid hdid;
	
	if((ret = HD_Is_Active(hd_no)) != 1 )
		return ret;	
	
	if((ret = do_df( HD_Table[hd_no].mp_string, info)) != 0 )
		ret = -3;
	
	info->stat = HD_Get_Action_Flag(hd_no);
	
	HD_Get_Identification(hd_no,&hdid);
	
	strcpy(info->model_name,hdid.model);
	strcpy(info->serial_num,hdid.serial_no);
	
	return ret;
}


int HD_Set_Standby_Timeout(int hd_no, int timeout)
{
	int	fd, ret;
	char	*devname;
	unsigned char args[4] = { WIN_SETIDLE1, 0, 0, 0 };

	devname = HD_Table[hd_no].devname;
	
	if ( (fd=open(devname, O_RDONLY | O_NONBLOCK)) < 0 )
		return fd;
	
	args[1] = timeout;
	ret = ioctl(fd, HDIO_DRIVE_CMD, &args);

	close(fd);

	return ret;
}

int HD_Get_Power_State(int hd_no)
{
	int	fd;
	int	power_state;
	char	*devname;
	unsigned char args[4] = { WIN_CHECKPOWERMODE1, 0, 0, 0 };

	devname = HD_Table[hd_no].devname;

	if ( (fd=open(devname, O_RDONLY | O_NONBLOCK)) < 0 )
		return fd;

	if ( ioctl(fd, HDIO_DRIVE_CMD, &args) && (args[0]=WIN_CHECKPOWERMODE2 && ioctl(fd, HDIO_DRIVE_CMD, &args)) ) {
		if ( errno != EIO && args[0] != 0 && args[1] != 0 )
			power_state = HDPS_UNKNOWN;
		else
			power_state = HDPS_SLEEPING;
	} else {
		power_state = (args[2] == 255) ? HDPS_ACTIVE : HDPS_SLEEPING;
	}

	close(fd);

	return power_state;
}
			

/*
int main(int argc,char **argv)
{
	pid_t pid;
	unsigned char stat = HD_UNINITIAL;
	HD_INFO info;
	
	printf("%s\n",HD_Table[1].devname);
	
	
	pid = fork();
	switch(pid)
	{
	case 0:
		HD_Remove_Label(1);
		HD_Initialization(1);
		break;	
	default:
		while(stat != HD_READY)	{ 
			stat = HD_Get_Action_Flag(1);
			printf("flag=%2x\n",stat);
			sleep(1);
		}
		break;	
	}
	if(pid!=0) {
		int sat_val;
		pid_t child_pid;
		
		child_pid = wait (&sat_val);	
		HD_Get_Information(1,&info);
		printf("Model=%s SerNo=%s Capacity=%9ld Available=%9ld used_percent=%3ld MP=%s STS=%2x\n",
			info.model_name,
			info.serial_num,
			info.capacity,
			info.available,
			info.used_percent,
			info.mount_point,
			info.stat);
	}
	
	
	return 0;
	
}
*/

/*
 *  ide-smart 1.3 - IDE S.M.A.R.T. cheking tool
 *  Copyright (C) 1998-1999 Ragnar Hojland Espinosa <ragnar@lightside.dhis.org>
 *                1998      Gadi Oxman <gadio@netvision.net.il>
 *
 *		  2002	    Modified by LouisTsai 
 */

#define NR_ATTRIBUTES	30

#ifndef TRUE
#define TRUE 1
#endif

typedef struct threshold_s {
		__u8		id;
   	__u8		threshold;
   	__u8		reserved[10];
} __attribute__ ((packed)) threshold_t;


typedef struct thresholds_s {
   	__u16		revision;
   	threshold_t	thresholds[NR_ATTRIBUTES];
   	__u8		reserved[18];
   	__u8		vendor[131];
   	__u8		checksum;
} __attribute__ ((packed)) thresholds_t;


typedef struct value_s {
   	__u8		id;
   	__u16		status;
   	__u8		value;
   	__u8		vendor[8];
} __attribute__ ((packed)) value_t;


typedef struct values_s {
   	__u16		revision;
   	value_t		values[NR_ATTRIBUTES];
   	__u8		offline_status;
   	__u8		vendor1;
   	__u16		offline_timeout;
   	__u8		vendor2;
   	__u8		offline_capability;
   	__u16		smart_capability;
   	__u8		reserved[16];
   	__u8		vendor[125];
   	__u8		checksum;
} __attribute__ ((packed)) values_t;


struct {
   	__u8		value;
   	char		*text;
} offline_status_text[] = {
   	{ 0x00,	"NeverStarted" },
   	{ 0x02,	"Completed"    },
   	{ 0x04,	"Suspended"    },
   	{ 0x05,	"Aborted"      },
   	{ 0x06,	"Failed"       },
   	{ 0, 0 }
};


struct {
   	__u8		value;
   	char		*text;
} smart_command[] = {
   	{ SMART_ENABLE,	 		"SMART_ENABLE"		  },
   	{ SMART_DISABLE,	 	"SMART_DISABLE"           },
   	{ SMART_IMMEDIATE_OFFLINE,	"SMART_IMMEDIATE_OFFLINE" },
   	{ SMART_AUTO_OFFLINE,		"SMART_AUTO_OFFLINE"      },
};


/* Index to smart_command table, keep in order */
enum SmartCommand
{ 
	SMART_CMD_ENABLE,
	SMART_CMD_DISABLE, 
	SMART_CMD_IMMEDIATE_OFFLINE, 
   	SMART_CMD_AUTO_OFFLINE 
};


int smart_read_values (int fd, values_t* values)
{   
   	__u8 args[4 + 512] = {WIN_SMART, 0, SMART_READ_VALUES, 1, };

   	if (ioctl(fd, HDIO_DRIVE_CMD, &args)) {
      		int e = errno;
      		perror ("SMART_READ_VALUES");
	      	return e;
   	}

   	memcpy (values, args + 4, 512);
   	return 0;
}


int values_not_passed (values_t *p, thresholds_t *t)
{
   	value_t *value = p->values;
   	threshold_t *threshold = t->thresholds;
   	int failed = 0;
   	int passed = 0;
   	int i;
   
   	for (i = 0; i < NR_ATTRIBUTES; i++) {
      		if (value->id && threshold->id && value->id == threshold->id) {
	 		if (value->value <= threshold->threshold) {
	    			++failed;
	 		} else {
	    			++passed;
	 		}
      		}
      
      		++value;
      		++threshold;
   	}
   
   	return ((failed==0) ? 1 : 0);
}



void print_thresholds (thresholds_t *p)
{
   	threshold_t *threshold = p->thresholds;
   	int i;

   	printf ("\n");
   	printf ("SmartRevision=%d\n", p->revision);

   	for (i = 0; i < NR_ATTRIBUTES; i++) {
      		if (threshold->id) {
	 		printf ("Id=%3d, Threshold=%3d\n", threshold->id, threshold->threshold);
      		}
      		++threshold;
   	}
   
   	printf ("CheckSum=%d\n", p->checksum);
}


int smart_cmd_simple (int fd, enum SmartCommand command, __u8 val0, char show_error)
{
   	__u8 args[4] = { WIN_SMART, val0, smart_command[command].value, 0 };
   	int e = 0;
   
   	if (ioctl (fd, HDIO_DRIVE_CMD, &args)) {
      		e = errno;
      		if (show_error) {
	 		perror (smart_command[command].text);
      		}
   	}
   
   	return e;
}
   

int smart_read_thresholds (int fd, thresholds_t *thresholds)
{
   	__u8 args[4 + 512] = {WIN_SMART, 0, SMART_READ_THRESHOLDS, 1, };

   	if (ioctl (fd, HDIO_DRIVE_CMD, &args)) {
      		int e = errno;
      		perror("SMART_READ_THRESHOLDS");
      		return e;
   	}
   
   	memcpy (thresholds, args + 4, 512);
   	return 0;
}

int ataSmartSupport ( struct hd_driveid drive)
{
#ifdef __NEW_HD_DRIVE_ID
	if ( drive.command_set_1 & 0x0001 ){
#else
	if ( drive.command_sets & 0x0001 ){
#endif
		return (1); /* drive supports S.M.A.R.T. and is disabled */
	}
			
	return (0);

}
/*----------------------------------------------------------------------*
 * HD SMART status				  			*
 *									*
 * return =  0 , ok							*
 *	  = -1 , not supported SMART					*
 *	  = -2 , SMART is disable					*
 *----------------------------------------------------------------------*/


int HD_SMART_Status (int hd_no)
{	
	struct hd_driveid drive;
	int fd,ret=0;
	char* devname;
	unsigned char parms[4] = { WIN_SMART, 0, SMART_STATUS, 0};
	
	
	HD_Get_Identification(hd_no,&drive);
	
	if(!ataSmartSupport(drive)) return -1;
	
	devname = HD_Table[hd_no].devname;
	if ((fd = open(devname, O_RDONLY)) < 0) {
		printf("cannot open device %s\n",devname);
		return -3;
	}
	

  	if (ioctl ( fd , HDIO_DRIVE_CMD,  &parms) != 0) {
      		ret = -2;
   	}	
	close(fd);
	
	return ret;
}


/*----------------------------------------------------------------------*
 * HD SMART enable function			  			*
 *									*
 * enable = 0 , disable smart 						*
 *	    1 , enable smart						*
 *----------------------------------------------------------------------*/


int HD_Enable_SMART(int hd_no,int enable)
{
	int fd;
	char* devname;
	
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDONLY)) < 0) {
		printf("cannot open device %s\n",devname);
		return -1;
	}
	
	if (smart_cmd_simple (fd, SMART_CMD_ENABLE, 0, enable))
	 	return -1;
	 	
	 close(fd);
	 
	 return 0;
}

/*----------------------------------------------------------------------*
 * HD SMART test function			  			*
 *									*
 *  return = 0 , pass							*
 *  	   = -1, failed							*		
 *----------------------------------------------------------------------*/
int HD_SMART_Check(int hd_no)
{
	thresholds_t thresholds;
      	values_t values;      
      	int fd;
	char* devname;
	
	devname = HD_Table[hd_no].devname;
	
	if ((fd = open(devname, O_RDONLY)) < 0) {
		printf("cannot open device %s\n",devname);
		return -1;
	}
      	
      	smart_read_values(fd, &values);
	smart_read_thresholds(fd, &thresholds);
	
	close(fd);
	
	return values_not_passed (&values, &thresholds);
}


/*------------------------------------------------------------- */
/* Return mounting path for each hd				*/
/*------------------------------------------------------------- */

char* HD_Get_Sharing_Path(int hd_no)
{
	static char path[128];
	
	strcpy(path,HD_Table[hd_no].data_path);
	
	return path;
}

char* HD_Get_Mounting_Path(int hd_no)
{
	static char path[128];
	
	strcpy(path,HD_Table[hd_no].mp_string);
	
	return path;
	
}

char* HD_Get_Partition_Path(int hd_no)
{
	static char path[128];
	
	strcpy(path,HD_Table[hd_no].parname);
	
	return path;
	
}

/*----------------------------------------------------------*
 * Backup current firmware to hardisk, the space is after
 * first partition, C/H/S format cannot use all sectors. But 
 * the remaining size maybe not enough to put firmware file.
 * Therefore, we will decrease the one cylinder sectors from 
 * 1st partitions. 	
 *----------------------------------------------------------*/
 
int FW_Backup_To_HD(int hd_no)
{
	int 	fd, i;
	char*	devname;
	struct 	partition *p;
	unsigned int	fw_start_sec, ret;
	unsigned long 	fw_addr_high, fw_addr_low;
	loff_t tmp;
	unsigned char 	*fwbuffer;
	unsigned long 	fwsize, dest_offset, count;
	unsigned char 	tstbuf[128];
	unsigned int	boot_version;
	MEM_ACCESS mem_data;
	
	devname = HD_Table[hd_no].devname;

	if ((fd = open(devname, O_RDWR)) < 0) {
		printf("No HD\n");
		return -1;
	}
	
	mem_data.dest_addr = 0xbfc0c004;
 	mem_data.size	   = 2;
 	mem_data.buf	   = (unsigned char *) malloc(2);
 	Config_Read(MC_RWMEM,&mem_data);
 	boot_version = *((unsigned int*) mem_data.buf);
 	free(mem_data.buf);
 	
 	if(boot_version < 0x0108) {
 		printf("boot loader don't support the firmware backup\n");	
 		return -1;
 	}
 		
	
	read_mbr(fd);
 	
	p = pt_offset(MBRbuffer,hd_no);
 	fw_start_sec = get_start_sect(p) +  get_nr_sects(p);
 	

	fw_addr_high = fw_start_sec>>23;
	fw_addr_low  = (unsigned long) (fw_start_sec * 512) & 0xffffffff;
//printf("fw_addr_high = %x , fw_addr_low= %x \n",fw_addr_high , fw_addr_low);
 	_llseek(fd, fw_addr_high, fw_addr_low,&tmp,SEEK_SET);
 	//_llseek(fd,0x1b,0xf3e80800,&tmp,SEEK_SET);
 	
 	
 	mem_data.dest_addr = 0xbfc20018;
 	mem_data.size	   = 4;
 	mem_data.buf	   = (unsigned char *) malloc(4);
 	Config_Read(MC_RWMEM,&mem_data);
 	fwsize		= (*(unsigned long *) mem_data.buf+ 0x10) - 0xbfc20000;
 	free(mem_data.buf);
 	
 	count=0;
 	dest_offset = 0x80000;
 	while(count < fwsize){
 		mem_data.dest_addr = 0xbfc20000+count;
 		mem_data.size	   = dest_offset;
 		mem_data.buf	   = (unsigned char *) malloc(mem_data.size);
 		Config_Read(MC_RWMEM,&mem_data);
 		ret = write(fd, (unsigned char *)mem_data.buf, mem_data.size);
 		free(mem_data.buf);
 		count += dest_offset;
 		if(count+ dest_offset > fwsize)
 			dest_offset = fwsize - count;
 	}

	_llseek(fd, fw_addr_high, fw_addr_low,&tmp,SEEK_SET);
	
 	close(fd);
 	
 	return ret;
}
 
 
 
 
 
