/*   ____  __  _ _____ ____     _ _            _   
**  / ___||  \/ |_   _|  _ \___| (_) ___ _ __ | |_ 
**  \___ \| |\/| || | | |_)/ __| | |/ _ \ '_ \| __|
**   ___) | |  | || | |  _| (__| | |  __/ | | | |_ 
**  |____/|_|  |_||_| |_|  \___|_|_|\___|_| |_|\__|
**   
**  SMTPclient -- simple SMTP client
**
**  This program is a minimal SMTP client that takes an email
**  message body and passes it on to a SMTP server (default is the
**  MTA on the local host). Since it is completely self-supporting,
**  it is especially suitable for use in restricted environments.
**
**  ======================================================================
**
**  Copyright (c) 1997 Ralf S. Engelschall, All rights reserved.
**
**  This program is free software; it may be redistributed and/or modified
**  only under the terms of either the Artistic License or the GNU General
**  Public License, which may be found in the SMTP source distribution.
**  Look at the file COPYING. 
**
**  This program is distributed in the hope that it will be useful, but
**  WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  ======================================================================
**
**  smtpclient_main.c -- program source
**
**  Based on smtp.c as of August 11, 1995 from
**      W.Z. Venema,
**      Eindhoven University of Technology,
**      Department of Mathematics and Computer Science,
**      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands.
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "libconf.h"

static char *from_addr  = NULL;
static char *mailhost   = NULL;
static int   mailport   = 25;
static char *reply_addr = 0;
static int   mime_style = 0;
static int   verbose    = 0;

static FILE *sfp;
static FILE *rfp;

#define dprintf  if (verbose) printf
#define dvprintf if (verbose) vprintf

/*
 *  examine message from server 
 *  here can also check smtp server connected or not, if first time
 *  fgets return nothing, can be judged as "not connected to SMTPServer"
 *  Return: 0 means success, 1 means Fail
*/
int get_response(void)
{
	char buf[BUFSIZ];
    	int bFirst = 1;

    	while (fgets(buf, sizeof(buf), rfp)) {
		bFirst = 0;
        	buf[strlen(buf)-1] = 0;
        	dprintf("%s --> %s\n", mailhost, buf);
        	if (!isdigit(buf[0]) || buf[0] > '3') {
            		printf("unexpected reply: %s", buf);
            		exit(1);
	        }
        	if (buf[4] != '-')
            	break;
    	}
    	
    	return bFirst;
}

/*
**  say something to server and check the response
*/
int chat(char *fmt, ...)
{
    	va_list ap;
    	int len;

    	va_start(ap, fmt);
    	len = vfprintf(sfp, fmt, ap);
    	va_end(ap);
  
    	va_start(ap, fmt);
    	dprintf("%s <-- ", mailhost);
    	dvprintf(fmt, ap);
    	va_end(ap);

    	fflush(sfp);
   	return get_response();
}

/*
**  transform to MIME-style quoted printable
**
**  Extracted from the METAMAIL version 2.7 source code (codes.c)
**  and modified to emit \r\n at line boundaries.
*/

static char basis_hex[] = "0123456789ABCDEF";

void toqp(FILE *infile, FILE *outfile)
{
    	int c;
    	int ct = 0;
    	int prevc = 255;

    	while ((c = getc(infile)) != EOF) {
        	if (   (c < 32 && (c != '\n' && c != '\t'))
          	  || (c == '=')
            	|| (c >= 127)
            	|| (ct == 0 && c == '.')               ) {
        	putc('=', outfile);
        	putc(basis_hex[c >> 4], outfile);
        	putc(basis_hex[c & 0xF], outfile);
        	ct += 3;
        	prevc = 'A'; /* close enough */
    	}
    	else if (c == '\n') {
        	if (prevc == ' ' || prevc == '\t') {
	    		putc('=', outfile);  /* soft & hard lines */
	    		putc(c, outfile);
       	 	}
        	putc(c, outfile);
        	ct = 0;
        	prevc = c;
    	} 
    	else {
        	if (c == 'F' && prevc == '\n') {
        	/*
         	* HORRIBLE but clever hack suggested by MTR for
         	* sendmail-avoidance
         	*/
        	c = getc(infile);
        	if (c == 'r') {
            		c = getc(infile);
            		if (c == 'o') {
            			c = getc(infile);
            			if (c == 'm') {
                			c = getc(infile);
                			if (c == ' ') {
                				/* This is the case we are looking for */
                				fputs("=46rom", outfile);
               	 				ct += 6;
                			} 
                			else {
                				fputs("From", outfile);
                				ct += 4;
                			}
            			} 
            			else {
                			fputs("Fro", outfile);
                			ct += 3;
            			}
            		} 
            		else {
            			fputs("Fr", outfile);
            			ct += 2;
            		}
        	}
        	else {
            		putc('F', outfile);
            		++ct;
        	}
        	ungetc(c, infile);
        	prevc = 'x'; /* close enough -- printable */
        } 
        else { 
        	putc(c, outfile);
        	++ct;
        	prevc = c;
        }
    }
    if (ct > 72) {
        putc('=', outfile);
        putc('\r', outfile); 
        putc('\n', outfile);
        ct = 0;
        prevc = '\n';
    }
    }
    if (ct) {
    putc('=', outfile);
    putc('\r', outfile); 
    putc('\n', outfile);
    }
    return;
}



int send_single_mail(char *sender, char *Subject, char *Msg, char *To, char *SMTP_IP, char *Level)
{
    char From[BUFSIZ];
    struct sockaddr_in sin;
    struct hostent *hp;
    int s;
    int r;
    char date_time[128];
    time_t current_time;
    NIC_INFO wan_info;
    
    (void)time(&current_time);
    sprintf(date_time,"%s",ctime(&current_time));
    
    NIC_Get_WAN_Info(&wan_info);
    
    strcpy(From, To);
    /*
     *  Determine SMTP server
     */
    mailhost = SMTP_IP;
//	printf(" Subject=%s\n", Subject);
//	printf(" Msg=%s\n", Msg);
//	printf(" To=%s\n", To);
//	printf(" mailhost=%s\n", SMTP_IP);

    /*
     *  Determine from address.
     */
    from_addr = From;

    /*
     *  Connect to smtp daemon on mailhost.
     */
    if ((hp = gethostbyname(mailhost)) == NULL) {
		goto SendFail;
    }
    if (hp->h_addrtype != AF_INET) {
		goto SendFail;
    }
    memset((char *)&sin, 0, sizeof(sin));
    memcpy((char *)&sin.sin_addr, hp->h_addr, hp->h_length);
    sin.sin_family = hp->h_addrtype;
    sin.sin_port = htons(mailport);
    if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		goto SendFail;
    }
    if (connect(s, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		goto SendFail;
    }
    if ((r = dup(s)) < 0) {
		goto SendFail;
    }
    if ((sfp = fdopen(s, "w")) == 0) {
		goto SendFail;
    }
    if ((rfp = fdopen(r, "r")) == 0) {
		goto SendFail;
    }

    /* 
     *  Give out SMTP headers.
     */
    if(get_response()) /* banner */
		goto SendFail;
	
    if(chat("HELO %s\r\n", "mct"))
		goto SendFail;

    if(chat("MAIL FROM: <%s>\r\n", from_addr))
		goto SendFail;

    if(chat("RCPT TO: <%s>\r\n", from_addr))
		goto SendFail;

    if(chat("DATA\r\n"))
		goto SendFail;

    /* 
     *  Give out Message header. 
     */
    fprintf(sfp, "From: %s\r\n", sender);
    fprintf(sfp, "Subject: %s\r\n", Subject);

    if (reply_addr)
        fprintf(sfp, "Reply-To: %s\r\n", reply_addr);
    fprintf(sfp, "Sender: %s\r\n", sender);

    fprintf(sfp, "To: %s", To);
    fprintf(sfp, "\r\n");

    if (mime_style) {
        fprintf(sfp, "MIME-Version: 1.0\r\n");
        fprintf(sfp, "Content-Type: text/plain; charset=ISO-8859-1\r\n");
        fprintf(sfp, "Content-Transfer-Encoding: quoted-printable\r\n");
    }

    fprintf(sfp, "\r\n");


    // Title information
    fprintf(sfp, "Server Name: %s\r\n", sender);
    fprintf(sfp, "IP Address: %s\r\n", wan_info.ipaddr);
    fprintf(sfp, "Date/Time: %s\r\n", date_time);
    if(Level != NULL)
    	fprintf(sfp, "Level: %s\r\n", Level);
    fprintf(sfp, "\r\n");

    /* 
     *  Give out Message body.
     */
    fprintf(sfp, "%s\r\n", Msg);

    /* 
     *  Give out SMTP end.
     */
    if(chat(".\r\n"))
		goto SendFail;
    if(chat("QUIT\r\n"))
		goto SendFail;

    /* 
     *  Die gracefully ...
     */
    return(0);
SendFail:
    return(1);
}

int ALARM_Get_Level()
{
	REG_ALARM_CONF conf;
	
	Config_Read(MC_ALARM_CONFIG, &conf);
	return conf.alarm_level;
}

BOOL ALARM_Is_Enable()
{
	REG_ALARM_CONF conf;
	
	Config_Read(MC_ALARM_CONFIG, &conf);
	return conf.enable;
}

/*
 * send multiple mails via specified SMTP Server  
 * Return:
 *	0			pass			
 *	ERROR_FAIL	FAIL to send mails
 */
 
 typedef struct  {
 	int msg_id;
 	char *map_str;
 } LOG_STR_TABLE;
 
LOG_STR_TABLE log_str_map[] = {
 	{IDS_SYSTEM_START, 		"System startup"},
	{IDS_DHCPC_NO_LEASE,		"There is no DHCP lease obtained from external interface"},
	{IDS_DHCP_COLLISION, 		"%u.%u.%u.%u was occupied by someone without registering"},
	{IDS_DHCP_AUTO_ADD_NEW_RANGE, 	"No legal IP, assigned new IP lease range automatically!!!"},
	{IDS_IPSEC_CONFLICTING_SETTING, "IPSec connection[%s] : Local subnet is the same with remote subnet!!!"},
	{ 255 }
};
 
 
int ALARM_SendMail(char *Subject,char *Msg,char *level)
{
	REG_NET_CONF	netconf;
	REG_ALARM_CONF	conf;
	
	Config_Read(MC_NET_CONFIG, &netconf);
	Config_Read(MC_ALARM_CONFIG, &conf);
	
	return send_single_mail(netconf.hostname,Subject,Msg,conf.admin_email,conf.server,level);
}

int Add_Logs(int level, int msgid, void *arg)
{
	EVENTLOG_ENTRY  log;
	char msgstr[256];
	
	log.level = level;
	log.msgid = msgid;
	
	switch(msgid) {
	case IDS_IPSEC_CONFLICTING_SETTING :
		log.u.vpn_conn_id  = *((unsigned char *) arg);
		break;	
	case IDS_DHCP_COLLISION :
		log.collision_ip = *((unsigned long *) arg);
		break;
	}
        
        Config_Write(MC_EVENTLOG_ADD, &log);
	
	if(ALARM_Is_Enable() && (ALARM_Get_Level()<=level)) {
		char 		level_str[15];
		char 		*pstr;
		int  		mapid=0;
		u_int32_t	addr;
		LOG_STR_TABLE 	*plocale_strmap;
	
		for (plocale_strmap = log_str_map; plocale_strmap->msg_id != 255; plocale_strmap++) {
			if( plocale_strmap->msg_id == msgid )
				break;	
			mapid++;
		}
	
	
		pstr = log_str_map[mapid].map_str;
		
		switch(level) {
		case LEVEL_NORMAL:
			strcpy(level_str,"NORMAL");
			break;
		case LEVEL_WARNING:
			strcpy(level_str,"WARNING");
			break;
		case LEVEL_URGENT:
			strcpy(level_str,"URGENT");
			break;
		}	
		
		switch ( msgid ) {
		case IDS_DHCP_COLLISION:
			addr = log.collision_ip;
			sprintf(msgstr, pstr, (addr & 0xff), (addr>>8) & 0xff, (addr>>16) & 0xff, (addr>>24) & 0xff);
			break;
		case IDS_IPSEC_CONFLICTING_SETTING:
			{
			REG_IPSEC_CONF ipsecconf;
			Config_Read(MC_IPSEC_CONFIG, &ipsecconf);
			sprintf(msgstr, pstr,ipsecconf.conn[log.u.vpn_conn_id].name);
			}
			break;
		default:
			strncpy(msgstr, pstr, 256);
		}
		
			
		ALARM_SendMail("System Information", msgstr, level_str);
	}
}

