/*
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 2000 MIPS Technologies, Inc.  All rights reserved.
 * Modified for TOSHIBA boards (Nov/2001)
 * Copyright (C) 2001 Toshiba Corporation
 *
 * ########################################################################
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ########################################################################
 *
 * This is the interface to the remote debugger stub.
 *
 */

#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/serialP.h>
#include <linux/serial_reg.h>

#include <asm/serial.h>
#include <asm/io.h>

static struct irqaction kgdb_irqaction;
static struct resource kgdb_resource;

static int (*generic_putDebugChar)(char);
static char (*generic_getDebugChar)(void);
static void (*generic_setupDebugInterrupt)(void);

int putDebugChar(char c)
{
	return generic_putDebugChar(c);
}

char getDebugChar(void) 
{
	return generic_getDebugChar();
}

void setupDebugInterrupt(void) 
{
	if (generic_setupDebugInterrupt)
		generic_setupDebugInterrupt();
}

static void do_stop_command(struct pt_regs *regs)
{
	extern void set_async_breakpoint(unsigned int epc);
	extern void breakpoint(void);
	if (!user_mode(regs))
		set_async_breakpoint(read_32bit_cp0_register(CP0_EPC));
	else
		breakpoint();
}

#if defined(CONFIG_TOSHIBA_SDB) || defined(CONFIG_TOSHIBA_TX4927EVB)

static struct serial_state rs_kgdb_table[2];
#define NR_PORTS	(sizeof(rs_kgdb_table)/sizeof(struct serial_state))

static struct async_struct kdb_port_info = {0};
static int kgdb_baud = 9600;

static __inline__ unsigned int serial_in(struct async_struct *info, int offset)
{
	return inb(info->port + offset);
}

static __inline__ void serial_out(struct async_struct *info, int offset,
				int value)
{
	outb(value, info->port+offset);
}

static void rs_debug_chipinit(int intson, int baud)
{
	int t;

	/*
	 * Clear all interrupts
	 */
	serial_in(&kdb_port_info, UART_LSR);
	serial_in(&kdb_port_info, UART_RX);
	serial_in(&kdb_port_info, UART_IIR);
	serial_in(&kdb_port_info, UART_MSR);

	/*
	 * Now, initialize the UART 
	 */
	serial_out(&kdb_port_info, UART_LCR, UART_LCR_WLEN8);	/* reset DLAB */
	if (kdb_port_info.flags & ASYNC_FOURPORT) {
		kdb_port_info.MCR = UART_MCR_DTR | UART_MCR_RTS;
		t = UART_MCR_DTR | UART_MCR_OUT1;
	} else {
		kdb_port_info.MCR 
			= UART_MCR_DTR | UART_MCR_RTS | UART_MCR_OUT2;
		t = UART_MCR_DTR | UART_MCR_RTS;
	}

	if (!intson)
		kdb_port_info.MCR = t;		/* no interrupts, please */
	serial_out(&kdb_port_info, UART_MCR, kdb_port_info.MCR);
	if (intson) {
		/* enable receiver data interrupt */
		serial_out(&kdb_port_info, UART_IER, UART_IER_RDI);
	}
	
	/*
	 * and set the speed of the serial port
	 */

	t = kdb_port_info.state->baud_base / baud;
	/* set DLAB */
	serial_out(&kdb_port_info, UART_LCR, UART_LCR_WLEN8 | UART_LCR_DLAB);
	serial_out(&kdb_port_info, UART_DLL, t & 0xff);/* LS of divisor */
	serial_out(&kdb_port_info, UART_DLM, t >> 8);  /* MS of divisor */
	/* reset DLAB */
	serial_out(&kdb_port_info, UART_LCR, UART_LCR_WLEN8);
}

static int rs_putDebugChar(char c)
{

	if (!kdb_port_info.state) { 	/* need to init device first */
		return 0;
	}

	while ((serial_in(&kdb_port_info, UART_LSR) & UART_LSR_THRE) == 0)
		;

	serial_out(&kdb_port_info, UART_TX, c);

	return 1;
}

static char rs_getDebugChar(void)
{
	if (!kdb_port_info.state) { 	/* need to init device first */
		return 0;
	}

	while (!(serial_in(&kdb_port_info, UART_LSR) & 1))
		;

	return(serial_in(&kdb_port_info, UART_RX));
}

/*
 * This is the serial driver's interrupt routine for a debug port
 */
static void rs_interrupt_debug(int irq, void *dev_id, struct pt_regs *regs)
{
	int status;
	unsigned char ch;
	
	/* Look for kgdb 'stop' character, consult the gdb
	 * documentation for remote target debugging and
	 * arch/sparc/kernel/sparc-stub.c (or arch/mips/kernel/gdb-stub.c)
	 * to see how all this works.
	 */
	status = serial_in(&kdb_port_info, UART_LSR);
	if (status & UART_LSR_DR) {
		ch = serial_in(&kdb_port_info, UART_RX);
		if ((ch =='\003'))
			do_stop_command(regs);
	}
	/* clear interrupt source... (required?) */
	while (!(serial_in(&kdb_port_info, UART_IIR) & UART_IIR_NO_INT)) {
		if (serial_in(&kdb_port_info, UART_LSR) & UART_LSR_DR) {
			(void)serial_in(&kdb_port_info, UART_RX);
		}
		(void)serial_in(&kdb_port_info, UART_MSR);
	}
}


static void __init rs_setupDebugInterrupt(void)
{
	unsigned long flags;

	/* Enable interrupts because we now want to receive the
	 * 'control-c' character from the client attached to us
	 * asynchronously.
	 */
	if (!kdb_port_info.state)
		return;
	kgdb_irqaction.handler = rs_interrupt_debug;
	kgdb_irqaction.flags = SA_INTERRUPT;
	kgdb_irqaction.name = "serial(debug)";
	save_and_cli(flags);
	if (setup_irq(kdb_port_info.state->irq, &kgdb_irqaction) == 0) {
		rs_debug_chipinit(1, kgdb_baud);
		printk(KERN_INFO "ttyS%02d: enable interrupts for C-c.\n",
		       kdb_port_info.line);
	}
	restore_flags(flags);
}

void __init rs_kgdb_hook(int tty_no, int baud)
{
	struct serial_state *ser = &rs_kgdb_table[tty_no];

	if (tty_no >= NR_PORTS)
		return;
	kdb_port_info.state = ser;
	kdb_port_info.magic = SERIAL_MAGIC;
	kdb_port_info.port = ser->port;
	kdb_port_info.flags = ser->flags;
	kdb_port_info.line = tty_no;

	kgdb_baud = baud;

	/* prevent initialization by driver */
	kgdb_resource.name = "serial(debug)";
	kgdb_resource.start = ser->port;
	kgdb_resource.end = ser->port + 8 - 1;
	kgdb_resource.flags = IORESOURCE_IO | IORESOURCE_BUSY;
	request_resource(&ioport_resource, &kgdb_resource);

	rs_debug_chipinit(0, baud);

	generic_putDebugChar = rs_putDebugChar;
	generic_getDebugChar = rs_getDebugChar;
	generic_setupDebugInterrupt = rs_setupDebugInterrupt;
}

int __init early_serial_kgdb_setup(struct serial_struct *req)
{
	int i = req->line;

	if (i >= NR_PORTS)
		return(-ENOENT);
	rs_kgdb_table[i].magic = 0;
	rs_kgdb_table[i].baud_base = req->baud_base;
	rs_kgdb_table[i].port = req->port;
	rs_kgdb_table[i].irq = req->irq;
	rs_kgdb_table[i].flags = req->flags;
	rs_kgdb_table[i].close_delay = req->close_delay;
	rs_kgdb_table[i].io_type = req->io_type;
	rs_kgdb_table[i].hub6 = req->hub6;
	rs_kgdb_table[i].iomem_base = req->iomem_base;
	rs_kgdb_table[i].iomem_reg_shift = req->iomem_reg_shift;
	rs_kgdb_table[i].type = req->type;
	rs_kgdb_table[i].xmit_fifo_size = req->xmit_fifo_size;
	rs_kgdb_table[i].custom_divisor = req->custom_divisor;
	rs_kgdb_table[i].closing_wait = req->closing_wait;
	return(0);
}

#endif

#if defined(CONFIG_TOSHIBA_JMR3927) || defined(CONFIG_TOSHIBA_TX4927EVB) || \
	defined(CONFIG_TOSHIBA_RBTX4927) || defined(CONFIG_TOSHIBA_RBTX4925)

#include <asm/txx927.h>

#define TXX927_NR_PORTS	2
static int txx927_kdb_port = -1;

/*
 * Hardware specific serial port structure
 */
static struct rs_kgdb_port { 	
	unsigned long		base;
	int			irq;
	int			baud_base;
	int			flags;
} rs_kgdb_ports[TXX927_NR_PORTS]; 

static inline struct txx927_sio_reg *sio_reg(struct rs_kgdb_port *port)
{
	return (struct txx927_sio_reg *)port->base;
}

int txx927_rs_putDebugChar(char c)
{
	struct rs_kgdb_port *port = &rs_kgdb_ports[txx927_kdb_port];
	if (txx927_kdb_port < 0) { 	/* need to init device first */
		return 0;
	}

	while (!(sio_reg(port)->cisr & TXx927_SICISR_TXALS))
		;
	sio_reg(port)->tfifo = c;

	return 1;
}

char txx927_rs_getDebugChar(void)
{
	int dicr;
	char c;
	struct rs_kgdb_port *port = &rs_kgdb_ports[txx927_kdb_port];
	if (txx927_kdb_port < 0) { 	/* need to init device first */
		return 0;
	}
	/* diable RX int. */
	dicr = sio_reg(port)->dicr;
	sio_reg(port)->dicr = 0;

	/* read char */
	while (sio_reg(port)->disr & TXx927_SIDISR_UVALID)
		;
	c = sio_reg(port)->rfifo;

	/* clear RX int. status */
	sio_reg(port)->disr &= ~TXx927_SIDISR_RDIS;
	/* enable RX int. */
	sio_reg(port)->dicr = dicr;

	return c;
}

static void txx927_rs_interrupt_debug(int irq, void *dev_id, struct pt_regs *regs)
{
	struct rs_kgdb_port *port = &rs_kgdb_ports[txx927_kdb_port];
	int status;
	unsigned char ch;
	
	/* Look for kgdb 'stop' character, consult the gdb
	 * documentation for remote target debugging and
	 * arch/sparc/kernel/sparc-stub.c (or arch/mips/kernel/gdb-stub.c)
	 * to see how all this works.
	 */
	status = sio_reg(port)->disr;
	if (!(status & TXx927_SIDISR_UVALID)) {
		ch = sio_reg(port)->rfifo;
		if ((ch =='\003'))
			do_stop_command(regs);
	}
	/* clear interrupt source... (required?) */
	while (sio_reg(port)->disr & (TXx927_SIDISR_RDIS|TXx927_SIDISR_TOUT)) {
		if (!(sio_reg(port)->disr & TXx927_SIDISR_UVALID)) {
			(void)sio_reg(port)->rfifo;
		}
		sio_reg(port)->disr &= ~(TXx927_SIDISR_RDIS|TXx927_SIDISR_TOUT);
	}
}

static void __init txx927_rs_setupDebugInterrupt(void)
{
	unsigned long flags;
	struct rs_kgdb_port *port = &rs_kgdb_ports[txx927_kdb_port];

	/* Enable interrupts because we now want to receive the
	 * 'control-c' character from the client attached to us
	 * asynchronously.
	 */
	if (txx927_kdb_port < 0)
		return;
	kgdb_irqaction.handler = txx927_rs_interrupt_debug;
	kgdb_irqaction.flags = SA_INTERRUPT;
	kgdb_irqaction.name = "serial_txx927(debug)";
	save_and_cli(flags);
	if (setup_irq(port->irq, &kgdb_irqaction) == 0) {
		/* clear FIFO */
		sio_reg(port)->fcr |=
			TXx927_SIFCR_TFRST | TXx927_SIFCR_RFRST |
			TXx927_SIFCR_FRSTE;
		sio_reg(port)->fcr &=
			~(TXx927_SIFCR_TFRST | TXx927_SIFCR_RFRST |
			  TXx927_SIFCR_FRSTE);
		/* clear interrupts */
		sio_reg(port)->disr = 0;
		/* enable receiver data interrupt */
		sio_reg(port)->dicr |= TXx927_SIDICR_RIE;
		printk(KERN_INFO "ttySC%02d: enable interrupts for C-c.\n",
		       txx927_kdb_port);
	}
	restore_flags(flags);
}

void __init txx927_rs_kgdb_hook(int tty_no, int baud)
{
	struct rs_kgdb_port *port = &rs_kgdb_ports[tty_no];

	if (tty_no < 0 || tty_no >= TXX927_NR_PORTS)
		return;
	if (!port->base)
		return;
	txx927_kdb_port = tty_no;

	/*
	 * Reset the UART.
	 */
	sio_reg(port)->fcr = TXx927_SIFCR_SWRST;
	while (sio_reg(port)->fcr & TXx927_SIFCR_SWRST)
		;

	/*
	 * and set the speed of the serial port
	 * (currently hardwired to 9600 8N1
	 */

	sio_reg(port)->lcr = TXx927_SILCR_UMODE_8BIT |
		TXx927_SILCR_USBL_1BIT |
		TXx927_SILCR_SCS_IMCLK_BG;
	sio_reg(port)->bgr =
		((port->baud_base + baud / 2) / baud) |
		TXx927_SIBGR_BCLK_T0;

	/* no RTS/CTS control */
	sio_reg(port)->flcr = TXx927_SIFLCR_RTSTL_MAX /* 15 */;
	/* Enable RX/TX */
	sio_reg(port)->flcr &= ~(TXx927_SIFLCR_RSDE | TXx927_SIFLCR_TSDE);

	/* prevent initialization by driver */
	kgdb_resource.name = "serial_txx927(debug)";
	kgdb_resource.start = port->base;
	kgdb_resource.end = port->base + 36 - 1;
	kgdb_resource.flags = IORESOURCE_MEM | IORESOURCE_BUSY;
	if (request_resource(&iomem_resource, &kgdb_resource) == -EBUSY) {
		/* we must find TX4927 resource... */
		struct resource *child = iomem_resource.child;
		while (child) {
			if (request_resource(child, &kgdb_resource) == 0)
				break;
			child = child->sibling;
		}
	}

	generic_putDebugChar = txx927_rs_putDebugChar;
	generic_getDebugChar = txx927_rs_getDebugChar;
	generic_setupDebugInterrupt = txx927_rs_setupDebugInterrupt;
}

int __init early_serial_txx927_kgdb_setup(int line, unsigned long base, int irq, int baud_base)
{
	if (line >= TXX927_NR_PORTS)
		return(-ENOENT);
	rs_kgdb_ports[line].base = base;
	rs_kgdb_ports[line].irq = irq;
	rs_kgdb_ports[line].baud_base = baud_base;
	return(0);
}

#endif
