/*
 *  linux/arch/mips/toshiba-boards/generic/irq.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 *
 * $Id: irq.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/ptrace.h>
#include <linux/sched.h>
#include <linux/string.h>

#include <asm/system.h>
#include <asm/mipsregs.h>
#include <asm/addrspace.h>
#include <asm/cpu.h>
#include <asm/bootinfo.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/mct-boards/pmon.h>
#include <asm/mct-boards/irq.h>

/* irq_setup() overrides this */
int (*toshibaboards_irqdispatch)(struct pt_regs *regs);

/*
 * PCI-ISA brigde interrupt controller
 */
static int i8259_pci_irqno;

/* Explicitly generate a acknowledge cycle. Returns irq number. */
/* generic but slow... */
static int i8259_gen_iack(void)
{
	int irq;
	unsigned char isr;

	/* generate acknowledge cycle */
	outb(0x0c, 0x20);
	inb(0x20);
	/* read ISR */
	outb(0x0b, 0x20);
	isr = inb(0x20);
	outb(0x0a, 0x20);	/* restore */
	if (isr & 0x04) {
		/* generate acknowledge cycle (slave) */
		outb(0x0c, 0xa0);
		inb(0xa0);
		/* read ISR (slave) */
		outb(0x0b, 0xa0);
		isr = inb(0xa0);
		outb(0x0a, 0xa0);	/* restore */
		for (irq = 0; irq < 8; irq++)
			if (isr & (1 << irq))
				return irq + 8;
	} else {
		for (irq = 0; irq < 8; irq++)
			if (isr & (1 << irq))
				return irq;
	}
	return -1;
}

/* machine dependent setup routine override this */
int (*toshibaboards_gen_iack)(void) = i8259_gen_iack;

/* If this is an interrupt from PCI-ISA bridge, convert to i8259 IRQ number. */
int toshibaboards_i8259_irqroute(int irq)
{
	if (irq != i8259_pci_irqno)
		return irq;	/* return unchanged number */
	irq = toshibaboards_gen_iack();
	if (irq >= NR_ISA_IRQS) {
		printk(KERN_ERR "gen_iack:invalid irq (%d)\n", irq);
		return -1;
	}
	return irq;
}

static struct irqaction i8259_action = {
	no_action, SA_INTERRUPT|SA_SHIRQ, 0, "cascade(i8259)", NULL, NULL,
};

int __init
toshibaboards_i8259_irq_setup(int irq)
{
	int err;
	unsigned long flags;
	extern void init_i8259_irqs (void);

	if (i8259_pci_irqno)
		return 0;

	save_and_cli(flags);

	init_i8259_irqs();
	err = setup_irq(irq, &i8259_action);
	if (!err) {
		printk(KERN_INFO "PCI-ISA bridge PIC (irq %d)\n", irq);
		i8259_pci_irqno = irq;
	}
	restore_flags(flags);
	return err;
}


#ifdef CONFIG_TX_BRANCH_LIKELY_BUG_WORKAROUND
static int tx_branch_likely_bug_count = 0;
static int have_tx_branch_likely_bug = 0;
void tx_branch_likely_bug_fixup(struct pt_regs *regs)
{
	/* TX39/49-BUG: Under this condition, the insn in delay slot
           of the branch likely insn is executed (not nullified) even
           the branch condition is false. */
	if (!have_tx_branch_likely_bug)
		return;
	if ((regs->cp0_epc & 0xfff) == 0xffc &&
	    KSEGX(regs->cp0_epc) != KSEG0 &&
	    KSEGX(regs->cp0_epc) != KSEG1) {
		unsigned int insn;
		if (get_user(insn, (unsigned int *)(regs->cp0_epc - 4)))
			return;
		/* beql,bnel,blezl,bgtzl */
		/* bltzl,bgezl,blezall,bgezall */
		/* bczfl, bcztl */
		if ((insn & 0xf0000000) == 0x50000000 ||
		    (insn & 0xfc0e0000) == 0x04020000 ||
		    (insn & 0xf3fe0000) == 0x41020000) {
			regs->cp0_epc -= 4;
			tx_branch_likely_bug_count++;
			printk(KERN_INFO
			       "fix branch-likery bug in %s (insn %08x)\n",
			       current->comm, insn);
		}
	}
}
#endif

#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
static int tx_fpu_c_bug_count = 0;
static int have_tx_fpu_c_bug = 0;
void tx_fpu_c_bug_fixup(struct pt_regs *regs)
{
	/* TX49-BUG: If FP ALU insn were executed immediately after
	   C.cond.d insn and the FP ALU insn ware interrupted, the
	   result of the C.cond.d may not be correct. */
	if (!have_tx_fpu_c_bug)
		return;
	if ((regs->cp0_cause & CAUSEF_BD) == 0 &&
	    KSEGX(regs->cp0_epc) != KSEG0 &&
	    KSEGX(regs->cp0_epc) != KSEG1) {
		unsigned int insn, insn2;
		extern void save_fp(struct task_struct *);
		extern void restore_fp(struct task_struct *);
		extern int fpu_emulator_cop1Handler(struct pt_regs *);
		if (get_user(insn, (unsigned int *)(regs->cp0_epc - 4)))
			return;
		/* c.cond.d */
		if ((insn & 0xffe00030) != 0x46200030)
			return;
		if (get_user(insn2, (unsigned int *)regs->cp0_epc))
			return;
		/* COP1(s,d) */
		if ((insn2 & 0xff000000) != 0x46000000)
			return;
		/* abs, add, ceil, cvt, floor, mov, neg, round, sub, trunc */
		if ((2 <= (insn2 & 0x3f) && (insn2 & 0x3f) < 5) ||
		    16 <= (insn2 & 0x3f))
			return;
		printk(KERN_WARNING
		       "fix FPU bug in %s epc == %08lx, insn == %08x\n",
		       current->comm, regs->cp0_epc, insn2);
		tx_fpu_c_bug_count++;
		regs->cp0_epc -= 4;
		/* emulate C.cond.d insn.  see do_fpe() */
		save_fp(current);
		fpu_emulator_cop1Handler(regs);
		current->thread.fpu.soft.sr &= ~FPU_CSR_UNI_X;
		restore_fp(current);
	}
}
#endif

#ifdef CONFIG_REMOTE_DEBUG
extern char *prom_getcmdline(void);
extern void rs_kgdb_hook(int, int);
extern void txx927_rs_kgdb_hook(int, int);
extern void breakpoint(void);
extern void set_debug_traps(void);
extern void setupDebugInterrupt(void);
#endif

extern asmlinkage void toshibaboards_IRQ(void);

void (*irq_setup)(void);

void __init init_IRQ(void)
{
#ifdef CONFIG_REMOTE_DEBUG
	char *ctype;
	int line = 0;
	int baud = 9600;
	int use_gdb_on_std_serial = 0;
	int use_gdb_on_txx927_serial = 0;
#endif
#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
	unsigned long revid;
#endif

#ifdef CONFIG_TX_BRANCH_LIKELY_BUG_WORKAROUND
	/* TX3927A, TX49/H */
	switch (mips_cpu.processor_id & 0xffff) {
	case (PRID_IMP_TX39|PRID_REV_TX3927):
	case (PRID_IMP_TX49|0x2d11): /* TX49/H */
		have_tx_branch_likely_bug = 1;
		break;
	}
#endif
#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
	switch (mips_cpu.processor_id & 0xffff) {
	case (PRID_IMP_TX49|0x2d11): /* TX4955 (TX49/H) */
	case (PRID_IMP_TX49|0x2d20): /* TX4955A (TX49/H2)*/
	case (PRID_IMP_TX49|0x2d21): /* TX4927R1 (TX49/H2) */
	case (PRID_IMP_TX49|0x2d22): /* TX4927R2, TX4925, etc (TX49/H2) */
	case (PRID_IMP_TX49|0x2d23):
		/* TX4927#4 does not have this bug. */
		if (mips_machtype == MACH_TOSHIBA_TX4927EVB ||
		    mips_machtype == MACH_TOSHIBA_RBTX4927) {
#ifdef __BIG_ENDIAN
			revid = *(volatile unsigned long *)0xff1fe00c;
#else
			revid = *(volatile unsigned long *)0xff1fe008;
#endif
			if ((revid & 0xffff0000) == 0x49270000 &&
			    (revid & 0x000000ff) >= 0x40)
				break;
		}
		have_tx_fpu_c_bug = 1;
		break;
	}
#endif

	init_generic_irq();
	set_except_vector(0, toshibaboards_IRQ);
	irq_setup();	/* board specific setup */

#if 0 /* EXCEPTION_DEBUG */
/* XXX debug buggy CPU... (see int-handler.S also) */
	{
		extern char excdbg_vec0_handler, excdbg_vec3_handler;
#if defined(CONFIG_CPU_TX49XX)
		memcpy((void *)KSEG0, &excdbg_vec0_handler, 0x80);
		memcpy((void *)(KSEG0 + 0x180), &excdbg_vec3_handler, 0x80);
#endif
#if defined(CONFIG_CPU_TX39XX)
		memcpy((void *)KSEG0, &excdbg_vec0_handler, 0x80);
		memcpy((void *)(KSEG0 + 0x80), &excdbg_vec3_handler, 0x80);
#endif
		flush_icache_range(KSEG0, KSEG0 + 0x200);
	}
#endif /* EXCEPTION_DEBUG */
#ifdef CONFIG_REMOTE_DEBUG
	for (ctype = prom_getcmdline(); *ctype; ctype++) {
		/*
		 * If we have both standard and txx927 serial,
		 * use ttyS0,1 for Standard-SIO, ttySC0,1 for CPU-SIO.
		 */
		if ((strncmp(ctype, "kgdb=ttyd", 9) == 0 ||
		     strncmp(ctype, "kgdb=ttyS", 9) == 0) &&
		    (ctype[9] == '0' || ctype[9] == '1')) {
			ctype += 9;
#if defined(CONFIG_SERIAL)
			use_gdb_on_std_serial = 1;
#elif defined(CONFIG_SERIAL_TXX927)
			use_gdb_on_txx927_serial = 1;
#endif
			break;
		}
#if defined(CONFIG_SERIAL) && defined(CONFIG_SERIAL_TXX927)
		if ((strncmp(ctype, "kgdb=ttydc", 10) == 0 ||
		     strncmp(ctype, "kgdb=ttySC", 10) == 0) &&
		    (ctype[10] == '0' || ctype[10] == '1')) {
			ctype += 10;
			use_gdb_on_txx927_serial = 1;
			break;
		}
#endif
	}
	if (use_gdb_on_std_serial || use_gdb_on_txx927_serial) {
		line = *ctype - '0';
		ctype++;
		if (*ctype == ',') {
			ctype++;
			baud = simple_strtoul(ctype, NULL, 10);
		}

		printk("KGDB: Using serial line /dev/%s%d for "
		       "session (%dbps)\n",
#if defined(CONFIG_SERIAL) && defined(CONFIG_SERIAL_TXX927)
		       use_gdb_on_txx927_serial ? "ttySC" : "ttyS",
#else
		       "ttyS",
#endif
		       line, baud);
		pmon_printf("Now you can start KGDB session (%dbps).\n", baud);
		/* flush FIFO :-) */
		pmon_printf("\r\r\r\r\r\r\r\r\r");
	}
#ifdef CONFIG_SERIAL
	if (use_gdb_on_std_serial) {
		rs_kgdb_hook(line, baud);
	}
#endif
#ifdef CONFIG_SERIAL_TXX927
	if (use_gdb_on_txx927_serial) {
		txx927_rs_kgdb_hook(line, baud);
	}
#endif
	if (use_gdb_on_std_serial || use_gdb_on_txx927_serial) {
		set_debug_traps();
		breakpoint();
		setupDebugInterrupt();
	} else {
		set_pmon_debug_traps();
	}
#else /* CONFIG_REMOTE_DEBUG */
	set_pmon_debug_traps();
#endif /* CONFIG_REMOTE_DEBUG */
}
