/*
 * linux/arch/mips/toshiba-boards/generic/irq_tx3927.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 *
 * $Id: irq_tx3927.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */

/*
 * TX3927 defines 32 IRQs.
 *
 * This file exports one function:
 *	tx3927_irq_init(int irq_base);
 */

#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/ptrace.h>

#include <asm/tx3927.h>

#define irc_dlevel	0
#define irc_elevel	1
static unsigned char irc_level[TX3927_NUM_IR] = {
	5, 5, 5, 5, 5, 5,	/* INT[5:0] */
	7, 7,			/* SIO */
	5, 5, 5, 0, 0,		/* DMA, PIO, PCI */
	6, 6, 6			/* TMR */
};

static int tx3927_irq_base = -1;

static void tx3927_irq_enable(unsigned int irq)
{
	int irq_nr = irq - tx3927_irq_base;
	volatile unsigned long *ilrp = &tx3927_ircptr->ilr[irq_nr / 2];
	if (irq_nr & 1)
		*ilrp = (*ilrp & 0x00ff) | (irc_level[irq_nr] << 8);
	else
		*ilrp = (*ilrp & 0xff00) | irc_level[irq_nr];
	/* update IRCSR */
	tx3927_ircptr->imr = 0;
	tx3927_ircptr->imr = irc_elevel;
}

static void tx3927_irq_disable(unsigned int irq)
{
	int irq_nr = irq - tx3927_irq_base;
	volatile unsigned long *ilrp = &tx3927_ircptr->ilr[irq_nr / 2];
	if (irq_nr & 1)
		*ilrp = (*ilrp & 0x00ff) | (irc_dlevel << 8);
	else
		*ilrp = (*ilrp & 0xff00) | irc_dlevel;
	/* update IRCSR */
	tx3927_ircptr->imr = 0;
	tx3927_ircptr->imr = irc_elevel;
}

static unsigned int tx3927_irq_startup(unsigned int irq)
{
	tx3927_irq_enable(irq);
	return 0;
}
#define	tx3927_irq_shutdown	tx3927_irq_disable

static void tx3927_irq_ack(unsigned int irq)
{
	tx3927_irq_disable(irq);
}
static void tx3927_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		tx3927_irq_enable(irq);
}

static hw_irq_controller tx3927_irq_controller = {
	typename:	"tx3927_irq",
	startup:	tx3927_irq_startup,
	shutdown:	tx3927_irq_shutdown,
	enable:		tx3927_irq_enable,
	disable:	tx3927_irq_disable,
	ack:		tx3927_irq_ack,
	end:		tx3927_irq_end,
	set_affinity:	NULL,
};

void __init
tx3927_irq_init(int irq_base)
{
	int i;

	for (i= irq_base; i< irq_base + TX3927_NUM_IR; i++) {
		/* INT[4], INT[5] pins are multiplexed. */
		if ((i - irq_base == TX3927_IR_INT(4) ||
		     i - irq_base == TX3927_IR_INT(5)) &&
		    (tx3927_ccfgptr->pcfg & TX3927_PCFG_SELSIOC(0)))
			continue;
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &tx3927_irq_controller;
	}

	tx3927_irq_base = irq_base;

	/* enable interrupt control */
	tx3927_ircptr->cer = TX3927_IRCER_ICE;
	tx3927_ircptr->imr = irc_elevel;
}

int tx3927_irq_to_irq(int irq)
{
	if (irq < 0 || irq > TX3927_NUM_IR)
		return -1;

	return irq + tx3927_irq_base;
}

EXPORT_SYMBOL(tx3927_irq_to_irq);
