/*
 * linux/arch/mips/toshiba-boards/generic/irq_jmi39io2isac.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 *
 * $Id: irq_jmi39io2isac.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */

/*
 * JMI39IO2 ISAC defines 8 IRQs.
 *
 * This file exports one function:
 *	jmi39io2_isac_irq_init();
 */

#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/ptrace.h>

#include <asm/system.h>
#include <asm/toshiba-boards/jmi39io2.h>

static void jmi39io2_isac_irq_enable(unsigned int irq)
{
	int irq_nr = irq - jmi39io2_isac_irq_base;
	/* 0: mask */
	unsigned char imask =
		jmi39io2_isac_reg_in(JMI39IO2_ISAC_INTM_ADDR(jmi39io2_iob_base));
	unsigned int bit  = 1 << irq_nr;
	jmi39io2_isac_reg_out(imask | bit, JMI39IO2_ISAC_INTM_ADDR(jmi39io2_iob_base));
	wbflush();
}
static void jmi39io2_isac_irq_disable(unsigned int irq)
{
	int irq_nr = irq - jmi39io2_isac_irq_base;
	/* 0: mask */
	unsigned char imask =
		jmi39io2_isac_reg_in(JMI39IO2_ISAC_INTM_ADDR(jmi39io2_iob_base));
	unsigned int bit  = 1 << irq_nr;
	jmi39io2_isac_reg_out(imask & ~bit, JMI39IO2_ISAC_INTM_ADDR(jmi39io2_iob_base));
	wbflush();
}
static unsigned int jmi39io2_isac_irq_startup(unsigned int irq)
{
	jmi39io2_isac_irq_enable(irq);
	return 0;
}
#define	jmi39io2_isac_irq_shutdown	jmi39io2_isac_irq_disable

static void jmi39io2_isac_irq_ack(unsigned int irq)
{
	jmi39io2_isac_irq_disable(irq);
}
static void jmi39io2_isac_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		jmi39io2_isac_irq_enable(irq);
}

static hw_irq_controller jmi39io2_isac_irq_controller = {
	typename:	"JMI39IO2-ISAC",
	startup:	jmi39io2_isac_irq_startup,
	shutdown:	jmi39io2_isac_irq_shutdown,
	enable:		jmi39io2_isac_irq_enable,
	disable:	jmi39io2_isac_irq_disable,
	ack:		jmi39io2_isac_irq_ack,
	end:		jmi39io2_isac_irq_end,
	set_affinity:	NULL,
};

/* for standard irq mappings */
static int irqmap[16] = {
	-1, -1, -1, JMI39IO2_ISAC_INTB_IRQ3,
	JMI39IO2_ISAC_INTB_IRQ4, JMI39IO2_ISAC_INTB_IRQ5, -1, -1,
	-1, -1, JMI39IO2_ISAC_INTB_IRQ10, -1,
	JMI39IO2_ISAC_INTB_IRQ12, -1, -1, -1
};
static int irqrmap[JMI39IO2_NR_IRQ_ISAC] = {
	5, -1, -1, 4, 12, 3, 10, -1
};

static void jmi39io2_isac_stdirq_enable(unsigned int irq)
{
	jmi39io2_isac_irq_enable(irqmap[irq] + jmi39io2_isac_irq_base);
}
static void jmi39io2_isac_stdirq_disable(unsigned int irq)
{
	jmi39io2_isac_irq_disable(irqmap[irq] + jmi39io2_isac_irq_base);
}
static unsigned int jmi39io2_isac_stdirq_startup(unsigned int irq)
{
	return jmi39io2_isac_irq_startup(irqmap[irq] + jmi39io2_isac_irq_base);
}
#define	jmi39io2_isac_stdirq_shutdown	jmi39io2_isac_stdirq_disable

static void jmi39io2_isac_stdirq_ack(unsigned int irq)
{
	jmi39io2_isac_irq_ack(irqmap[irq] + jmi39io2_isac_irq_base);
}
static void jmi39io2_isac_stdirq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		jmi39io2_isac_irq_enable(irqmap[irq] + jmi39io2_isac_irq_base);
}

static hw_irq_controller jmi39io2_isac_stdirq_controller = {
	typename:	"JMI39IO2-ISAC",
	startup:	jmi39io2_isac_stdirq_startup,
	shutdown:	jmi39io2_isac_stdirq_shutdown,
	enable:		jmi39io2_isac_stdirq_enable,
	disable:	jmi39io2_isac_stdirq_disable,
	ack:		jmi39io2_isac_stdirq_ack,
	end:		jmi39io2_isac_stdirq_end,
	set_affinity:	NULL,
};

void __init
jmi39io2_isac_irq_init(void)
{
	int i;

	for (i = jmi39io2_isac_irq_base; i < jmi39io2_isac_irq_base + JMI39IO2_NR_IRQ_ISAC; i++) {
		if (irqrmap[i - jmi39io2_isac_irq_base] >= 0)
			continue; /* jmi39io2_isac_stdirq_controller */
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &jmi39io2_isac_irq_controller;
	}
	for (i = 0; i < 16; i++) {
		if (irqmap[i] < 0)
			continue;
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &jmi39io2_isac_stdirq_controller;
	}

	/* mask all ISAC interrupts */
	jmi39io2_isac_reg_out(0, JMI39IO2_ISAC_INTM_ADDR(jmi39io2_iob_base));
	/* setup ISAC interrupt mode (ISAIRQ3,ISAIRQ5:Low Active ???) */
	jmi39io2_isac_reg_out(JMI39IO2_ISAC_INTF_IRQ3|JMI39IO2_ISAC_INTF_IRQ5,
			      JMI39IO2_ISAC_INTM_ADDR(jmi39io2_iob_base));
}

/* find first bit set */
static inline int ffs8(unsigned char x)
{
	int r = 1;

	if (!x)
		return 0;
	if (!(x & 0xf)) {
		x >>= 4;
		r += 4;
	}
	if (!(x & 3)) {
		x >>= 2;
		r += 2;
	}
	if (!(x & 1)) {
		x >>= 1;
		r += 1;
	}
	return r;
}

int jmi39io2_isac_irqroute(void)
{
	unsigned char istat = jmi39io2_isac_reg_in(JMI39IO2_ISAC_INTS2_ADDR(jmi39io2_iob_base));
	int irq = ffs8(istat);
	if (irq == 0)
		return -1;
	irq--;
	if (irqrmap[irq] >= 0)
		return irqrmap[irq];
	return jmi39io2_isac_irq_base + irq;
}
