/*
 * linux/arch/mips/toshiba-boards/generic/irq_tx4925.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 *
 * $Id: irq_tx4925.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */

/*
 * TX4925 defines 32 IRQs.
 *
 * This file exports one function:
 *	tx4925_irq_init(int irq_base);
 */

#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/ptrace.h>

#include <asm/tx4925.h>

#define irc_dlevel	0
#define irc_elevel	1
static unsigned char irc_level[TX4925_NUM_IR] = {
	7, 7,			/* reserved, WTOERR */
	5, 5, 5, 5, 5, 5, 5, 5,	/* INT[7:0] */
	5, 5,			/* reserved, NAND */
	7, 7,			/* SIO */
	5, 5, 5, 5, 5, 5, 5,	/* DMA[3:0], PIO, PDMAC, PCIC */
	6, 6, 6,		/* TMR[2:0] */
	5, 5,			/* SPI, RTC */
	5, 5, 5,		/* ACLC, ACLCPME, CHI */
	7, 7,			/* PCIERR, PCIPME */
	0
};

static int tx4925_irq_base = -1;

static void tx4925_irq_enable(unsigned int irq)
{
	int irq_nr = irq - tx4925_irq_base;
	volatile unsigned long *ilrp = &tx4925_ircptr->ilr[(irq_nr % 16 ) / 2];
	int ofs = (irq_nr >= 16 ? 16 : 0) + ((irq_nr & 1) ? 8 : 0);
	*ilrp = (*ilrp & ~(0xff << ofs)) | (irc_level[irq_nr] << ofs);
	/* update IRCSR */
	tx4925_ircptr->imr = 0;
	tx4925_ircptr->imr = irc_elevel;
}

static void tx4925_irq_disable(unsigned int irq)
{
	int irq_nr = irq - tx4925_irq_base;
	volatile unsigned long *ilrp = &tx4925_ircptr->ilr[(irq_nr % 16) / 2];
	int ofs = (irq_nr >= 16 ? 16 : 0) + ((irq_nr & 1) ? 8 : 0);
	*ilrp = (*ilrp & ~(0xff << ofs)) | (irc_dlevel << ofs);
	/* update IRCSR */
	tx4925_ircptr->imr = 0;
	tx4925_ircptr->imr = irc_elevel;
}

static unsigned int tx4925_irq_startup(unsigned int irq)
{
	tx4925_irq_enable(irq);
	return 0;
}
#define	tx4925_irq_shutdown	tx4925_irq_disable

static void tx4925_irq_ack(unsigned int irq)
{
	tx4925_irq_disable(irq);
}
static void tx4925_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		tx4925_irq_enable(irq);
}

static hw_irq_controller tx4925_irq_controller = {
	typename:	"tx4925_irq",
	startup:	tx4925_irq_startup,
	shutdown:	tx4925_irq_shutdown,
	enable:		tx4925_irq_enable,
	disable:	tx4925_irq_disable,
	ack:		tx4925_irq_ack,
	end:		tx4925_irq_end,
	set_affinity:	NULL,
};

void __init
tx4925_irq_init(int irq_base)
{
	int i;

	for (i= irq_base; i< irq_base + TX4925_NUM_IR; i++) {
		/* INT[2:7] pins are multiplexed. */
		unsigned long pcfg = tx4925_ccfgptr->pcfg;
		if ((i - irq_base == TX4925_IR_INT(2) ||
		     i - irq_base == TX4925_IR_INT(3)) &&
		    ((pcfg & (TX4925_PCFG_SELSIOC(0)|TX4925_PCFG_SELSIO(0))) ==
		     (TX4925_PCFG_SELSIOC(0)|TX4925_PCFG_SELSIO(0))))
			continue;
		if ((i - irq_base == TX4925_IR_INT(4) ||
		     i - irq_base == TX4925_IR_INT(5)) &&
		    ((pcfg & (TX4925_PCFG_SELACLC|TX4925_PCFG_SELNAND)) ||
		     (pcfg & (TX4925_PCFG_SELSIOC(1)|TX4925_PCFG_SELSIO(1))) ==
		     (TX4925_PCFG_SELSIOC(1)|TX4925_PCFG_SELSIO(1))))
			continue;
		if ((i - irq_base == TX4925_IR_INT(6) ||
		     i - irq_base == TX4925_IR_INT(7)) &&
		    ((tx4925_ccfgptr->ccfg & TX4925_CCFG_PCTRCE) ||
		     (pcfg & (TX4925_PCFG_SELCARD(1)|TX4925_PCFG_SELCE(1)))))
			continue;
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &tx4925_irq_controller;
	}

	tx4925_irq_base = irq_base;

	/* enable interrupt control */
	tx4925_ircptr->cer = TX4925_IRCER_ICE;
	tx4925_ircptr->imr = irc_elevel;
}

int tx4925_irq_to_irq(int irq)
{
	if (irq < 0 || irq > TX4925_NUM_IR)
		return -1;

	return irq + tx4925_irq_base;
}

EXPORT_SYMBOL(tx4925_irq_to_irq);
