/*
 *  linux/arch/mips/toshiba-boards/generic/pci.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 *
 * $Id: pci.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/version.h>
#include <linux/pci.h>
#include <asm/system.h>
#include <asm/mipsregs.h>
#include <asm/io.h>
#include <asm/toshiba-boards/pci.h>
#include <asm/toshiba-boards/irq.h>

static int toshibaboards_pci66_check(void)
{
	struct pci_dev *dev;
	u16 stat;
	int cap66 = -1;

	/* check 66MHz capability */
	pci_for_each_dev(dev) {
		if (cap66 < 0)
			cap66 = 1;
		if (cap66) {
			pci_read_config_word(dev, PCI_STATUS, &stat);
			if (!(stat & PCI_STATUS_66MHZ)) {
				printk(KERN_DEBUG "PCI: %02x:%02x not 66MHz capable.\n",
				       dev->bus->number, dev->devfn);
				cap66 = 0;
			}
		}
	}
	return cap66 > 0;
}

static inline u8
bridge_swizzle(u8 pin, u8 slot) 
{
	return (((pin-1) + slot) % 4) + 1;
}

static u8 __init
pci_swizzle(struct pci_dev *dev, u8 *pinp)
{
	u8 pin = *pinp;

	while (dev->bus->self) {	/* Move up the chain of bridges. */
		pin = bridge_swizzle(pin, PCI_SLOT(dev->devfn));
		dev = dev->bus->self;
	}
	*pinp = pin;

	return PCI_SLOT(dev->devfn);
}

int (*toshibaboards_pci_map_irq)(struct pci_dev *dev, u8 slot, u8 pin);
void (*toshibaboards_pci66_init)(void);
struct pci_ops *toshibaboards_pci_ops;

struct resource toshibaboards_pci_io_resource =
	{ "PCI IO", 0x0000, IO_SPACE_LIMIT, IORESOURCE_IO };
struct resource toshibaboards_pci_mem_resource =
	{ "PCI mem", 0x0000, 0xffffffff, IORESOURCE_MEM };
static struct resource toshibaboards_pci_mmio_resource =
	{ "PCI MMIO", 0x0000, IO_SPACE_LIMIT, IORESOURCE_MEM|IORESOURCE_BUSY };

/* arch-dependent pci initialization */
void __init pcibios_init(void)
{
	struct pci_dev *dev;

	if (!toshibaboards_pci_ops)
		return;
	printk("PCI: Probing PCI hardware\n");

	/* reserve MMIO area */
	toshibaboards_pci_mmio_resource.start =
		virt_to_phys((void *)(toshibaboards_pci_io_resource.start + mips_io_port_base));
	toshibaboards_pci_mmio_resource.end =
		virt_to_phys((void *)(toshibaboards_pci_io_resource.end + mips_io_port_base));
	if (request_resource(&iomem_resource,
			     &toshibaboards_pci_mmio_resource))
		return;

	printk(KERN_DEBUG "PCI: IO 0x%08lx-0x%08lx MEM 0x%08lx-0x%08lx\n",
	       toshibaboards_pci_io_resource.start,
	       toshibaboards_pci_io_resource.end,
	       toshibaboards_pci_mem_resource.start,
	       toshibaboards_pci_mem_resource.end);

	pci_scan_bus(0, toshibaboards_pci_ops, NULL);

	if (toshibaboards_pci66_init) {
		if (toshibaboards_pci66_check()) {
			toshibaboards_pci66_init();
		}
	}

	pci_assign_unassigned_resources();

	pci_fixup_irqs(pci_swizzle, toshibaboards_pci_map_irq);

	pci_for_each_dev(dev) {
#if 1 /* XXX PCI */
		unsigned char line_size, bist, latency;
		line_size = 32 / 4;
		pci_write_config_byte(dev, PCI_CACHE_LINE_SIZE, line_size);
		pci_read_config_byte(dev, PCI_CACHE_LINE_SIZE, &line_size);
		pci_write_config_byte(dev, PCI_LATENCY_TIMER, 64);
		pci_read_config_byte(dev, PCI_LATENCY_TIMER, &latency);
		printk(KERN_DEBUG "PCI: %02x:%02x LINE %02x LATENCY %02x\n",
		       dev->bus->number, dev->devfn, line_size, latency);

		/* Do build-in self test */
		pci_read_config_byte(dev, PCI_BIST, &bist);
		if (bist & PCI_BIST_CAPABLE) {
			printk(KERN_INFO "PCI: %02x:%02x BIST...",
			       dev->bus->number, dev->devfn);
			pci_write_config_byte(dev, PCI_BIST, PCI_BIST_START);
			do {
				pci_read_config_byte(dev, PCI_BIST, &bist);
			} while (bist & PCI_BIST_START);
			bist &= PCI_BIST_CODE_MASK;
			if (bist)
				printk("failed. (0x%x)\n", bist);
			else
				printk("OK.\n");
		}
#endif
	}
}

void __init pcibios_fixup_bus(struct pci_bus *bus)
{
}

int pcibios_enable_device(struct pci_dev *dev)
{
	/* Nothing to do, since we enable all devices at startup.  */
	return 0;
}

char *pcibios_setup(char *str)
{
	if (!strcmp(str, "off")) {
		toshibaboards_pci_ops = NULL;
		return NULL;
	}
	return str;
}

void pcibios_align_resource(void *data, struct resource *res, unsigned long size)
{
#if 0
	struct pci_dev *dev = (struct pci_dev *)data;
#endif
	unsigned long start = res->start;
	/* Some PCI-IDE chips (Promise, etc.) needs
	 * extra space for DMA (reg 4) :-<
	 * See ide-dma.c and ide-pci.c for more details.
	 */
	/* always want the addresses 256 byte aligned... */
	start = (start + 1024 - 1) & ~(1024 - 1);
	res->start = start;
	/* check PCI resource limit */
	if (res->flags & IORESOURCE_IO) {
		if (res->end > toshibaboards_pci_io_resource.end - toshibaboards_pci_io_resource.start)
			res->end = toshibaboards_pci_io_resource.end - toshibaboards_pci_io_resource.start;
	} else {
		if (res->start < toshibaboards_pci_mem_resource.start)
			res->start = toshibaboards_pci_mem_resource.start;
		if (res->end > toshibaboards_pci_mem_resource.end)
			res->end = toshibaboards_pci_mem_resource.end;
	}
}

void pcibios_update_resource(struct pci_dev *dev, struct resource *root,
			     struct resource *res, int resource)
{
	u32 new, check;
	int reg;

	new = res->start | (res->flags & PCI_REGION_FLAG_MASK);
	if (new & PCI_BASE_ADDRESS_SPACE_IO) {
		struct pci_bus *root_bus;
		root_bus = dev->bus;
		while (root_bus->parent)
			root_bus = root_bus->parent;
		/* convert to PCI I/O space address */
		new -= root_bus->resource[0]->start;
	}
	if (resource < 6) {
		reg = PCI_BASE_ADDRESS_0 + 4*resource;
	} else if (resource == PCI_ROM_RESOURCE) {
		res->flags |= PCI_ROM_ADDRESS_ENABLE;
		new |= PCI_ROM_ADDRESS_ENABLE;
		reg = dev->rom_base_reg;
	} else {
		/* Somebody might have asked allocation of a non-standard resource */
		return;
	}
	
	pci_write_config_dword(dev, reg, new);
	pci_read_config_dword(dev, reg, &check);
	if ((new ^ check) & ((new & PCI_BASE_ADDRESS_SPACE_IO) ? PCI_BASE_ADDRESS_IO_MASK : PCI_BASE_ADDRESS_MEM_MASK)) {
		printk(KERN_ERR "PCI: Error while updating region "
		       "%s/%d (%08x != %08x)\n", dev->slot_name, resource,
		       new, check);
	}
}

void __init pcibios_update_irq(struct pci_dev *dev, int irq)
{
	u8 line;
	pci_write_config_byte(dev, PCI_INTERRUPT_LINE, irq);
	pci_read_config_byte(dev, PCI_INTERRUPT_LINE, &line);
	printk(KERN_DEBUG "PCI: %02x:%02x IRQ %02x\n",
	       dev->bus->number, dev->devfn, line);
}

void __init pcibios_fixup_pbus_ranges(struct pci_bus *bus,
				      struct pbus_set_ranges_data *ranges)
{
	ranges->io_start -= bus->resource[0]->start;
	ranges->io_end -= bus->resource[0]->start;
	ranges->mem_start -= bus->resource[1]->start;
	ranges->mem_end -= bus->resource[1]->start;
}

unsigned __init int pcibios_assign_all_busses(void)
{
	return 1;
}

static void __init quirk_slc90e66_bridge(struct pci_dev *dev)
{
	int irq;	/* PCI/ISA Bridge interrupt */
	unsigned char dat;
	u32 ddat;
	extern void toshibaboards_fdc37m81x_init(unsigned long port_base);

	printk(KERN_INFO "PCI: %02x:%02x",
	       dev->bus->number, dev->devfn);
	/* enable Serial IRQ */
	pci_read_config_dword(dev, 0xb0, &ddat);
	ddat |= 0x10000;	/* SERIRQ/GP17 select */
	pci_write_config_dword(dev, 0xb0, ddat);
	pci_read_config_dword(dev, 0xb0, &ddat);
	printk(" GENCFG %08x", ddat);
	pci_read_config_byte(dev, 0x64, &dat);
	dat |= 0xc0;	/* Serial IRQ enable (Continuous mode) */
	pci_write_config_byte(dev, 0x64, dat);
	pci_read_config_byte(dev, 0x64, &dat);
	printk(" SERIRQC %02x", dat);
	printk("\n");

	irq = toshibaboards_pci_map_irq(dev, PCI_SLOT(dev->devfn), 1 /* INTA */);
	if (!irq)
		return;
	toshibaboards_i8259_irq_setup(irq);

	/* initialize SuperIO on PCI Backplane board */
	toshibaboards_fdc37m81x_init(0);
}

static void __init quirk_slc90e66_ide(struct pci_dev *dev)
{
	unsigned char dat;
	int regs[2] = {0x41, 0x43};
	int i;

	printk(KERN_INFO "PCI: %02x:%02x",
	       dev->bus->number, dev->devfn);
	/* SMSC SLC90E66 IDE uses irq 14, 15 (default) */
	pci_write_config_byte(dev, PCI_INTERRUPT_LINE, 14);
	pci_read_config_byte(dev, PCI_INTERRUPT_LINE, &dat);
	printk(" IRQ %02x", dat);
#if 1
	pci_write_config_byte(dev, PCI_LATENCY_TIMER, 64);
	pci_read_config_byte(dev, PCI_LATENCY_TIMER, &dat);
	printk(" LATENCY %02x", dat);
#endif
	/* enable SMSC SLC90E66 IDE */
	for (i = 0; i < sizeof(regs) / sizeof(regs[0]); i++) {
		pci_read_config_byte(dev, regs[i], &dat);
		pci_write_config_byte(dev, regs[i], dat | 0x80);
		pci_read_config_byte(dev, regs[i], &dat);
		printk(" IDETIM%d %02x", i, dat);
	}
	printk("\n");
}

struct pci_fixup pcibios_fixups[] = {
	{ PCI_FIXUP_FINAL,	PCI_VENDOR_ID_EFAR,	PCI_DEVICE_ID_EFAR_SLC90E66_0,	quirk_slc90e66_bridge },
	{ PCI_FIXUP_FINAL,	PCI_VENDOR_ID_EFAR,	PCI_DEVICE_ID_EFAR_SLC90E66_1,	quirk_slc90e66_ide },
	{ 0 }
};

/* debugging... */
void toshibaboards_dump_pci_config(void)
{
	struct pci_dev *dev;
	unsigned short w;
	int i, j;

	pci_for_each_dev(dev) {
		printk("PCICONFIG: %02x:%02x\n",
		       dev->bus->number, dev->devfn);
		for (i = 0; i < 0x40; i+=0x10) {
			printk("%02x:", i);
			for (j = 0; j < 0x10; j+=2) {
				pci_read_config_word(dev, i + j, &w);
				printk(" %04x", w);
			}
			printk("\n");
		}
	}
}
