/*
 * $Id: rtc_ds1553.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 *
 * RTC routines for Dallas DS1553 chip.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1999-2001 Toshiba Corporation
 */

#include <linux/kernel.h>
#include <linux/rtc.h>
#include <asm/time.h>
#include <asm/mc146818rtc.h>	/* bad name... */

#define RTC_BRAM_SIZE		0x2000
#define RTC_OFFSET		0x1ff0

/**********************************************************************
 * register summary
 **********************************************************************/
#define RTC_FLAGS		(RTC_OFFSET + 0)
#define RTC_SECONDS_ALARM	(RTC_OFFSET + 2)
#define RTC_MINUTES_ALARM	(RTC_OFFSET + 3)
#define RTC_HOURS_ALARM		(RTC_OFFSET + 4)
#define RTC_DATE_ALARM		(RTC_OFFSET + 5)
#define RTC_INTERRUPTS		(RTC_OFFSET + 6)
#define RTC_WATCHDOG		(RTC_OFFSET + 7)
#define RTC_CONTROL		(RTC_OFFSET + 8)
#define RTC_CENTURY		(RTC_OFFSET + 8)
#define RTC_SECONDS		(RTC_OFFSET + 9)
#define RTC_MINUTES		(RTC_OFFSET + 10)
#define RTC_HOURS		(RTC_OFFSET + 11)
#define RTC_DAY			(RTC_OFFSET + 12)
#define RTC_DATE		(RTC_OFFSET + 13)
#define RTC_MONTH		(RTC_OFFSET + 14)
#define RTC_YEAR		(RTC_OFFSET + 15)

#define RTC_CENTURY_MASK	0x3f
#define RTC_SECONDS_MASK	0x7f
#define RTC_DAY_MASK		0x07

/*
 * Bits in the Control/Century register
 */
#define RTC_WRITE		0x80
#define RTC_READ		0x40

/*
 * Bits in the Seconds register
 */
#define RTC_STOP		0x80

/*
 * Bits in the Day register
 */
#define RTC_FREQ_TEST		0x40

/*
 * Conversion between binary and BCD.
 */
#ifndef BCD_TO_BIN
#define BCD_TO_BIN(val) ((val)=((val)&15) + ((val)>>4)*10)
#endif

#ifndef BIN_TO_BCD
#define BIN_TO_BCD(val) ((val)=(((val)/10)<<4) + (val)%10)
#endif

/* RTC-dependent code for time.c */

static int
rtc_ds1553_set_time(unsigned long t)
{
	struct rtc_time tm;
	u8 year, month, day, hour, minute, second, century;

	/* convert */
	to_tm(t, &tm);

	year = tm.tm_year;
	month = tm.tm_mon+1;	/* tm_mon starts from 0 to 11 */
	day = tm.tm_mday;
	hour = tm.tm_hour;
	minute = tm.tm_min;
	second = tm.tm_sec;
	century = year / 100;
	year %= 100;

	BIN_TO_BCD(second);
	BIN_TO_BCD(minute);
	BIN_TO_BCD(hour);
	BIN_TO_BCD(day);
	BIN_TO_BCD(month);
	BIN_TO_BCD(year);
	BIN_TO_BCD(century);

	CMOS_WRITE(RTC_WRITE, RTC_CONTROL);

	CMOS_WRITE(year, RTC_YEAR);
	CMOS_WRITE(month, RTC_MONTH);
	CMOS_WRITE(day, RTC_DATE);
	CMOS_WRITE(hour, RTC_HOURS);
	CMOS_WRITE(minute, RTC_MINUTES);
	CMOS_WRITE(second & RTC_SECONDS_MASK, RTC_SECONDS);

	/* RTC_CENTURY and RTC_CONTROL share same address... */
#if 1
	CMOS_WRITE(RTC_WRITE | (century & RTC_CENTURY_MASK), RTC_CENTURY);
	CMOS_WRITE(century & RTC_CENTURY_MASK, RTC_CONTROL);
#else
	CMOS_WRITE(century & RTC_CENTURY_MASK, RTC_CENTURY);
	CMOS_WRITE(0, RTC_CONTROL);
#endif
	return 0;
}

static unsigned long
rtc_ds1553_get_time(void)
{
	unsigned int year, month, day, hour, minute, second;
	unsigned int century;

	CMOS_WRITE(RTC_READ, RTC_CONTROL);
	second = CMOS_READ(RTC_SECONDS) & RTC_SECONDS_MASK;
	minute = CMOS_READ(RTC_MINUTES);
	hour = CMOS_READ(RTC_HOURS);
	day = CMOS_READ(RTC_DATE);
	month = CMOS_READ(RTC_MONTH);
	year = CMOS_READ(RTC_YEAR);
	century = CMOS_READ(RTC_CENTURY) & RTC_CENTURY_MASK;
	CMOS_WRITE(0, RTC_CONTROL);
	BCD_TO_BIN(second);
	BCD_TO_BIN(minute);
	BCD_TO_BIN(hour);
	BCD_TO_BIN(day);
	BCD_TO_BIN(month);
	BCD_TO_BIN(year);
	BCD_TO_BIN(century);

	year += century * 100;
	return mktime(year, month, day, hour, minute, second);
}

void
rtc_ds1553_init(void)
{
	unsigned int cen, sec;
	/* turn on RTC if it is not on */
	sec = CMOS_READ(RTC_SECONDS);
	if (sec & RTC_STOP) {
		sec &= RTC_SECONDS_MASK;
		cen = CMOS_READ(RTC_CENTURY) & RTC_CENTURY_MASK;
		CMOS_WRITE(RTC_WRITE, RTC_CONTROL);
		CMOS_WRITE(sec, RTC_SECONDS);
		CMOS_WRITE(cen & RTC_CENTURY_MASK, RTC_CONTROL);
	}

	/* set the function pointers */
	rtc_get_time = rtc_ds1553_get_time;
	rtc_set_time = rtc_ds1553_set_time;
}
