/*
 * linux/arch/mips/toshiba-boards/generic/setup.c
 * $Id: setup.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 *
 * Setup pointers to hardware-dependent routines.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 */

#include <linux/config.h>
#include <linux/console.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/notifier.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include <asm/mipsregs.h>
#include <asm/system.h>
#include <asm/bootinfo.h>
#include <asm/cpu.h>
#include <asm/fdc37m81x.h>
#include <asm/toshiba-boards/pmon.h>
#include <asm/toshiba-boards/jmi39io2.h>
#ifdef CONFIG_BLK_DEV_IDE
#include <asm/toshiba-boards/pci.h>
#endif
#ifdef CONFIG_BLK_DEV_IDEPCI
#include <linux/hdreg.h>
#include <asm/ptrace.h>
#include <linux/ide.h>
extern struct ide_ops std_ide_ops;
#endif
#ifdef CONFIG_PC_KEYB
#include <asm/keyboard.h>
extern struct kbd_ops no_kbd_ops;
extern struct kbd_ops std_kbd_ops;
#endif

extern int mips_configk0;
extern int mips_ic_disable, mips_dc_disable;
extern int mips_config_cwfon;
extern int mips_config_wbon;

static void __init toshibaboards_cache_setup(void)
{
	unsigned int conf;

	/* enable cache here */
	switch (mips_cpu.processor_id & 0xff00) {
	case PRID_IMP_TX39:	/* TX39 core */
		conf = read_32bit_cp0_register(CP0_CONF);
		conf &= ~(TX39_CONF_ICE | TX39_CONF_DCE | TX39_CONF_WBON | TX39_CONF_CWFON);
		conf |= mips_ic_disable ? 0 : TX39_CONF_ICE;
		conf |= mips_dc_disable ? 0 : TX39_CONF_DCE;
		conf |= mips_config_wbon ? TX39_CONF_WBON : 0;
		conf |= mips_config_cwfon ? TX39_CONF_CWFON : 0;
		write_32bit_cp0_register(CP0_CONF, conf);
		write_32bit_cp0_register(CP0_TX39_CACHE, 0);
		break;
	case PRID_IMP_TX49:	/* TX49 core */
		conf = read_32bit_cp0_register(CP0_CONFIG);
		conf &= ~(TX49_CONF_IC | TX49_CONF_DC | TX49_CONF_CWFON);
		conf |= mips_ic_disable ? TX49_CONF_IC : 0;
		conf |= mips_dc_disable ? TX49_CONF_DC : 0;
		/* only TX49/H2(and upper) have CWFON bit */
		if ((mips_cpu.processor_id & 0x00ff) != 0x11) /* TX4955 (TX49/H) */
			conf |= (mips_config_cwfon && !mips_dc_disable) ? TX49_CONF_CWFON : 0;
		write_32bit_cp0_register(CP0_CONFIG, conf);
		break;
	}
}

static void __init toshibaboards_cache_report(void)
{
	unsigned int conf;

	switch (mips_cpu.processor_id & 0xff00) {
	case PRID_IMP_TX39:	/* TX39 core */
		conf = read_32bit_cp0_register(CP0_CONF);
		if (!(conf & TX39_CONF_ICE))
			printk("TX39XX I-Cache disabled.\n");
		if (!(conf & TX39_CONF_DCE))
			printk("TX39XX D-Cache disabled.\n");
		else if (!(conf & TX39_CONF_WBON))
			printk("TX39XX D-Cache WriteThrough.\n");
		else if (!(conf & TX39_CONF_CWFON))
			printk("TX39XX D-Cache WriteBack.\n");
		else
			printk("TX39XX D-Cache WriteBack (CWF) .\n");
		break;
	case PRID_IMP_TX49:	/* TX49 core */
		conf = read_32bit_cp0_register(CP0_CONFIG);
		if (conf & TX49_CONF_IC)
			printk("TX49XX I-Cache disabled.\n");
		if (conf & TX49_CONF_DC)
			printk("TX49XX D-Cache disabled.\n");
		break;
	}

	switch(mips_cpu.cputype) {
	case CPU_R4300:
	case CPU_TX49XX:
		/* r4k generic */
		conf = read_32bit_cp0_register(CP0_CONFIG);
		switch (conf & CONF_CM_CMASK) {
		case CONF_CM_CACHABLE_NO_WA:
			printk("KSEG0 writethru(no-WA).\n");
			break;
		case CONF_CM_CACHABLE_WA:
			printk("KSEG0 writethru(WA).\n");
			break;
		case CONF_CM_UNCACHED:
			printk("KSEG0 uncached.\n");
			break;
		case CONF_CM_CACHABLE_NONCOHERENT:
			if (mips_cpu.cputype == CPU_TX49XX &&
			    (conf & TX49_CONF_CWFON))
				printk("TX49XX D-Cache WriteBack (CWF).\n");
			break;
		}
		break;
	}
}

static int panic_event(struct notifier_block *this, unsigned long event,
                      void *ptr)
{
	extern void breakpoint(void);
	extern int panic_timeout;
	/* called when panic happens. enter to monitor. */
#ifdef CONFIG_REMOTE_DEBUG
	breakpoint();
#endif
	if (!panic_timeout)
		pmon_halt();
	return NOTIFY_DONE;
}

static struct notifier_block panic_block = {
	panic_event,
	NULL,
	0
};

int __init toshibaboards_probe_ieee1394_board(unsigned long base)
{
	volatile unsigned long *pr3;
	unsigned long pr3_ver;
	int found = 0;

	/* check JMY-IEEE1394IF board */
	pr3 = (volatile unsigned long *)base;
	pr3_ver = pr3[0];
	pr3[0] = 0xffffffff;
	if ((pr3[0] & 0xff00ff00) == 0) { /* PR3_VERSION */
		unsigned long tmp = pr3[0x50>>2];
		pr3[0x50>>2] = 0xffff0000; /* PR3_ADDR: initial value */
		if (pr3[0x50>>2] == 0xffff0000) {
			pr3[0x50>>2] = 0x0000ffff; /* PR3_ADDR: reverse */
			if (pr3[0x50>>2] == 0x0000ffff)
				found = 1;
		}
		pr3[0x50>>2] = tmp;
	} else {
		pr3[0] = pr3_ver;
	}
	if (found)
		printk(KERN_DEBUG "IEEE1394 Link Board found.\n");
	else
		printk(KERN_DEBUG "IEEE1394 Link Board not found.\n");
	return found ? 0 : -ENODEV;
}

unsigned long jmi39io2_iob_base;
int jmi39io2_isac_irq_base;
extern int early_i82365_setup(u_long iobase, u_long membase, u_int irqmask, int levelirq);
void __init jmi39io2_init(unsigned long base, int isac_irq)
{
	jmi39io2_iob_base = base;
	jmi39io2_isac_irq_base = isac_irq;
	if (!base)
		return;
	printk("TJSYS JMI-TX39IO2 (at %08lx) --- ISAC(Rev %d) DIPSW:%01x\n",
	       jmi39io2_iob_base,
	       jmi39io2_isac_reg_in(JMI39IO2_ISAC_REV_ADDR(base)) & JMI39IO2_REV_MASK,
	       jmi39io2_io_dipsw(base));
	jmi39io2_io_led_set(base, 0);
#if defined(CONFIG_PCMCIA) && defined(CONFIG_I82365)
	/* IRQ3, IRQ5 are used for PC Card Controller */
	early_i82365_setup(JMI39IO2_ISAIO_BASE(base) - mips_io_port_base,
			   PHYSADDR(JMI39IO2_ISAMEM_BASE(base)),
			   (1 << 3) | (1 << 5), 1 /* level */);
#endif
}

/* Initialize SuperIO on PCI Backplane board */
void __init toshibaboards_fdc37m81x_init(unsigned long port_base)
{
	unsigned long port = port_base + 0x3f0;

	outb(FDC37M81x_CONFIG_ENTER, port + FDC37M81x_CONFIG_INDEX);
	if (fdc37m81x_readb(port, FDC37M81x_DID) != 0x4d)
		return;

	/* Init Serial port1 */
	fdc37m81x_writeb(port, FDC37M81x_DNUM, FDC37M81x_SERIAL1);
	fdc37m81x_writeb(port, FDC37M81x_ACTIVE, 0x01);
	fdc37m81x_writeb(port, FDC37M81x_BASEADDR0, 0x03);
	fdc37m81x_writeb(port, FDC37M81x_BASEADDR1, 0xf8);

	/* Init Serial port2 */
	fdc37m81x_writeb(port, FDC37M81x_DNUM, FDC37M81x_SERIAL2);
	fdc37m81x_writeb(port, FDC37M81x_ACTIVE, 0x01);
	fdc37m81x_writeb(port, FDC37M81x_BASEADDR0, 0x02);
	fdc37m81x_writeb(port, FDC37M81x_BASEADDR1, 0xf8);

	/* Init Keyboard */
	fdc37m81x_writeb(port, FDC37M81x_DNUM, FDC37M81x_KBD);
	fdc37m81x_writeb(port, FDC37M81x_ACTIVE, 0x01);
	fdc37m81x_writeb(port, FDC37M81x_INT, 1);
	fdc37m81x_writeb(port, FDC37M81x_INT2, 12);

	outb(FDC37M81x_CONFIG_EXIT, port + FDC37M81x_CONFIG_INDEX);

#ifdef CONFIG_PC_KEYB
	/* do not override any onboard keyboard controller */
	if (kbd_ops == &no_kbd_ops)
		kbd_ops = &std_kbd_ops;
#endif
}

#ifdef CONFIG_HAVE_BOARD_IO_FUNCS
#ifdef __LITTLE_ENDIAN
/* for Little Endian, we can use standard IO macros. */
#warning using CONFIG_HAVE_BOARD_IO_FUNCS in LittleEndian
#endif
/* default IO functions for BigEndian. */
/*
 * We do byte-swap by software for PCI region in big endian because
 * TX4927/TX3927 PCIC does not have "smart" swap feature :-<
 */
/* Board specific setup routine can override those functions. */
#define IS_SWAPPED_PORT(port) \
	(toshibaboards_pci_io_resource.start <= (port) && \
	 (port) <= toshibaboards_pci_io_resource.end)
#define __inb(port) (*(volatile u8 *)(mips_io_port_base + (port)))
#define __inw(port) (*(volatile u16 *)(mips_io_port_base + (port)))
#define __inl(port) (*(volatile u32 *)(mips_io_port_base + (port)))
#define __outb(val,port)						\
do {									\
	*(volatile u8 *)(mips_io_port_base + (port)) = (val);		\
} while(0)

#define __outw(val,port)						\
do {									\
	*(volatile u16 *)(mips_io_port_base + (port)) = (val);		\
} while(0)

#define __outl(val,port)						\
do {									\
	*(volatile u32 *)(mips_io_port_base + (port)) = (val);		\
} while(0)
static inline void __outsb(unsigned long port, const void *addr, unsigned int count)
{
	while (count--) {
		__outb(*(u8 *)addr, port);
		addr++;
	}
}
static inline void __insb(unsigned long port, void *addr, unsigned int count)
{
	while (count--) {
		*(u8 *)addr = __inb(port);
		addr++;
	}
}
static inline void __outsw(unsigned long port, const void *addr, unsigned int count)
{
	while (count--) {
		__outw(*(u16 *)addr, port);
		addr += 2;
	}
}
static inline void __insw(unsigned long port, void *addr, unsigned int count)
{
	while (count--) {
		*(u16 *)addr = __inw(port);
		addr += 2;
	}
}
static inline void __outsl(unsigned long port, const void *addr, unsigned int count)
{
	while (count--) {
		__outl(*(u32 *)addr, port);
		addr += 4;
	}
}
static inline void __insl(unsigned long port, void *addr, unsigned int count)
{
	while (count--) {
		*(u32 *)addr = __inl(port);
		addr += 4;
	}
}
static inline void insw_swap(unsigned long port, void *addr, unsigned int count)
{
	unsigned short *ptr = (unsigned short *)addr;
	while (count--) {
		*ptr++ = le16_to_cpu(__inw(port));
	}
}
static inline void insl_swap(unsigned long port, void *addr, unsigned int count)
{
	unsigned int *ptr = (unsigned int *)addr;
	while (count--) {
		*ptr++ = le32_to_cpu(__inl(port));
	}
}
static inline void outsw_swap(unsigned long port, const void *addr, unsigned int count)
{
	unsigned short *ptr = (unsigned short *)addr;
	while (count--) {
		__outw(cpu_to_le16(*ptr), port);
		ptr++;
	}
}
static inline void outsl_swap(unsigned long port, const void *addr, unsigned int count)
{
	unsigned int *ptr = (unsigned int *)addr;
	while (count--) {
		__outl(cpu_to_le32(*ptr), port);
		ptr++;
	}
}
static void tx_writeb(unsigned char b, volatile unsigned char *addr) {
	*addr = b;
}
static void tx_writew(unsigned short b, volatile unsigned short *addr) {
	*addr = cpu_to_le16(b);
}
static void tx_writel(unsigned int b, volatile unsigned int *addr) {
	*addr = cpu_to_le32(b);
}
static unsigned char tx_readb(volatile unsigned char *addr) {
	return *addr;
}
static unsigned short tx_readw(volatile unsigned short *addr) {
	return le16_to_cpu(*addr);
}
static unsigned int tx_readl(volatile unsigned int *addr) {
	return le32_to_cpu(*addr);
}
static void tx_outb(unsigned int value, unsigned long port) {
	__outb(value, port);
}
static void tx_outw(unsigned int value, unsigned long port) {
	if (IS_SWAPPED_PORT(port))
		__outw(cpu_to_le16(value), port);
	else
		__outw(value, port);
}
static void tx_outl(unsigned int value, unsigned long port) {
	if (IS_SWAPPED_PORT(port))
		__outl(cpu_to_le32(value), port);
	else
		__outw(value, port);
}
static unsigned char tx_inb(unsigned long port) {
	return __inb(port);
}
static unsigned short tx_inw(unsigned long port) {
	if (IS_SWAPPED_PORT(port))
		return le16_to_cpu(__inw(port));
	return __inw(port);
}
static unsigned int tx_inl(unsigned long port) {
	if (IS_SWAPPED_PORT(port))
		return le32_to_cpu(__inl(port));
	return __inl(port);
}
static void tx_outsb(unsigned long port, const void *addr, unsigned int count) {
	__outsb(port, addr, count);
}
static void tx_outsw(unsigned long port, const void *addr, unsigned int count) {
	if (IS_SWAPPED_PORT(port)) {
		outsw_swap(port, addr, count);
	} else {
		__outsw(port, addr, count);
	}
}
static void tx_outsl(unsigned long port, const void *addr, unsigned int count) {
	if (IS_SWAPPED_PORT(port)) {
		outsl_swap(port, addr, count);
	} else {
		__outsl(port, addr, count);
	}
}
static void tx_insb(unsigned long port, void *addr, unsigned int count) {
	__insb(port, addr, count);
}
static void tx_insw(unsigned long port, void *addr, unsigned int count) {
	if (IS_SWAPPED_PORT(port)) {
		insw_swap(port, addr, count);
	} else {
		__insw(port, addr, count);
	}
}
static void tx_insl(unsigned long port, void *addr, unsigned int count) {
	if (IS_SWAPPED_PORT(port)) {
		insl_swap(port, addr, count);
	} else {
		__insl(port, addr, count);
	}
}
static void tx_memset_io(volatile void *addr, int c, int len) {
	volatile unsigned char *ptr = (volatile unsigned char *)addr;
	while (len--)
		*ptr++ = c;
}
static void tx_memcpy_fromio(void *to, volatile void *from, int len) {
	volatile unsigned char *fromptr = (volatile unsigned char *)from;
	volatile unsigned char *toptr = (volatile unsigned char *)to;
	while (len--)
		*toptr++ = *fromptr++;
}
static void tx_memcpy_toio(volatile void *to, const void *from, int len) {
	volatile unsigned char *fromptr = (volatile unsigned char *)from;
	volatile unsigned char *toptr = (volatile unsigned char *)to;
	while (len--)
		*toptr++ = *fromptr++;
}

struct mips_io_funcs mips_io_funcs = {
	writeb:	tx_writeb,
	writew:	tx_writew,
	writel:	tx_writel,
	readb:	tx_readb,
	readw:	tx_readw,
	readl:	tx_readl,
	outb:	tx_outb,
	outw:	tx_outw,
	outl:	tx_outl,
	inb:	tx_inb,
	inw:	tx_inw,
	inl:	tx_inl,
	outsb:	tx_outsb,
	outsw:	tx_outsw,
	outsl:	tx_outsl,
	insb:	tx_insb,
	insw:	tx_insw,
	insl:	tx_insl,
	memset_io:	tx_memset_io,
	memcpy_fromio:	tx_memcpy_fromio,
	memcpy_toio:	tx_memcpy_toio,
};
EXPORT_SYMBOL(mips_io_funcs);
#endif /* CONFIG_HAVE_BOARD_IO_FUNCS */

#if defined(CONFIG_BLK_DEV_IDE) && defined(__BIG_ENDIAN)
/* Ugly hack... */
static void
ideproc_noswap(ide_ide_action_t action,
	       ide_drive_t *drive, void *buffer, unsigned int wcount)
{
	byte io_32bit;
	io_32bit = drive->io_32bit;
	switch (action) {
	case ideproc_ide_input_data:
	case ideproc_atapi_input_bytes:
		if (io_32bit)
			__insl(IDE_DATA_REG, buffer, wcount);
		else
			__insw(IDE_DATA_REG, buffer, wcount<<1);
		break;
	case ideproc_ide_output_data:
	case ideproc_atapi_output_bytes:
		if (io_32bit)
			__outsl(IDE_DATA_REG, buffer, wcount);
		else
			__outsw(IDE_DATA_REG, buffer, wcount<<1);
		break;
	}
}
static void
ideproc_swap(ide_ide_action_t action,
	     ide_drive_t *drive, void *buffer, unsigned int wcount)
{
	byte io_32bit;
	io_32bit = drive->io_32bit;
	switch (action) {
	case ideproc_ide_input_data:
	case ideproc_atapi_input_bytes:
		if (io_32bit)
			insl_swap(IDE_DATA_REG, buffer, wcount);
		else
			insw_swap(IDE_DATA_REG, buffer, wcount<<1);
		break;
	case ideproc_ide_output_data:
	case ideproc_atapi_output_bytes:
		if (io_32bit)
			outsl_swap(IDE_DATA_REG, buffer, wcount);
		else
			outsw_swap(IDE_DATA_REG, buffer, wcount<<1);
		break;
	}
}
static ide_ioreg_t (*org_ide_default_io_base)(int index);
static ide_ioreg_t ide_initialize_hook(int index)
{
	ide_ioreg_t base = org_ide_default_io_base(index);
	/* install our ideproc */
	/* use swapped IO for JMI39IO2 IDE */
	if(jmi39io2_iob_base &&
	   base == JMI39IO2_IDE_ADDR(jmi39io2_iob_base) - mips_io_port_base)
		ide_hwifs[index].ideproc = ideproc_swap;
	else
		ide_hwifs[index].ideproc = ideproc_noswap;
	return base;
}
#endif /* CONFIG_BLK_DEV_IDE && __BIG_ENDIAN */

int mips_parport_base;
int mips_parport_irq;
extern void wbflush_setup(void);
extern void toshibaboards_dma_setup(void);
extern void tsdb_setup(void);
extern void jmr3927_setup(void);
extern void tx4927evb_setup(void);
extern void rbtx4927_setup(void);
extern void rbtx4925_setup(void);

void __init toshibaboards_setup(void)
{
	clear_cp0_status(ST0_RE);

	toshibaboards_cache_setup();

	wbflush_setup();

	notifier_chain_register(&panic_notifier_list, &panic_block);

	toshibaboards_dma_setup();

	/* setup resource limits */
	ioport_resource.start = 0;
	ioport_resource.end = 0xffffffff;	/* no limit */
	iomem_resource.start = 0;
	iomem_resource.end = 0x1fffffff;	/* 512MB */

#ifdef CONFIG_VT
	conswitchp = &dummy_con;
#endif
#ifdef CONFIG_BLK_DEV_IDEPCI
	ide_ops = &std_ide_ops;
#endif
	switch (mips_machtype) {
#ifdef CONFIG_TOSHIBA_SDB
	case MACH_TOSHIBA_SDB:
		tsdb_setup();
		break;
#endif
#ifdef CONFIG_TOSHIBA_JMR3927
	case MACH_TOSHIBA_JMR3927:
		jmr3927_setup();
		break;
#endif
#ifdef CONFIG_TOSHIBA_TX4927EVB
	case MACH_TOSHIBA_TX4927EVB:
		tx4927evb_setup();
		break;
#endif
#ifdef CONFIG_TOSHIBA_RBTX4927
	case MACH_TOSHIBA_RBTX4927:
		rbtx4927_setup();
		break;
#endif
#ifdef CONFIG_TOSHIBA_RBTX4925
	case MACH_TOSHIBA_RBTX4925:
		rbtx4925_setup();
		break;
#endif
	default:
		panic("Unsupported type");
	}
	printk(KERN_DEBUG "IO port base: %08lx\n", mips_io_port_base);

#if defined(CONFIG_BLK_DEV_IDE) && defined(__BIG_ENDIAN)
	/* Ugly hack... */
	org_ide_default_io_base = ide_ops->ide_default_io_base;
	ide_ops->ide_default_io_base = ide_initialize_hook;
#endif
	toshibaboards_cache_report();
}
