/*
 *  linux/arch/mips/toshiba-boards/rbtx4927/irq.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 *
 * $Id: irq.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */
#include <linux/config.h>
#include <linux/init.h>

#include <linux/errno.h>
#include <linux/kernel_stat.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/timex.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#ifdef CONFIG_BLK_DEV_IDEPCI
#include <linux/pci.h>
#endif

#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/mipsregs.h>
#include <asm/system.h>

#include <asm/ptrace.h>
#include <asm/processor.h>
#include <asm/toshiba-boards/irq.h>
#include <asm/toshiba-boards/rbtx4927.h>
#include <asm/toshiba-boards/jmi39io2.h>

#if RBTX4927_IRQ_END > NR_IRQS
#error RBTX4927_IRQ_END > NR_IRQS
#endif

#define RBTX4927_IRQ_ISACINT	RBTX4927_IRQ_IRC_INT(JMI39IO2_INT_ISAC)

static int rbtx4927_gen_iack(void)
{
	/* generate ACK cycle */
#if 1
	return tx4927_pcicptr->g2pintack & (NR_ISA_IRQS - 1);
#else
	return tx4927_pcicptr->g2pintack & 0xff;
#endif
}

/*
 * RBTX4927 IOC controller definition
 */

#define flush_wb()	\
	__asm__ __volatile__( \
		".set push\n\t" \
		".set mips2\n\t" \
		"sync\n\t" \
		".set pop\n\t")

static void rbtx4927_ioc_irq_enable(unsigned int irq)
{
	int irq_nr = irq - RBTX4927_IRQ_IOC;
	/* 0: mask */
	*rbtx4927_imask_ptr |= 1 << irq_nr;
	flush_wb();
}

static void rbtx4927_ioc_irq_disable(unsigned int irq)
{
	int irq_nr = irq - RBTX4927_IRQ_IOC;
	/* 0: mask */
	*rbtx4927_imask_ptr &= ~(1 << irq_nr);
	flush_wb();
}

static unsigned int rbtx4927_ioc_irq_startup(unsigned int irq)
{
	rbtx4927_ioc_irq_enable(irq);
	return 0;
}
#define	rbtx4927_ioc_irq_shutdown	rbtx4927_ioc_irq_disable

#define rbtx4927_ioc_irq_ack	rbtx4927_ioc_irq_disable
static void rbtx4927_ioc_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		rbtx4927_ioc_irq_enable(irq);
}

static hw_irq_controller rbtx4927_ioc_irq_controller = {
	typename:	"IOC",
	startup:	rbtx4927_ioc_irq_startup,
	shutdown:	rbtx4927_ioc_irq_shutdown,
	enable:		rbtx4927_ioc_irq_enable,
	disable:	rbtx4927_ioc_irq_disable,
	ack:		rbtx4927_ioc_irq_ack,
	end:		rbtx4927_ioc_irq_end,
	set_affinity:	NULL,
};

/* find first bit set */
static inline int ffs8(unsigned char x)
{
	int r = 1;

	if (!x)
		return 0;
	if (!(x & 0xf)) {
		x >>= 4;
		r += 4;
	}
	if (!(x & 3)) {
		x >>= 2;
		r += 2;
	}
	if (!(x & 1)) {
		x >>= 1;
		r += 1;
	}
	return r;
}

static int rbtx4927_ioc_irqroute(void)
{
	unsigned char istat = *rbtx4927_imstat_ptr;
	int irq = ffs8(istat);
	if (irq == 0)
		return -1;
	return RBTX4927_IRQ_IOC + (irq - 1);
}

extern int rbtx4927_use_r4k_timer;
static int rbtx4927_irc_irqdispatch(struct pt_regs *regs)
{
	int irq = 0;
	unsigned int csr = (unsigned int)-1;

#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
	tx_fpu_c_bug_fixup(regs);
#endif
	if (rbtx4927_use_r4k_timer && (regs->cp0_cause & CAUSEF_IP7)) {
		irq = RBTX4927_IRQ_LOCAL_TIMER;
	} else if ((regs->cp0_cause & CAUSEF_IP0)) {
		irq = RBTX4927_IRQ_LOCAL_SOFT0;
	} else if ((regs->cp0_cause & CAUSEF_IP1)) {
		irq = RBTX4927_IRQ_LOCAL_SOFT1;
	} else {
		csr = tx4927_ircptr->csr;
		if (csr & TX4927_IRCSR_IF)
			return -1;
		irq = RBTX4927_IRQ_IRC + (csr & 0x1f);

		/* redirect IOC interrupts */
		switch (irq) {
		case RBTX4927_IRQ_IOCINT:
			irq = rbtx4927_ioc_irqroute();
#if 1	/* someone forgot flushing writebuffer...? */
			if (irq < 0)
				return -1;
#endif
			irq = toshibaboards_i8259_irqroute(irq);
			break;
		case RBTX4927_IRQ_ISACINT:
			irq = jmi39io2_isac_irqroute();
			break;
		}
		if (irq < 0)
			return -1;
	}
#if 1 /* XXX DEBUG */
	*rbtx4927_led_ptr = ~irq;
#endif
	do_IRQ(irq, regs);
	flush_wb();
	return 0;
}

static struct irqaction ioc_action = {
	no_action, 0, 0, "cascade(IOC)", NULL, NULL,
};
static struct irqaction isac_action = {
	no_action, 0, 0, "cascade(JMI39IO2 ISAC)", NULL, NULL,
};
static struct irqaction irc_action = {
	no_action, 0, 0, "cascade(IRC)", NULL, NULL,
};

static void rbtx4927_pcierr_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
	extern void tx4927_report_pcic_status(void);
	extern void tx4927_dump_pcic_settings(void);
	extern void toshibaboards_dump_pci_config(void);
	extern struct pci_ops *toshibaboards_pci_ops;
	if (!toshibaboards_pci_ops) {
		printk("PCI disabled. disable %s interrupt (%d)\n",
		       irq == RBTX4927_IRQ_IRC_PCIC ? "PCIC" : "PCIERR", irq);
		disable_irq_nosync(irq);
		return;
	}
#ifdef CONFIG_BLK_DEV_IDEPCI
	/* ignore MasterAbort for ide probing... */
	if (irq == RBTX4927_IRQ_IRC_PCIERR &&
	    ((tx4927_pcicptr->pcistatus >> 16) & 0xf900) == PCI_STATUS_REC_MASTER_ABORT) {
		tx4927_pcicptr->pcistatus =
			(tx4927_pcicptr->pcistatus & 0x0000ffff) |
			(PCI_STATUS_REC_MASTER_ABORT << 16);
		return;
	}
#endif
	if (irq == RBTX4927_IRQ_IRC_PCIC)
		printk("PCIC interrupt (irq 0x%x)", irq);
	else
		printk("PCIERR interrupt (irq 0x%x)", irq);
	printk(" at 0x%08lx.\n", regs->cp0_epc);
        printk("ccfg:%08lx, tear:%02lx_%08lx\n",
               (unsigned long)tx4927_ccfgptr->ccfg,
	       (unsigned long)(tx4927_ccfgptr->tear >> 32),
	       (unsigned long)tx4927_ccfgptr->tear);
	tx4927_report_pcic_status();
	show_regs(regs);
	tx4927_dump_pcic_settings();
	toshibaboards_dump_pci_config();
	panic("PCI error.");
}
static struct irqaction pcierr_action = {
	rbtx4927_pcierr_interrupt, SA_INTERRUPT, 0, "PCI error", NULL, NULL,
};

static void rbtx4927_isaerr_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
#if 1
	printk("ISA error interrupt (irq 0x%x).\n", irq);
#endif
}
static struct irqaction isaerr_action = {
	rbtx4927_isaerr_interrupt, SA_INTERRUPT, 0, "ISA error", NULL, NULL,
};

static int tx4927_print_irc_settings = 0;
extern void rbtx4927_setup_be_board_handler(void);

void __init rbtx4927_irq_setup(void)
{
	int i;

	/* Now, interrupt control disabled, */
	/* all IRC interrupts are masked, */
	/* all IRC interrupt mode are Low Active. */

	/* mask all IOC interrupts */
	*rbtx4927_imask_ptr = 0;

	/* clear SoftInt interrupts */
	*rbtx4927_softint_ptr = 0;

	/* Onboard Ether: High Active */
	tx4927_ircptr->cr[(RBTX4927_IRQ_ETHER - RBTX4927_IRQ_IRC) / 8] |=
		TX4927_IRCR_HIGH << (((RBTX4927_IRQ_ETHER - RBTX4927_IRQ_IRC) % 8) * 2);

	toshibaboards_irqdispatch = rbtx4927_irc_irqdispatch;

	/* setup irq descriptors */
	mips_cpu_irq_init(RBTX4927_IRQ_LOCAL);
	tx4927_irq_init(RBTX4927_IRQ_IRC);
	for (i = RBTX4927_IRQ_IOC; i < RBTX4927_IRQ_IOC + RBTX4927_NR_IRQ_IOC; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &rbtx4927_ioc_irq_controller;
	}

	setup_irq(RBTX4927_IRQ_LOCAL_IRC, &irc_action);
	setup_irq(RBTX4927_IRQ_IRC_PCIC, &pcierr_action);
	setup_irq(RBTX4927_IRQ_IRC_PCIERR, &pcierr_action);
	setup_irq(RBTX4927_IRQ_IOCINT, &ioc_action);
	if (have_jmi39io2()) {
		jmi39io2_isac_irq_init();
		setup_irq(RBTX4927_IRQ_ISACINT, &isac_action);
		setup_irq(RBTX4927_IRQ_ISAC + JMI39IO2_ISAC_INTB_ISAER, &isaerr_action);
	}

	toshibaboards_gen_iack = rbtx4927_gen_iack;

	if (tx4927_print_irc_settings) {
		extern void tx4927_dump_irc_settings(void);
		tx4927_dump_irc_settings();
	}

	rbtx4927_setup_be_board_handler();
}
