/*
 * linux/arch/mips/toshiba-boards/rbtx4927/setup.c
 * $Id: setup.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 *
 * Setup pointers to hardware-dependent routines.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/ioport.h>
#include <linux/kdev_t.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <asm/system.h>
#include <asm/reboot.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/toshiba-boards/pmon.h>
#include <asm/toshiba-boards/pci.h>
#include <asm/toshiba-boards/rbtx4927.h>
#include <asm/toshiba-boards/jmi39io2.h>
#include <asm/mc146818rtc.h>	/* bad name... */
#include <asm/bootinfo.h>
#include <asm/cpu.h>
#include <asm/pci.h>
#include <asm/time.h>
#include <asm/mipsregs.h>
#include <asm/ptrace.h>
#if defined(CONFIG_SERIAL_TXX927)
#include <asm/serial_txx927.h>
#endif

/*static*/ int rbtx4927_ccfg_toeon = 1;
/*static*/ int rbtx4927_pcic_trdyto = 0;	/* default: disabled */

extern struct rtc_ops rbtx4927_rtc_ops;
extern struct pci_ops rbtx4927_pci_ops;
extern int rbtx4927_pci_map_irq(struct pci_dev *dev, u8 slot, u8 pin);
extern void rbtx4927_irq_setup(void) __init;
extern void rbtx4927_time_init(void) __init;
extern void rbtx4927_timer_setup(struct irqaction *irq) __init;

#ifdef CONFIG_REMOTE_DEBUG
extern int __init early_serial_txx927_kgdb_setup(int line, unsigned long base, int irq, int baud_base);
extern void breakpoint(void);
#define TOMON(s) printk(s); breakpoint(); pmon_halt()
#else
#define TOMON(s) pmon_printf(s); pmon_halt()
#endif
extern void wait_for_keypress(void);

static inline void printk_ll(const char *fmt, unsigned long long val)
{
	printk(fmt, (unsigned long)(val >> 32), (unsigned long)val);
}

static void rbtx4927_machine_restart(char *command)
{
	cli();
	printk("Rebooting...");
	*rbtx4927_softresetlock_ptr = 1;
	*rbtx4927_softreset_ptr = 1;
	wbflush();
	while(1);
}

static void rbtx4927_machine_halt(void)
{
	cli();
	printk("Press any key to reboot.\n");
	wait_for_keypress();
	*rbtx4927_softresetlock_ptr = 1;
	*rbtx4927_softreset_ptr = 1;
	wbflush();
	while(1);
}

static void rbtx4927_machine_power_off(void)
{
	cli();
	printk("Press any key to reboot.\n");
	wait_for_keypress();
	*rbtx4927_softresetlock_ptr = 1;
	*rbtx4927_softreset_ptr = 1;
	wbflush();
	while(1);
}

extern void (*ibe_board_handler)(struct pt_regs *regs);
extern void (*dbe_board_handler)(struct pt_regs *regs);
static void rbtx4927_be_board_handler(struct pt_regs *regs)
{
	extern void tx4927_report_pcic_status(void);
	printk("%cBE exception at 0x%08lx\n",
	       (regs->cp0_cause & (1<<2)) ? 'D' : 'I', regs->cp0_epc);
	printk("ccfg:%08lx, ", (unsigned long)tx4927_ccfgptr->ccfg);
	printk_ll("tear:%02lx_%08lx\n", tx4927_ccfgptr->tear);
	tx4927_report_pcic_status();
	show_regs(regs);
	TOMON("BusError!\n");
	while(1);
}
void __init rbtx4927_setup_be_board_handler(void)
{
	ibe_board_handler = rbtx4927_be_board_handler;
	dbe_board_handler = rbtx4927_be_board_handler;
}

int rbtx4927_gbus_clock = 100000000; /* 100MHz */
static int rbtx4927_master_clock = 25000000; /* 25MHz */
static int rbtx4927_cpu_clock = 200000000; /* 200MHz */
int rbtx4927_pci66;	/* 0:auto, 1:on(force), -1:off(disable) */

unsigned long rbtx4927_ce_base[8];
static unsigned long rbtx4927_ce_size[8];

static unsigned long __init
rbtx4927_find_free_region(int n,
			  unsigned long *bases,
			  unsigned long *sizes,
			  unsigned long begin,
			  unsigned long end,
			  unsigned long *size)
{
	unsigned long base;
	unsigned long sz = *size;
	int i;

	for (; sz; sz /= 2) {
		for (base = begin / sz * sz; base < end; base += sz) {
			for (i = 0; i < n; i++) {
				if (sizes[i] &&
				    base <= bases[i] + (sizes[i] - 1) &&
				    base + (sz - 1) >= bases[i])
					break;
			}
			if (i == n) {
				*size = sz;
				return base;
			}
		}
	}
	return 0;
}

#ifdef CONFIG_PCI
static void __init rbtx4927_pci66_setup(void)
{
	if (tx4927_ccfgptr->ccfg & TX4927_CCFG_PCI66)
		return;

	/* Reset PCI Bus */
	*rbtx4927_pcireset_ptr = 1;
	/* Reset PCIC */
	tx4927_ccfgptr->clkctr |= TX4927_CLKCTR_PCIRST;

	/* Assert M66EN */
	tx4927_ccfgptr->ccfg |= TX4927_CCFG_PCI66;
	/* set PCICLK 66MHz */
	if (tx4927_ccfgptr->pcfg & TX4927_PCFG_PCICLKEN_ALL) {
		unsigned int pcidivmode = 0;
		/* CPU 200MHz */
		pcidivmode = TX4927_CCFG_PCIDIVMODE_3;
		tx4927_ccfgptr->ccfg =
			(tx4927_ccfgptr->ccfg & ~TX4927_CCFG_PCIDIVMODE_MASK)
			| pcidivmode;
		printk(KERN_DEBUG "PCICLK: ccfg:%08lx\n",
		       (unsigned long)tx4927_ccfgptr->ccfg);
	}

	udelay(10000);

	/* clear PCIC reset */
	tx4927_ccfgptr->clkctr &= ~TX4927_CLKCTR_PCIRST;
	/* clear PCI reset */
	*rbtx4927_pcireset_ptr = 0;
}

/* setup rbtx4927_pci_{mem,io}_{base,size} */
static unsigned long rbtx4927_pci_mem_base;
static unsigned long rbtx4927_pci_mem_size = 0x08000000; /* MAX 128M */
static unsigned long rbtx4927_pci_io_base;
static unsigned long rbtx4927_pci_io_size = 0x01000000;	/* MAX 16M */

static void __init rbtx4927_setup_pci_regions(void)
{
	unsigned long base[4+8+1];
	unsigned long size[4+8+1];
	unsigned long mbegin, mend;
	int i;

	if (rbtx4927_pci_mem_base || rbtx4927_pci_io_base)
		return;	/* done */

	for (i = 0; i < sizeof(base) / sizeof(base[0]); i++)
		base[i] = size[i] = 0;
	for (i = 0; i < 4; i++) {
		if (!((unsigned long)tx4927_sdramcptr->cr[i] & 0x00000400))
			continue;	/* disabled */
		base[i] = (unsigned long)(tx4927_sdramcptr->cr[i] >> 49) << 21;
		size[i] = ((unsigned long)(tx4927_sdramcptr->cr[i] >> 33) + 1) << 21;
	}
	for (i = 0; i < 8; i++) {
		if (!((unsigned long)tx4927_ebuscptr->cr[i] & 0x8))
			continue;	/* disabled */
		base[i+4] = rbtx4927_ce_base[i];
		size[i+4] = rbtx4927_ce_size[i];
	}
	mbegin = base[0] + size[0];	/* end of SDRAM ch 0 */
	mend = base[4];	/* begin of CE0 */

	rbtx4927_pci_mem_base =
		rbtx4927_find_free_region(sizeof(base) / sizeof(base[0]),
					  base, size, mbegin, mend,
					  &rbtx4927_pci_mem_size);
	base[4+8] = rbtx4927_pci_mem_base;
	size[4+8] = rbtx4927_pci_mem_size;

	rbtx4927_pci_io_base =
		rbtx4927_find_free_region(sizeof(base) / sizeof(base[0]),
					  base, size, mbegin, mend,
					  &rbtx4927_pci_io_size);
}

static void __init rbtx4927_pci_setup(void)
{
	static int called = 0;
	int i;
	if (!called) {
		printk("TX4927 PCIC -- DID:%04x VID:%04x RID:%02x Arbiter:%s\n",
		       (unsigned short)(tx4927_pcicptr->pciid >> 16),
		       (unsigned short)(tx4927_pcicptr->pciid & 0xffff),
		       (unsigned short)(tx4927_pcicptr->pciccrev & 0xff),
		       (!(tx4927_ccfgptr->ccfg & TX4927_CCFG_PCIXARB)) ? "External" : "Internal");
		called = 1;
	}
	printk("TX4927 PCIC --%s PCICLK:",
	       (tx4927_ccfgptr->ccfg & TX4927_CCFG_PCI66) ? " PCI66" : "");
	if (tx4927_ccfgptr->pcfg & TX4927_PCFG_PCICLKEN_ALL) {
		int pciclk = 0;
		switch ((unsigned long)tx4927_ccfgptr->ccfg & TX4927_CCFG_PCIDIVMODE_MASK) {
		case TX4927_CCFG_PCIDIVMODE_2_5:
			pciclk = rbtx4927_cpu_clock * 2 / 5; break;
		case TX4927_CCFG_PCIDIVMODE_3:
			pciclk = rbtx4927_cpu_clock / 3; break;
		case TX4927_CCFG_PCIDIVMODE_5:
			pciclk = rbtx4927_cpu_clock / 5; break;
		case TX4927_CCFG_PCIDIVMODE_6:
			pciclk = rbtx4927_cpu_clock / 6; break;
		}
		printk("%dMHz", pciclk / 1000000);
	}
	printk("\n");

	/* GB->PCI mappings */
	tx4927_pcicptr->g2piomask = (rbtx4927_pci_io_size - 1) >> 4;
	tx4927_pcicptr->g2piogbase = rbtx4927_pci_io_base |
#ifdef __BIG_ENDIAN
		TX4927_PCIC_G2PIOGBASE_ECHG
#else
		TX4927_PCIC_G2PIOGBASE_BSDIS
#endif
		;
	tx4927_pcicptr->g2piopbase = 0;
	for (i = 0; i < 3; i++) {
		tx4927_pcicptr->g2pmmask[i] = 0;
		tx4927_pcicptr->g2pmgbase[i] = 0;
		tx4927_pcicptr->g2pmpbase[i] = 0;
	}
	if (rbtx4927_pci_mem_base) {
		tx4927_pcicptr->g2pmmask[0] = (rbtx4927_pci_mem_size - 1) >> 4;
		tx4927_pcicptr->g2pmgbase[0] = rbtx4927_pci_mem_base |
#ifdef __BIG_ENDIAN
			TX4927_PCIC_G2PMnGBASE_ECHG
#else
			TX4927_PCIC_G2PMnGBASE_BSDIS
#endif
			;
		tx4927_pcicptr->g2pmpbase[0] = rbtx4927_pci_mem_base;
	}
	/* PCI->GB mappings (I/O 256B) */
	tx4927_pcicptr->p2giopbase = 0; /* 256B */
	tx4927_pcicptr->p2giogbase = 0;
	/* PCI->GB mappings (MEM 512MB (64MB on R1.x)) */
	tx4927_pcicptr->p2gm0plbase = 0;
	tx4927_pcicptr->p2gm0pubase = 0;
	tx4927_pcicptr->p2gmgbase[0] = 0 |
		TX4927_PCIC_P2GMnGBASE_TMEMEN |
#ifdef __BIG_ENDIAN
		TX4927_PCIC_P2GMnGBASE_TECHG
#else
		TX4927_PCIC_P2GMnGBASE_TBSDIS
#endif
		;
	/* PCI->GB mappings (MEM 16MB) */
	tx4927_pcicptr->p2gm1plbase = 0xffffffff;
	tx4927_pcicptr->p2gm1pubase = 0xffffffff;
	tx4927_pcicptr->p2gmgbase[1] = 0;
	/* PCI->GB mappings (MEM 1MB) */
	tx4927_pcicptr->p2gm2pbase = 0xffffffff; /* 1MB */
	tx4927_pcicptr->p2gmgbase[2] = 0;

	tx4927_pcicptr->pciccfg &= TX4927_PCIC_PCICCFG_LBWC_MASK;
	/* Enable Initiator Memory Space */
	if (rbtx4927_pci_mem_base)
		tx4927_pcicptr->pciccfg |= TX4927_PCIC_PCICCFG_IMSE(0);
	/* Enable Initiator I/O Space, Config */
	tx4927_pcicptr->pciccfg |=
		TX4927_PCIC_PCICCFG_IISE |
		TX4927_PCIC_PCICCFG_ICAE |
		TX4927_PCIC_PCICCFG_ATR;

	/* Do not use MEMMUL, MEMINF: YMFPCI card causes M_ABORT. */
	tx4927_pcicptr->pcicfg1 = 0;

	if (rbtx4927_pcic_trdyto >= 0) {
		tx4927_pcicptr->g2ptocnt &= ~0xff;
		tx4927_pcicptr->g2ptocnt |= (rbtx4927_pcic_trdyto & 0xff);
		printk(KERN_DEBUG "TX4927 PCIC -- TRDYTO:%02lx\n",
		       tx4927_pcicptr->g2ptocnt & 0xff);
	}

	/* Clear All Local Bus Status */
	tx4927_pcicptr->pcicstatus = TX4927_PCIC_PCICSTATUS_ALL;
	/* Enable All Local Bus Interrupts */
	tx4927_pcicptr->pcicmask = TX4927_PCIC_PCICSTATUS_ALL;
	/* Clear All Initiator Status */
	tx4927_pcicptr->g2pstatus = TX4927_PCIC_G2PSTATUS_ALL;
	/* Enable All Initiator Interrupts */
	tx4927_pcicptr->g2pmask = TX4927_PCIC_G2PSTATUS_ALL;
	/* Clear All PCI Status Error */
	tx4927_pcicptr->pcistatus =
		(tx4927_pcicptr->pcistatus & 0x0000ffff) |
		(TX4927_PCIC_PCISTATUS_ALL << 16);
	/* Enable All PCI Status Error Interrupts */
	tx4927_pcicptr->pcimask = TX4927_PCIC_PCISTATUS_ALL;

	/* PCIC Int => IRC IRQ16 */
	tx4927_pcicptr->pcicfg2 =
		(tx4927_pcicptr->pcicfg2 & 0xffffff00) |
		TX4927_IR_PCIC;

	if (!(tx4927_ccfgptr->ccfg & TX4927_CCFG_PCIXARB)) {
		/* XXX */
	} else {
		/* Reset Bus Arbiter */
		tx4927_pcicptr->pbacfg = TX4927_PCIC_PBACFG_RPBA;
		/*
		 * swap reqBP and reqXP (raise priority of PCI-ISA bridge).
		 * SLC90E66(PCI-ISA bridge) are connected to REQ2 on
		 * PCI Backplane board.
		 */
		tx4927_pcicptr->pbareqport = 0x72543610;
		tx4927_pcicptr->pbabm = 0;
		/* Use Fixed ParkMaster (required by PCI-ISA bridge) */
		tx4927_pcicptr->pbacfg = TX4927_PCIC_PBACFG_FIXPA;
		/* Enable Bus Arbiter */
		tx4927_pcicptr->pbacfg |= TX4927_PCIC_PBACFG_PBAEN;
	}

	tx4927_pcicptr->pcistatus = PCI_COMMAND_MASTER |
		PCI_COMMAND_MEMORY |
		PCI_COMMAND_PARITY | PCI_COMMAND_SERR;
}
#endif /* CONFIG_PCI */

#define JMY_CE	5
#define JMY_SIZE	0x400000

#ifdef __LITTLE_ENDIAN	/* TX4927 EBUSC bug workaround */
static inline void store_ll(volatile void *addr, unsigned long long val)
{
	*(volatile unsigned long *)((unsigned long)addr + 4) = (unsigned long)(val >> 32);
	*(volatile unsigned long *)((unsigned long)addr + 0) = (unsigned long)val;
}
#endif
static void __init tx4927_setup(void)
{
	int i;
	int cpuclk = 0;
	unsigned long divmode;

	/* SDRAMC,EBUSC are configured by PROM */
	for (i = 0; i < 8; i++) {
		if (!(tx4927_ebuscptr->cr[i] & 0x8))
			continue;	/* disabled */
		rbtx4927_ce_base[i] = (unsigned long)(tx4927_ebuscptr->cr[i] >> 48) << 20;
		rbtx4927_ce_size[i] = 0x00100000 << ((unsigned long)(tx4927_ebuscptr->cr[i] >> 8) & 0xf);
	}
	/* use CE5 for IEEE1394 board */
	if (rbtx4927_ce_base[JMY_CE] == 0) {
		/* find free 4MB region */
		unsigned long ce_base, ce_size = JMY_SIZE;
		ce_base = rbtx4927_find_free_region(8,
						    rbtx4927_ce_base,
						    rbtx4927_ce_size,
						    0x1c000000, 0x1e000000,
						    &ce_size);
		if (ce_base && ce_size == JMY_SIZE) {
			/* storing longlong may not be atomic. set BA first */
#ifdef __LITTLE_ENDIAN	/* TX4927 EBUSC bug workaround */
			store_ll(&tx4927_ebuscptr->cr[JMY_CE],
				((unsigned long long)(ce_base >> 20) << 48) |
				0x10f210); /* 1/3speed, 4MB */
			/* then enable */
			store_ll(&tx4927_ebuscptr->cr[JMY_CE], tx4927_ebuscptr->cr[JMY_CE] | 8);
#else
			tx4927_ebuscptr->cr[JMY_CE] =
				((unsigned long long)(ce_base >> 20) << 48) |
				0x10f200; /* 4MB */
			/* then enable */
			tx4927_ebuscptr->cr[JMY_CE] |= 8;
#endif
			rbtx4927_ce_base[JMY_CE] = ce_base;
			rbtx4927_ce_size[JMY_CE] = ce_size;
		}
	}

	switch ((unsigned long)tx4927_ccfgptr->ccfg & TX4927_CCFG_DIVMODE2_MASK) {
	case TX4927_CCFG_DIVMODE2_1:
		rbtx4927_gbus_clock = rbtx4927_master_clock; break;
	case TX4927_CCFG_DIVMODE2_4:
		rbtx4927_gbus_clock = rbtx4927_master_clock * 4; break;
	}
	divmode = (unsigned long)tx4927_ccfgptr->ccfg & TX4927_CCFG_DIVMODE_MASK;
	switch (divmode) {
	case TX4927_CCFG_DIVMODE_2:
		cpuclk = rbtx4927_gbus_clock * 2; break;
	case TX4927_CCFG_DIVMODE_3:
		cpuclk = rbtx4927_gbus_clock * 3; break;
	case TX4927_CCFG_DIVMODE_4:
		cpuclk = rbtx4927_gbus_clock * 4; break;
	case TX4927_CCFG_DIVMODE_2_5:
		cpuclk = rbtx4927_gbus_clock * 5 / 2; break;
	}
	rbtx4927_cpu_clock = cpuclk;

	/* CCFG */
	/* enable Timeout BusError */
	if (rbtx4927_ccfg_toeon)
		tx4927_ccfgptr->ccfg |= TX4927_CCFG_TOE;

	/* clear BusErrorOnWrite flag */
	tx4927_ccfgptr->ccfg &= ~TX4927_CCFG_BEOW;

	/* DMA selection */
	tx4927_ccfgptr->pcfg &= ~TX4927_PCFG_DMASEL_ALL;

	printk("TX4927 -- %dMHz(M%dMHz) CRIR:%08lx ",
	       cpuclk / 1000000, rbtx4927_master_clock / 1000000,
	       (unsigned long)tx4927_ccfgptr->crir);
	printk_ll("CCFG:%03lx_%08lx ", tx4927_ccfgptr->ccfg);
	printk_ll("PCFG:%08lx_%08lx\n", tx4927_ccfgptr->pcfg);

	printk("TX4927 SDRAMC --");
	for (i = 0; i < 4; i++) {
		if (!((unsigned long)tx4927_sdramcptr->cr[i] & 0x00000400))
			continue;	/* disabled */
		printk(" CR%d:", i);
		printk_ll("%08lx_%08lx", tx4927_sdramcptr->cr[i]);
	}
	printk_ll(" TR:%01lx_%08lx\n", tx4927_sdramcptr->tr);

	/* IRC */
	/* disable interrupt control */
	tx4927_ircptr->cer = 0;
	/* mask all IRC interrupts */
	tx4927_ircptr->imr = 0;
	for (i = 0; i < TX4927_NUM_IR / 4; i++) {
		tx4927_ircptr->ilr[i] = 0;
	}
	/* setup IRC interrupt mode (Low Active) */
	for (i = 0; i < TX4927_NUM_IR / 16; i++) {
		tx4927_ircptr->cr[i] = 0;
	}

	/* TMR */
	/* disable all timers */
	for (i = 0; i < TX4927_NR_TMR; i++) {
		tx4927_tmrptr(i)->tcr = TXx927_TMTCR_CRE;
		tx4927_tmrptr(i)->tisr = 0;
		tx4927_tmrptr(i)->cpra = 0xffffffff;
		tx4927_tmrptr(i)->itmr = 0;
		tx4927_tmrptr(i)->ccdr = 0;
		tx4927_tmrptr(i)->pgmr = 0;
	}

	/* DMA */
	tx4927_dmaptr->mcr = 0;
	for (i = 0; i < sizeof(tx4927_dmaptr->ch) / sizeof(tx4927_dmaptr->ch[0]); i++) {
		/* reset channel */
		tx4927_dmaptr->ch[i].ccr = TX4927_DMA_CCR_CHRST;
#ifdef __BIG_ENDIAN
		tx4927_dmaptr->ch[i].ccr = 0;
#else
		tx4927_dmaptr->ch[i].ccr = TX4927_DMA_CCR_LE;
#endif
	}
	/* enable DMA */
	tx4927_dmaptr->mcr = TX4927_DMA_MCR_MSTEN;

	/* PCIC */
#ifdef CONFIG_PCI
	rbtx4927_setup_pci_regions();
	if (rbtx4927_pci66 > 0) {
		rbtx4927_pci66_setup();
	} else {
		/* Reset PCI Bus */
		*rbtx4927_pcireset_ptr = 1;
		/* Reset PCIC */
		tx4927_ccfgptr->clkctr |= TX4927_CLKCTR_PCIRST;
		udelay(10000);
		/* clear PCIC reset */
		tx4927_ccfgptr->clkctr &= ~TX4927_CLKCTR_PCIRST;
		*rbtx4927_pcireset_ptr = 0;
	}
	rbtx4927_pci_setup();
#endif

	/* PIO */
	/* PIO[15]: DTR , PIO[14:13]: DIPSW , PIO[1:0]: LEDs */
	tx4927_pioptr->dir = 0x00008003;
	tx4927_pioptr->maskcpu = 0;
	tx4927_pioptr->maskext = 0;
}

/* callback routine from pcibios_init() */
static void __init rbtx4927_pci66_init(void)
{
	rbtx4927_pci66_setup();
	/* Reinitialize PCIC */
	rbtx4927_pci_setup();
}

extern int early_premier3_setup(unsigned long base, int irq);
extern int toshibaboards_probe_ieee1394_board(unsigned long base);
static void __init rbtx4927_check_ieee1394_board(void)
{
	unsigned long long ebccr;

	if (rbtx4927_ce_base[JMY_CE] == 0 ||
	    rbtx4927_ce_size[JMY_CE] < JMY_SIZE)
		return;
	ebccr = tx4927_ebuscptr->cr[JMY_CE];
	/* Normal submode (1/3 speed 4MB) */
#ifdef __LITTLE_ENDIAN	/* TX4927 EBUSC bug workaround */
	store_ll(&tx4927_ebuscptr->cr[JMY_CE],
		 (ebccr & 0xffffffffffc00000ull) | 0x0010f218);
#else
	tx4927_ebuscptr->cr[JMY_CE] =
		(ebccr & 0xffffffffffc00000ull) | 0x0010f218;
#endif
	printk(KERN_DEBUG "EBCCR%d ", JMY_CE);
	printk_ll("%08lx_%08lx\n", tx4927_ebuscptr->cr[JMY_CE]);
	if (toshibaboards_probe_ieee1394_board(RBTX4927_PREMIER3_BASE) == 0) {
		/* External ACK* submode (1/3 speed 4MB) */
#ifdef __LITTLE_ENDIAN	/* TX4927 EBUSC bug workaround */
		store_ll(&tx4927_ebuscptr->cr[JMY_CE],
			 (ebccr & 0xffffffffffc00000ull) | 0x0013f218);
#else
		tx4927_ebuscptr->cr[JMY_CE] =
			(ebccr & 0xffffffffffc00000ull) | 0x0013f218;
#endif
#ifdef CONFIG_IEEE1394_PREMIER3
		early_premier3_setup(RBTX4927_PREMIER3_BASE, RBTX4927_IRQ_PREMIER3);
#endif
	} else {
#ifdef __LITTLE_ENDIAN	/* TX4927 EBUSC bug workaround */
		store_ll(&tx4927_ebuscptr->cr[JMY_CE], ebccr);
#else
		tx4927_ebuscptr->cr[JMY_CE] = ebccr;
#endif
	}
	printk(KERN_DEBUG "EBCCR%d ", JMY_CE);
	printk_ll("%08lx_%08lx\n", tx4927_ebuscptr->cr[JMY_CE]);
}

#define JMI_CE	3

static int rbtx4927_check_iob(void)
{
	unsigned char idt;
	unsigned long flags;
	unsigned long long romcr;
	save_and_cli(flags);
	romcr = tx4927_ebuscptr->cr[JMI_CE];
	tx4927_ebuscptr->cr[JMI_CE] = romcr & ~0x1000ull;	/* do not wait infinitely */
	idt = jmi39io2_isac_reg_in(JMI39IO2_ISAC_REV_ADDR(RBTX4927_IOB_BASE))
		& JMI39IO2_IDT_MASK;
	tx4927_ebuscptr->cr[JMI_CE] = romcr;
	restore_flags(flags);
	return idt == JMI39IO2_ISAC_IDT;
}

static struct resource rbtx4927_ebus_resources[8] = {
	{ "EBUS0", 0, 0, IORESOURCE_MEM },
	{ "EBUS1", 0, 0, IORESOURCE_MEM },
	{ "EBUS2", 0, 0, IORESOURCE_MEM },
	{ "EBUS3", 0, 0, IORESOURCE_MEM },
	{ "EBUS4", 0, 0, IORESOURCE_MEM },
	{ "EBUS5", 0, 0, IORESOURCE_MEM },
	{ "EBUS6", 0, 0, IORESOURCE_MEM },
	{ "EBUS7", 0, 0, IORESOURCE_MEM },
};
static struct resource tx4927_reg_resource = {
	"TX4927", TX4927_REG_BASE, TX4927_REG_BASE+TX4927_REG_SIZE, IORESOURCE_MEM
};

static struct resource rbtx4927_fpga_resource;

void __init rbtx4927_setup(void)
{
	int i;

	tx4927_setup();

	board_time_init = rbtx4927_time_init;
	board_timer_setup = rbtx4927_timer_setup;
	irq_setup = rbtx4927_irq_setup;
	/* map ioport 0 to PCI I/O space address 0 */
	set_io_port_base(KSEG1ADDR(rbtx4927_pci_io_base));

	/* setup PCI area */
	toshibaboards_pci_io_resource.start = 0;
	toshibaboards_pci_io_resource.end = rbtx4927_pci_io_size - 1;
	toshibaboards_pci_mem_resource.start = rbtx4927_pci_mem_base;
	toshibaboards_pci_mem_resource.end =
		rbtx4927_pci_mem_base + rbtx4927_pci_mem_size - 1;

	/* TX4927 internal registers */
	iomem_resource.end = 0xffffffff;	/* 4GB */
	request_resource(&iomem_resource, &tx4927_reg_resource);
	/* reserve EBUS area */
	for (i = 0; i < 8; i++) {
		rbtx4927_ebus_resources[i].start = rbtx4927_ce_base[i];
		rbtx4927_ebus_resources[i].end =
			rbtx4927_ce_base[i] + rbtx4927_ce_size[i] - 1;
		if (rbtx4927_ebus_resources[i].start) {
			request_resource(&iomem_resource,
					 &rbtx4927_ebus_resources[i]);
		}
	}
	rbtx4927_fpga_resource.name = "FPGA Registers";
	rbtx4927_fpga_resource.start =
		virt_to_bus((void *)RBTX4927_FPGA_REG_ADDR);
	rbtx4927_fpga_resource.end =
		virt_to_bus((void *)RBTX4927_FPGA_REG_ADDR) + 0xffff;
	rbtx4927_fpga_resource.flags = IORESOURCE_MEM | IORESOURCE_BUSY;
	request_resource(&rbtx4927_ebus_resources[2], &rbtx4927_fpga_resource);

	/* disable all OnBoard I/O interrupts */
	*rbtx4927_imask_ptr = 0;

	_machine_restart = rbtx4927_machine_restart;
	_machine_halt = rbtx4927_machine_halt;
	_machine_power_off = rbtx4927_machine_power_off;

	rtc_ops = &rbtx4927_rtc_ops;
	toshibaboards_pci_ops = &rbtx4927_pci_ops;
	if (rbtx4927_pci66 == 0)
		toshibaboards_pci66_init = rbtx4927_pci66_init;
	toshibaboards_pci_map_irq = rbtx4927_pci_map_irq;

	/* TX4927-SIO DTR on (PIO[15]) */
	tx4927_pioptr->dout = 0x00008003;

	*rbtx4927_led_ptr = 0xff;
	printk("TOSHIBA RBTX4927 (Rev %02x) --- FPGA(Rev %02x)",
	       *rbtx4927_board_rev_ptr, *rbtx4927_fpga_rev_ptr);
	printk(" DIPSW:%02x\n",
	       *rbtx4927_dipsw_ptr);

	if (rbtx4927_check_iob())
		jmi39io2_init(RBTX4927_IOB_BASE, RBTX4927_IRQ_ISAC);
	if (have_jmi39io2()) {
		jmi39io2_kbd_init();
#ifdef CONFIG_BLK_DEV_IDE
		/* overrides PCI-IDE */
		jmi39io2_ide_init();
#endif
	}

	rbtx4927_check_ieee1394_board();

#if defined(CONFIG_SERIAL_TXX927)
	for(i = 0; i < 2; i++) {
		/* use Pre-scaler T0 (1/2) */
		early_serial_txx927_setup(i,
					  TX4927_SIO_REG(i),
					  RBTX4927_IRQ_IRC_SIO(i),
					  RBTX4927_IMCLK / 2 / 16, /* not constant! */
					  TXx927_SERIAL_HAVE_CTS_LINE);
#ifdef CONFIG_REMOTE_DEBUG
		early_serial_txx927_kgdb_setup(i,
					       TX4927_SIO_REG(i),
					       RBTX4927_IRQ_IRC_SIO(i),
					       RBTX4927_IMCLK / 2 / 16);
#endif
	}
#endif

#ifdef CONFIG_JMI39IO2_NET
	if (have_jmi39io2()) {
		early_jmi39io2_ether_setup(JMI39IO2_ETHER_ADDR(RBTX4927_IOB_BASE) - mips_io_port_base,
					   RBTX4927_IRQ_IRC_INT(JMI39IO2_INT_ETHER));
	}
#endif
#ifdef CONFIG_FB_E1355
	if (have_jmi39io2()) {
		early_e1355_setup(JMI39IO2_LCDVGA_REG_BASE(RBTX4927_IOB_BASE),
				  JMI39IO2_LCDVGA_MEM_BASE(RBTX4927_IOB_BASE));
	}
#endif
}
