/*
 *  linux/arch/mips/toshiba-boards/tsdb/irq.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1999-2001 Toshiba Corporation
 *
 * $Id: irq.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 */
#include <linux/config.h>
#include <linux/init.h>

#include <linux/errno.h>
#include <linux/kernel_stat.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/timex.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#ifdef CONFIG_BLK_DEV_IDEPCI
#include <linux/pci.h>
#endif

#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/mipsregs.h>
#include <asm/system.h>

#include <asm/ptrace.h>
#include <asm/processor.h>
#include <asm/toshiba-boards/irq.h>
#include <asm/toshiba-boards/tsdb.h>

#if TSDB_IRQ_END > NR_IRQS
#error TSDB_IRQ_END > NR_IRQS
#endif

#if 0
static int tsdb_gen_iack(void)
{
	/* generate ACK cycle */
	unsigned long saved_base;
	unsigned long vector;
	struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;
	/* generate ACK cycle */
	saved_base = usc->lb_pci_base[0];
	usc->lb_pci_base[0] =
		(saved_base & ~(USC_LB_PCI_BASE_PCI_CMD_MASK |
				USC_LB_PCI_BASE_PREFETCH |
				USC_LB_PCI_BASE_ERR_EN)) |
		USC_LB_PCI_BASE_PCI_CMD_IACK;
	vector = *(volatile unsigned long *)KSEG1ADDR(TSDB_LB_PCI_APERTURE_0);
	usc->lb_pci_base[0] = saved_base;
	return vector & 0xff;
}
#endif

extern asmlinkage void tsdb_IRQ(void);

static void tsdb_pci_irq_enable(unsigned int irq)
{
	int irq_nr = irq - TSDB_IRQ_PCI;
	if (irq_nr == TSDB_INTB_PCI_ENUM) {
		struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;
		usc->int_cfg[1] |= USC_INTF_INT3;
	} else {
		/* 0: mask */
		*tsdb_pci_imask_ptr = *tsdb_pci_imask_ptr | (1 << irq_nr);
		(void)*tsdb_pci_imask_ptr;	/* flush... */
	}
	wbflush();
}
static void tsdb_pci_irq_disable(unsigned int irq)
{
	int irq_nr = irq - TSDB_IRQ_PCI;
	if (irq_nr == TSDB_INTB_PCI_ENUM) {
		struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;
		usc->int_cfg[1] &= ~USC_INTF_INT3;
	} else {
		/* 0: mask */
		*tsdb_pci_imask_ptr = *tsdb_pci_imask_ptr & ~(1 << irq_nr);
		(void)*tsdb_pci_imask_ptr;	/* flush... */
	}
	wbflush();
}
static unsigned int tsdb_pci_irq_startup(unsigned int irq)
{
	tsdb_pci_irq_enable(irq);
	return 0;
}
#define	tsdb_pci_irq_shutdown	tsdb_pci_irq_disable

static void tsdb_pci_irq_ack(unsigned int irq)
{
	tsdb_pci_irq_disable(irq);
}
static void tsdb_pci_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		tsdb_pci_irq_enable(irq);
}

static hw_irq_controller tsdb_pci_irq_controller = {
	typename:	"PCI",
	startup:	tsdb_pci_irq_startup,
	shutdown:	tsdb_pci_irq_shutdown,
	enable:		tsdb_pci_irq_enable,
	disable:	tsdb_pci_irq_disable,
	ack:		tsdb_pci_irq_ack,
	end:		tsdb_pci_irq_end,
	set_affinity:	NULL,
};


static void tsdb_ioc_irq_enable(unsigned int irq)
{
	int irq_nr = irq - TSDB_IRQ_IOC;
	/* 0: mask */
	*tsdb_ioc_imask_ptr = *tsdb_ioc_imask_ptr | (1 << irq_nr);
	(void)*tsdb_ioc_imask_ptr;	/* flush... */
	wbflush();
}
static void tsdb_ioc_irq_disable(unsigned int irq)
{
	int irq_nr = irq - TSDB_IRQ_IOC;
	/* 0: mask */
	*tsdb_ioc_imask_ptr = *tsdb_ioc_imask_ptr & ~(1 << irq_nr);
	(void)*tsdb_ioc_imask_ptr;	/* flush... */
	wbflush();
}
static unsigned int tsdb_ioc_irq_startup(unsigned int irq)
{
	tsdb_ioc_irq_enable(irq);
	return 0;
}
#define	tsdb_ioc_irq_shutdown	tsdb_ioc_irq_disable

static void tsdb_ioc_irq_ack(unsigned int irq)
{
	tsdb_ioc_irq_disable(irq);
}
static void tsdb_ioc_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		tsdb_ioc_irq_enable(irq);
}

static hw_irq_controller tsdb_ioc_irq_controller = {
	typename:	"IOC",
	startup:	tsdb_ioc_irq_startup,
	shutdown:	tsdb_ioc_irq_shutdown,
	enable:		tsdb_ioc_irq_enable,
	disable:	tsdb_ioc_irq_disable,
	ack:		tsdb_ioc_irq_ack,
	end:		tsdb_ioc_irq_end,
	set_affinity:	NULL,
};


static void tsdb_usc_irq_enable(unsigned int irq)
{
	int irq_nr = irq - TSDB_IRQ_USC;
	/* 1: enable */
	struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;
	usc->int_cfg[0] |= 1 << irq_nr;
	wbflush();
}
static void tsdb_usc_irq_disable(unsigned int irq)
{
	int irq_nr = irq - TSDB_IRQ_USC;
	/* 1: enable */
	struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;
	usc->int_cfg[0] &= ~(1 << irq_nr);
	wbflush();
}
static unsigned int tsdb_usc_irq_startup(unsigned int irq)
{
	tsdb_usc_irq_enable(irq);
	return 0;
}
#define	tsdb_usc_irq_shutdown	tsdb_usc_irq_disable

static void tsdb_usc_irq_ack(unsigned int irq)
{
	struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;

	tsdb_usc_irq_disable(irq);

	/* clear interrupts */
	switch (irq - TSDB_IRQ_USC) {
	case USC_INTB_DRAM_PI:
		break;
	case USC_INTB_DI0:
	case USC_INTB_DI1:
		usc->int_stat = USC_INTF_DI_EN;
		break;
	default:
		usc->int_stat = 1 << (irq - TSDB_IRQ_USC);
	}
}
static void tsdb_usc_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		tsdb_usc_irq_enable(irq);
}

static hw_irq_controller tsdb_usc_irq_controller = {
	typename:	"USC",
	startup:	tsdb_usc_irq_startup,
	shutdown:	tsdb_usc_irq_shutdown,
	enable:		tsdb_usc_irq_enable,
	disable:	tsdb_usc_irq_disable,
	ack:		tsdb_usc_irq_ack,
	end:		tsdb_usc_irq_end,
	set_affinity:	NULL,
};

static struct irqaction pci_action = {
	no_action, 0, 0, "cascade(PCI)", NULL, NULL
};
static struct irqaction ioc_action = {
	no_action, 0, 0, "cascade(IOC)", NULL, NULL
};
static struct irqaction usc_action = {
	no_action, 0, 0, "cascade(USC)", NULL, NULL
};


static void tsdb_pushsw_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
#if 1
	tsdb_leddisp_put(regs->cp0_epc);
#endif
	*tsdb_ioc_istat_ptr = TSDB_INTF_IOC_PUSHSW;
}
static struct irqaction pushsw_action = {
	tsdb_pushsw_interrupt, SA_INTERRUPT, 0, "INT Switch", NULL, NULL,
};

static void tsdb_uscerr_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
	char *s;
#ifdef CONFIG_BLK_DEV_IDEPCI
	/* ignore MasterAbort for ide probing... */
	if (irq == TSDB_IRQ_USC_PCI_M_ABORT) {
		struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;
		usc->pci_stat |= PCI_STATUS_REC_MASTER_ABORT;
		usc->int_stat |= USC_INTF_PCI_M_ABORT;
		/* avoid spurious interrupt...(why?) */
		*tsdb_fpga_rev_ptr = *tsdb_fpga_rev_ptr;
		wbflush();
		return;
	}
#endif
	switch (irq) {
	case TSDB_IRQ_USC_PSLAVE_PI: s = "PSLAVE_PI"; break;
	case TSDB_IRQ_USC_PMASTER_PI: s = "PMASTER_PI"; break;
	case TSDB_IRQ_USC_PCI_T_ABORT: s = "PCI_T_ABORT"; break;
	case TSDB_IRQ_USC_PCI_M_ABORT: s = "PCI_M_ABORT"; break;
	case TSDB_IRQ_USC_DRAM_PI: s = "DRAM_PI"; break;
	default: s = "UNKNOWN";
	}
	printk(KERN_ERR "USC error (%s) interrupt at 0x%08lx.\n", s, regs->cp0_epc);
}
static struct irqaction uscerr_action = {
	tsdb_uscerr_interrupt, SA_INTERRUPT, 0, "USC error", NULL, NULL,
};
static void tsdb_pcierr_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
#if 1
	printk("PCI error interrupt (irq 0x%x).\n", irq);
#endif
}
static struct irqaction pci_perr_action = {
	tsdb_pcierr_interrupt, SA_INTERRUPT, 0, "PCI PERR", NULL, NULL,
};
static struct irqaction pci_serr_action = {
	tsdb_pcierr_interrupt, SA_INTERRUPT, 0, "PCI SERR", NULL, NULL,
};

void tsdb_local_irqdispatch(struct pt_regs *regs, int irq)
{
#ifdef CONFIG_TX_BRANCH_LIKELY_BUG_WORKAROUND
	tx_branch_likely_bug_fixup(regs);
#endif
#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
	tx_fpu_c_bug_fixup(regs);
#endif
	do_IRQ(irq, regs);
	wbflush();
}

void tsdb_usc_irqdispatch(struct pt_regs *regs)
{
	struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;
	unsigned long istat = usc->int_stat;
	int i;
#ifdef CONFIG_TX_BRANCH_LIKELY_BUG_WORKAROUND
	tx_branch_likely_bug_fixup(regs);
#endif
#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
	tx_fpu_c_bug_fixup(regs);
#endif
	/* only check INT_CFG0 */
	istat &= usc->int_cfg[0];
	for (i = 0; i < TSDB_NR_IRQ_USC; i++) {
		if (istat & (1 << i)) {
			do_IRQ(TSDB_IRQ_USC + i, regs);
			break;
		}
	}
	wbflush();
}

void tsdb_ioc_irqdispatch(struct pt_regs *regs)
{
	unsigned char istat = *tsdb_ioc_istat_ptr;	/* 0: int */
	unsigned char imask = *tsdb_ioc_imask_ptr;	/* 0: mask */
	int i;
#ifdef CONFIG_TX_BRANCH_LIKELY_BUG_WORKAROUND
	tx_branch_likely_bug_fixup(regs);
#endif
#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
	tx_fpu_c_bug_fixup(regs);
#endif
	istat = (~istat) & imask;
	for (i = 0; i < TSDB_NR_IRQ_IOC; i++) {
		if (istat & (1 << i)) {
			do_IRQ(TSDB_IRQ_IOC + i, regs);
			break;
		}
	}
	wbflush();
}

int tsdb_pci_irqdispatch(struct pt_regs *regs)
{
	struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;
	unsigned char istat = *tsdb_pci_istat_ptr;	/* 0: int */
	unsigned char imask = *tsdb_pci_imask_ptr;	/* 0: mask */
	int i;
	int irq;
	istat = (~istat) & imask;

#ifdef CONFIG_TX_BRANCH_LIKELY_BUG_WORKAROUND
	tx_branch_likely_bug_fixup(regs);
#endif
#ifdef CONFIG_TX_FPU_C_BUG_WORKAROUND
	tx_fpu_c_bug_fixup(regs);
#endif
	/* check ENUM */
	if ((usc->int_cfg[1] & USC_INTF_INT3) &&
	    (usc->int_stat & USC_INTF_INT3)) {
		do_IRQ(TSDB_IRQ_PCI_ENUM, regs);
		wbflush();
		return 0;
	}

	for (i = 0; i < TSDB_NR_IRQ_PCI; i++) {
		if (istat & (1 << i)) {
			irq = TSDB_IRQ_PCI + i;
			irq = toshibaboards_i8259_irqroute(irq);
			if (irq < 0)
				return -1;
			do_IRQ(irq, regs);
			break;
		}
	}
	wbflush();
	return 0;
}

void __init tsdb_irq_setup(void)
{
	struct v320usc_reg *usc = (struct v320usc_reg *)TSDB_USC_BASE;
	int i;

	/* mask all IOC/PCI interrupts */
	*tsdb_ioc_imask_ptr = 0;
	*tsdb_pci_imask_ptr = 0;

	/* clear PushSW/NMI interrupts */
	*tsdb_ioc_istat_ptr = TSDB_INTF_IOC_PUSHSW;
	*tsdb_nmi_istat_ptr = TSDB_INTF_NMI_NMI;

	/* clear SoftReset/SoftInt interrupts */
	*tsdb_softreset_ptr = 1;
	*tsdb_softint_ptr = 1;

	/* clear PCI Reset interrupts */
	*tsdb_pcireset_ptr = 1;

	/* inputs from INT2,INT3 generate an INT1 interrupt. */
	usc->int_cfg[0] = 0;
	usc->int_cfg[1] = USC_INTF_INT2;
	usc->int_cfg[2] = 0;
	usc->int_cfg3 = 0;

	set_except_vector(0, tsdb_IRQ);

	/* setup irq descriptors */
	mips_cpu_irq_init(TSDB_IRQ_LOCAL);
	for (i = TSDB_IRQ_PCI; i < TSDB_IRQ_PCI + TSDB_NR_IRQ_PCI; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &tsdb_pci_irq_controller;
	}
	for (i = TSDB_IRQ_IOC; i < TSDB_IRQ_IOC + TSDB_NR_IRQ_IOC; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &tsdb_ioc_irq_controller;
	}
	for (i = TSDB_IRQ_USC; i < TSDB_IRQ_USC + TSDB_NR_IRQ_USC; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &tsdb_usc_irq_controller;
	}

	setup_irq(TSDB_IRQ_LOCAL_PCI, &pci_action);
	setup_irq(TSDB_IRQ_LOCAL_IOC, &ioc_action);
	setup_irq(TSDB_IRQ_LOCAL_USC, &usc_action);

	/* enable pushsw int. */
	setup_irq(TSDB_IRQ_IOC_PUSHSW, &pushsw_action);

#if 0	/* XXX */
	/* enable ENUM int. */
	if ((usc->int_stat & USC_INTF_INT3) == 0) {
		setup_irq(TSDB_IRQ_PCI_ENUM, &enum_action);
	}
#endif

	setup_irq(TSDB_IRQ_USC_PSLAVE_PI, &uscerr_action);
	setup_irq(TSDB_IRQ_USC_PMASTER_PI, &uscerr_action);
	setup_irq(TSDB_IRQ_USC_PCI_T_ABORT, &uscerr_action);
	setup_irq(TSDB_IRQ_USC_PCI_M_ABORT, &uscerr_action);
	setup_irq(TSDB_IRQ_USC_DRAM_PI, &uscerr_action);
	
	setup_irq(TSDB_IRQ_PCI_SERR, &pci_serr_action);
	setup_irq(TSDB_IRQ_PCI_PERR, &pci_perr_action);

#if 0
	toshibaboards_gen_iack = tsdb_gen_iack;
#endif
}
