/* $Id: pci.c,v 1.1.1.1 2004/04/07 08:36:50 louistsai Exp $
 *
 * TX4927 specific PCI support.
 *
 * based on arch/mips/sni/pci.c
 * Copyright (C) 1997, 1998 Ralf Baechle
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation
 */
#include <linux/config.h>
#include <linux/pci.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <asm/byteorder.h>
#include <asm/toshiba-boards/tx4927evb.h>

#ifdef CONFIG_PCI

int tx4927evb_pci_map_irq(struct pci_dev *dev, u8 slot, u8 pin)
{
	int irq = pin;

	/* IRQ rotation */
	irq--;	/* 0-3 */
	if (dev->bus->parent == NULL &&
	    (slot == TX4927_PCIC_IDSEL_AD_TO_SLOT(21) ||
	     slot == TX4927_PCIC_IDSEL_AD_TO_SLOT(22))) {
		/* PCI CardSlot (IDSEL=A21,A22) */
		/* PCIA => PCIC (IDSEL=A21) */
		/* PCIA => PCID (IDSEL=A22) */
		irq = (irq + 2 + slot) % 4;
	} else {
		/* PCI Backplane */
		irq = (irq + 3 + slot) % 4;
	}
	irq++;	/* 1-4 */

	switch (irq) {
	case 1:
		irq = TX4927EVB_IRQ_IOC_PCIA;
		break;
	case 2:
		irq = TX4927EVB_IRQ_IOC_PCIB;
		break;
	case 3:
		irq = TX4927EVB_IRQ_IOC_PCIC;
		break;
	case 4:
		irq = TX4927EVB_IRQ_IOC_PCID;
		break;
	}
	return irq;
}


static int
mkaddr(struct pci_dev *dev, int where)
{
	if (dev->bus->number == 0 &&
	    dev->devfn >= PCI_DEVFN(TX4927_PCIC_MAX_DEVNU, 0))
		return -1;
	tx4927_pcicptr->g2pcfgadrs = ((dev->bus->number & 0xff) << 0x10) |
				     ((dev->devfn & 0xff) << 0x08) |
				     (where  & 0xfc);
	/* clear M_ABORT and Disable M_ABORT Int. */
	tx4927_pcicptr->pcistatus =
		(tx4927_pcicptr->pcistatus & 0x0000ffff) |
		(PCI_STATUS_REC_MASTER_ABORT << 16);
	return 0;
}

static int
check_abort(void)
{
	int code = PCIBIOS_SUCCESSFUL;
	if (tx4927_pcicptr->pcistatus & (PCI_STATUS_REC_MASTER_ABORT << 16)) {
		tx4927_pcicptr->pcistatus =
			(tx4927_pcicptr->pcistatus & 0x0000ffff) |
			(PCI_STATUS_REC_MASTER_ABORT << 16);
		code = PCIBIOS_DEVICE_NOT_FOUND;
	}
#if 1 /* TX4927 PCIC BUG WORKAROND */
	/* if G2PCFGADRS[2] is 0, G2PINTACK may cause problems. */
	tx4927_pcicptr->g2pcfgadrs |= 4;
#endif
	return code;
}

static inline u8 icd_readb(int offset)
{
#ifdef __BIG_ENDIAN
	offset ^= 3;
#endif
	return *(volatile u8 *)((ulong)&tx4927_pcicptr->g2pcfgdata | offset);
}
static inline u16 icd_readw(int offset)
{
#ifdef __BIG_ENDIAN
	offset ^= 2;
#endif
	return *(volatile u16 *)((ulong)&tx4927_pcicptr->g2pcfgdata | offset);
}
static inline u32 icd_readl(void)
{
	return tx4927_pcicptr->g2pcfgdata;
}
static inline void icd_writeb(u8 val, int offset)
{
#ifdef __BIG_ENDIAN
	offset ^= 3;
#endif
	*(volatile u8 *)((ulong)&tx4927_pcicptr->g2pcfgdata | offset) = val;
}
static inline void icd_writew(u16 val, int offset)
{
#ifdef __BIG_ENDIAN
	offset ^= 2;
#endif
	*(volatile u16 *)((ulong)&tx4927_pcicptr->g2pcfgdata | offset) = val;
}
static inline void icd_writel(u32 val)
{
	tx4927_pcicptr->g2pcfgdata = val;
}

/*
 * We can't address 8 and 16 bit words directly.  Instead we have to
 * read/write a 32bit word and mask/modify the data we actually want.
 */
static int tx4927evb_pci_read_config_byte(struct pci_dev *dev, int where,
					  u8 *val)
{
	if (mkaddr(dev, where))
		return -1;
	*val = icd_readb(where & 3);
	return check_abort();
}

static int tx4927evb_pci_read_config_word(struct pci_dev *dev, int where,
					  u16 *val)
{
	if (mkaddr(dev, where))
		return -1;
	*val = icd_readw(where & 3);
	return check_abort();
}

static int tx4927evb_pci_read_config_dword(struct pci_dev *dev, int where,
					   u32 *val)
{
	if (mkaddr(dev, where))
		return -1;
	*val = icd_readl();
	return check_abort();
}

static int tx4927evb_pci_write_config_byte(struct pci_dev *dev, int where,
					   u8 val)
{
	if (mkaddr(dev, where))
		return -1;
	icd_writeb(val, where & 3);
	return check_abort();
}

static int tx4927evb_pci_write_config_word(struct pci_dev *dev, int where,
					   u16 val)
{
	if (mkaddr(dev, where))
		return -1;
	icd_writew(val, where & 3);
	return check_abort();
}

static int tx4927evb_pci_write_config_dword(struct pci_dev *dev, int where,
					    u32 val)
{
	if (mkaddr(dev, where))
		return -1;
	icd_writel(val);
	return check_abort();
}

struct pci_ops tx4927evb_pci_ops = {
	read_byte:      tx4927evb_pci_read_config_byte,
	read_word:      tx4927evb_pci_read_config_word,
	read_dword:     tx4927evb_pci_read_config_dword,
	write_byte:     tx4927evb_pci_write_config_byte,
	write_word:     tx4927evb_pci_write_config_word,
	write_dword:    tx4927evb_pci_write_config_dword
};

#endif /* CONFIG_PCI */
