/*
 * $Id: toshiba-tx.c,v 1.1.1.1 2004/04/07 08:36:54 louistsai Exp $
 *
 * Mapping of a custom board with both AMD CFI and JEDEC flash in partitions.
 * Config with both CFI and JEDEC device support.
 *
 * Basically physmap.c with the addition of partitions and 
 * an array of mapping info to accomodate more than one flash type per board.
 *
 * Based on cstm_mips_ixx.c by MontaVista Software Inc.
 * Copyright 2000 MontaVista Software Inc.
 * Copyright 2001 Toshiba Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <linux/config.h>

#if defined(CONFIG_TOSHIBA_SDB)
#include <linux/delay.h>
#endif
#include <asm/bootinfo.h>
#ifdef CONFIG_TOSHIBA_SDB
#include <asm/toshiba-boards/tsdb.h>
#endif
#ifdef CONFIG_TOSHIBA_JMR3927
#include <asm/toshiba-boards/jmr3927.h>
#endif
#ifdef CONFIG_TOSHIBA_TX4927EVB
#include <asm/toshiba-boards/tx4927evb.h>
#endif
#ifdef CONFIG_TOSHIBA_RBTX4927
#include <asm/toshiba-boards/rbtx4927.h>
#endif
#ifdef CONFIG_TOSHIBA_RBTX4925
#include <asm/toshiba-boards/rbtx4925.h>
#endif

#define IGNORE_UNALIGNED /* XXX really needed? */

static __u8 toshiba_tx_read8(struct map_info *map, unsigned long ofs)
{
	return *(volatile __u8 *)(map->map_priv_1 + ofs);
}

static __u16 toshiba_tx_read16(struct map_info *map, unsigned long ofs)
{
#ifdef IGNORE_UNALIGNED
	if ((map->map_priv_1 + ofs) & 1) return 0xffff;
#endif
	return *(volatile __u16 *)(map->map_priv_1 + ofs);
}

static __u32 toshiba_tx_read32(struct map_info *map, unsigned long ofs)
{
#ifdef IGNORE_UNALIGNED
	if ((map->map_priv_1 + ofs) & 3) return 0xffffffff;
#endif
	return *(volatile __u32 *)(map->map_priv_1 + ofs);
}

static void toshiba_tx_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	//printk(KERN_NOTICE "flash: from = %04x, len = %04x, first word = %04x\n", from, len, *(volatile __u16 *) (map->map_priv_1+from));
	memcpy_fromio(to, map->map_priv_1 + from, len);
}

static void toshiba_tx_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	*(volatile __u8 *)(map->map_priv_1 + adr) = d;
}

static void toshiba_tx_write16(struct map_info *map, __u16 d, unsigned long adr)
{
#ifdef IGNORE_UNALIGNED
	if ((map->map_priv_1 + adr) & 1) return;
#endif
	*(volatile __u16 *)(map->map_priv_1 + adr) = d;
}

static void toshiba_tx_write32(struct map_info *map, __u32 d, unsigned long adr)
{
#ifdef IGNORE_UNALIGNED
	if ((map->map_priv_1 + adr) & 3) return;
#endif
	*(volatile __u32 *)(map->map_priv_1 + adr) = d;
}

static void toshiba_tx_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	memcpy_toio(map->map_priv_1 + to, from, len);
}

static void toshiba_tx_set_vpp(struct map_info *map,int vpp)
{
  if (vpp) {
#ifdef CONFIG_TOSHIBA_SDB
	if (mips_machtype == MACH_TOSHIBA_SDB) {
		static u8 first = 1;
		*tsdb_flashwp_ptr = 0;
		if (first) {
			first = 0;
			udelay(1000);
		}
	}
#endif
  }
  else {
#ifdef CONFIG_TOSHIBA_SDB
	if (mips_machtype == MACH_TOSHIBA_SDB) {
		*tsdb_flashwp_ptr = 1;
	}
#endif
  }
}

static const struct map_info basic_toshiba_tx_map = {
	NULL,
	0,
	0,
	toshiba_tx_read8,
	toshiba_tx_read16,
	toshiba_tx_read32,
	toshiba_tx_copy_from,
	toshiba_tx_write8,
	toshiba_tx_write16,
	toshiba_tx_write32,
	toshiba_tx_copy_to,
        toshiba_tx_set_vpp,
	0,
	0
};

/* board and partition description */

#define MAX_PHYSMAP_PARTITIONS    8
struct toshiba_tx_info {
	char *name;
	unsigned long window_addr;
	unsigned long window_size;
	int buswidth;
	int num_partitions;
};

#define PHYSMAP_NUMBER  3  // number of board desc structs needed, one per contiguous flash type 
static struct toshiba_tx_info toshiba_tx_board_desc[PHYSMAP_NUMBER] = 
{
    {  
	NULL,	// name
	0,	// window_addr
	0,	// window_size
	0,	// buswidth
	1,	// num_partitions
    },

};
#ifdef CONFIG_MTD_PARTITIONS
static struct mtd_partition toshiba_tx_partitions[PHYSMAP_NUMBER][MAX_PHYSMAP_PARTITIONS] = {
{ 
	{
		name: "main partition",
		size:  0,
		offset: 0,
	},
},
};
#endif

struct map_info toshiba_tx_map[PHYSMAP_NUMBER];

int __init init_toshiba_tx(void)
{
	int i;
        struct mtd_info *mymtd = NULL;
#ifdef CONFIG_MTD_PARTITIONS
        struct mtd_partition *parts;
#endif

	/* override toshiba_tx_board_desc[] */
#ifdef CONFIG_TOSHIBA_SDB
	if (mips_machtype == MACH_TOSHIBA_SDB) {
		if (*tsdb_dipsw1_ptr & TSDB_DIPSW1_BOOTROM) {
			/* boot from Boot PROM */
			toshiba_tx_board_desc[0].name = "Boot PROM";
			toshiba_tx_board_desc[0].window_addr =
				TSDB_LB_ROM_APERTURE + 0x03c00000;
			toshiba_tx_board_desc[0].window_size = 0x80000;
			toshiba_tx_board_desc[0].buswidth = 4;
			toshiba_tx_board_desc[1].name = "SystemFlash1";
			toshiba_tx_board_desc[1].window_addr =
				TSDB_LB_ROM_APERTURE + 0x01000000;
			toshiba_tx_board_desc[1].window_size = 0x1000000;
			toshiba_tx_board_desc[1].buswidth = 4;
			toshiba_tx_board_desc[2].name = "SystemFlash2";
			toshiba_tx_board_desc[2].window_addr =
				TSDB_LB_ROM_APERTURE;
			toshiba_tx_board_desc[2].window_size = 0x1000000;
			toshiba_tx_board_desc[2].buswidth = 4;
		} else {
			/* boot from System Flash */
			toshiba_tx_board_desc[0].name = "SystemFlash1";
			toshiba_tx_board_desc[0].window_addr =
				TSDB_LB_ROM_APERTURE + 0x03000000;
			toshiba_tx_board_desc[0].window_size = 0x1000000;
			toshiba_tx_board_desc[0].buswidth = 4;
			toshiba_tx_board_desc[1].name = "SystemFlash2";
			toshiba_tx_board_desc[1].window_addr =
				TSDB_LB_ROM_APERTURE + 0x02000000;
			toshiba_tx_board_desc[1].window_size = 0x1000000;
			toshiba_tx_board_desc[1].buswidth = 4;
			if ((*tsdb_dipsw2_ptr &
			     (TSDB_DIPSW2_EXTROM_AC|TSDB_DIPSW2_BPROM_AC)) ==
			    TSDB_DIPSW2_BPROM_AC) {
				toshiba_tx_board_desc[2].name = "Boot PROM";
				toshiba_tx_board_desc[2].window_addr =
					TSDB_LB_ROM_APERTURE;
				toshiba_tx_board_desc[2].window_size = 0x80000;
				toshiba_tx_board_desc[2].buswidth = 4;
			}
		}
	}
#endif /* CONFIG_TOSHIBA_SDB */
#ifdef CONFIG_TOSHIBA_JMR3927
	if (mips_machtype == MACH_TOSHIBA_JMR3927) {
		toshiba_tx_board_desc[0].window_addr = JMR3927_ROMCE0;
		toshiba_tx_board_desc[1].window_addr = JMR3927_ROMCE1;
		if (tx3927_romcptr->cr[0] & 0x80) {
			toshiba_tx_board_desc[0].name = "Flash-3";
			toshiba_tx_board_desc[0].window_size = 0x100000;
			toshiba_tx_board_desc[0].buswidth = 2;
			toshiba_tx_board_desc[1].name = "Flash-1";
			toshiba_tx_board_desc[1].window_size = 0x400000;
			toshiba_tx_board_desc[1].buswidth = 4;
		} else {
			toshiba_tx_board_desc[0].name = "Flash-1";
			toshiba_tx_board_desc[0].window_size = 0x400000;
			toshiba_tx_board_desc[0].buswidth = 4;
			toshiba_tx_board_desc[1].name = "Flash-3";
			toshiba_tx_board_desc[1].window_size = 0x100000;
			toshiba_tx_board_desc[1].buswidth = 2;
		}
	}
#endif /* CONFIG_TOSHIBA_JMR3927 */
#ifdef CONFIG_TOSHIBA_TX4927EVB
	if (mips_machtype == MACH_TOSHIBA_TX4927EVB) {
		toshiba_tx_board_desc[0].name = "Flash CE0";
		toshiba_tx_board_desc[0].window_addr = TX4927EVB_CE0;
		toshiba_tx_board_desc[1].name = "Flash CE1";
		toshiba_tx_board_desc[1].window_addr = TX4927EVB_CE1;
		toshiba_tx_board_desc[2].name = "Flash CE2";
		toshiba_tx_board_desc[2].window_addr = TX4927EVB_CE2;
		for (i = 0; i < 3; i++) {
			/* temporary map size */
			toshiba_tx_board_desc[i].window_size = 0x400000;
			toshiba_tx_board_desc[i].buswidth = 4;
			/* ignore if ExternalACK* mode */
			if ((tx4927_ebuscptr->cr[i] & 0x0003f000) == 0x0003f000)
				toshiba_tx_board_desc[i].name = 0;
		}
	}
#endif /* CONFIG_TOSHIBA_TX4927EVB */
#ifdef CONFIG_TOSHIBA_RBTX4927
	if (mips_machtype == MACH_TOSHIBA_RBTX4927) {
		toshiba_tx_board_desc[0].window_addr = RBTX4927_CE0;
		toshiba_tx_board_desc[1].window_addr = RBTX4927_CE1;
		if ((tx4927_ebuscptr->cr[0] & 0x300000) == 0x200000) {
			toshiba_tx_board_desc[0].name = "Flash 2";
			toshiba_tx_board_desc[0].buswidth = 2;
			toshiba_tx_board_desc[1].name = "Flash 1";
			toshiba_tx_board_desc[1].buswidth = 4;
		} else {
			toshiba_tx_board_desc[0].name = "Flash 1";
			toshiba_tx_board_desc[0].buswidth = 4;
			toshiba_tx_board_desc[1].name = "Flash 2";
			toshiba_tx_board_desc[1].buswidth = 2;
		}
		for (i = 0; i < 2; i++) {
			/* temporary map size */
			toshiba_tx_board_desc[i].window_size = 0x200000;
		}
	}
#endif /* CONFIG_TOSHIBA_RBTX4927 */
#ifdef CONFIG_TOSHIBA_RBTX4925
	if (mips_machtype == MACH_TOSHIBA_RBTX4925) {
		toshiba_tx_board_desc[0].window_addr = RBTX4925_CE0+0x1000000;
		toshiba_tx_board_desc[1].window_addr = RBTX4925_CE1+0x1000000;
		if ((tx4925_ebuscptr->ch[0].cr & 0x300000) == 0x200000) {
			toshiba_tx_board_desc[0].name = "Flash 2";
			toshiba_tx_board_desc[0].buswidth = 2;
			toshiba_tx_board_desc[1].name = "Flash 1(Top)";
			toshiba_tx_board_desc[1].buswidth = 4;
			toshiba_tx_board_desc[2].window_addr = RBTX4925_CE1;
			toshiba_tx_board_desc[2].name = "Flash 1(Bottom)";
			toshiba_tx_board_desc[2].buswidth = 4;
		} else {
			toshiba_tx_board_desc[0].name = "Flash 1(Top)";
			toshiba_tx_board_desc[0].buswidth = 4;
			toshiba_tx_board_desc[1].name = "Flash 2";
			toshiba_tx_board_desc[1].buswidth = 2;
			toshiba_tx_board_desc[2].window_addr = RBTX4925_CE0;
			toshiba_tx_board_desc[2].name = "Flash 1(Bottom)";
			toshiba_tx_board_desc[2].buswidth = 4;
		}
		for (i = 0; i < 3; i++) {
			/* temporary map size */
			toshiba_tx_board_desc[i].window_size = 0x200000;
		}
	}
#endif /* CONFIG_TOSHIBA_RBTX4925 */
	/* Initialize mapping */
	for (i=0;i<PHYSMAP_NUMBER;i++) {
		if (!toshiba_tx_board_desc[i].name)
			continue;
		printk(KERN_DEBUG "toshiba_tx flash device: probe %lx at %lx\n", toshiba_tx_board_desc[i].window_size, toshiba_tx_board_desc[i].window_addr);
                memcpy((char *)&toshiba_tx_map[i],(char *)&basic_toshiba_tx_map,sizeof(struct map_info));
		toshiba_tx_map[i].map_priv_1 = (unsigned long)ioremap(toshiba_tx_board_desc[i].window_addr, toshiba_tx_board_desc[i].window_size);
		if (!toshiba_tx_map[i].map_priv_1) {
			printk(KERN_WARNING "Failed to ioremap\n");
			return -EIO;
	        }
		toshiba_tx_map[i].name = toshiba_tx_board_desc[i].name;
		toshiba_tx_map[i].size = toshiba_tx_board_desc[i].window_size;
		toshiba_tx_map[i].buswidth = toshiba_tx_board_desc[i].buswidth;
		//printk(KERN_NOTICE "toshiba_tx: ioremap is %x\n",(unsigned int)(toshiba_tx_map[i].map_priv_1));
	}

	for (i=0;i<PHYSMAP_NUMBER;i++) {
		struct toshiba_tx_info *txinfo = &toshiba_tx_board_desc[i];
		if (!toshiba_tx_map[i].name)
			continue;
#ifdef CONFIG_MTD_PARTITIONS
                parts = &toshiba_tx_partitions[i][0];
#endif
		mymtd = (struct mtd_info *)do_map_probe("cfi_probe", &toshiba_tx_map[i]);
		//printk(KERN_NOTICE "phymap %d cfi_probe: mymtd is %x\n",i,(unsigned int)mymtd);
		if (!mymtd) {
			mymtd = (struct mtd_info *)do_map_probe("jedec_probe", &toshiba_tx_map[i]);
			//printk(KERN_NOTICE "toshiba_tx %d jedec: mymtd is %x\n",i,(unsigned int)mymtd);
		}
		if (mymtd) {
			mymtd->module = THIS_MODULE;
#if defined(CONFIG_TOSHIBA_TX4927EVB) || defined(CONFIG_TOSHIBA_RBTX4927) \
	|| defined(CONFIG_TOSHIBA_RBTX4925)
			/* true map size */
			toshiba_tx_map[i].size = mymtd->size;
			/* remap 1f000000-1f3fffff to 1fc00000-1fffffff */
			if (txinfo->window_addr == 0x1f000000 &&
			    toshiba_tx_map[i].size <= 0x00400000) {
				txinfo->window_addr = 0x1fc00000;
				iounmap((void *)toshiba_tx_map[i].map_priv_1);
				toshiba_tx_map[i].map_priv_1 =
					(unsigned long)ioremap(txinfo->window_addr, txinfo->window_size);
			}
#endif
			printk(KERN_NOTICE "toshiba_tx flash device(%s): %lx at %lx\n",
			       toshiba_tx_map[i].name, toshiba_tx_map[i].size,
			       txinfo->window_addr);
#ifdef CONFIG_MTD_PARTITIONS
			/* override toshiba_tx_partitions[] */
			toshiba_tx_partitions[i][0].size = txinfo->window_size;
#endif

	                toshiba_tx_map[i].map_priv_2 = (unsigned long)mymtd;
#ifdef CONFIG_MTD_PARTITIONS
		        add_mtd_partitions(mymtd, parts, txinfo->num_partitions);
#else
			add_mtd_device(mymtd);
#endif
		}
	}
	if (!mymtd)
		return -ENXIO;
	return 0;
}

static void __exit cleanup_toshiba_tx(void)
{
	int i;
        struct mtd_info *mymtd;

	for (i=0;i<PHYSMAP_NUMBER;i++) {
	        mymtd = (struct mtd_info *)toshiba_tx_map[i].map_priv_2;
		if (mymtd) {
#ifdef CONFIG_MTD_PARTITIONS
			del_mtd_partitions(mymtd);
#else
			del_mtd_device(mymtd);
#endif
			map_destroy(mymtd);
		}
		if (toshiba_tx_map[i].map_priv_1) {
			iounmap((void *)toshiba_tx_map[i].map_priv_1);
			toshiba_tx_map[i].map_priv_1 = 0;
		}
	}
}

module_init(init_toshiba_tx);
module_exit(cleanup_toshiba_tx);


MODULE_LICENSE("GPL");
MODULE_AUTHOR("TOSHIBA Corporation");
MODULE_DESCRIPTION("MTD map driver for TOSHIBA TX boards");
