/* $Id: tx4925_rbtx4925.c,v 1.1.1.1 2004/04/07 08:36:58 louistsai Exp $
 * tx4925_rbtx4925.c: RBTX4925 board specific pcmcia routines.
 *
 * RBTX4925 has 1 PCMCIA slot.  The slot is connected to TX4925 via
 * LinkUp L1121 chip.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 2001 Toshiba Corporation
 */
#include <linux/kernel.h>
#include <linux/sched.h>

#include <asm/toshiba-boards/rbtx4925.h>
#include "tx4925_pcmcia.h"

static int rbtx4925_pcmcia_init(struct pcmcia_init *init)
{
	int irq;

	/* EBUSC are configured in arch-specific setup routine */

	/* L1121 are configured in arch-specific setup routine */

	if (*rbtx4925_piosel_ptr & RBTX4925_PIOSEL_NOPCMCIA) {
		printk(KERN_ERR "%s: disabled by PIOSEL\n", __FUNCTION__);
		return -1;
	}

	if (L1121_inb(L1121_IDR1) != L1121_IDR1_VAL ||
	    L1121_inb(L1121_IDR2) != L1121_IDR2_VAL ||
	    L1121_inb(L1121_IDR3) != L1121_IDR3_VAL ||
	    L1121_inb(L1121_IDR3) != L1121_IDR3_VAL) {
		printk(KERN_ERR "%s: L1121 ID mismatch\n", __FUNCTION__);
		return -1;
	}

	/* Register interrupts */
	irq = RBTX4925_IRQ_PCMCIA + L1121_CD1;
	if (request_irq(irq, init->handler, SA_INTERRUPT, "PCMCIA CD1", NULL) < 0)
		goto irq_err;
	irq = RBTX4925_IRQ_PCMCIA + L1121_CD2;
	if (request_irq(irq, init->handler, SA_INTERRUPT, "PCMCIA CD2", NULL) < 0)
		goto irq_err;
	irq = RBTX4925_IRQ_PCMCIA + L1121_VS1;
	if (request_irq(irq, init->handler, SA_INTERRUPT, "PCMCIA VS1", NULL) < 0)
		goto irq_err;
	irq = RBTX4925_IRQ_PCMCIA + L1121_VS2;
	if (request_irq(irq, init->handler, SA_INTERRUPT, "PCMCIA VS2", NULL) < 0)
		goto irq_err;

	/* enable on-chip power switch */
	L1121_outb(L1121_CR2_PDCS, L1121_CR2);

	/* There's two sockets, but only the first one, 0, is used */
	return 1;
irq_err:
	printk(KERN_ERR "%s: Request for IRQ %u failed\n", __FUNCTION__, irq);
	return -1;
}

static int rbtx4925_pcmcia_shutdown(void)
{
	/* disable on-chip power switch */
	L1121_outb(0, L1121_CR2);
	return 0;
}

static int 
rbtx4925_pcmcia_socket_state(unsigned sock, struct pcmcia_state *state)
{
	u8 sr = L1121_inb(L1121_SR);

	if (sock != 0)
		return -1;

	state->ready = 0;
	state->vs_Xv = 0;
	state->vs_3v = 0;
	state->detect = 0;

	/* 
	 * This is tricky. The READY pin is also the #IRQ pin.  We'll treat
	 * READY as #IRQ and set state->ready to 1 whenever state->detect 
	 * is true.
	 */
	if (!(sr & (L1121_SR_CD1 | L1121_SR_CD2))) {
		state->detect = 1;
		if (sr & L1121_SR_VS1)
			state->vs_3v = 1;
	}

	if (state->detect)
		state->ready = 1;

	state->bvd1 = (sr & L1121_SR_BVD1) ? 0 : 1;
	state->bvd2 = (sr & L1121_SR_BVD2) ? 0 : 1;
	state->wrprot = (sr & L1121_SR_IOIS16) ? 0 : 1;
	return 1;
}


static int rbtx4925_pcmcia_get_irq_info(struct pcmcia_irq_info *info)
{
	if (info->sock != 0)
		return -1;
	info->irq = RBTX4925_IRQ_PCMCIA + L1121_RDY;
	return 0;
}


static int 
rbtx4925_pcmcia_configure_socket(const struct pcmcia_configure *configure)
{
	u8 cr;

	if (configure->sock != 0)
		return -1;

	cr = L1121_inb(L1121_CR1);
	cr &= ~(L1121_CR1_RESET | L1121_CR1_SOE);
	if (configure->reset)
		cr |= L1121_CR1_RESET;
	if (configure->output)
		cr |= L1121_CR1_SOE;
	L1121_outb(cr, L1121_CR1);

	cr = L1121_inb(L1121_CR2);
	cr &= ~(L1121_CR2_S3 | L1121_CR2_S4);
	switch(configure->vcc){
		case 0:  /* Vcc 0 */
			break;
		case 50: /* Vcc 5V */
			cr |= L1121_CR2_S3;
			break;
		case 33: /* Vcc 3.3V */
			cr |= L1121_CR2_S4;
			break;
		default: /* what's this ? */
			printk(KERN_ERR "%s: bad Vcc %d\n", 
					__FUNCTION__, configure->vcc);
			break;
	}
	L1121_outb(cr, L1121_CR2);

	cr = L1121_inb(L1121_CR3);
	cr &= ~L1121_CR3_MIO;
	if (!configure->iocard)
		cr |= L1121_CR3_MIO;
	L1121_outb(cr, L1121_CR3);
	return 0;
}

struct pcmcia_low_level rbtx4925_pcmcia_ops = { 
	rbtx4925_pcmcia_init,
	rbtx4925_pcmcia_shutdown,
	rbtx4925_pcmcia_socket_state,
	rbtx4925_pcmcia_get_irq_info,
	rbtx4925_pcmcia_configure_socket
};

