/*
This file is a framebuffer driver for Chips And Technologies
(later Intel and now Asiliant) grafic chips and is subject to the terms and
conditions of the GNU General Public License.
Created in Jan - July 2000 by Thomas Hhenleitner <th@visuelle-maschinen.de>

At the end of this file you'll find some README information.

Code starts about 50 line from here. This file includes itself a few times
to avoid several files and to avoid repeating the same words again and again.
Before each include of this file the macro PAR is defined with different
meanings and to enable only the next few lines.
*/
/*
 * Modified for TOSHIBA Boards (Jul/2001)
 * Copyright (C) 2001 Toshiba Corporation
 */

// ############################################################################
#ifdef PAR // here starts later included part #################################
// ############################################################################

// macro names PAR and PARAMETERS are declared later in this file always
// if this file is included.

#ifdef PARAMETERS
PAR( mode, mode, "initial screen mode - see init_fb_var_screeninfo for details" )
PAR( crtfp, crtfp, "CRT/FP mode - 0:crt, 1:fp, 2:both" )
#undef PARAMETERS
#endif

#define DEFAULTDESCRIPTION "see fb.h struct var_screeninfo"
// following names are exactly the names of the members of struct
// fb_var_screeninfo, do not change them - otherwise some macros will not work

// append = "video=ctfb:x:1280,xv:1280,y:1024,yv:1024,depth:16,pclk:7800,le:248,ri:16,up:38,lo:1,hs:144,vs:3,sync=3,vmode:0,accel:0"

        PAR( x,         xres,          "horizotal resolution" )
        PAR( xv,        xres_virtual,  "virtual horz. res." )
        PAR( y,         yres,          "vertical resolution" )
        PAR( yv,        yres_virtual,  "virtual vert. res." )
        PAR( xo,        xoffset,       DEFAULTDESRIPTION )
        PAR( yo,        yoffset,       DEFAULTDESRIPTION )
        PAR( depth,     bits_per_pixel,DEFAULTDESRIPTION )
        PAR( gs,        grayscale,     DEFAULTDESRIPTION )
        PAR( nonstd,    nonstd,        DEFAULTDESRIPTION )
        PAR( act,       activate,      DEFAULTDESRIPTION )
        PAR( h,         height,        DEFAULTDESRIPTION )
        PAR( w,         width,         DEFAULTDESRIPTION )
        PAR( accel,     accel_flags,   DEFAULTDESRIPTION )
        PAR( pclk,      pixclock,      DEFAULTDESRIPTION )
        PAR( le,        left_margin,   DEFAULTDESRIPTION )
        PAR( ri,        right_margin,  DEFAULTDESRIPTION )
        PAR( up,        upper_margin,  DEFAULTDESRIPTION )
        PAR( lo,        lower_margin,  DEFAULTDESRIPTION )
        PAR( hs,        hsync_len,     DEFAULTDESRIPTION )
        PAR( vs,        vsync_len,     DEFAULTDESRIPTION )
        PAR( sync,      sync,          DEFAULTDESRIPTION )
        PAR( vmode,     vmode,         DEFAULTDESRIPTION )


#undef DEFAULTDESCRIPTION

#undef PAR

// ############################################################################
#else // here starts the real framebuffer code ################################
// ############################################################################


//_____________________________________________________________________________

// module interface

#define FB_VERSION "ver 0.52+"

  #include <linux/version.h>
  #include <linux/module.h>
  #include <linux/mm.h>
  #include <linux/fb.h>
  #include <linux/init.h>
  #include <linux/errno.h>
  #include <linux/kernel.h>
  #include <asm/io.h>                                   // todo: include path
#ifdef MODULE
  #include </usr/src/linux/include/video/fbcon.h>       // <video/fbcon.h>
  #include </usr/src/linux/include/video/fbcon-cfb8.h>  // <video/fbcon-cfb8.h>
  #include </usr/src/linux/include/video/fbcon-cfb16.h> // <video/fbcon-cfb16.h>
  #include </usr/src/linux/include/video/fbcon-cfb24.h> // <video/fbcon-cfb24.h>
#else
  #include <video/fbcon.h>       // <video/fbcon.h>
  #include <video/fbcon-cfb8.h>  // <video/fbcon-cfb8.h>
  #include <video/fbcon-cfb16.h> // <video/fbcon-cfb16.h>
  #include <video/fbcon-cfb24.h> // <video/fbcon-cfb24.h>
#endif
  #include <linux/console.h>
  #include <asm/uaccess.h>
  #include <linux/selection.h> // color_table[]
  #include <linux/slab.h> // was malloc.h
  #include <linux/delay.h>
  #include <linux/pci.h>
#ifdef CONFIG_TOSHIBA_BOARDS
/* values for "crtfp" ooption */
#define CRTFP_MODE_CRT	0
#define CRTFP_MODE_LCD	1
#define CRTFP_MODE_BOTH	2
#define CRTFP_MODE_COUNT	3
#endif

#define PCI_DEVICE_ID_CT_69000 0x00c0 // ct69000 datasheet Rev. 1.3 chapter 7-2
#define PCI_DEVICE_ID_CT_69030 0x0c30 // see assiliantfb.c from Adrian Cox

#define KERNEL24_INTERFACE ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,00) )
/* probably the new interface is already a little bit older than 2.4.0 */

//_____________________________________________________________________________

// general defines

#define PRINTK( fmt, args... ) \
	printk( "%s%s: " fmt, ctfb, __FUNCTION__ , ##args )

#ifndef min
#define min( a, b ) ( ( a ) < ( b ) ) ? ( a ) : ( b )
#endif
#ifndef max
#define max( a, b ) ( ( a ) > ( b ) ) ? ( a ) : ( b )
#endif
#define N_ELTS( x ) ( sizeof( x ) / sizeof( x[ 0 ] ) )

//_____________________________________________________________________________

// internal interface

/*****************************************************************************/
// There are only dynamic instances of this struct, see functions ct_...()
// and fbgen.c. device dependend data struct: The hardware specific data in
// this structure uniquely defines a video mode.
struct ct_par
{
	struct fb_var_screeninfo var; // we get away with that
	struct fb_fix_screeninfo fix; // we get away with that
};

/*****************************************************************************/
// init modes count - adapt init_screeninfo_var(), if you change this number
#define SETUPMODECOUNT (sizeof(fbxxx_init) / sizeof(fbxxx_init[0]))

/*****************************************************************************/
struct pci_dev_info
{
	unsigned short	vendor;		/* vendor id */
	unsigned short	device;		/* device id */
	const char	*name;		/* device name */
};

/*****************************************************************************/
struct ctfb_info
{
	struct fb_info_gen gen; // starts internally with fb_info info - nust
				// be the first - see <linux/fb.h>
	struct fb_monspecs ctfb_monspecs; // not really used, see ct_detect()
	unsigned int frame_buffer_phys;   // address
//	unsigned long blitter_regs_phys;
//	unsigned long blitter_data_phys;
	unsigned int      io_base_phys;   // address
	__u8  *frame_buffer;              // mapped address
//	__u32 *blitter_regs;
//	__u8  *blitter_data;
//	__u8  *io_base;
	unsigned int current_bpp;         // only used in ct_setcol()
	struct fb_fix_screeninfo fix;
	struct fb_var_screeninfo var;
	struct display init_disp;
	union cfb //struct cfb // part of disp, filled by ctfb_setcolreg
	{
		#ifdef FBCON_HAS_CFB8
			u16 cfb8[ 16 ];
		#endif
		#ifdef FBCON_HAS_CFB16
			u16 cfb16[ 16 ];
		#endif
		#ifdef FBCON_HAS_CFB24
			u32 cfb24[ 16 ];
                #endif
	} cmap ;
	struct pci_dev *pci;             // pci device info
};

// defined in hw dependend part at the end and used in following general part
static int ct_blank( int blank_mode, struct fb_info_gen *info );
static int ct_getcolreg( unsigned regno, unsigned *red, unsigned *green,
	unsigned *blue, unsigned *transp, struct fb_info *info );
static int ct_setcolreg( unsigned regno, unsigned  red, unsigned  green,
	unsigned  blue, unsigned  transp, struct fb_info *info );
static void ct_hw_write( const struct ct_par * pp );
static void ct_hw_read ( struct ct_par * pp );
static void ct_hw_init( struct ctfb_info *p ) __init;



//_____________________________________________________________________________

// general part of frame buffer code
// (mainly without hw depencies exept pci stuff)

char* ctfb = "ctfb: ";
#define PARAMETERS
#define PAR( s, a, b ) static int a;
#include "ctfb.c"
extern struct fb_ops ctfb_ops;
static void set_bitfields( struct fb_var_screeninfo *pvar );
static int check_var_screeninfo( struct fb_var_screeninfo *var );
static void set_dispsw( unsigned bpp, struct display *pd, struct ctfb_info *pi );


/*****************************************************************************/
struct ctfb_info // this is a module global pointer to our data struct
	*p_ctfb_info = NULL;	

#define DEVICE(vid,did,name) \
	{ PCI_VENDOR_ID_##vid, PCI_DEVICE_ID_##did, (name) }

static struct pci_dev_info pcidev_info[ ] __initdata =
{
	DEVICE( CT, CT_65554, "CT65554"),   // max 16 chars for strings
	DEVICE( CT, CT_69000, "CT69000"),
	DEVICE( CT, CT_69030, "CT69030"),   // untested
	DEVICE( CT, CT_65550, "CT65550"),   // untested
	DEVICE( CT, CT_65555, "CT65555"),   // untested
};

static unsigned int __initdata FoundPciDeviceIndex = 0;

/*****************************************************************************/
static int __init
SearchPciCtDevice( struct pci_dev **pdev )
{
	static const int CheckPciDeviceCount = N_ELTS( pcidev_info );
	int i = 0;
	for( i = 0; i < CheckPciDeviceCount; i++ ){
		if( ( *pdev = pci_find_device( pcidev_info[ i ].vendor,
		pcidev_info[ i ].device, NULL ) ) ){
			FoundPciDeviceIndex = i;
			PRINTK( "%s found.\n", pcidev_info[ i ].name );
			return 0;			
		}	
	}
	printk(KERN_DEBUG);
	PRINTK( "No device of type " );
	for( i = 0; i < CheckPciDeviceCount; i++ )
		printk( "%s ", pcidev_info[ i ].name ); 	
	printk( "found.\n" );
	return -ENODEV;
}

/*****************************************************************************/
static int __init
init_pcidevice( struct pci_dev **p )
{
	short int cmd = 0;
	int err = 0;
        char *s0 = "pci_", *s1 = "", *s2 = "_config_word failed\n";
	if( SearchPciCtDevice( p ) ) return -ENODEV;
        err = pci_read_config_word( *p, PCI_COMMAND, &cmd );
     	if( err ){ s1 = "read";  PRINTK( "%s%s%s", s0, s1, s2 ); return err; }
	// enable memory and IO space
	err = pci_write_config_word( *p, PCI_COMMAND, cmd | 3 );
        if( err ){ s1 = "write"; PRINTK( "%s%s%s", s0, s1, s2 ); return err; }
	return 0;
}

/*****************************************************************************/
static unsigned int
FbMemLen( void )
{
	unsigned int m = 0x100000;
	switch( p_ctfb_info->pci->device ){
		case PCI_DEVICE_ID_CT_69030: return 0x400000;
		case PCI_DEVICE_ID_CT_65550:
		case PCI_DEVICE_ID_CT_65555:
			return m;
		case PCI_DEVICE_ID_CT_65554: return m;
		case PCI_DEVICE_ID_CT_69000: return 0x200000;
		default:PRINTK( "Unexpected p_ctfb_info->pci->device \
				PCI_DEVICE_ID = 0x%x! Using value 0x%x\n",
			 	p_ctfb_info->pci->device, m );
				return m;
	}
}

/*****************************************************************************/
// struct fb_memspecs is used by fbmon_valid_timings() in fbmem.c,
// but it does nothing (code is not active)
// struct fb_memspecs is used by fbmon_dpms(),
// but I know nobody calling this function
void __init
init_ctfb_monspecs( struct ctfb_info *p )
{
	struct fb_monspecs *pm = &p->ctfb_monspecs;
	pm->hfmin = 1;			/* hfreq lower limit (Hz) */
	pm->hfmax = 200000;             /* hfreq upper limit (Hz) */
	pm->vfmin = 20;			/* vfreq lower limit (Hz) */
	pm->vfmax = 200;		/* vfreq upper limit (Hz) */
	pm->dpms  = 0;			/* supports DPMS */
}

/*****************************************************************************/
static void
ct_detect( void ) // dummy
{
}

/*****************************************************************************/
static int  // par->fix
ct_encode_fix( struct fb_fix_screeninfo *fix, const void *par,
	       struct fb_info_gen *info )
{
	const struct ct_par *pp = par;
	struct ctfb_info    *pi = ( struct ctfb_info * )info;
	if( !fix || !par || !info ){ return -EINVAL; }
	// overwrite line length (bytes), because it could have changed
	pi->fix.line_length =( pp->var.xres_virtual
			     * pp->var.bits_per_pixel ) >> 3; // bytes
	pi->fix.visual = pp->fix.visual; // the only value we read from the hw
	*fix = pi->fix;  // copy struct
	return 0;
}

/*****************************************************************************/
static int // var->par
ct_decode_var( const struct fb_var_screeninfo *var, void *par,
	       struct fb_info_gen *info )
{
	struct ct_par *pp = par;
	struct fb_var_screeninfo v;
	if( !var || !par || !info ){ return -EINVAL; }
	v = *var; // copy struct for save manipulation
	set_bitfields( &v );
	if( check_var_screeninfo( &v ) ) return -EINVAL;
	pp->var = v; // copy struct
	pp->fix = p_ctfb_info->fix; // copy struct, we need that because
		//var->par is followed by par->hw, which uses par.fix.visual	
	return 0;
}

/*****************************************************************************/
static int // par->var
ct_encode_var( struct fb_var_screeninfo *var, const void *par,
	       struct fb_info_gen *info )
{
	const struct ct_par *pp = par;
	if( !var || !par || !info ) return -EINVAL;
	*var = pp->var; // copy struct
	return 0;
}

/*****************************************************************************/
static void // hw->par
ct_get_par( void *par, struct fb_info_gen *info )
{
	ct_hw_read( ( struct ct_par * )par );
}

/*****************************************************************************/
static void  // par->hw
ct_set_par( const void *par, struct fb_info_gen *info )
{
	const struct ct_par *pp = par;
	ct_hw_write( pp );
	// we need to copy only for ct_setcol()
	( ( struct ctfb_info * )info )->current_bpp = pp->var.bits_per_pixel;
}

/*****************************************************************************/
static void //par->disp
ct_set_disp( const void *par, struct display *disp, struct fb_info_gen *info)
{ // this function ic called, if we change the resolution
	struct ctfb_info *pi = ( struct ctfb_info * )info;
	struct fb_var_screeninfo *pv = &( ( struct ct_par * )par )->var;
	set_dispsw( pv->bits_per_pixel, disp, pi );
	disp->line_length = pi->fix.line_length;  // set the linelength too?
	disp->var = *pv; // copy struct, todo: probaby not nessesary
	ct_set_par( par, info ); // par->hw
}

/*****************************************************************************/
static struct fbgen_hwswitch // Interface to hardware functions
ct_switch = {
	ct_detect,	
	ct_encode_fix,	
	ct_decode_var,	
	ct_encode_var,	
	ct_get_par,	
	ct_set_par,	
	ct_getcolreg,	
	ct_setcolreg,	
	NULL,         // pan_display
	ct_blank, 	
	ct_set_disp,	
};

#define IOLEN 0x800
#ifdef __BIG_ENDIAN  //see 69000 data sheet page 5-7
#ifdef __mips__
#define ENDIAN_OFFS   0
#else
#define ENDIAN_OFFS   0x800000
#endif
#else
#define ENDIAN_OFFS   0
#endif
/*****************************************************************************/
static void __init
init_fb_addresses( struct ctfb_info *pi )
{
#if KERNEL24_INTERFACE
	pi->frame_buffer_phys = pci_resource_start( p_ctfb_info->pci, 0 )
			      + ENDIAN_OFFS;
        pi->io_base_phys      = pci_resource_start( p_ctfb_info->pci, 0 ) 
                              + 0x400000;
#else
	pi->frame_buffer_phys = p_ctfb_info->pci->base_address[ 0 ]
			      + ENDIAN_OFFS;
        pi->io_base_phys      = p_ctfb_info->pci->base_address[ 0 ] + 0x400000;
#endif
//	pi->blitter_regs_phys = p_ctfb_info->pci->base_address[ 0 ] + 0x400000;
//	pi->ER_regs	      = p_ctfb_info->pci->base_address[ 0 ] + 0x400600;
//	pi->VGA_and_subregs   = p_ctfb_info->pci->base_address[ 0 ] + 0x400700;
//	pi->blitter_data_phys = pi->frame_buffer_phys  + 0x410000;   //  64 KB
	// todo?: _PAGE_NO_CACHE
        pi->frame_buffer      = ioremap( pi->frame_buffer_phys, FbMemLen( ) );
//	pi->io_base           = ioremap( pi->io_base_phys, IOLEN );
//	pi->blitter_regs      = ioremap( pi->blitter_regs_phys, 0x1000 );
//	pi->blitter_data      = ioremap( pi->blitter_data_phys, 0x10000 );
}

/*****************************************************************************/
static int
check_var_screeninfo( struct fb_var_screeninfo *v )
{
	int retval = 0;
	#define c { 0, 0, 0 }                     //     x     y    xv    yv xo xo bpp  g  r  g  b  t  n  a   h   w  a  pixclk   le   ri   up   lo   hs  vs  s  m
	static const struct fb_var_screeninfo min = {  256,  128,  256,  128, 0, 0,  8, 0, c, c, c, c, 0, 0, -1, -1, 0,   4545,   0,   0,   0,   0,  10,  1, 0, 0 };
	static const struct fb_var_screeninfo max = { 1664, 1248, 1664, 1248, 0, 0, 24, 0, c, c, c, c, 0, 0, -1, -1, 0, 666665, 400, 400, 100, 100, 400, 20, 0, 0 };
	#undef c
        // XF86_FBDev sets xres_virtual and yres_virtual to -1
	v->xres_virtual = ( v->xres_virtual == -1 ) ? v->xres : v->xres_virtual;
	v->yres_virtual = ( v->yres_virtual == -1 ) ? v->yres : v->yres_virtual;
	#define PAR( a,x,b )                                  \
	if( ( v->##x < min.##x ) || ( v->##x > max.##x ) ){   \
		retval = -EINVAL;                             \
		PRINTK( #x" = %d invalid value!\n", v->##x ); \
	}
	#include "ctfb.c"
	if ( v->xres > v->xres_virtual ){ 
		retval = -EINVAL;
		PRINTK( "xres_virtual = %d invalid value!\n", v->xres_virtual );
	}
	if ( v->yres > v->yres_virtual ){ 
		retval = -EINVAL;
		PRINTK( "yres_virtual = %d invalid value!\n", v->yres_virtual );
	}
	// todo: check ColorBitsInfo
	// todo: check mutual influence of parameters
	return retval;
}

/*****************************************************************************/
static void
set_bitfields( struct fb_var_screeninfo *pvar ) // supports only 8, 16, 24 bpp
{
#ifdef __BIG_ENDIAN
	/* R5G3B5 for 16bpp */
	static struct fb_bitfield red   [ 3 ] =
		{ { 0, 8, 0 }, {  3, 5, 0 }, {  0, 8, 0 } };
	static struct fb_bitfield green [ 3 ] =
		{ { 0, 8, 0 }, {  0, 3, 0 }, {  8, 8, 0 } };
	static struct fb_bitfield blue  [ 3 ] =
		{ { 0, 8, 0 }, {  8, 5, 0 }, { 16, 8, 0 } };
#else
	static struct fb_bitfield red   [ 3 ] =
		{ { 0, 8, 0 }, { 11, 5, 0 }, { 16, 8, 0 } };
	static struct fb_bitfield green [ 3 ] =
		{ { 0, 8, 0 }, {  5, 6, 0 }, {  8, 8, 0 } };
	static struct fb_bitfield blue  [ 3 ] =
		{ { 0, 8, 0 }, {  0, 5, 0 }, {  0, 8, 0 } };
#endif
	static struct fb_bitfield transp      =   { 0, 0, 0 };
	int i = ( pvar->bits_per_pixel >> 3 ) - 1;
	pvar->red    = red[ i ];
	pvar->green  = green[ i ];
	pvar->blue   = blue[ i ];
	pvar->transp = transp;
}

/*****************************************************************************/
struct initvalue{
	struct fb_var_screeninfo var;
	char* name;
};

#define c { 0, 0, 0 }
static const struct initvalue
fbxxx_init[] = { // todo:  __initdata causes a section type conflict if compiled into the kernel - why?
	//{    x     y    xv    yv xo xo bpp  g  r  g  b  t  n  a   h   w         activate  pixclk   le   ri   up   lo   hs  vs  s                 vmode   },          name,     nessesary sram
	{ {  640,  480,   -1,   -1, 0, 0, 24, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,  41000,  56,  16,  24,   4,  80,  4, 0, FB_VMODE_NONINTERLACED },    "640x480x24",  }, // 1 MB
#ifdef CONFIG_TOSHIBA_BOARDS
	{ {  800,  600,   -1,   -1, 0, 0, 16, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,  25195,  96,  48,  32,  10,  80,  4, 0, FB_VMODE_NONINTERLACED },    "800x600x16",  }, // 1 MB (60Hz)
#else
	{ {  800,  600,   -1,   -1, 0, 0, 16, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,  16004,  96,  48,  32,  10,  80,  4, 0, FB_VMODE_NONINTERLACED },    "800x600x16",  }, // 1 MB
#endif
	{ { 1024,  768,   -1,   -1, 0, 0,  8, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,  12004, 160,  40,  32,  10,  80,  4, 0, FB_VMODE_NONINTERLACED },    "1024x768x8",  }, // 1 MB
	{ {  960,  720,   -1,   -1, 0, 0, 24, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,  13100, 160,  40,  32,   8,  80,  4, 0, FB_VMODE_NONINTERLACED },    "960x720x24",  }, // 2 MB
	{ { 1152,  864,   -1,   -1, 0, 0, 16, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,   9400, 200,  64,  32,  16,  80,  4, 0, FB_VMODE_NONINTERLACED },   "1152x864x16",  }, // 2 MB
	{ { 1664, 1248,   -1,   -1, 0, 0,  8, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,   6678, 240, 100,  32,  16,  80,  4, 0, FB_VMODE_NONINTERLACED },   "1664x1248x8",  }, // 2 MB, only 55 Hz vfreq!
	{ {  640,  480,   -1,   -1, 0, 0, 24, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,  71990, 104,  80,  84,  28,  64,  8, 0, FB_VMODE_INTERLACED    }, "PALi640x480x24", },
	{ {  320,  320,   -1,   -1, 0, 0, 24, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,  14309,  24,  40,  40,  24,  64,  8, 0, FB_VMODE_NONINTERLACED }, "PALu320x240x24", },
#ifdef CONFIG_TOSHIBA_BOARDS
	{ {  800,  480,   -1,   -1, 0, 0, 24, 0, c, c, c, c, 0, 0, -1, -1, FB_ACTIVATE_NOW,  30090,  88,  40,  31,  12,  128,  2, 0, FB_VMODE_NONINTERLACED },    "800x480x24",  }, // 2 MB (VSync 59.6Hz HSync 30.5KHz)
#endif
	};
#undef c


/*****************************************************************************/
static int __init
init_fb_var_screeninfo( struct ctfb_info *pi )
{
	pi->var = fbxxx_init[ mode ].var; // copy struct
	#define PAR( s, val, x )                                \
		do{                                             \
			if( val != -1 ){                        \
			pi->var.val = val;                      \
			}                                       \
		}                                               \
		while( 0 ); // overwrite with params, if specified
        #include "ctfb.c"
	set_bitfields( &pi->var );
	return check_var_screeninfo( &pi->var );
}

/*****************************************************************************/
unsigned int
fb_visual_color( unsigned int bpp )
{			  // use color table      // no mapping
	return bpp == 8 ? FB_VISUAL_PSEUDOCOLOR : FB_VISUAL_TRUECOLOR;
}	

/*****************************************************************************/
static void __init
init_fb_fix_screeninfo( struct ctfb_info *pi )
{
	int bpp = pi->var.bits_per_pixel;
	int vxres = pi->var.xres_virtual;
	strcpy( pi->fix.id, pcidev_info[ FoundPciDeviceIndex ].name ); // 16 ch
#if KERNEL24_INTERFACE
	pi->fix.smem_start  =  pi->frame_buffer_phys;	// phys. addr
#else
	pi->fix.smem_start  = ( char* ) pi->frame_buffer_phys;	// phys. addr
#endif
	pi->fix.smem_len    = FbMemLen( );	// Length of frame buffer mem
	pi->fix.type        = FB_TYPE_PACKED_PIXELS; // see FB_TYPE_* 		
//	pi->fix.type_aux    = 0; // Interleave for interleaved Planes
	pi->fix.visual      = fb_visual_color( bpp );
//	pi->fix.xpanstep    = 0; // zero if no hardware panning
//	pi->fix.ypanstep    = 0; // zero if no hardware panning
//	pi->fix.ywrapstep   = 0; // zero if no hardware ywrap
	pi->fix.line_length = ( vxres * bpp ) >> 3; // in bytes
#if KERNEL24_INTERFACE
	pi->fix.mmio_start  = pi->io_base_phys; // phys. addr
#else	
	pi->fix.mmio_start  = (char *) pi->io_base_phys; // phys. addr
#endif
	pi->fix.mmio_len    = IOLEN;	// Length of Memory Mapped I/O
	pi->fix.accel       = FB_ACCEL_NONE; // Type of acceleration available
	//__u16 reserved[3];		// Reserved for future compatibility
}

/*****************************************************************************/
static void
set_dispsw( unsigned bpp, struct display *pd, struct ctfb_info *pi )
{
	switch( bpp ){
	case 24:
		#ifdef FBCON_HAS_CFB24
		pd->dispsw      = &fbcon_cfb24;    // low level operations
		pd->dispsw_data = &pi->cmap.cfb24; // opt. dispsw helper data
		#endif
		break;
	case 16:
		#ifdef FBCON_HAS_CFB16
		pd->dispsw      = &fbcon_cfb16;    // low level operations
		pd->dispsw_data = &pi->cmap.cfb16; // opt.dispsw helper data
		#endif
		break;
	case 8:
		#ifdef FBCON_HAS_CFB8
		pd->dispsw      = &fbcon_cfb8;     // low level operations
		pd->dispsw_data = &pi->cmap.cfb8;  // opt. dispsw helper data
		#endif
		break;
	default:
		pd->dispsw      = &fbcon_dummy;
		pd->dispsw_data = NULL; //todo: (?) fb_default_cmap( 16 );
	}
}

/*****************************************************************************/
static void __init
init_fb_display_struct( struct ctfb_info *pi ) // fix,var->disp
{
	struct fb_var_screeninfo *pv = &pi->var;
	pi->init_disp.var               = *pv;  // copy struct
/*      pi->init_disp.cmap.start        = 0;    // colormap
        pi->init_disp.cmap.len          = 0;
        pi->init_disp.cmap.red          = NULL;
        pi->init_disp.cmap.green        = NULL;
        pi->init_disp.cmap.blue         = NULL;
        pi->init_disp.cmap.transp       = NULL;   */
	pi->init_disp.screen_base       = pi->frame_buffer; // virtual addres
	pi->init_disp.visual            = pi->fix.visual;
	pi->init_disp.type              = pi->fix.type;     // see FB_TYPE_*
	pi->init_disp.type_aux          = pi->fix.type_aux; // Interleave
	pi->init_disp.ypanstep          = pi->fix.xpanstep; // 0 if no hw ypan
	pi->init_disp.ywrapstep         = pi->fix.ypanstep; // 0 f no hw ywrap
	pi->init_disp.line_length       = pi->fix.line_length;// in bytes
	pi->init_disp.can_soft_blank    = 1; // zero if no hardware blanking
        pi->init_disp.inverse           = 1; // != 0 text black on white
        set_dispsw( pv->bits_per_pixel, &pi->init_disp, pi );
	pi->init_disp.scrollmode 	   = SCROLL_YREDRAW;
	// other members of displa filled in by the low-level console driver
}

/*****************************************************************************/
static void __init
init_fb_info_struct( struct ctfb_info *pi )
{
	// default video mode (max len 40 chars)
        strcpy( pi->gen.info.modename, fbxxx_init[ mode ].name );
	pi->gen.info.node          = -1;            // todo: Who needs that?
        pi->gen.info.flags         = FBINFO_FLAG_DEFAULT;// todo: sense?
	pi->gen.info.fbops         = &ctfb_ops;
	pi->gen.info.monspecs      = pi->ctfb_monspecs;  // copy struct
	pi->gen.info.disp          = &pi->init_disp; // initial disp. variable
	pi->gen.info.display_fg    = NULL; // Console visible on this display
	pi->gen.info.fontname[ 0 ] = '\0'; // default font name, max 40 chars
//	pi->gen.info.changevar is initialized with &fbcon_changevar in fbcon.c
	// Interface to the low level console driver:
	pi->gen.info.switch_con    = &fbgen_switch;      // switch console
	pi->gen.info.updatevar     = &fbgen_update_var;  // update the vars
	pi->gen.info.blank         = &fbgen_blank;    // (un)blank the screen
}

/*****************************************************************************/
static void // Entries for a generic frame buffer device
init_fb_info_gen_struct( struct ctfb_info *pi )
{
	pi->gen.parsize = sizeof( struct ct_par );
	pi->gen.fbhw = &ct_switch;
}

/*****************************************************************************/
static int __init
alloc_chipsinfo_mem( void )
{
	unsigned size = sizeof( struct ctfb_info );
	p_ctfb_info = ( struct ctfb_info * )kmalloc( size, GFP_ATOMIC );
	if( p_ctfb_info == 0 ){
		PRINTK( "kmalloc failed\n" ); return  -ENOMEM;
	}
	memset( p_ctfb_info, 0, size );
        return 0;
}

/*****************************************************************************/
static int // Open the frame buffer device
ctfb_open( struct fb_info *pi, int user )
{
	MOD_INC_USE_COUNT;
 	return 0 ;
}

/*****************************************************************************/
static int // Release the frame buffer device
ctfb_release( struct fb_info *pi, int user )
{
	MOD_DEC_USE_COUNT;
 	return 0;
}

/*****************************************************************************/
static int
ctfb_get_fix( struct fb_fix_screeninfo *fix, int con, struct fb_info *info )
{
	return fbgen_get_fix( fix, con, info);
}

/*****************************************************************************/
static int
ctfb_get_var( struct fb_var_screeninfo *var, int con, struct fb_info *info )
{
	return fbgen_get_var( var, con, info );
}

/*****************************************************************************/
static int
ctfb_set_var( struct fb_var_screeninfo *var, int con, struct fb_info *info )
{
	return fbgen_set_var( var, con, info );
}

/*****************************************************************************/
static int
ctfb_get_cmap( struct fb_cmap *cmap, int kspc, int con, struct fb_info *info )
{
	return fbgen_get_cmap( cmap, kspc, con, info );
}

/*****************************************************************************/
static int
ctfb_set_cmap( struct fb_cmap *cmap, int kspc, int con, struct fb_info *info )
{
	return fbgen_set_cmap( cmap, kspc, con, info );
}

/*****************************************************************************/
static int
ctfb_pan_display( struct fb_var_screeninfo *var, int con, struct fb_info *info )
{
	return fbgen_pan_display( var, con, info );
}

/*****************************************************************************/
struct fb_ops // In most cases the `generic' routines (fbgen_*) should be ok
ctfb_ops = {  // However, you're free to fill in your own replacements.
#if KERNEL24_INTERFACE
        owner:          THIS_MODULE,
#endif
	fb_open:	ctfb_open,
	fb_release:	ctfb_release,
	fb_get_fix:	ctfb_get_fix,
	fb_get_var:	ctfb_get_var,
	fb_set_var:	ctfb_set_var,
	fb_get_cmap:	ctfb_get_cmap,
	fb_set_cmap:	ctfb_set_cmap,
	fb_pan_display:	ctfb_pan_display,
	fb_ioctl:	NULL,	
	fb_mmap:	NULL,// perform fb specific mmap - todo: sense?
	fb_rasterimg:	NULL // switch to/from raster image mode -todo: sense?
		// (used for example in fbcon_show_logo() in fbcon.c)
};

#define REPEAT_KMSGS 1
#if REPEAT_KMSGS
#ifndef MODULE
int do_syslog( int type, char * buf, int len ); // code in kernel/printk.c
#endif // MODULE
#endif // REPEAT_KMSGS

/*****************************************************************************/
static int
init_ctfb( void )
{
	int err = 0;
	struct pci_dev *p = NULL;
	struct fb_info     *pi = NULL;									
	struct fb_var_screeninfo *v;
	unsigned int q, vfreq;
	
	mode = mode  == -1 ? 0 : mode; // set to 0 if not specified in cmdline 									
	if( ( err = init_pcidevice( &p ) ) ) return err;						
        if( ( err = alloc_chipsinfo_mem( ) ) ) return err;
	p_ctfb_info->pci = p;
	pi = &( p_ctfb_info->gen.info );					
	v  =   &p_ctfb_info->var;					
        init_fb_info_struct    ( p_ctfb_info );
        init_fb_info_gen_struct( p_ctfb_info );								
	init_ctfb_monspecs     ( p_ctfb_info );										
 	init_fb_addresses      ( p_ctfb_info );          						
        if( ( err = init_fb_var_screeninfo ( p_ctfb_info ) ) ) return err;          						
        init_fb_fix_screeninfo ( p_ctfb_info ); // after init_fb_var_ !		
        init_fb_display_struct ( p_ctfb_info ); // after init_fb_fix_ !
	ct_hw_init             ( p_ctfb_info );                      						
	err = register_framebuffer( pi );                						
	if( err < 0 ) {                                                						
	//	PRINTK( "could not register_framebuffer\n" ); 						
		kfree( p_ctfb_info );                      						
		return err;                                						
	}
	#if REPEAT_KMSG
	#ifndef MODULE
        else{   // reprint kmsg
		const int len = 4096;
		char buf[ len ];
		memset( buf, 0, len );
		err = do_syslog( 3, buf, len ); // read the last messages in the ring buffer.
                if( err >= 0 ){
			char msg[ 64 ];
			sprintf( msg, "%s%s kmsgs:\n", ctfb, FB_VERSION );
			console_print( msg );
			console_print( buf ); // print last messages again to new console buffer
		}
		else printk( "do_syslog error=%d\n", err );
	}
	#endif // MODULE
	#endif // REPEAT_KMSGS
 
//      if( ( err = init_fb_var_screeninfo ( p_ctfb_info ) ) ) return err;          						
//      init_fb_fix_screeninfo ( p_ctfb_info ); // after init_fb_var_ !		
//      init_fb_display_struct ( p_ctfb_info ); // after init_fb_fix_ !
//	ct_hw_init             ( p_ctfb_info );
	q = ( v->left_margin  + v->xres + v->right_margin + v->hsync_len )
	  * ( v->upper_margin + v->yres + v->lower_margin + v->vsync_len )
	  / 100 * v->pixclock / 10;                                          						
	vfreq = ( 1000000000 + ( q / 2 ) ) / q;
	sprintf( pi->modename, "%dx%dx%d-%d",
		v->xres, v->yres, v->bits_per_pixel, vfreq );
   	PRINTK("fb%d: %s\n", GET_FB_IDX( pi->node ), pi->modename );																			
	fb_con.con_startup( );
	fb_display[ 0 ].can_soft_blank = 1; // for con 0 separately, why?
	fb_set_cmap( fb_default_cmap( 16 ), 1, ct_switch.setcolreg, pi );
	return 0;
}

#ifndef MODULE


/*****************************************************************************/
// is registered to the kernel, in case we do no modularization
#if KERNEL24_INTERFACE
int  __init ctfb_init( void )
{
	int err = init_ctfb( );
	MOD_INC_USE_COUNT;
	return err;
}
#else
void  __init ctfb_init( void )
{
	init_ctfb( );
	MOD_INC_USE_COUNT;
}
#endif

/*****************************************************************************/
// Is registered to the kernel and is used to pass card specific options from
// the boot prompt - see http://www.linux-fbdev.org/HOWTO/4.html
// if we compile as a separate module this function is not needed
#if KERNEL24_INTERFACE
int __init ctfb_setup(char* options)
#else
int __init ctfb_setup(char *options, int *ints)
#endif
{ 
  // Parse user speficied options (`video=ctfb:')
  // if you specify a mode this set of init parameters is used,
  // values of the selected mode are overwritten if they specified
  // Example: append = "video=ctfb:mode:2,pixclock:12000"
	char *this_opt;
	if( !options || !*options ){
		PRINTK( "no options, using default values.\n" ); return 0;
	}
	PRINTK( "option string=%s\n", options );	
	for( this_opt = strtok( options, "," ); this_opt;
	     this_opt = strtok( NULL, "," ) ){
		int i;
		/* specify mode by name */
		for (i = 0; i < SETUPMODECOUNT; i++)
			if (strcmp(this_opt, fbxxx_init[i].name) == 0)
				break;
		if (i < SETUPMODECOUNT) {
			mode = i;
			continue;
		}
		#define PARAMETERS
		#define PAR( s, a, b )                                      \
		if( !strncmp( this_opt, #s":", strlen( #s ) + 1 ) ) {       \
			a = simple_strtoul( this_opt + strlen( #s ) + 1, NULL, 0 ); \
			continue;                                                   \
		}
		#include "ctfb.c" 
		else { PRINTK( KERN_ERR "unknown parameter %s\n", this_opt ); }
        }
	if( mode < 0 || SETUPMODECOUNT <= mode ) {
		PRINTK( KERN_ERR "mode %d unsupported, using mode 0\n", mode );
		mode = 0;
	}
	if( crtfp < 0 || CRTFP_MODE_COUNT <= crtfp ) {
		PRINTK( KERN_ERR "crtfp %d unsupported, using crtfp 0\n", crtfp );
		crtfp = 0;
	}
	if( bits_per_pixel != -1 && bits_per_pixel != 8
	 && bits_per_pixel != 16 && bits_per_pixel != 24
	 && bits_per_pixel != 32 ) {
		PRINTK( KERN_ERR "%d bpp unsupported, using 8 bpp\n",
			bits_per_pixel );
		bits_per_pixel = 8;
	}
	return 0;
}

#define PARAMETERS
#define PAR( s, a, b ) static int a = -1;
#include "ctfb.c"

#else // Modularization

/*****************************************************************************/
int __init
init_module( void )
{
	EXPORT_NO_SYMBOLS; // todo: nessesary?
	return init_ctfb( );
}

/*****************************************************************************/
void cleanup_module( void )
{
	struct fb_info *pi = ( struct fb_info * )p_ctfb_info;
	int err = 0;
	err = unregister_framebuffer( pi );
	if( !err ) {
		kfree( pi );
		p_ctfb_info = 0;
	}
	else {
		PRINTK( "error unregister_framebuffer = %x\n", err );
	}
}

MODULE_AUTHOR("(c) 2000 Thomas Hhenleitner <th@visuelle-maschinen.de>");
MODULE_DESCRIPTION("FBDev driver for CT65550, CT65554, CT65555, CT69000, \
	CT69030");

#define PARAMETERS
#define PAR( s, a, b ) \
	static int a = -1; MODULE_PARM( a, "i" ); MODULE_PARM_DESC( a, #b );
#include "ctfb.c"

#endif



//_____________________________________________________________________________


// hardware read and write access
//
// This part contains all functionality to access the hardware.
// In case you wish to write a framebuffer driver for different hardware
// the main thing you have to do, is to exchange the pci stuff and this part.
// The struct par and these functions are the interface to the upper part
// ct_blank, ct_getcolreg, ct_setcolreg, ct_hw_write, ct_hw_read, ct_hw_init

// register access - This is matter of future changes
/* Adrian Cox suggested i.e.:
> > We could do with some indirection to switch between IO and
> > memory-mapping depending on the chip. That would be a good place to
> > start.
> >

Ive got my driver working in this mode. It looks something like this:
struct fb_info_asiliant;
typedef void (* write_reg_fn)(struct fb_info_asiliant *p, u8 reg, u8
data);
struct asiliant_reg_ops {
write_reg_fn write_xr;
write_reg_fn write_fr;
write_reg_fn write_cr;
write_reg_fn write_gr;
write_reg_fn write_sr;
write_reg_fn write_ar;
};
#define write_xr(num, val)       p->ops->write_xr(p, num, val)
static struct asiliant_reg_ops mm_reg_ops = {
mm_write_xr,
mm_write_fr,
mm_write_cr,
mm_write_gr,
mm_write_sr,
mm_write_ar
};
*/

#define Index_ar         0x3c0
#define wData_ar         0x3c0
#define rData_ar         0x3c1
#define ST00_3c2_RO      0x3c2
#define MSR_3c2_WO       0x3c2

#define Index_sr         0x3c4
#define wData_sr         0x3c5
#define rData_sr         0x3c5
#define DACMASK_3C6      0x3c6 // Pixel_Data_Mask_Register_RW_3C6h
#define DACSTATE_3c7_RO  0x3c7 // DAC_State_Register_RO_3C7h
#define DACRX_3c7_WO     0x3c7 // Palette_Read_Index_Register_RO_3C7h
#define DACWX_3c8        0x3c8 // Palette_Write_Index_Register_RW_3C8h
#define DACDATA_3c9      0x3c9 // Palette_Data_Register_RW_3C9h
#define FCR_3ca_RO       0x3ca
#define MSR_3cc_RO       0x3cc

#define Index_gr         0x3ce
#define wData_gr         0x3cf
#define rData_gr         0x3cf
#define Index_fr         0x3d0
#define wData_fr         0x3d1
#define rData_fr         0x3d1
#define Index_mr         0x3d2
#define wData_mr         0x3d3
#define rData_mr         0x3d3
#define Index_cr         0x3d4
#define wData_cr         0x3d5
#define rData_cr         0x3d5
#define Index_xr         0x3d6
#define wData_xr         0x3d7
#define rData_xr         0x3d7

#define ST01_3da_RO      0x3da
#define FCR_3da_WO       0x3da

#if 1 // IOACCESS
#define out_8( P, V ) outb( ( V ), ( P ) )
#define in_8( P ) inb( P )
#else // MMACCESS
//todo for ct69000 (not really nessesary now for the hw I know)
#endif

// attribute registers slightly strange
#define S( r ) if( Index_##r == Index_ar ) \
		in_8( ST01_3da_RO )

#define ri( r, idx )                      \
	do {                              \
		S( r );                   \
		idx =  in_8( Index_##r ); \
	} while( 0 ) // read index

#define wi( r, val )                     \
	do { 	                         \
		S( r );                  \
		out_8( Index_##r, val ); \
	} while( 0 ) // write index

#define rr( r, i, var )                  \
	do {                             \
		S( r );                  \
      		out_8( Index_##r, i );   \
		var = in_8( rData_##r ); \
	} while( 0 ) // read register r##i into v

#define wr( r, i, val )                  \
	do {                             \
		S( r );                  \
		out_8( Index_##r, i );   \
		out_8( wData_##r, val ); \
	} while( 0 ) // write v into register r##i

/*****************************************************************************/
#define fntwidth 8 // fixed for now
#define text     0 // fixed for now

// pixelclock control

/*****************************************************************************/
// We have a rational number p/q and need an m/n which is very close to p/q
// but has m and n within mnmin and mnmax. We have no floating point in the
// kernel. We can use long long without divide. And we have time to compute...
static unsigned int
FindBestPQFittingMN( unsigned int p, unsigned int q, unsigned int mnmin,
		     unsigned int mnmax, unsigned int *pm, unsigned int *pn )
{ // this code is not for general purpose usable but good for our number ranges
	unsigned int n = mnmin, m = 0;
	long long int L = 0, P = p, Q = q, H = P >> 1;
	long long int D = 0x7ffffffffffffffLL;
	for( n = mnmin; n <= mnmax; n++ ) {
		m = mnmin; // p/q ~ m/n -> p*n ~ m*q -> p*n-x*q ~ 0
		L = P * n - m * Q;
		while( L > 0  && m < mnmax  ){ L -= q; m++; }
		if( m > mnmax ) break; // bigger n causing the same situation
 		// L is <= 0 now
		if( -L > H && m > mnmin ){ L += q; m--; } // closer to 0
		L = ( L < 0 ) ? -L : +L;     // abs
		if( D < L ) continue;
		D = L;  *pm = m;  *pn = n;   // keep improved data
		if( D == 0 ) break;          // best result we can get		
	}
	return ( unsigned int )( 0xffffffff & D );
}

#if 0 // that is the hardware we have to manage
  ---------    -------------------    ----------------------    --
 | REFCLK  |__|NTSC Divisor Select|__|FVCO Reference Divisor|__|N|__
 | 14.3MHz |  |(NTSCDS) (1, 5)  |  |Select (RDS) (1, 4) |  |  |  |
  ---------    -------------------    ----------------------    --   |
  ___________________________________________________________________|
 |
 |                                    fvco                      fout
 |  --------    ------------    -----       -------------------     ----
 --| Phase  |__|Charge Pump |__| VCO |_____|Post Divisor (PD)  |___|CLK |--->
 --| Detect |  |& Filter VCO|  |     |  |  |1, 2, 4, 8, 16, 32|   |    |
 |  --------    ------------    -----   |   -------------------     ----
 |                                      |
 |     --     ---------------           |
 |____|M|___|VCO Loop Divide|__________|
      |  |   |(VLD)(4, 16) |
       --     ---------------
#endif

#define COMMON_VARIABLES \
	unsigned int m = 0;              /* Loop Divisor M            */ \
	unsigned int n = 0;              /* Reference Divisor N       */ \
        unsigned int    vld = 4;         /* VCO Loop Divide (4 or 16) */ \
	unsigned int     pd = 1, PD = 0; /* Post Divisor              */ \
	unsigned int fref   = 14318180;  /* Hz                        */ \
        unsigned int xr_cb = 0;

/*****************************************************************************/
static int
ReadPixClckFromXrRegsBack( void )
{
	COMMON_VARIABLES
	unsigned int i = 0, pixclock = -EINVAL;
	rr( xr, 0xc8, m );	
	rr( xr, 0xc9, n );	
	rr( xr, 0xcb, xr_cb );	PD =  ( 0x70 & xr_cb ) >> 4;
	for( i = 0; i < PD; i++ ) { pd *=2; }
	vld    = ( 0x04 & xr_cb ) ? 16 : 4;
	if( n * vld * m ){
                unsigned long long p = 1000000000000LL * pd * n;
                unsigned long long q = ( long long )fref * vld * m;
                #define COND ( ( p > 0xffffffffLL ) || ( q > 0xffffffffLL ) )
		if( COND )
			do{ // can't divide with long long so we scale down
				p >>= 1; q >>= 1;
			}while( COND );
		#undef COND
		pixclock = ( unsigned )p / ( unsigned )q;
	}
	else PRINTK( "Invalid data in xr regs.\n" );
	return pixclock;
}

/*****************************************************************************/
static void
FindAndSetPllParamIntoXrRegs( unsigned int pixelclock )
{
	COMMON_VARIABLES
	unsigned int fvcomin = ( p_ctfb_info->pci->device
				== PCI_DEVICE_ID_CT_69030 ) ? 100 : 48; // MHz
	unsigned int fvcomax =   220;       // MHz
	unsigned int pclckmin =  1000000 / fvcomax + 1; //   4546
	unsigned int pclckmax = 32000000 / fvcomin - 1; // 666665
    unsigned int l = min( pixelclock, pclckmax );
	unsigned int pclk = max( pclckmin, l); //minmax( pclckmin, pixelclock, pclckmax ); // ps pp
	unsigned int pfreq   = 250 * ( 4000000000U / pclk );
	unsigned int fvco = pfreq; // Hz
	unsigned int new_pixclock = 0;
	#define COND ( fvco < fvcomin * 1000000 )
	if( COND ){
		do { // todo one or two times more
			fvco *= 2;  pd *= 2;   PD++;
		} while( COND ); } // fvco=48000120...219973603 Hz
	FindBestPQFittingMN( fvco / vld, fref, 3, 255, &m, &n ); // rds = 1
	xr_cb = ( ( 0x7 & PD ) << 4 ) | ( vld == 16 ? 0x04 : 0 );
	// All four of the registers used for dot clock 2 (XRC8 - XRCB) must be
	// written, and in order from XRC8 to XRCB, before the hardware will
	// update the synthesizer s settings.
	wr( xr, 0xc8, m );
	wr( xr, 0xc9, n );      // xrca does not exist in CT69000 and CT69030
	wr( xr, 0xca, 0 );      // because of a hw bug I guess, but we write
	wr( xr, 0xcb, xr_cb );  // 0 to it for savety
	new_pixclock = ReadPixClckFromXrRegsBack( );
	if( abs( pixelclock - new_pixclock ) > 10 ){
		PRINTK( "pixelclock.set = %d, pixelclock.real = %d \n",
			pixelclock, new_pixclock ); }
}
#undef COMMON_VARIABLES

// picture geometry control

/*****************************************************************************/
static int
CR( int idx )
{
	int var;
	rr( cr, idx, var );
	return var;
}

/*****************************************************************************/
static void
ReadTimingsFromCrRegs( struct fb_var_screeninfo *pvar )
{
	const int cc = 8; // char clocks (font size)
	int dblscan = 0;
	int interlaced = 0;
	int ri, ht, hd, le, hs, h, he;
	int up, vt, vd, lo, vs, v, ve;
	ht = CR( 0x00 ) + 5;
	hd = CR( 0x02 );
	hs = CR( 0x04 );
	ri = hs - hd;
	h  = ( ( CR( 0x05 ) | 0xe0 ) - hs ) & 0x1f;
	le = ht - hs - h;
 	he = hs + h;

	vt = ( ( ( CR( 0x30 ) & 0x0f ) << 8 ) | CR( 0x06 ) )+ 2;
	vd = ( ( ( CR( 0x31 ) & 0x0f ) << 8 ) | CR( 0x12 ) )+ 1;
	vs = ( ( ( CR( 0x33 ) & 0x0f ) << 8 ) | CR( 0x15 ) );
	lo = vs - vd;
	v  = ( ( CR( 0x11 ) | 0xf0 ) - vs ) & 0x0f;
	up = vt - vs - v;
        ve = vs + v;
	dblscan    = ( 0x80 & CR( 0x09 ) ) ? FB_VMODE_DOUBLE : 0;
	interlaced = ( 0x80 & CR( 0x70 ) ) ? FB_VMODE_INTERLACED : 0;
	if( interlaced ){
		vs *= 2;
		ve *= 2;	
		vt *= 2;
		v *= 2;
	}
	if( pvar )
	{
		pvar->xres         = cc * hd;
		pvar->left_margin  = cc * ( ht - he );
		pvar->right_margin = cc * ( hs - hd );
		pvar->hsync_len    = cc * h;
                pvar->yres         = vd;
                pvar->upper_margin = vt - ve;
		pvar->lower_margin = vs - vd;
		pvar->vsync_len    = v;
		pvar->vmode &= ~( FB_VMODE_DOUBLE
				| FB_VMODE_INTERLACED ); // mask
		pvar->vmode |= dblscan | interlaced; // set bits
	}
}


static unsigned int msr0 = 0; // todo: eliminate this variable

/*****************************************************************************/
static void
ReadSyncStateFromRegs( struct fb_var_screeninfo *pvar )
{
	unsigned int vHighActive = 0x80 & msr0 ? 0 : FB_SYNC_VERT_HIGH_ACT;
	unsigned int hHighActive = 0x40 & msr0 ? 0 : FB_SYNC_HOR_HIGH_ACT;
        unsigned int bcast       = 0;
	rr( fr, 0x0b, bcast ); // bcast influences are also in some cr regs
	bcast = ( 0x20 & bcast ) ? FB_SYNC_BROADCAST : 0;
	if( pvar )
	{
		pvar->sync |= FB_SYNC_BROADCAST
			    | FB_SYNC_VERT_HIGH_ACT
			    | FB_SYNC_HOR_HIGH_ACT; // mask bits
		pvar->sync &= bcast | hHighActive | vHighActive; // set bits
	}
}

/*****************************************************************************/
static void
ReadModesFromRegs( struct fb_var_screeninfo *pvar )
{
	unsigned int bpp = 0, DisplayColorMode = 0, vxres = 0,
                     OffsetBits = 0, CRT_ControllerExtensionsEnable = 0;
	unsigned int BPP[] = { 4, 0, 8, 0, 15, 16, 24, 32 };
	unsigned int dblscan = 0, DoubleScanning = 0, interlaced = 0,
                     InterlaceEnable = 0, tmp = 0;
	rr( xr, 0x09, tmp );	
	CRT_ControllerExtensionsEnable = 0x01 & tmp;
	
	rr( cr, 0x09, tmp );	
	DoubleScanning  = ( 0x80 & tmp ) >> 7;

	rr( cr, 0x70, tmp );	
	InterlaceEnable = ( 0x80 & tmp ) >> 7;
	
	rr( xr, 0x81, tmp );
	DisplayColorMode = 0x07 & tmp;
	
	rr( cr, 0x13, tmp );
        OffsetBits |=   0xff & tmp;           // bits 7-0
        rr( cr, 0x41, tmp );
	if( CRT_ControllerExtensionsEnable ){
		OffsetBits |= ( 0x0f & tmp ) << 8;
	} // bits 11-8
	
	dblscan    = DoubleScanning  ? FB_VMODE_DOUBLE : 0;
	interlaced = InterlaceEnable ? FB_VMODE_INTERLACED : 0;
	bpp = BPP[ DisplayColorMode ];
	vxres = OffsetBits * 64 / bpp;
	if( pvar ){
		pvar->bits_per_pixel = bpp;
		pvar->xres_virtual   = vxres;
	      //yres_virtual is maintained in driver software only
		pvar->vmode &= ~( FB_VMODE_DOUBLE
				| FB_VMODE_INTERLACED  ); // mask
		pvar->vmode |= dblscan | interlaced; // set bits
        }
}

/*****************************************************************************/
static void
VarToMsr0( struct fb_var_screeninfo* var ) // todo: read back
{
	unsigned hHighActive= ( var->sync & FB_SYNC_HOR_HIGH_ACT  ) ? 0 : 0x40;
	unsigned vHighActive= ( var->sync & FB_SYNC_VERT_HIGH_ACT ) ? 0 : 0x80;
//	unsigned cHighActive= ( var->sync & FB_SYNC_COMP_HIGH_ACT ) ? 1 : 0;
	msr0 = vHighActive | hHighActive | 0x29;
	// upper64K==0x20, CLC2select==0x08, RAMenable==0x02!(todo), CGA==0x01	
	// Selects the upper 64KB page.Bit5=1
 	// CLK2 (left reserved in standard VGA) Bit3|2=1|0
	// Disables CPU access to frame buffer. Bit1=0
	// Sets the I/O address decode for ST01, FCR, and all CR registers
	// to the 3Dx I/O address range (CGA emulation). Bit0=1
	out_8( MSR_3c2_WO, msr0 ); // set misc output reg
}

/*****************************************************************************/
static void
SetBcastSyncIntoFrReg( struct fb_var_screeninfo* var )  // FR_init
{
	unsigned int bcast = var->sync  & FB_SYNC_BROADCAST; // todo: check
	unsigned int SimpleCompositeSync = bcast ? 1 : 0;
	// Note: Effective only when XR0B[2] = 0,
	// 0== HSYNC output pin provides CRT HSYNC signal,
	// 1== HSYNC output pin provides CRT CSYNC signal
	// (horizontal sync OR'd with vertical sync).
	unsigned int tmp = 0;					
	rr( fr, 0x0b, tmp );
	tmp = SimpleCompositeSync ? ( tmp |  ( SimpleCompositeSync << 5 ) )
				  : ( tmp & ~( SimpleCompositeSync << 5 ) );
	wr( fr, 0x0b, tmp );
}

/*****************************************************************************/
static void
SetBitsPerPixelIntoXrRegs( unsigned bpp )
{       //todo: set also br04 (probably not nessesary)
	unsigned int n = ( bpp >> 3 ) - 1, tmp; // only for 8, 16, 24 bpp
        static char md [ 3 ] = {  0x02,  0x05,  0x06 }; // DisplayColorMode
	static char off[ 3 ] = { ~0x30, ~0x20, ~0x10 }; // mask
	static char on [ 3 ] = {  0x00,  0x10,  0x20 }; // mask
        rr( xr, 0x20, tmp );
	tmp &= off[ n ];
	tmp |= on[ n ];
	wr( xr, 0x20, tmp ); // BitBLT Configuration
	wr( xr, 0x81, md[ n ] );
}

static unsigned char Cr[ 0x7a ];
static unsigned char Fr[ 0x75 ];

/*****************************************************************************/
static void
init_cr_reg( void )
{
	#define set_cr( r ) do{ wr( cr, 0x##r, Cr[ 0x##r ] ); } while( 0 )
	unsigned int tmp = 0;
	set_cr( 00 );	set_cr( 01 );	set_cr( 02 );	set_cr( 03 );	
	set_cr( 04 );	set_cr( 05 );	set_cr( 06 );	set_cr( 07 );
	set_cr( 08 );	set_cr( 09 );	set_cr( 0A );	set_cr( 0B );	
	set_cr( 0C );	set_cr( 0D ); // 0E and 0F ignored in graphics modes.
	set_cr( 10 );	set_cr( 11 );	set_cr( 12 );	set_cr( 13 );	
	set_cr( 14 );	set_cr( 15 );	set_cr( 16 );	set_cr( 17 );
	set_cr( 18 );	set_cr( 22 );	set_cr( 30 );	set_cr( 31 );	
	set_cr( 32 );	set_cr( 33 );	set_cr( 38 );	set_cr( 3C );
	set_cr( 40 );	set_cr( 41 );	set_cr( 70 );	set_cr( 71 );	
	set_cr( 72 );	set_cr( 73 );	set_cr( 74 );
	rr( cr, 0x40, tmp );
	tmp &= 0x0f;
	tmp |= 0x80;
	wr( cr, 0x40, tmp ); // StartAddressEnable
	#undef set_cr
}

/*****************************************************************************/
#define LeM var->left_margin
#define RiM var->right_margin
#define UpM var->upper_margin
#define LoM var->lower_margin
static void  // todo: check against matroxfb.c
CR_timingarray_init( struct fb_var_screeninfo* var )
{	          // he -le-   ht|0    hd -ri- hs     -h-      he
	unsigned hd = (       var->xres                        ) / 8; // HDisp.
	unsigned hs = (       var->xres + RiM                  ) / 8; // HsStrt
	unsigned he = (       var->xres + RiM + var->hsync_len ) / 8; // HsEnd
	unsigned ht = ( LeM + var->xres + RiM + var->hsync_len ) / 8; // HTotal
//	unsigned h = ht - hd;
	unsigned hbe = ht - 1; // todo docu wants ht here, but it does not work
		 // ve -up-  vt|0    vd -lo- vs     -v-      ve
	unsigned vd =        var->yres;                           // VDisplay
	unsigned vs =        var->yres + LoM;                     // VSyncStart
	unsigned ve =        var->yres + LoM + var->vsync_len;    // VSyncEnd
	unsigned vt =  UpM + var->yres + LoM + var->vsync_len;    // VTotal
        unsigned bpp = var->bits_per_pixel;
	unsigned wd = var->xres_virtual * bpp / 64; // double words per line
	unsigned dblscan     = ( var->vmode & FB_VMODE_DOUBLE ) ? 1 : 0;
	unsigned interlaced  =   var->vmode & FB_VMODE_INTERLACED;
	unsigned bcast       =   var->sync  & FB_SYNC_BROADCAST;
	unsigned CrtHalfLine = bcast ? ( hd >> 1 ) : 0;
	unsigned BlDelayCtrl = bcast ? 1 : 0;
	unsigned CompSyncCharClkDelay = 0;  // 2 bit
	unsigned CompSyncPixelClkDelay = 0; // 3 bit
	unsigned NTSC_PAL_HorizontalPulseWidth =
		bcast ? /*( var->hsync_len >> 1 ) + 1*/ 7 : 0;
	// 4 bit: hsync pulse width = ( ( CR74[4:0] - CR74[5] )
	//		/ 2 ) + 1 --> CR74[4:0] = 2*(hs-1) + CR74[5]
	unsigned HorizontalEqualizationPulses = bcast ? 0 : 1; // inverse value
	unsigned HorizontalSerration1Start = bcast ? 31 : 0;  // ( ht >> 1 )
	unsigned HorizontalSerration2Start = bcast ? 89 : 0;  // ( ht >> 1 )
	unsigned TextScanLines = 1; // this is in fact a vertical zoom factor
	unsigned RAMDAC_BlankPedestalEnable = 0; // 1=en-, 0=disable, see XR82
	const int LineCompare = 0x3ff;
	if( interlaced ){ // we divide all vertical timings, exept vd
		vs >>= 1;
		ve >>= 1;
		vt >>= 1;
	}
	memset( Cr, 0, sizeof( Cr ) );
	Cr[ 0x00 ] = 0xff & ( ht - 5 );
	Cr[ 0x01 ] = hd - 1; // soll:4f ist 59
	Cr[ 0x02 ] = hd;
	Cr[ 0x03 ] = ( hbe & 0x1F ) | 0x80; // hd + ht - hd
	Cr[ 0x04 ] = hs;
	Cr[ 0x05 ] = ( ( hbe & 0x20 ) << 2 ) | ( he & 0x1f );
	Cr[ 0x06 ] = ( vt - 2 ) & 0xFF;
	Cr[ 0x30 ] = ( vt - 2 ) >> 8;
	Cr[ 0x07 ] = ( ( vt & 0x100 ) >> 8 )
		   | ( ( vd & 0x100 ) >> 7 )
		   | ( ( vs & 0x100 ) >> 6 )
		   | ( ( vs & 0x100 ) >> 5 )
		   | ( ( LineCompare & 0x100 ) >> 4 )
		   | ( ( vt & 0x200 ) >> 4 )
		   | ( ( vd & 0x200 ) >> 3 )
		   | ( ( vs & 0x200 ) >> 2 );
	Cr[ 0x08 ] =  0x00;
	Cr[ 0x09 ] = ( dblscan << 7 )
		   | ( ( LineCompare & 0x200 ) >> 3 )
		   | ( ( vs & 0x200 ) >> 4)
		   | ( TextScanLines - 1 );
	Cr[ 0x10 ] =   vs &  0xff;                    // VSyncPulseStart
	Cr[ 0x32 ] = ( vs & 0xf00 ) >> 8;             // VSyncPulseStart
	Cr[ 0x11 ] = ( ve & 0x0f );                   // | 0x20;
	Cr[ 0x12 ] =   ( vd - 1 ) &  0xff;            // LineCount
	Cr[ 0x31 ] = ( ( vd - 1 ) & 0xf00 ) >> 8;     // LineCount
	Cr[ 0x13 ] =   wd &  0xff;
	Cr[ 0x41 ] = ( wd & 0xf00 ) >> 8;
	Cr[ 0x15 ] =   vs &  0xff;
	Cr[ 0x33 ] = ( vs & 0xf00 ) >> 8;
	Cr[ 0x38 ] = ( 0x100 & ( ht - 5 ) ) >> 8;
	Cr[ 0x3C ] = 0xc0 & hbe;
	Cr[ 0x16 ] = ( vt - 1 ) & 0xff;	// vbe - docu wants vt here,
	Cr[ 0x17 ] = 0xe3;                  // but it does not work
	Cr[ 0x18 ] = 0xff & LineCompare;
	Cr[ 0x22 ] = 0xff; // todo?
	Cr[ 0x70 ] = interlaced ? ( 0x80 | CrtHalfLine ) : 0x00; // check:0xa6
	Cr[ 0x71 ] = 0x80
		   | ( RAMDAC_BlankPedestalEnable << 6 )
		   | ( BlDelayCtrl << 5 )
		   | ( ( 0x03 & CompSyncCharClkDelay ) << 3 )
		   | ( 0x07 & CompSyncPixelClkDelay ); // todo: see XR82
	Cr[ 0x72 ] = HorizontalSerration1Start;
	Cr[ 0x73 ] = HorizontalSerration2Start;
	Cr[ 0x74 ] = ( HorizontalEqualizationPulses << 5 )
		   | NTSC_PAL_HorizontalPulseWidth;
	// todo: ct69000 has also 0x75-79
	if (crtfp != CRTFP_MODE_CRT) {
		Fr[ 0x20 ] = (hd - 1) & 0xff;
		Fr[ 0x21 ] = (hs - 1) & 0xff;
		Fr[ 0x22 ] = he & 0x1f;
		Fr[ 0x23 ] = (ht - 5) & 0xff;
		Fr[ 0x24 ] = hs & 0xff;
		Fr[ 0x25 ] = (((hs - 1) & 0xf00) >> 4) | (((hd - 1) & 0xf00) >> 8);
		Fr[ 0x26 ] = ((hs & 0xf00) >> 4) | (((ht - 5) & 0xf00) >> 8);
		Fr[ 0x27 ] = 0x0f;	/* XXX LPwidth=16 */
		Fr[ 0x30 ] = (vd - 1) & 0xff;
		Fr[ 0x31 ] = (vs - 1) & 0xff;
		Fr[ 0x32 ] = (ve - 1) & 0xf;
		Fr[ 0x33 ] = (vt - 2) & 0xff;
		Fr[ 0x34 ] = 0;	/* not used */
		Fr[ 0x35 ] = (((vs - 1) & 0xf00) >> 4) | (((vd - 1) & 0xf00) >> 8);
		Fr[ 0x36 ] = 0 | (((vt - 2) & 0xf00) >> 8);
		Fr[ 0x37 ] = 0x88;	/* XXX FLMwidth=2 */
		#define set_fr( r ) do{ wr( fr, 0x##r, Fr[ 0x##r ] ); } while( 0 )
		set_fr( 20 );	set_fr( 21 );	set_fr( 22 );	set_fr( 23 );	
		set_fr( 24 );	set_fr( 25 );	set_fr( 26 );	set_fr( 27 );
		set_fr( 30 );	set_fr( 31 );	set_fr( 32 );	set_fr( 33 );	
		set_fr( 34 );	set_fr( 35 );	set_fr( 36 );	set_fr( 37 );
		#undef set_fr
	}
}
#undef LeM
#undef RiM
#undef UpM
#undef LoM

/*****************************************************************************/
static void
CR_init( struct fb_var_screeninfo *var )
{
	CR_timingarray_init( var );
	init_cr_reg( );  // set cr regs
}

/*****************************************************************************/
static void
var2hw( struct fb_var_screeninfo *pv )
{
	VarToMsr0( pv );
	CR_init( pv );
	SetBcastSyncIntoFrReg( pv );//FR_init( var );
	SetBitsPerPixelIntoXrRegs( pv->bits_per_pixel ); //XR_init( var );
	FindAndSetPllParamIntoXrRegs( pv->pixclock );
	// following 2 values can change even they part of struct
	// fb_fix_sereeninfo, so we treat p_ctfb_info.fix as part of the hw
	p_ctfb_info->fix.line_length =
		( pv->xres_virtual * pv->bits_per_pixel ) >> 3; // bytes
	p_ctfb_info->fix.visual = fb_visual_color( pv->bits_per_pixel );
	p_ctfb_info->var = *pv; // save fields not red back from hw
}

/*****************************************************************************/
static void
fix2hw( struct fb_fix_screeninfo *pf )
{
//	XR82:3 Graphics Data Gamma Correction Enable
//	0: Graphics data bypasses the palette when the graphics system is set
// 	   to a color depth of 16, 24 or 32 bits per pixel. This is the default
//	   after reset.
//	1: Graphics data goes through the palette when the graphics system is
//	   set to a color depth of 16, 24 or 32 bits per pixel, allowing the
//         palette to be used to perform gamma correction.
	char tmp;
	rr( xr, 0x82, tmp );
	switch( pf->visual ) // this is the only value we have to set here
	{
	case FB_VISUAL_TRUECOLOR:    tmp &= 0xf7; break; // no gamma corr
	case FB_VISUAL_DIRECTCOLOR:  tmp |= 0x08; break; // gamma on
	case FB_VISUAL_PSEUDOCOLOR:  break; // no need for that in 8 bpp mode
	default: PRINTK( "fix.visual = %d not supported!\n", pf->visual );
	}
	wr( xr, 0x82, tmp );
}

/*****************************************************************************/
static void
hw2var( struct fb_var_screeninfo *pv )
{
	*pv = p_ctfb_info->var; // fill fields not red back from hw
	ReadTimingsFromCrRegs( pv );
	ReadSyncStateFromRegs( pv );
	ReadModesFromRegs( pv );
	set_bitfields( pv );
	pv->pixclock = ReadPixClckFromXrRegsBack( );
}

/*****************************************************************************/
static void
hw2fix( struct fb_fix_screeninfo *pf )
{
	*pf = p_ctfb_info->fix; // copy struct is all we have to do, because
}       // it is consistent with the hw, if nobody hacks into the registers 	

/*****************************************************************************/
static void __init
SetArRegs( void ) // like matroxfb
{ // old ok code: rr( ar, 0x10, tmp ); tmp &= ~0x08; tmp |= 0x01;
// wr( ar, 0x10, tmp ); wi( ar, 0x20 ); Set Graphics Mode (needed for ct65554)
	int i, tmp;
	for( i = 0; i < 0x10; i++ )
		wr( ar, i, i );
	if( text ) tmp = 0x04; else tmp = 0x41;
	wr( ar, 0x10, tmp );          // Mode Control Register
	wr( ar, 0x11, 0x00 );         // Overscan Color Register
	wr( ar, 0x12, 0x0f );         // Memory Plane Enable Register
	if( fntwidth == 9 ) tmp = 0x08; else tmp = 0x00;
	wr( ar, 0x13, tmp );          // Horizontal Pixel Panning
	wr( ar, 0x14, 0x00 );         // Color Select Register
	wi( ar, 0x20 );               // enable video
}

/*****************************************************************************/
static void __init
SetGrRegs( void ) // like matroxfb
{ // old ok code: rr( gr, 0x06, tmp );	tmp |= 0x01;	
//wr( gr, 0x06, tmp ); // Set Graphics Mode
	int i;
	for( i = 0; i < 0x05; i++ ) wr( gr, i, 0 );
	if( text ){ wr( gr, 0x05, 0x10 ); wr( gr, 0x06, 0x02 ); }
	else      { wr( gr, 0x05, 0x40 ); wr( gr, 0x06, 0x05 ); }
	wr( gr, 0x07, 0x0f );
	wr( gr, 0x08, 0xff );
}

/*****************************************************************************/
static void __init
SetSrRegs( void ) // like matroxfb
{
	int tmp = 0;
	wr( sr, 0x00, 0x00 ); // reset
	rr( sr, 0x01, tmp );
	if( fntwidth == 8 ) tmp |= 0x01; else tmp &= ~0x01;
	wr( sr, 0x01, tmp );  // Clocking Mode Register
	wr( sr, 0x02, 0x0f ); // Enable CPU wr access to given memory plane
        wr( sr, 0x03, 0x00 ); // Character Map Select Register
        if( text ) tmp = 0x02; else tmp = 0x0e;
	wr( sr, 0x04, tmp );  // Enable CPU accesses to the rest of the 256KB
	    // total VGA memory beyond the first 64KB and set fb mapping mode.
	wr( sr, 0x00, 0x03 ); // enable
}

/*****************************************************************************/
static void __init
SetXrRegs( void )
{
	int tmp = 0;
	 // use internal clock sources (this is the default)
      //rr( xr, 0xcf, tmp );	tmp |= 0x06;	wr( xr, 0xce, tmp );
	// set default memory clock 40 MHz (xrcc,xrcd,xrce[6-0] are ignored):
	// LET MCLK AS BIOS INITALIZED IT (PROBLEMS WITH VIDEO IN CAN OCCUR)
      //rr( xr, 0xce, tmp );	tmp &= 0x7f;	wr( xr, 0xce, tmp );

	rr( xr, 0x09, tmp );	
	tmp |= 0x01;
	wr( xr, 0x09, tmp );  // CRT Controller Extensions Enable
	rr( xr, 0x0a, tmp );	
	tmp |= 0x02;	
	wr( xr, 0x0a, tmp ); // Frame Buffer Linear Mapping Enable
        wr( xr, 0x40, 0x03 ); // Selects the use of 64-bit accesses to
		// accommodate high resolution modes. All memory address bits
		// are used, allowing access to all of the graphics memory.
	rr( xr, 0x80, tmp );	
	tmp &= 0xf7;	
	wr( xr, 0x80, tmp ); // Disable extended status read feature to permit
		// normal accesses to the registers and color data locations
		// within the palette.
	rr( xr, 0x80, tmp );	
	tmp |= 0x80;	
	wr( xr, 0x80, tmp ); // D-to-A converters are set for 8-bit operation.
}

/*****************************************************************************/
static void __init
SetFrRegs( void )
{
	// nothing for now
}

/*****************************************************************************/
#ifdef CONFIG_TOSHIBA_BOARDS
struct ctfb_init_reg {
	unsigned char addr;
	unsigned char data;
};
static struct ctfb_init_reg ctfb_init_xr_69000_640x480_8[] __initdata = {
	{0x08, 0x03},
#if 0 /* __BIG_ENDIAN: if we swap here, we can not display 24bpp image */
	{0x0A, 0x12}, {0x0B, 0x01},
#else
	{0x0A, 0x02}, {0x0B, 0x01},
#endif
	{0x0E, 0x00},
	{0x10, 0x2C}, {0x11, 0x10},
	{0x12, 0x2C}, {0x13, 0x10},
	{0x20, /*0x0C*/0x00},
	{0x40, 0x03},
	{0x60, 0x00}, {0x61, 0x00},
	{0x70, 0xFE}, {0x71, 0xFF},
	{0x80, 0x00}, {0x81, 0x02},
	{0x82, 0x00},
	{0xC0, 0x7D}, {0xC1, 0x07},
	{0xC2, 0x00}, {0xC3, 0x34},
	{0xC4, 0x55}, {0xC5, 0x09},
	{0xC6, 0x00}, {0xC7, 0x24},
	{0xC8, 0x77}, {0xC9, 0x0A},/*XXX 0x77=M(121-2) 0x09=N(12-2) */
	{0xCA, 0x00}, {0xCB, 0x24},/*XXX 0x24=PD:4,VLD:1=>36.1MHz*/
	{0xCC, 0x38}, {0xCD, 0x03},
	{0xCE, 0x90}, {0xCF, 0x06},
	{0xD0, 0x0F}, {0xD1, 0x01},
	{0xD2, 0x00},
};
static struct ctfb_init_reg ctfb_init_fr_69000_640x480_8_crt[] __initdata = {
	{0x03, 0x08},
	{0x04, 0x81}, {0x05, 0x01},
	{0x06, 0x03},
	{0x0F, 0x02},
	{0x10, 0x0C}, {0x11, 0xE0},
	{0x12, 0x50}, {0x13, 0x00},
	{0x16, 0x00}, {0x17, 0xBD},
	{0x18, 0x00}, {0x19, 0x88},
	{0x1A, 0x00},
	{0x1E, 0x80}, {0x1F, 0x80},
	{0x40, 0x03}, {0x41, 0x00},
	{0x48, 0x13}, {0x49, 0x00},
	{0x4A, 0x01}, {0x4B, 0x80},
	{0x4C, 0x00}, {0x4D, 0x60},
	{0x4E, 0x0F},
	{0x70, 0x00}, {0x71, 0x55},
	{0x72, 0xAA}, {0x73, 0x00},
	{0x74, 0x5F},
};
static struct ctfb_init_reg ctfb_init_fr_69000_640x480_8_lcd[] __initdata = {
	{0x03, 0x08},
	{0x04, 0x81}, {0x05, 0x21},/*XXX*/
	{0x06, 0x03},
	{0x0F, 0x02},
	{0x10, 0x0C}, {0x11, 0xE0},
	{0x12, 0x50}, {0x13, 0x00},
	{0x16, 0x03}, {0x17, 0xBD},/*XXX*/
	{0x18, 0x00}, {0x19, 0x88},
	{0x1A, 0x00},
	{0x1E, 0x80}, {0x1F, 0x80},
	{0x40, 0x83}, {0x41, 0x00},/*XXXXXX*/
	{0x48, 0x13}, {0x49, 0x00},
	{0x4A, 0x01}, {0x4B, 0x80},
	{0x4C, 0x00}, {0x4D, 0x60},
	{0x4E, 0x0F},
	{0x70, 0x00}, {0x71, 0x55},
	{0x72, 0xAA}, {0x73, 0x00},
	{0x74, 0x5F},
};
static struct ctfb_init_reg ctfb_init_fr_69000_640x480_8_both[] __initdata = {
	{0x03, 0x08},
	{0x04, 0x81}, {0x05, 0x21},/*XXX*/
	{0x06, 0x03},
	{0x0F, 0x02},
	{0x10, 0x0C}, {0x11, 0xE0},
	{0x12, 0x50}, {0x13, 0x00},
	{0x16, 0x03}, {0x17, 0xBD},/*XXX*/
	{0x18, 0x00}, {0x19, 0x88},
	{0x1A, 0x00},
	{0x1E, 0x80}, {0x1F, 0x80},
	{0x40, 0x83}, {0x41, 0x00},/*XXX*/
	{0x48, 0x13}, {0x49, 0x00},
	{0x4A, 0x01}, {0x4B, 0x80},
	{0x4C, 0x00}, {0x4D, 0x60},
	{0x4E, 0x0F},
	{0x70, 0x00}, {0x71, 0x55},
	{0x72, 0xAA}, {0x73, 0x00},
	{0x74, 0x5F},
};
#if 0
static int ctfb_do_regdump = 0;
static void
ct_hw_regdump(void)
{
	int i;
	unsigned char dat;
	if (!ctfb_do_regdump)
		return;
	dat = in_8(MSR_3cc_RO);
	printk("MSR   %02X;  ", dat);
	dat = in_8(FCR_3ca_RO);
	printk("FCR   %02X;  ", dat);
#if 1
	for (i = 0; i < 5; i++) {
		if (i % 2 == 0)
			printk("\n");
		rr(sr, i, dat);
		printk("SR%02X  %02X;  ", i, dat);
	}
	for (i = 0; i < 0x19; i++) {
		if (i % 2 == 0)
			printk("\n");
		rr(cr, i, dat);
		printk("CR%02X  %02X;  ", i, dat);
	}
	for (i = 0; i < 9; i++) {
		if (i % 2 == 0)
			printk("\n");
		rr(gr, i, dat);
		printk("GR%02X  %02X;  ", i, dat);
	}
	for (i = 0; i < 0x15; i++) {
		if (i % 2 == 0)
			printk("\n");
		rr(ar, i, dat);
		printk("AR%02X  %02X;  ", i, dat);
	}
	wi( ar, 0x20 );               // enable video
#endif
	for (i = 0; i < 256; i++) {
		if (i % 2 == 0)
			printk("\n");
		rr(xr, i, dat);
		printk("XR%02X  %02X;  ", i, dat);
	}
	for (i = 0; i < 128; i++) {
		if (i % 2 == 0)
			printk("\n");
		rr(fr, i, dat);
		printk("FR%02X  %02X;  ", i, dat);
	}
#if 0
	for (i = 0; i < 128; i++) {
		if (i % 2 == 0)
			printk("\n");
		rr(mr, i, dat);
		printk("MR%02X  %02X;  ", i, dat);
	}
	for (i = 0; i < 256; i++) {
		unsigned char p[3];
		if (i % 4 == 0)
			printk("\n");
		out_8(DACRX_3c7_WO, i);
		p[0] = in_8(DACDATA_3c9);
		p[1] = in_8(DACDATA_3c9);
		p[2] = in_8(DACDATA_3c9);
		printk("VP%02X  %02X %02X %02X;  ", i, p[0], p[1], p[2]);
	}
#endif
	printk("\n");
}
#endif

static void __init
ct_hw_pre_init( struct ctfb_info *p )
{
	/* we have no BIOS. setup for initial state... */
	int i;
	struct ctfb_init_reg *xrp, *frp;
	int nxr, nfr;

	xrp = ctfb_init_xr_69000_640x480_8;
	frp = ctfb_init_fr_69000_640x480_8_crt;
	nxr = N_ELTS(ctfb_init_xr_69000_640x480_8);
	nfr = N_ELTS(ctfb_init_fr_69000_640x480_8_crt);
	switch (crtfp) {
	case CRTFP_MODE_LCD:
		frp = ctfb_init_fr_69000_640x480_8_lcd;
		nfr = N_ELTS(ctfb_init_fr_69000_640x480_8_lcd);
		break;
	case CRTFP_MODE_BOTH:
		frp = ctfb_init_fr_69000_640x480_8_both;
		nfr = N_ELTS(ctfb_init_fr_69000_640x480_8_both);
		break;
	}

	out_8(MSR_3c2_WO, 0xeb); /* 640x480 (0x2b for 800x600) */

	wr(fr, 0x08, 0xcc);	/* FP Pin Polarity */
	wr(fr, 0x0a, 0x08);	/* Programmable Output Drive */
	wr(fr, 0x0b, 0x01);	/* Pin Control 1 */
	wr(fr, 0x0c, 0x18);	/* Pin Control 2 */
	wr(xr, 0x62, 0x00);	/* GPIO Pin Control */

	for (i = 0; i < nxr; ++i)
		wr(xr, xrp[i].addr, xrp[i].data);
	switch (crtfp) {
	case CRTFP_MODE_LCD:
		wr(xr, 0xd0, 0x0e);	/* Disable D-A converter */
		break;
	}
	for (i = 0; i < nfr; ++i)
		wr(fr, frp[i].addr, frp[i].data);
#if 0
	ct_hw_regdump();
#endif
}
#endif /* CONFIG_TOSHIBA_BOARDS */
static void __init
ct_hw_init( struct ctfb_info *p )
{ // we do not need to setup all registers - this is the job of the bios
  // (I think so) - setting just what we need (more todo?)
#ifdef CONFIG_TOSHIBA_BOARDS
	ct_hw_pre_init(p);
#endif
	SetArRegs( );
	SetGrRegs( );
	SetSrRegs( );
        SetFrRegs( );
	SetXrRegs( );
#ifdef CONFIG_TOSHIBA_BOARDS
	memset(p->frame_buffer, 0, p->fix.smem_len);
#endif
}

/*****************************************************************************/
static void
ct_hw_write( const struct ct_par * pp )
{
#if 1
	u8 tmp, fr01;
#endif
	var2hw( &( ( struct ct_par * )pp )->var );
	fix2hw( &( ( struct ct_par * )pp )->fix );
#if 1
	switch (crtfp) {
	case CRTFP_MODE_LCD:
	case CRTFP_MODE_BOTH:
		fr01 = 0x02;
		break;
	default:
		fr01 = 0x01;
	}
	rr( fr, 0x01, tmp);
	if (tmp != fr01) {
		//printk("ctfb: FP %sable\n", (fr01 & 0x02) ? "En" : "Dis");
		wr( fr, 0x01, fr01);
	}
#endif
#if 0
	ct_hw_regdump();
#endif
}

/*****************************************************************************/
static void
ct_hw_read ( struct ct_par * pp )
{
	hw2var( &pp->var );
	hw2fix( &pp->fix );
}

/*****************************************************************************/
static void
ct_screen( int mode ) // 0 == screen off, else on
{ // Clocking Mode Register Bit 5: Screen Off: 1: Disables all graphics output
// except for video playback windows and turns off picture-generating logic
// allowing full memory bandwidth to be available for both host CPU accesses
// and accesses by multimedia engine for video capture and playback functions.
// Synchronization pulses to the display, however, are maintained. Setting
// this bit to 1 can be used as a way to more rapidly update the frame buffer.
// 0: Normal Operation	
	int tmp = 0;
	rr( sr, 0x01, tmp );      			
	wr( sr, 0x00, 0x00 );     // reset
	tmp = mode ? ( ~0x20 & tmp ) : ( 0x20 | tmp );
	wr( sr, 0x01, tmp );
	wr( sr, 0x00, 0x03 );     // enable
	wi( ar, 0x20 );	          // enable video
}

/*****************************************************************************/
static void
ct_sync( int mode ) // 0 == hsync, vsync off, else on
{
	unsigned char tmp;
	rr( cr, 0x17, tmp ); // bit 7: CRT Controller Reset, 0:Forces horiz.
	tmp = mode ? ( 0x80 | tmp ) : ( ~0x80 & tmp ); // and vert.sync signals
        wr( cr, 0x17, tmp ); // to be inactive. No other registers or outputs
}                            // are affected., 1: normal operation

/*****************************************************************************/
static int // display: 0 unblank, 1 blank, 2 no vsync, 3 no hsync, 4 off
ct_blank( int blank_mode, struct fb_info_gen *info )
{
	switch( blank_mode )
	{
	case 0: ct_screen( 1 ); ct_sync( 1 ); break;
	case 1: ct_screen( 0 ); 	      break;
	case 2:
	case 3:
	case 4: ct_screen( 0 ); ct_sync( 0 ); break;
	default:PRINTK( "blank_mode = %d  unexpected, using 0\n", blank_mode );
		ct_screen( 1 ); ct_sync( 1 ); return -EINVAL;
	}
	return 0;
}

/*****************************************************************************/
static int // Read a single color register and split into colors/transparent.
ct_getcolreg( unsigned regno, unsigned *red, unsigned *green, unsigned *blue,
	      unsigned *transp, struct fb_info *info )
{ // Return != 0 for invalid regno.
	if( ( regno > 255 ) ){
		PRINTK( "Error! regno=%d\n", regno );
		return -EINVAL;
	}
	out_8( DACRX_3c7_WO, regno );	
	udelay( 1 );	
	*red    = in_8( DACDATA_3c9 );    *red   |= *red   << 8;
	*green  = in_8( DACDATA_3c9 );    *green |= *green << 8;
	*blue   = in_8( DACDATA_3c9 );    *blue  |= *blue  << 8;
	*transp = 0;
							   	         	
	return 0;
}

/*****************************************************************************/
// Set a single color register. The values supplied are already rounded down
// to the hardware's capabilities (todo: what does that mean in detail?)
// according to the entries in var structure). Return != 0 for invalid regno.
static int
ct_setcolreg( unsigned regno, unsigned  red, unsigned  green, unsigned  blue,
	      unsigned transp, struct fb_info *info )
{
	int tmp = 0;
	if( ( regno > 255 ) ){
		PRINTK( "Error! regno=%d\n", regno );
	return -EINVAL;
	}
	rr( xr, 0x80, tmp );        // Disable extended status read feature to
	tmp &= 0xf7;	        // permit normal accesses to the registers and
	wr( xr, 0x80, 0x80 | tmp ); // color data locations within the palette.
	out_8( DACWX_3c8, regno );
	udelay( 1 );	
	out_8( DACDATA_3c9, red   >> 8 );
	out_8( DACDATA_3c9, green >> 8 );
	out_8( DACDATA_3c9, blue  >> 8 );
	if( regno < 16 ) // Make the first 16 colors of palette available to
	{ // fbcon: The first 16 colormap indexes are used for colouring text,
	  // rest is used if we have 8 bit color depth or if we access indexed.
	  // Ususllay we access in 16 od 24 bit mode mot thru the cmap - XR82:3
		struct ctfb_info *pi = ( struct ctfb_info* )info;
		switch( pi->current_bpp ) // modified copy from skeletonfb.c
		{
		#ifdef FBCON_HAS_CFB16
		case 16:/* RGB 565 */
			pi->cmap.cfb16[ regno ] =   ( red   & 0xf800 )
					        | ( ( green & 0xfc00 ) >>  5 )
						| ( ( blue  & 0xf800 ) >> 11 );
#ifdef __BIG_ENDIAN
			/* we do not use XR20 swap feature. swap here. */
			swab16s(&pi->cmap.cfb16[regno]);
#endif
			break;
		#endif
		#ifdef FBCON_HAS_CFB24
		case 24:/* RGB 888 */
#ifdef __BIG_ENDIAN
			pi->cmap.cfb24[ regno ] = ( ( red   & 0xff00 ) >>  8 )
					        |   ( green & 0xff00 )
						| ( ( blue  & 0xff00 ) <<  8 );
#else
			pi->cmap.cfb24[ regno ] = ( ( red   & 0xff00 ) <<  8 )
					        |   ( green & 0xff00 )
						| ( ( blue  & 0xff00 ) >>  8 );
#endif
			break;
		#endif
		#ifdef FBCON_HAS_CFB32
 		case 32:/* RGBA 8888 */
			pi->cmap.cfb32[ regno ] = ( ( red   & 0xff00 ) << 16 )
						| ( ( green & 0xff00 ) <<  8 )
						|   ( blue  & 0xff00 )
						| ( ( transp & 0xff00 ) >> 8 );
			 break;
		#endif
		}
	}
	return 0;
}



// ############################################################################
#endif // #####################################################################
// ############################################################################


/*			README information

Thanks
======
First of all thank to the Linux comunity, which gives so much hope for the
computer future.

Christian Gruner from Aglaia GmbH <csg@aglaia-gmbh.de> brought me on this way
and without his help I would have had a much harder time creating this driver.

Thanks also to members of the fbdev mailing list, helping with some hints:
James Simmons, Petr Vandrovec and others

Roots
=====
This software is influenced by several other framebuffer drivers, some code I
simply copied. I started over with the Powermac "chips" driver - (c) 1997
Fabio Riccardi. Later I reworked the stuff using hints in skeletonfb.c -
(c) 1997 Geert Uytterhoeven. Pleas check also assiliantfb from Adrian Cox,
framebuffer driver for the CT69030, derived from chipsfb.


Compiling as a separate module
==============================
If you do not already have intergrated the functionality of fbgen.c in your
kernel you need fbgen.o. To generate a module named fbgen.o edit
/usr/src/linux/drivers/video/Makefile, change the line

CONFIG_FBGEN_MODULE :=

into

CONFIG_FBGEN_MODULE :=y

and execute "make modules" in directory /usr/src/linux. This will produce
/usr/src/linux/drivers/video/fbgen.o. You have to load this module before
ctfb.o, just say "insmod /usr/src/linux/drivers/video/fbgen.o".
Or, once you did a make modules_install give the command 'modprobe fbgen'.

Enter cmd line: gcc -O2 -Wall -DMODULE -D__KERNEL__ -c ctfb.c

PLEASE make a sync before the very first time trying this driver on your
hardware - I could test the driver only on my hardware.


Compiling into the kernel (valid for kernel versions 2.2.x and 2.4.x)
=====================================================================
Not perfect but it works this way:

copy ctfb.c into /usr/src/linux/drivers/video

#. Modify /usr/src/linux/drivers/video/Makefile: (kernel 2.2.x)
-------------------------------add following 9 lines
ifeq ($(CONFIG_FB_CT),y)
 L_OBJS       += ctfb.o
 CONFIG_FBGEN_BUILTIN = y
else
  ifeq ($(CONFIG_FB_CT),m)
  M_OBJS       += ctfb.o
  CONFIG_FBGEN_MODULE = y
  endif
endif
--------------------------------

#. Modify /usr/src/linux/drivers/video/Makefile: (kernel 2.4.x)
-------------------------------add following line
obj-$(CONFIG_FB_CT)               += ctfb.o fbgen.o
-------------------------------
also add ctfb.o to export-objs    :=


#. Modify  /usr/src/linux/drivers/video/fbmem.c:
--------------------------------
add folowing 2 lines to section initialization and setup routines
extern int ctfb_init(void);
extern int ctfb_setup(char* options, int *ints);

// for KERNEL24_INTERFACE:

extern int ctfb_init(void);
extern int ctfb_setup(char*);
 
--------------------------------
add following 3 lines to fb_drivers[] __initdata
#ifdef CONFIG_FB_CT
        { "ctfb", ctfb_init, ctfb_setup },
#endif
--------------------------------

#. Modify /usr/src/linux/drivers/video/Config.in
--------------------------------
add following 3 lines into a right place
if [ "$CONFIG_PCI" != "n" ]; then
    tristate 'CT69000 display support (EXPERIMENTAL)' CONFIG_FB_CT
fi
--------------------------------
add one of the next two lines on several positions where it is nessesary
"$CONFIG_FB_CT" = "y" -o \
"$CONFIG_FB_CT" = "m" -o \
--------------------------------
You will find an example in ctfb_addons/Config.in

#. Build a new kernel after selecting "CT69000 display support"

#. Modify /etc/modules.conf: (nessesary?)
--------------------------------add following line
pre-install ctfb modprobe "fbgen"
--------------------------------

#. Question: Is it nessesary to add to file offb.c stuff like that? Why?
--------------------------------
#ifdef CONFIG_FB_MATROX
extern int matrox_of_init(struct device_node *dp);
#endif // CONFIG_FB_MATROX


Tests
=====
The ctfb.o is testet on following

I386 achitectures:
Linux kernel 2.2.10 running on a PCC5   board from DIGITAL-LOGIC with a
CT65554 video controller, P54C       CPU, HX430 chipset (only older versions)
Linux kernel 2.2.14, 2.2.18, 2.4.0 
running on a PCCPII board from DIGITAL-LOGIC with a
CT60000 video controller, Pentium II CPU, BX440 chipset


PPC achitectures:
PPC405GP based SBC from MPL AG

This version is _not_ tested on CT65554 hw, because I have no access to it
right now.

Features
========
Supports bitdepths 8, 16 and 24 with different resolutions and different pixel
clocks. You can have different modes on different displays.

You can also use PAL and NTSC TV resolutions.

You can start with your special mode - see example lilo.conf append= line
at the beginning of this file. Or select a mode from the predeined ones and modify it.
An lilo.conf example line: : append = "video=ctfb:mode:0,depth:8"


Limits
======
Only I386 hw is supported now. Only crt output is programmed, not the flat
panel regs. - I have no accesss to other hw.
Standard VESA modes not impemented yet.

Bugs
====
The 640x480 pixels upper left corner is filled with some white during setup.
To avoid this disable VGA text console if you compile ctfb into the kernel.
Please report bugs to <th@aglaia-gmbh.de>

Todo list (this is no promize for later implementation - just ideas)
====================================================================
Support memory mapped register access.
Merge with assiliantfb.
switch of cursor in graphics mode
bitblt acceleration
mtrr register support

Code layout
===========
There are some unusual self include of this file for less repeats in the code
and still having only one file. It should disturb the readability not too much.
The used tabulator size is 8.

Updates
=======
Check www.visuelle-maschinen.de/ctfb/
There is no further development on this driver from my side. Feel free to modify the code.


Version log
===========
0.16 	pre initial public release
0.17    initial public release
0.18    bugfixes
0.20	compiles now ito the kernel
0.30	first usable relase
0.35    less files, less code, less bugs
0.37    Minor changes concerning CT69030 compatibility
0.38    Mainly cutted the long lines into 80 columns peaces
0.39    less header files
0.40    All in one file
0.41    Minor changes in comments
0.50    support for kernel versions 2.4 and 2.2
0.51    support for PPC architecture

Change log
==========
0.17	Minor changes in the ctfb_README file
0.18    obsolete pci device info stuff removed
	kernel oops bug in case no device found fixed (Thanks Brad Douglas!)
        proc fs files in separate dir and default compiling of that switched
	off Bugfix: color info now in var - see ModifyColorInfo()
	minor code clean ups
        GENERATE_PROC_FS_ENTRIES is off in default mode
	in fix now FB_VISUAL_DIRECTCOLOR instead of FB_VISUAL_TRUECOLOR
	renamed ctfb_makefile to Makefile
        Init now with mode=2 (is 1152x864x16) - Do NOT use with CT5554!!!!
        cmap now correct initialized (must be after and not before
	register_framebuffer)
0.20	Added kernel params for ctfb - see ctfb:setup() for details
	renamed struct pci_dev_info into pcidev_info to avoid namespace
	collision. All ctfb files not nessesary for compilation of the kernel
	are in a separate directory named ctfb_addons.
0.30    blanking now ok
	module loads with parameters
	ctfb_debug bitwise dynamically on and off using proc fs
	Should support beneath CT65554 and CT69000 also CT65550 and CT65555
	(untested!) several bugfixes
	a lot of code mainenance
0.35    pixclock date correction in initdata
	fixed /proc bug if compiled as a module
	no more special ioctls -> use ctfb_addons/ct_io now
	no more registers in /proc/chips -> use ctfb_addons/ct_regs now
	renamed /proc/chips into /proc/fbinfo
	added CT9030 support (not tested yet)
	merged files to module.c and accesshw.c
	added /proc/fbinfo/version file
0.37    changed default contence of AR11 from 0xff to 0x00 to avoid blue
	overscan borders, put pci device pointer into cftb_info for device
	dependend handling, code cleanup in pixclock setting
	(Thanks to Adrian Cox <apc@agelectronics.co.uk>)
0.38    Nearly only code layout changed (shorter lines)
0.39    put varelem.h and params.h into module.c and addes some include logic.
	No changes of fuctionality.
0.40    Merged all files into one. No changes of fuctionality.
0.50    Removed proc fs stuff
        Removed debug code
0.51 Minor changes
*/
