/*
 *
 * tdfx_base.h
 *
 *	Framebuffer driver for video cards with 3dfx chipsets
 *
 * Authors: 
 *		Hannu Mallat <hannu@firsthop.com>
 *   		Attila Kesmarki <danthe@aat.hu>
 * 
 * Copyright (C) 1999,2000 Hannu Mallat & Attila Kesmarki 
 * All rights reserved
 *
 */

#ifndef __TDFXFB_H__
#define __TDFXFB_H__

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/fb.h>
#include <linux/selection.h>
#include <linux/console.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/nvram.h>
#include <linux/kd.h>
#include <linux/vt_kern.h>
#include <asm/io.h>
#include <linux/timer.h>

#ifdef CONFIG_MTRR
#include <asm/mtrr.h>
#endif

#include <video/fbcon.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>
#include <video/fbcon-cfb24.h>
#include <video/fbcon-cfb32.h>
#include <linux/spinlock.h>

/* There is a bug in the X server for tdfx: It doesn't restore the
 * hwcursor state properly at VT switch. 
 */

#define XServerVTSwitchBug
//#define TDFXFB_DEBUG

#ifdef TDFXFB_DEBUG
#define DPRINTK(a,b...) printk(KERN_DEBUG "fb: %s: " a, __FUNCTION__ , ## b);
//#define DPRINTK(a,b...) printk("fb: %s: " a, __FUNCTION__ , ## b);
#else
#define DPRINTK(a,b...)
#endif

#ifndef TDFXFB_DEBUG
#define assert(expr) \
	if(!(expr)) { \
        printk( "Assertion failed! %s,%s,%s,line=%d\n",\
        #expr,__FILE__,__FUNCTION__,__LINE__); \
	BUG(); \
        }
#else
#define assert(expr)
#endif

#define PICOS2KHZ(a) (1000000000UL/(a))
#define KHZ2PICOS(a) (1000000000UL/(a))

#define PCI_SUBDEVICE_ID_VOODOO3_2000 0x0036
#define PCI_SUBDEVICE_ID_VOODOO3_3000 0x003a

#ifndef PCI_DEVICE_ID_3DFX_VOODOO5
#define PCI_DEVICE_ID_3DFX_VOODOO5	0x0009
#endif

/* COMMAND_2D reg. values */
#define ROP_COPY	0xcc	// src
#define ROP_INVERT      0x55	// NOT dst
#define ROP_XOR         0x66	// src XOR dst

#define TDFXF_HSYNC_ACT_HIGH	0x01
#define TDFXF_HSYNC_ACT_LOW	0x02
#define TDFXF_VSYNC_ACT_HIGH	0x04
#define TDFXF_VSYNC_ACT_LOW	0x08
#define TDFXF_LINE_DOUBLE	0x10
#define TDFXF_VIDEO_ENABLE	0x20

#define TDFXF_HSYNC_MASK	0x03
#define TDFXF_VSYNC_MASK	0x0c

struct tdfxfb_par {
	u32 pixclock;

	u32 baseline;

	u32 width;
	u32 height;
	u32 width_virt;
	u32 height_virt;
	u32 lpitch;		/* line pitch, in bytes */
	u32 ppitch;		/* pixel pitch, in bits */
	u32 bpp;

	u32 hdispend;
	u32 hsyncsta;
	u32 hsyncend;
	u32 htotal;

	u32 vdispend;
	u32 vsyncsta;
	u32 vsyncend;
	u32 vtotal;

	u32 video;
	u32 accel_flags;
	u32 cmap_len;
	void (*putc) (u32, u32, struct display *, int, int, int);
	void (*putcs) (u32, u32, struct display *, const unsigned short *,
		       int, int, int);
};

struct board {
	short vendor, device, sdid;
	unsigned int maxclk;
	const char *name;
	int cardid;		// ID used by the driver
};

struct fb_info_tdfx {
	struct fb_info fb_info;

	struct list_head next_fb;
	struct pci_dev *pdev;

	unsigned int usecount;
	int dead;

	struct board *card;
	u16 dev;
	u32 max_pixclock;

   int lfbsize;
	void *regbase_virt;
	unsigned long iobase;

	struct {
		unsigned red, green, blue, pad;
	} palette[256];
	
	// init flags
	struct {
		int noinit;
		int hwcursor;
		int accel;
		int inverse;
		} devflags;
		
	struct tdfxfb_par default_par;
	struct tdfxfb_par current_par;
	struct display *disp;
	struct display_switch dispsw;
#if defined(FBCON_HAS_CFB16) || defined(FBCON_HAS_CFB24) || defined(FBCON_HAS_CFB32)
	union {
#ifdef FBCON_HAS_CFB16
		u16 cfb16[16];
#endif
#ifdef FBCON_HAS_CFB24
		u32 cfb24[16];
#endif
#ifdef FBCON_HAS_CFB32
		u32 cfb32[16];
#endif
	} fbcon_cmap;
#endif

	struct {
		int type;
		int state;
		int w, u, d;
		int x, y, redraw;
		unsigned long enable, disable;
		unsigned long cursorimage;
		struct timer_list timer;
	} cursor;
	int currcon;
	struct display *currcon_display;
	spinlock_t DAClock;

#ifdef CONFIG_MTRR
	int mtrr_idx;
#endif

};

/*
 *  Frame buffer device API
 */
int tdfxfb_open(struct fb_info *info, int user);
int tdfxfb_release(struct fb_info *info, int user);
int tdfxfb_get_fix(struct fb_fix_screeninfo *fix,
		   int con, struct fb_info *fb);
int tdfxfb_get_var(struct fb_var_screeninfo *var,
		   int con, struct fb_info *fb);
int tdfxfb_set_var(struct fb_var_screeninfo *var,
		   int con, struct fb_info *fb);
int tdfxfb_pan_display(struct fb_var_screeninfo *var,
		       int con, struct fb_info *fb);
int tdfxfb_get_cmap(struct fb_cmap *cmap,
		    int kspc, int con, struct fb_info *info);
int tdfxfb_set_cmap(struct fb_cmap *cmap,
		    int kspc, int con, struct fb_info *info);
int tdfxfb_ioctl(struct inode *inode,
		 struct file *file,
		 u_int cmd, u_long arg, int con, struct fb_info *info);

/*
 *  Interface to the low level console driver
 */
int tdfxfb_switch_con(int con, struct fb_info *fb);
int tdfxfb_updatevar(int con, struct fb_info *fb);
void tdfxfb_blank(int blank, struct fb_info *fb);

/*
 *  Internal routines
 */
int tdfxfb_decode_var(const struct fb_var_screeninfo *var,
		      struct tdfxfb_par *par,
		      const struct fb_info_tdfx *info);
int tdfxfb_encode_var(struct fb_var_screeninfo *var,
		      const struct tdfxfb_par *par,
		      const struct fb_info_tdfx *info);
void tdfxfb_set_disp(struct display *disp,
		     struct fb_info *info, struct tdfxfb_par *par, int accel);
int tdfxfb_getcolreg(u_int regno,
		     u_int * red,
		     u_int * green,
		     u_int * blue, u_int * transp, struct fb_info *fb);
int tdfxfb_setcolreg(u_int regno,
		     u_int red,
		     u_int green,
		     u_int blue, u_int transp, struct fb_info *fb);
void tdfxfb_install_cmap(struct display *d, struct fb_info *info);

void tdfxfb_hwcursor_init(struct fb_info_tdfx *info);
void tdfxfb_createcursorshape(struct display *p);
void tdfxfb_createcursor(struct display *p);

/*
 * do_xxx: Hardware-specific functions
 */
void do_pan_var(struct fb_var_screeninfo *var, struct fb_info_tdfx *i);
void do_flashcursor(unsigned long ptr);
void do_bitblt(void *mmiobase, u32 curx, u32 cury, u32 dstx, u32 dsty,
	       u32 width, u32 height, u32 stride, u32 bpp);
void do_fillrect(void *mmiobase, u32 x, u32 y, u32 w, u32 h,
		 u32 color, u32 stride, u32 bpp, u32 rop);
void do_putc(u32 fgx, u32 bgx, struct display *p, int c, int yy, int xx);
void do_putcs(u32 fgx, u32 bgx, struct display *p,
	      const unsigned short *s, int count, int yy, int xx);
u32 do_calc_pll(int freq, int *freq_out);
void do_set_par(struct tdfxfb_par *par, struct fb_info *info);
unsigned long do_lfb_size(struct fb_info_tdfx *info);
void do_blank(struct fb_info_tdfx *info, int state, int vgablank);
void do_hwcursor_reset(struct fb_info_tdfx *info);
void do_hwcursor_enable(struct fb_info_tdfx *info);
void do_hwcursor_disable(struct fb_info_tdfx *info);
void do_hwcursor_pos(struct fb_info_tdfx *info, int x, int y);
inline void do_setpalentry(void *mmiobase, unsigned regno, u32 c);
int do_get_monitor_sense(struct fb_info_tdfx *info);
int do_card_preinit(struct fb_info_tdfx *info);
int do_inb(struct fb_info_tdfx *info,int reg);
void do_outb(struct fb_info_tdfx *info,int reg,int val);

extern int nohwcursor;

#endif // __TDFXFB_H__
