/*
 *
 * tdfx_hw.h
 *
 *	Framebuffer driver for video cards with 3dfx chipsets
 *
 * Authors: 
 *		Hannu Mallat <hannu@firsthop.com>
 *   		Attila Kesmarki <danthe@aat.hu>
 * 
 * Copyright (C) 1999,2000 Hannu Mallat & Attila Kesmarki 
 * All rights reserved
 *
 */

#ifndef __TDFXHW_H__
#define __TDFXHW_H__

#include "../vga.h"
#include "tdfx_base.h"

#include <asm/unaligned.h>

#define SST_SGRAM_OFLOP_DEL_ADJ_SHIFT   20
#define SST_SGRAM_CLK_NODELAY           BIT(13)
#define SST_DRAM_REFRESH_EN             BIT(0)
#define SST_DRAM_REFRESH_VALUE_SHIFT    1
#define SST_DRAM_REFRESH_VALUE          (0x1FF<<SST_DRAM_REFRESH_VALUE_SHIFT)
#define SST_SGRAM_TYPE_SHIFT            27
#define SST_SGRAM_TYPE                  (0x1L<<SST_SGRAM_TYPE_SHIFT)
#define SST_SGRAM_NUM_CHIPSETS          BIT(26)
#define SST_SGRAM_TYPE_8MBIT		(0x0L<<SST_SGRAM_TYPE_SHIFT)
#define SST_SGRAM_TYPE_16MBIT		(0x1L<<SST_SGRAM_TYPE_SHIFT)
#define SST_DISABLE_2D_BLOCK_WRITE      BIT(15)
#define SST_MCTL_TYPE_SDRAM             BIT(30)
#define SST_DAC_MODE_2X			BIT(0)
#define SST_VIDEO_2X_MODE_EN            BIT(26)
#define SST_VGA0_EXTENSIONS             BIT(6)
#define SST_WAKEUP_3C3                  1
#define SST_VGA0_WAKEUP_SELECT_SHIFT    8
#define SST_VGA0_LEGACY_DECODE_SHIFT    9
#define SST_VGA0_LEGACY_DECODE          (1 << SST_VGA0_LEGACY_DECODE_SHIFT)
#define SST_VGA0_ENABLE_DECODE          0
#define SST_ENABLE_ALT_READBACK         0
#define SST_VGA0_CLUT_SELECT_SHIFT      2
#define SST_CLUT_SELECT_6BIT            0
#define SST_CLUT_SELECT_8BIT            1
#define SST_VGA0_CONFIG_READBACK_SHIFT  10
#define SST_VIDEO_PROCESSOR_EN          BIT(0)
#define SST_CURSOR_MODE_SHIFT           1
#define SST_CURSOR_X11                  (1<<SST_CURSOR_MODE_SHIFT)
#define SST_DESKTOP_EN                  BIT(7)
#define SST_DESKTOP_PIXEL_FORMAT_SHIFT  18
#define SST_DESKTOP_CLUT_BYPASS         BIT(10)
#define SST_HALF_MODE                   BIT(4)
#define SST_CURSOR_EN                   BIT(27)
#define SST_BUSY                        BIT(9)
#define SST_RETRACE			BIT(6)

#define MEM_TYPE_SGRAM  0
#define MEM_TYPE_SDRAM  1

/* SRCFORMAT */
#define SRCFORMAT_BYTESWIZZLE		BIT(20)
#define SRCFORMAT_BYTEPACKED            BIT(22)
#define SRCFORMAT_WORDPACKED            BIT(23)
#define SRCFORMAT_DWORDPACKED           (BIT(22) | BIT(23))

/* membase0 register offsets */
#define STATUS		0x00
#define PCIINIT0	0x04
#define SIPMONITOR	0x08
#define LFBMEMORYCONFIG	0x0c
#define MISCINIT0	0x10
#define MISCINIT1	0x14
#define DRAMINIT0	0x18
#define DRAMINIT1	0x1c
#define AGPINIT		0x20
#define TMUGBEINIT	0x24
#define VGAINIT0	0x28
#define VGAINIT1	0x2c
#define DRAMCOMMAND	0x30
#define DRAMDATA	0x34
#define STRAPINFO       0x38 
/* reserved             0x3c */
#define PLLCTRL0	0x40
#define PLLCTRL1	0x44
#define PLLCTRL2	0x48
#define DACMODE		0x4c
#define DACADDR		0x50
#define DACDATA		0x54
#define RGBMAXDELTA	0x58
#define VIDPROCCFG	0x5c
#define HWCURPATADDR	0x60
#define HWCURLOC	0x64
#define HWCURC0		0x68
#define HWCURC1		0x6c
#define VIDINFORMAT	0x70
#define VIDINSTATUS	0x74
#define VIDSERPARPORT	0x78
#define VIDINXDELTA	0x7c
#define VIDININITERR	0x80
#define VIDINYDELTA	0x84
#define VIDPIXBUFTHOLD	0x88
#define VIDCHRMIN	0x8c
#define VIDCHRMAX	0x90
#define VIDCURLIN	0x94
#define VIDSCREENSIZE	0x98
#define VIDOVRSTARTCRD	0x9c
#define VIDOVRENDCRD	0xa0
#define VIDOVRDUDX	0xa4
#define VIDOVRDUDXOFF	0xa8
#define VIDOVRDVDY	0xac
/*  ... */
#define VIDOVRDVDYOFF	0xe0
#define VIDDESKSTART	0xe4
#define VIDDESKSTRIDE	0xe8
#define VIDINADDR0	0xec
#define VIDINADDR1	0xf0
#define VIDINADDR2	0xf4
#define VIDINSTRIDE	0xf8
#define VIDCUROVRSTART	0xfc

#define BASE_2D         0x00100000
#define INTCTRL		(BASE_2D + 0x04)
#define CLIP0MIN	(BASE_2D + 0x08)
#define CLIP0MAX	(BASE_2D + 0x0c)
#define DSTBASE		(BASE_2D + 0x10)
#define DSTFORMAT	(BASE_2D + 0x14)
#define SRCBASE		(BASE_2D + 0x34)
#define COMMANDEXTRA_2D	(BASE_2D + 0x38)
#define CLIP1MIN	(BASE_2D + 0x4c)
#define CLIP1MAX	(BASE_2D + 0x50)
#define SRCFORMAT	(BASE_2D + 0x54)
#define SRCSIZE		(BASE_2D + 0x58)
#define SRCXY		(BASE_2D + 0x5c)
#define COLORBACK	(BASE_2D + 0x60)
#define COLORFORE	(BASE_2D + 0x64)
#define DSTSIZE		(BASE_2D + 0x68)
#define DSTXY		(BASE_2D + 0x6c)
#define COMMAND_2D	(BASE_2D + 0x70)
#define LAUNCH_2D	(BASE_2D + 0x80)

#define BASE_3D		0x00200000
#define LFBMODE		(BASE_3D + 0x114)		
#define COMMAND_3D	(BASE_3D + 0x120)

/* register bitfields (not all, only as needed) */

#define BIT(x) (1UL << (x))

#define AUTOINC_DSTX                    BIT(10)
#define AUTOINC_DSTY                    BIT(11)
#define COMMAND_2D_FILLRECT		0x05
#define COMMAND_2D_S2S_BITBLT		0x01	// screen to screen
#define COMMAND_2D_H2S_BITBLT           0x03	// host to screen


#define COMMAND_3D_NOP			0x00
#define STATUS_RETRACE			BIT(6)
#define STATUS_BUSY			BIT(9)
#define MISCINIT1_CLUT_INV		BIT(0)
#define MISCINIT1_2DBLOCK_DIS		BIT(15)
#define MISCINIT1_ROM_SIZE		BIT(25)
#define DRAMINIT0_SGRAM_NUM		BIT(26)
#define DRAMINIT0_SGRAM_TYPE		BIT(27)
#define DRAMINIT1_MEM_SDRAM		BIT(30)

// TODO: Put these defines (and the corresponding code) into tdfx_hw 
#define VGAINIT0_VGA_DISABLE		BIT(0)
#define VGAINIT0_EXT_TIMING		BIT(1)
#define VGAINIT0_8BIT_DAC		BIT(2)
#define VGAINIT0_EXT_ENABLE		BIT(6)
#define VGAINIT0_WAKEUP_3C3		BIT(8)
#define VGAINIT0_LEGACY_DISABLE		BIT(9)
#define VGAINIT0_ALT_READBACK		BIT(10)
#define VGAINIT0_FAST_BLINK		BIT(11)
#define VGAINIT0_EXTSHIFTOUT		BIT(12)
#define VGAINIT0_DECODE_3C6		BIT(13)
#define VGAINIT0_SGRAM_HBLANK_DISABLE	BIT(22)
#define VGAINIT1_MASK			0x1fffff
#define VIDCFG_VIDPROC_ENABLE		BIT(0)
#define VIDCFG_CURS_X11			BIT(1)
#define VIDCFG_HALF_MODE		BIT(4)
#define VIDCFG_DESK_ENABLE		BIT(7)
#define VIDCFG_CLUT_BYPASS		BIT(10)
#define VIDCFG_2X			BIT(26)
#define VIDCFG_HWCURSOR_ENABLE          BIT(27)
#define VIDCFG_PIXFMT_SHIFT		18
#define DACMODE_2X			BIT(0)

#define REFFREQ 			1431818	// = ref. freq * 100

#ifndef FB_ACCEL_3DFX_BANSHEE
#define FB_ACCEL_3DFX_BANSHEE 		31
#endif

// This is missing from vga.h
#ifndef VGA_IS0_R
#define VGA_IS0_R			0x3c2
#endif

#if defined(__mips__) && defined(__BIG_ENDIAN)
extern inline unsigned short do_safe_readw(void * addr)
{
	return readb(addr) | (readb(addr + 1) << 8);
}
extern inline unsigned long do_safe_readl(void * addr)
{
	return readb(addr) | (readb(addr + 1) << 8) |
		(readb(addr + 2) << 16) | (readb(addr + 3) << 24);
}
#else
#define do_safe_readw(addr) get_unaligned((u16 *)(addr))
#define do_safe_readl(addr) get_unaligned((u32 *)(addr))
#endif

struct banshee_reg {
	/* VGA rubbish */
	unsigned char att[21];
	unsigned char crt[25];
	unsigned char gra[9];
	unsigned char misc[1];
	unsigned char seq[5];

	/* Banshee extensions */
	unsigned char ext[2];
	unsigned long vidcfg;
	unsigned long vidpll;
	unsigned long mempll;
	unsigned long gfxpll;
	unsigned long dacmode;
	unsigned long vgainit0;
	unsigned long vgainit1;
	unsigned long screensize;
	unsigned long stride;
	unsigned long cursloc;
	unsigned long curspataddr;
	unsigned long cursc0;
	unsigned long cursc1;
	unsigned long startaddr;
	unsigned long clip0min;
	unsigned long clip0max;
	unsigned long clip1min;
	unsigned long clip1max;
	unsigned long srcbase;
	unsigned long dstbase;
	unsigned long miscinit0;
};

void do_write_regs(struct fb_info_tdfx *info, struct banshee_reg *reg);

/* ------------------------------------------------------------------------- 
 *                      Hardware-specific inline funcions
 * ------------------------------------------------------------------------- */

static inline u8 vga_inb(unsigned long iobase, u32 reg)
{
	return inb(iobase + reg - 0x300);
}
static inline u16 vga_inl(unsigned long iobase, u32 reg)
{   
	return inl(iobase + reg - 0x300);
}

static inline void vga_outb(unsigned long iobase, u32 reg, u8 val)
{
	outb(val, iobase + reg - 0x300);
}
static inline void vga_outl(unsigned long iobase, u32 reg, u32 val)
{
	outl(val, iobase + reg - 0x300);
}

static inline void gra_outb(unsigned long iobase, u32 idx, u8 val)
{
	vga_outb(iobase, VGA_GFX_I, idx);
	vga_outb(iobase, VGA_GFX_D, val);
}

static inline u8 gra_inb(unsigned long iobase, u32 idx)
{
	vga_outb(iobase, VGA_GFX_I, idx);
	return vga_inb(iobase, VGA_GFX_D);
}

static inline void seq_outb(unsigned long iobase, u32 idx, u8 val)
{
	vga_outb(iobase, VGA_SEQ_I, idx);
	vga_outb(iobase, VGA_SEQ_D, val);
}

static inline u8 seq_inb(unsigned long iobase, u32 idx)
{
	vga_outb(iobase, VGA_SEQ_I, idx);
	return vga_inb(iobase, VGA_SEQ_D);
}

static inline void crt_outb(unsigned long iobase, u32 idx, u8 val)
{
	vga_outb(iobase, VGA_CRT_IC, idx);
	vga_outb(iobase, VGA_CRT_DC, val);
}

static inline u8 crt_inb(unsigned long iobase, u32 idx)
{
	vga_outb(iobase, VGA_CRT_IC, idx);
	return vga_inb(iobase, VGA_CRT_DC);
}

static inline void att_outb(unsigned long iobase, u32 idx, u8 val)
{
	unsigned char tmp;
	tmp = vga_inb(iobase, VGA_IS1_RC);
	vga_outb(iobase, VGA_ATT_IW, idx);
	vga_outb(iobase, VGA_ATT_IW, val);
}

static inline u8 att_inb(unsigned long iobase, u32 idx)
{
	unsigned char tmp;
	tmp = vga_inb(iobase, VGA_IS1_RC);
	vga_outb(iobase, VGA_ATT_IW, idx);
	return vga_inb(iobase, VGA_ATT_R);
}

static inline void vga_disable_video(unsigned long iobase)
{
	unsigned char s;
	s = seq_inb(iobase, 0x01) | 0x20;
	seq_outb(iobase, 0x00, 0x01);
	seq_outb(iobase, 0x01, s);
	seq_outb(iobase, 0x00, 0x03);
}

static inline void vga_enable_video(unsigned long iobase)
{
	unsigned char s;
	s = seq_inb(iobase, 0x01) & 0xdf;
	seq_outb(iobase, 0x00, 0x01);
	seq_outb(iobase, 0x01, s);
	seq_outb(iobase, 0x00, 0x03);
}

static inline void vga_disable_palette(unsigned long iobase)
{
	vga_inb(iobase, VGA_IS1_RC);
	vga_outb(iobase, VGA_ATT_IW, 0x00);
}

static inline void vga_enable_palette(unsigned long iobase)
{
	vga_inb(iobase, VGA_IS1_RC);
	vga_outb(iobase, VGA_ATT_IW, 0x20);
}

static inline u32 tdfx_inl(void *mmiobase, unsigned int reg)
{
#if defined(__powerpc__)
  	u32 val;
  		asm("lwbrx %0,%1,%2" : "=r"(val) : "b" (reg), "r" (mmiobase));
#endif
	return readl(mmiobase + reg);
}

static inline void tdfx_outl(void *mmiobase, unsigned int reg, u32 val)
{	
#if defined(__powerpc__)
	asm("stwbrx %0,%1,%2" : : "r" (val), "b" (reg), "r" (mmiobase) : "memory");
#endif
	writel(val, mmiobase + reg);
}

// Wait until at least <size> entries will be available 
// in the graphics processor's FIFO buffer.
static inline void do_make_room(void *mmiobase, int size)
{
	while ((tdfx_inl(mmiobase, STATUS) & 0x1f) < size);
}

static inline void do_wait_idle(void *mmiobase)
{
	int i = 0;

	do_make_room(mmiobase, 1);
	tdfx_outl(mmiobase, COMMAND_3D, COMMAND_3D_NOP);

	while (1) {
		i = (tdfx_inl(mmiobase, STATUS) & STATUS_BUSY) ? 0 : i + 1;
		if (i == 3)
			break;
	}
}

/* 
 * After these defines you can use the functions without indicating  
 * their first argument directly. (mmiobase/iobase)
 */
#define tdfx_inl(A) tdfx_inl(mmiobase,(A))
#define tdfx_outl(A,B) tdfx_outl(mmiobase,(A),(B))
#define do_make_room(A) do_make_room(mmiobase,(A))
#define do_wait_idle() do_wait_idle(mmiobase)

#define vga_enable_palette() vga_enable_palette(iobase)
#define vga_enable_video() vga_enable_video(iobase)
#define vga_disable_palette() vga_disable_palette(iobase)
#define vga_disable_video() vga_disable_video(iobase)

#endif // __TDFXHW_H__
