
/*-------------------------------------------------------------*/
/*-- Huffman coding low-level stuff                          --*/
/*--                                      lib_bzip_huffman.c --*/
/*-------------------------------------------------------------*/

/*--
  This file is part of lib_bzip, a block-sorting compression 
  library designed to compress 32kbyte blocks, for on-the-fly
  disk compression/decompression.

  Version 0.02, 4-Jan-1998

  Copyright (C) 1996, 1997, 1998 by Julian Seward.
     Guildford, Surrey, UK
     email: jseward@acm.org

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--*/


#include "lib_bzip_private.h"


/*---------------------------------------------------*/
/*--- Huffman coding low-level stuff              ---*/
/*---------------------------------------------------*/

#define WEIGHTOF(zz0)  ((zz0) & 0xffffff00U)
#define DEPTHOF(zz1)   ((zz1) & 0x000000ffU)
#define MYMAX(zz2,zz3) ((zz2) > (zz3) ? (zz2) : (zz3))

#define ADDWEIGHTS(zw1,zw2)                           \
   (WEIGHTOF(zw1)+WEIGHTOF(zw2)) |                    \
   (1 + MYMAX(DEPTHOF(zw1),DEPTHOF(zw2)))

#define UPHEAP(z)                                     \
do {                                                  \
   Int32 zz, tmp;                                     \
   zz = z; tmp = heap[zz];                            \
   while (weight[tmp] < weight[heap[zz >> 1]]) {      \
      heap[zz] = heap[zz >> 1];                       \
      zz >>= 1;                                       \
   }                                                  \
   heap[zz] = tmp;                                    \
} while(0)

#define DOWNHEAP(z)                                   \
do {                                                  \
   Int32 zz, yy, tmp;                                 \
   zz = z; tmp = heap[zz];                            \
   while (True) {                                     \
      yy = zz << 1;                                   \
      if (yy > nHeap) break;                          \
      if (yy < nHeap &&                               \
          weight[heap[yy+1]] < weight[heap[yy]])      \
         yy++;                                        \
      if (weight[tmp] < weight[heap[yy]]) break;      \
      heap[zz] = heap[yy];                            \
      zz = yy;                                        \
   }                                                  \
   heap[zz] = tmp;                                    \
} while(0)


/*---------------------------------------------*/
void hbMakeCodeLengths ( UChar* len, 
                         Int32* freq,
                         Int32  alphaSize,
                         Int32  maxLen,
                         // temporaries
                         Int32* heap,
                         Int32* weight,
                         Int32* parent)
{
   /*--
      Nodes and heap entries run from 1.  Entry 0
      for both the heap and nodes is a sentinel.
   --*/
   Int32 nNodes, nHeap, n1, n2, i, j, k;
   Bool  tooLong;

   /*--
      Now passed in from caller, so as to keep stack
      frame small.
   Int32 heap   [ MAX_ALPHA_SIZE + 2 ];
   Int32 weight [ MAX_ALPHA_SIZE * 2 ];
   Int32 parent [ MAX_ALPHA_SIZE * 2 ]; 
   --*/

   for (i = 0; i < alphaSize; i++)
      weight[i+1] = (freq[i] == 0 ? 1 : freq[i]) << 8;

   while (True) {

      nNodes = alphaSize;
      nHeap = 0;

      heap[0] = 0;
      weight[0] = 0;
      parent[0] = -2;

      for (i = 1; i <= alphaSize; i++) {
         parent[i] = -1;
         nHeap++;
         heap[nHeap] = i;
         UPHEAP(nHeap);
      }
   
      while (nHeap > 1) {
         n1 = heap[1]; heap[1] = heap[nHeap]; nHeap--; DOWNHEAP(1);
         n2 = heap[1]; heap[1] = heap[nHeap]; nHeap--; DOWNHEAP(1);
         nNodes++;
         parent[n1] = parent[n2] = nNodes;
         weight[nNodes] = ADDWEIGHTS(weight[n1], weight[n2]);
         parent[nNodes] = -1;
         nHeap++;
         heap[nHeap] = nNodes;
         UPHEAP(nHeap);
      }

      tooLong = False;
      for (i = 1; i <= alphaSize; i++) {
         j = 0;
         k = i;
         while (parent[k] >= 0) { k = parent[k]; j++; }
         len[i-1] = j;
         if (j > maxLen) tooLong = True;
      }
      
      if (! tooLong) break;

      for (i = 1; i < alphaSize; i++) {
         j = weight[i] >> 8;
         j = 1 + (j / 2);
         weight[i] = j << 8;
      }
   }
}


/*---------------------------------------------*/
void hbAssignCodes ( UInt32 *code,
                     UChar *length,
                     unsigned minLen,
                     unsigned maxLen,
                     Int32 alphaSize )
{
   Int32 i;
   UInt32 vec;
   unsigned n;

   vec = 0;
   for (n = minLen; n <= maxLen; n++) {
      for (i = 0; i < alphaSize; i++)
         if (length[i] == n) { code[i] = vec; vec++; };
      vec <<= 1;
   }
}


/*---------------------------------------------*/
void hbCreateDecodeTables ( Int32 *limit,
                            Int32 *base,
                            Int32 *perm,
                            UChar *length,
                            Int32 minLen,
                            Int32 maxLen,
                            Int32 alphaSize )
{
   Int32 pp, i, j, vec;

   pp = 0;
   for (i = minLen; i <= maxLen; i++)
      for (j = 0; j < alphaSize; j++)
         if (length[j] == i) { perm[pp] = j; pp++; };

   for (i = 0; i < MAX_CODE_LEN; i++) base[i] = 0;
   for (i = 0; i < alphaSize; i++) base[length[i]+1]++;

   for (i = 1; i < MAX_CODE_LEN; i++) base[i] += base[i-1];

   for (i = 0; i < MAX_CODE_LEN; i++) limit[i] = 0;
   vec = 0;

   for (i = minLen; i <= maxLen; i++) {
      vec += (base[i+1] - base[i]);
      limit[i] = vec-1;
      vec <<= 1;
   }
   for (i = minLen + 1; i <= maxLen; i++)
      base[i] = ((limit[i-1] + 1) << 1) - base[i];
}


/*-------------------------------------------------------------*/
/*-- end                                  lib_bzip_huffman.c --*/
/*-------------------------------------------------------------*/
