
/*-------------------------------------------------------------*/
/*-- private header for lib_bzip          lib_bzip_private.h --*/
/*-------------------------------------------------------------*/

/*--
  This file is part of lib_bzip, a block-sorting compression 
  library designed to compress 32kbyte blocks, for on-the-fly
  disk compression/decompression.

  Version 0.02, 3-Jan-1998

  Copyright (C) 1996, 1997, 1998 by Julian Seward.
     Guildford, Surrey, UK
     email: jseward@acm.org

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--*/


#ifndef LIB_BZIP_PRIVATE_H
#define LIB_BZIP_PRIVATE_H


#ifndef __KERNEL__
# include <stdlib.h>
# include <stdio.h>
/* unused: #include <assert.h> */
#endif


/*---------------------------------------------*/
// useful types

typedef char            Char;
typedef unsigned char   Bool;
typedef unsigned char   UChar;
typedef int             Int32;
typedef unsigned int    UInt32;
/* UInt31 is used for things that can safely be cast to either UInt32
   or Int32 */
typedef unsigned int    UInt31;
typedef short           Int16;
typedef unsigned short  UInt16;

#define True  ((Bool)1)
#define False ((Bool)0)



/*---------------------------------------------*/
// global constants

// max block size, 
// and max pointer denormalisation when sorting
#define N_BLOCK     32768
#define N_OVERSHOOT (1U+12U)

// parameters for run-len coding of zeroes
#define RUNBASE 2
#define RUNA 0
#define RUNB 1

// parameters for the back end
#define MAX_ALPHA_SIZE (256 + RUNBASE)
#define MAX_SELECTORS (2 + (N_BLOCK / G_SIZE))
#define MAX_CODE_LEN 23

// Don\'t change N_GROUPS; doing so will make the
// back end totally not work (not even slightly!)
// sendMTFValues() and the compressed file format
// now assume that N_GROUPS == 2.
#define N_GROUPS 2
#define G_SIZE   50
#define N_ITERS  4



/*---------------------------------------------*/
// a type containing all storage needed for compression
typedef
   struct {
      // verbosity of debugging output
      Int32 verb;

      // block size, block, pointers to block
      unsigned nblock;
      UInt16 block[N_BLOCK + N_OVERSHOOT];
      UInt16 zptr [N_BLOCK];

      // freq table and running order, both for sorting
      Int32  ftab  [769];
      Int32  rorder[256];

      // regarding block randomisation during sorting
      Bool  randomised;
      Int32 workDone;
      Int32 workLimit;

      // a record of which chars are used in block,
      // and a mapping into contiguous values 0 .. nInUse-1
      Bool  inUse[256];
      Int32 nInUse;
      UChar unseqToSeq[256];

      // to do with MTF values
      Int32 mtfFreq[MAX_ALPHA_SIZE];
      Int32 nMTF;

      // to do with calculating Huffman coding tables
      UChar  len  [N_GROUPS][MAX_ALPHA_SIZE];
      UInt32 code [N_GROUPS][MAX_ALPHA_SIZE];
      Int32  rfreq[N_GROUPS][MAX_ALPHA_SIZE];
      UChar  selector   [MAX_SELECTORS];

      // temporaries used only for calculating
      // Huffman code lengths
      Int32 heap_tmp   [ MAX_ALPHA_SIZE + 2 ];
      Int32 weight_tmp [ MAX_ALPHA_SIZE * 2 ];
      Int32 parent_tmp [ MAX_ALPHA_SIZE * 2 ]; 

      // bit stream buffering
      UInt32 bsBuff;
      Int32  bsLive;
      Int32  outctr;
      UChar* outbuff;
      Int32  totalSizeAllowed;
      Int32  predictedSize;  // only for debugging
   }
   Lib_Bzip_Encode_Storage_Ty;


/*---------------------------------------------*/
// constants for the fast MTF decoder
#define MTFA_SIZE 4096
#define MTFL_SIZE 16


// a type containing all storage needed for decompression
typedef
   struct {
      Bool   inUse[256];
      UInt16 origPtr;
      Int32  nInUse;
      UChar  seqToUnseq[256];
      Bool   randomised;

      // temporaries used only for MTF decoding
      UInt32 cc_tmp      [256];
      UChar  mtfa_tmp    [MTFA_SIZE];
      Int32  mtfbase_tmp [256 / MTFL_SIZE];

      // the selector tables
      UChar  selector   [MAX_SELECTORS];
 
      // huffman decoding tables
      UChar len    [N_GROUPS][MAX_ALPHA_SIZE];
      Int32 limit  [N_GROUPS][MAX_ALPHA_SIZE];
      Int32 base   [N_GROUPS][MAX_ALPHA_SIZE];
      Int32 perm   [N_GROUPS][MAX_ALPHA_SIZE];
      Int32 minLens[N_GROUPS];

      // for the inverse BWT
      UInt32 tvec[N_BLOCK];

      // bit stream buffering
      UInt32 bsBuff;
      Int32  bsLive;
      Int32  inctr;

      // buffers, and size of uncompressed block
      UChar* inbuff;
      UChar* outbuff;
      Int32  nblock;
   }
   Lib_Bzip_Decode_Storage_Ty;


/*---------------------------------------------*/
// Huffman coding helper functions

extern 
void 
hbMakeCodeLengths ( UChar* len, 
                    Int32* freq,
                    Int32  alphaSize,
                    Int32  maxLen,
                    Int32* heap,
                    Int32* weight,
                    Int32* parent );

extern 
void 
hbAssignCodes ( UInt32 *code,
                UChar *length,
                unsigned minLen,
                unsigned maxLen,
                Int32 alphaSize );

extern 
void 
hbCreateDecodeTables ( Int32 *limit,
                       Int32 *base,
                       Int32 *perm,
                       UChar *length,
                       Int32 minLen,
                       Int32 maxLen,
                       Int32 alphaSize );


/*---------------------------------------------*/
// stuff for randomising blocks

// numbers themselves are in lib_bzip_e.c
#define rNums bzip2_rNums /* don't pollute name space */
extern Int16 const rNums[64];

#define RAND_DECLS                                \
   Int16 rNToGo = 0;                              \
   Int16 rTPos  = 0;                              \
   Int16 rMask  = 1;

#define RAND_MASK ((rNToGo == 1) ? rMask : 0)

#define RAND_UPD_MASK                             \
   if (rNToGo == 0) {                             \
      rNToGo = rNums[rTPos] / 10;                 \
      rTPos++; if (rTPos == 64) rTPos = 0;        \
      rMask++; if (rMask == 256) rMask = 1;       \
   }                                              \
   rNToGo--;

#if 0
extern void *bzip2_work_area;
#endif

#endif

/*-------------------------------------------------------------*/
/*-- end                                  lib_bzip_private.h --*/
/*-------------------------------------------------------------*/
