#include <linux/string.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/ext2_fs_c.h>
#include <linux/module.h>

#include "zlib.h"

/*
 *	Deflate (as called by ext2_wGZIP) needs the following 
 *	memory:
 *	
 *		64K	hash table	(32K entries)
 *		64K	chain table	(32K entries)
 *		64K	item buffer	(16K items)
 *		 5K	internal state
 *
 *	The memory requirements could probably be reduced. 
 *	With cluster sizes less than 32K, the chain table and
 *	item buffer can be made smaller.
 *
 *	Inflate (as called by ext2_rGZIP) needs the following 
 *	memory:
 *
 *		32K	output window
 *			a few bytes for the code table ...
 *			... actually it's about another ...
 *		32K
 *
 */

#if 0
static char *ext2_gzip_heap_base = NULL;
static char *ext2_gzip_heap_ptr  = NULL;
static int   ext2_gzip_heap_used = 0;
static int   ext2_gzip_heap_size = (64 + 64 + 64 + 8) * 1024;
#else
# define	ext2_gzip_heap_size_C	((64 + 64 + 64 + 8) * 1024)
# define	ext2_gzip_heap_size_D	((32 + 32) * 1024)
#endif

#if 0
const char *z_errmsg[10] = {
"need dictionary",     /* Z_NEED_DICT       2  */
"stream end",          /* Z_STREAM_END      1  */
"",                    /* Z_OK              0  */
"file error",          /* Z_ERRNO         (-1) */
"stream error",        /* Z_STREAM_ERROR  (-2) */
"data error",          /* Z_DATA_ERROR    (-3) */
"insufficient memory", /* Z_MEM_ERROR     (-4) */
"buffer error",        /* Z_BUF_ERROR     (-5) */
"incompatible version",/* Z_VERSION_ERROR (-6) */
""};
#endif

struct heap_dets {
    unsigned char *curr_ptr; /* pointer to first unused byte on heap */
    unsigned char *end; /* pointer past the end of the available heap space */
};

voidp  ext2_zcalloc OF((voidp const opaque, unsigned items, unsigned size))
{
#define heap_details ((struct heap_dets *)opaque)
    unsigned bytes = items * size;
    void    *ptr;

    if (heap_details->curr_ptr + bytes > heap_details->end)
	return 0;

    ptr = heap_details->curr_ptr;
    heap_details->curr_ptr += bytes;
#if 0
    memset(ptr, 0, bytes);
    printf ("zcalloc (%p, %d, %d) \t-> %p\n", private, items, size, ptr);
#endif 

    return ptr;
#undef heap_details
}

/*
 *	Do nothing ... Could be improved ...!
 */

void ext2_zcfree  OF((voidp opaque, voidp ptr))
{
}


size_t ext2_iGZIP (int action)
{
	switch (action) {
		case EXT2_ALG_INIT_COMPRESS:
			return ext2_gzip_heap_size_C;
		case EXT2_ALG_INIT_DECOMPRESS:
			return ext2_gzip_heap_size_D;
		default:
			return 0;
	}
}

/*
 *	Compression
 */

size_t ext2_wGZIP (__u8 *ibuf, __u8 *obuf, void *heap,
		size_t ilen, size_t olen, int level)
{
    z_stream stream;
    int      err;

    MOD_INC_USE_COUNT;

    stream.next_in   = ibuf;
    stream.avail_in  = ilen;

    stream.zalloc    = ext2_zcalloc;
    stream.zfree     = ext2_zcfree;
    stream.opaque    = heap;
#define heap_details ((struct heap_dets *)heap)
    heap_details->curr_ptr = (unsigned char *)heap + sizeof(struct heap_dets);
    heap_details->end = (unsigned char *)heap + ext2_gzip_heap_size_C;
#undef heap_details

    if ((err = ext2_deflateInit2 (&stream, level, 8, -15, 8, Z_DEFAULT_STRATEGY)) != Z_OK) {
	MOD_DEC_USE_COUNT;
	return 0;
    }

    stream.next_out  = obuf;
    stream.avail_out = olen;

    err = ext2_deflate (&stream, Z_FINISH);

    if (err != Z_STREAM_END) {
	ext2_deflateEnd (&stream);
	MOD_DEC_USE_COUNT;
	return 0;
    }

    if ((err = ext2_deflateEnd (&stream)) != Z_OK) {
	MOD_DEC_USE_COUNT;
	return 0;
    }

    MOD_DEC_USE_COUNT;
    return stream.total_out;
}

/*
 *	Decompression
 */

size_t ext2_rGZIP (__u8 *ibuf, __u8 *obuf, void *heap,
		size_t ilen, size_t olen, int ignored)
{
    z_stream stream;
    int      err;

    MOD_INC_USE_COUNT;

    stream.next_in   = ibuf;
    stream.avail_in  = ilen;

    stream.zalloc    = ext2_zcalloc;
    stream.zfree     = ext2_zcfree;
    stream.opaque    = heap;
#define heap_details ((struct heap_dets *)heap)
    heap_details->curr_ptr = (unsigned char *)heap + sizeof(struct heap_dets);
    heap_details->end = (unsigned char *)heap + ext2_gzip_heap_size_D;
#undef heap_details

    if ((err = ext2_inflateInit2 (&stream, -15)) != Z_OK) {
        MOD_DEC_USE_COUNT;
	return 0;
    }

    stream.next_out  = obuf;
    stream.avail_out = olen;

    err = ext2_inflate (&stream, Z_FINISH);
  
    if (err != Z_STREAM_END && (err != Z_OK || stream.total_out < olen)) {
	ext2_inflateEnd (&stream);
        MOD_DEC_USE_COUNT;
	return err;
    }

    if ((err = ext2_inflateEnd (&stream)) != Z_OK) {
        MOD_DEC_USE_COUNT;
	return 0;
    }

    MOD_DEC_USE_COUNT;
    return stream.total_out;
}


#ifdef MODULE

int init_module(void)
{
	struct ext2_algorithm gzip_alg;

	gzip_alg.name = NULL;
	gzip_alg.avail = 1;
	gzip_alg.init = ext2_iGZIP;
	gzip_alg.compress = ext2_wGZIP;
	gzip_alg.decompress = ext2_rGZIP;

	return ext2_register_compression_module(EXT2_GZIP_ALG, 
			ext2_gzip_heap_size_C, ext2_gzip_heap_size_D,
			&gzip_alg);
}

void cleanup_module(void)
{
	ext2_unregister_compression_module(EXT2_GZIP_ALG);
}

#endif
