/*
 * linux/fs/ext2/ioctl.c
 *
 * Copyright (C) 1993, 1994, 1995
 * Remy Card (card@masi.ibp.fr)
 * Laboratoire MASI - Institut Blaise Pascal
 * Universite Pierre et Marie Curie (Paris VI)
 */

/*
 *  Copyright (C) 2001 Alcatel Business Systems - R&D Illkirch
 *  Added by Pierre Peiffer (pierre.peiffer@sxb.bsf.alcatel.fr) (2001-08-09)
 *  (Copied from pjm patch e2compr-0.4.39-patch-2.2.18)
 *      e2compress stuff.
 *
 * << Copyright (C) 1995  Antoine Dumesnil de Maricourt (dumesnil@etca.fr) >>
 * <<     (transparent compression code)				   >>
 */

#include <linux/fs.h>
#include <linux/ext2_fs.h>
#include <linux/sched.h>
#include <asm/uaccess.h>
#include <linux/ext2_fs_c.h>
#include <linux/stat.h>
#include <linux/config.h>

#ifdef CONFIG_EXT2_COMPRESS
#include <linux/kmod.h>
#endif

#ifndef MIN
# define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif

int ext2_ioctl (struct inode * inode, struct file * filp, unsigned int cmd,
		unsigned long arg)
{
	unsigned int flags;
#ifdef CONFIG_EXT2_COMPRESS
	unsigned long datum;
	int err;
#endif

	ext2_debug ("cmd = %u, arg = %lu\n", cmd, arg);

	switch (cmd) {
	case EXT2_IOC_GETFLAGS:
		flags = inode->u.ext2_i.i_flags & EXT2_FL_USER_VISIBLE;
		return put_user(flags, (int *) arg);
	case EXT2_IOC_SETFLAGS: {
		unsigned int oldflags;

		if (IS_RDONLY(inode))
			return -EROFS;

		if ((current->fsuid != inode->i_uid) && !capable(CAP_FOWNER))
			return -EPERM;

		if (get_user(flags, (int *) arg))
			return -EFAULT;

		oldflags = inode->u.ext2_i.i_flags;

		/*
		 * The IMMUTABLE and APPEND_ONLY flags can only be changed by
		 * the relevant capability.
		 *
		 * This test looks nicer. Thanks to Pauline Middelink
		 */
		if ((flags ^ oldflags) & (EXT2_APPEND_FL | EXT2_IMMUTABLE_FL)) {
			if (!capable(CAP_LINUX_IMMUTABLE))
				return -EPERM;
		}

		flags = flags & EXT2_FL_USER_MODIFIABLE;
#ifdef CONFIG_EXT2_COMPRESS
		if (S_ISREG (inode->i_mode) || S_ISDIR (inode->i_mode)) {

			/* pjm 1998-01-14: In previous versions of
			     e2compr, the kernel forbade raising
			     EXT2_ECOMPR_FL from userspace.  I can't
			     think of any purpose for forbidding this,
			     and I find it useful to raise
			     EXT2_ECOMPR_FL for testing purposes, so
			     I've removed the forbidding code. */

			if (S_ISREG (inode->i_mode)
			    && (EXT2_NOCOMPR_FL
				& (flags ^ inode->u.ext2_i.i_flags))) {

				/* NOCOMPR_FL can only be changed if
				   nobody else has the file opened.  */
				/* pjm 1998-02-16: inode->i_count is
				   useless to us because only dentries
				   use inodes now.  Unfortunately,
				   there isn't an easy way of finding
				   the equivalent.  We'd have to go
				   through all dentries using the
				   inode, and sum their d_count
				   values.  Rather than do that, I'd
				   rather get rid of the exclusion
				   constraint.  todo. */
				if (/*inode->i_count > 1*/ 0)
					return -ETXTBSY;
				else {
					/* pjm 970429: Discarding
					     cached pages is not very
					     clean, but should work. */
					/* pjm 980114: Not quite.  We
					     should also sync any
					     mappings to buffers first.
					     This isn't very important,
					     as none of the current
					     e2compr programs can
					     trigger this, but todo. */
					invalidate_inode_pages (inode);
				}
			}

			if (EXT2_COMPR_FL
			    & (flags ^ inode->u.ext2_i.i_flags)) {
				if (flags & EXT2_COMPR_FL) {
					if (inode->u.ext2_i.i_flags
					    & EXT2_COMPRBLK_FL) {
						/* There shouldn't actually be any
						   compressed blocks, AFAIK.  However,
						   this is still possible because sometimes
						   COMPRBLK gets raised just to stop 
						   us changing cluster size at the wrong
						   time.

						   todo: Call a function that just
						   checks that there are not compressed
						   clusters, and print a warning if any are
						   found. */
					} else {
						int bits = MIN(EXT2_DEFAULT_LOG2_CLU_NBLOCKS,
							       (EXT2_LOG2_MAX_CLUSTER_BYTES
								- inode->i_sb->s_blocksize_bits));

						inode->u.ext2_i.i_log2_clu_nblocks = bits;
						inode->u.ext2_i.i_clu_nblocks = 1 << bits;
					}
					inode->u.ext2_i.i_compr_method = EXT2_DEFAULT_COMPR_METHOD;
					if (S_ISREG (inode->i_mode)) {
						flags |= EXT2_DIRTY_FL;
						inode->u.ext2_i.i_compr_flags |= EXT2_CLEANUP_FL;
					}
				} else if (S_ISREG (inode->i_mode)) {
					if (inode->u.ext2_i.i_flags & EXT2_COMPRBLK_FL) {
						int err;

						err = ext2_decompress_inode(inode);
						if (err)
							return err;
					}
					inode->u.ext2_i.i_flags &= ~EXT2_DIRTY_FL;
					inode->u.ext2_i.i_compr_flags &= ~EXT2_CLEANUP_FL;
				}
			}
		}
#endif
		flags |= oldflags & ~EXT2_FL_USER_MODIFIABLE;
		inode->u.ext2_i.i_flags = flags;

		if (flags & EXT2_SYNC_FL)
			inode->i_flags |= S_SYNC;
		else
			inode->i_flags &= ~S_SYNC;
		if (flags & EXT2_APPEND_FL)
			inode->i_flags |= S_APPEND;
		else
			inode->i_flags &= ~S_APPEND;
		if (flags & EXT2_IMMUTABLE_FL)
			inode->i_flags |= S_IMMUTABLE;
		else
			inode->i_flags &= ~S_IMMUTABLE;
		if (flags & EXT2_NOATIME_FL)
			inode->i_flags |= S_NOATIME;
		else
			inode->i_flags &= ~S_NOATIME;
		inode->i_ctime = CURRENT_TIME;
		mark_inode_dirty(inode);
		return 0;
	}
	case EXT2_IOC_GETVERSION:
		return put_user(inode->i_generation, (int *) arg);
	case EXT2_IOC_SETVERSION:
		if ((current->fsuid != inode->i_uid) && !capable(CAP_FOWNER))
			return -EPERM;
		if (IS_RDONLY(inode))
			return -EROFS;
		if (get_user(inode->i_generation, (int *) arg))
			return -EFAULT;	
		inode->i_ctime = CURRENT_TIME;
		mark_inode_dirty(inode);
		return 0;
#ifdef CONFIG_EXT2_COMPRESS
	case EXT2_IOC_GETCOMPRMETHOD:	/* Result means nothing if COMPR_FL is not set */
		return put_user (inode->u.ext2_i.i_compr_method, (long *) arg);
	case EXT2_IOC_SETCOMPRMETHOD:
		if ((current->fsuid != inode->i_uid) && !fsuser ())
			return -EPERM;
		if (IS_RDONLY (inode))
			return -EROFS;
		if (get_user (datum, (long*) arg))
			return -EFAULT;
		if (!S_ISREG (inode->i_mode) && !S_ISDIR (inode->i_mode)) 
			return -ENOSYS;
		/* todo: Allow the below, but set initial value of
		   i_compr_meth at read_inode() time (using default if
		   !comprblk) instead of +c time.  Same for cluster
		   size. */
		if ((unsigned) datum >= EXT2_N_METHODS)
			return -EINVAL;
		if (inode->u.ext2_i.i_compr_method != datum) {
			if ((inode->u.ext2_i.i_compr_method == EXT2_NEVER_METH)
			    && (inode->u.ext2_i.i_flags & EXT2_COMPR_FL))
				return -EPERM;
			/* If the previous method was `defer' then
			   take a look at all uncompressed clusters
			   and try to compress them.  (pjm 1997-04-16) */
			if (inode->u.ext2_i.i_compr_method == EXT2_DEFER_METH) {
				inode->u.ext2_i.i_flags |= EXT2_DIRTY_FL;
				inode->u.ext2_i.i_compr_flags |= EXT2_CLEANUP_FL;
			}
			if (datum == EXT2_NEVER_METH) {
				if ((inode->u.ext2_i.i_flags & EXT2_COMPRBLK_FL)
				    && ((err = ext2_decompress_inode(inode))
					< 0))
					return err;
				inode->u.ext2_i.i_flags &= ~EXT2_DIRTY_FL;
				inode->u.ext2_i.i_compr_flags &= ~EXT2_CLEANUP_FL;
			}
			inode->u.ext2_i.i_compr_method = datum;
			inode->i_ctime = CURRENT_TIME;
			mark_inode_dirty(inode);
		}
#ifdef CONFIG_KMOD
		if (!ext2_algorithm_table[ext2_method_table[datum].alg].avail) {
			char str[32];

			sprintf(str, "ext2-compr-%d", ext2_method_table[datum].alg);
			request_module(str);
		}
#endif
		datum = ((datum < EXT2_N_METHODS)
			 && (ext2_algorithm_table[ext2_method_table[datum].alg].avail));
		return put_user(datum, (long *)arg);

	case EXT2_IOC_GETCLUSTERBIT:
		if (get_user (datum, (long*) arg))
			return -EFAULT;
		if (!S_ISREG (inode->i_mode))
			return -ENOSYS;
		/* We don't do `down(&inode->i_sem)' here because
		   there's no way for userspace to do the
		   corresponding up().  Userspace must rely on
		   EXT2_NOCOMPR_FL if it needs to lock. */
		err = ext2_cluster_is_compressed (inode, datum);
		if (err < 0)
			return err;
		return put_user ((err ? 1 : 0),
				 (long *) arg);

	case EXT2_IOC_RECOGNIZE_COMPRESSED:
		if (get_user (datum, (long*) arg))
			return -EFAULT;
		if (!S_ISREG (inode->i_mode))
			return -ENOSYS;
		if (IS_RDONLY (inode))
			return -EROFS;
		return ext2_recognize_compressed (inode, datum);

	case EXT2_IOC_GETCLUSTERSIZE:
		/* Result means nothing if COMPR_FL is not set (until
                   SETCLUSTERSIZE w/o COMPR_FL is implemented;
                   todo). */
		if (!S_ISREG (inode->i_mode)
		    && !S_ISDIR (inode->i_mode)) 
			return -ENOSYS;
		return put_user (inode->u.ext2_i.i_clu_nblocks, (long *) arg);

	case EXT2_IOC_GETFIRSTCLUSTERSIZE:
		/* Result means nothing if COMPR_FL is not set (until
                   SETCLUSTERSIZE w/o COMPR_FL is implemented;
                   todo). */
		if (!S_ISREG (inode->i_mode)
		    && !S_ISDIR (inode->i_mode)) 
			return -ENOSYS;
		return put_user (ext2_first_cluster_nblocks(inode), (long *) arg);

	case EXT2_IOC_SETCLUSTERSIZE:
		if ((current->fsuid != inode->i_uid) && !fsuser ())
			return -EPERM;
		if (IS_RDONLY (inode))
			return -EROFS;
		if (get_user (datum, (long *) arg))
			return -EFAULT;
		if (!S_ISREG (inode->i_mode)
		    && !S_ISDIR (inode->i_mode)) 
			return -ENOSYS;

		/* These are the only possible cluster sizes.  The
		   cluster size must be a power of two so that
		   clusters don't straddle address (aka indirect)
		   blocks.  At the moment, the upper limit is constrained
		   by how much memory is allocated for de/compression.
		   Also, the gzip algorithms have some optimisations
		   that assume tht the input is no more than 32KB,
		   and in compress.c we would need to zero more bits
		   of head->holemap.  (In previous releases, the file
		   format was limited to 32 blocks and under 64KB.) */
#if EXT2_MAX_CLUSTER_BLOCKS > 32 || EXT2_MAX_CLUSTER_NBYTES > 32768
# error "This code not updated for cluster size yet."
#endif
		switch (datum) {
		case (1 << 2): datum = 2; break;
		case (1 << 3): datum = 3; break;
		case (1 << 4): datum = 4; break;
		case (1 << 5): datum = 5; break;
		default: return -EINVAL;
		}

		assert (inode->u.ext2_i.i_clu_nblocks == (1 << inode->u.ext2_i.i_log2_clu_nblocks));
		if (datum == inode->u.ext2_i.i_log2_clu_nblocks)
			return 0;

		if (inode->u.ext2_i.i_flags & EXT2_ECOMPR_FL)
			return -EPERM;
		if (!(inode->u.ext2_i.i_flags & EXT2_COMPR_FL))
			return -ENOSYS;

		/* We currently lack a mechanism to change the cluster
		   size if there are already some compressed clusters.
		   The compression must be done in userspace
		   (e.g. with the e2compress program) instead.  */
		if (inode->u.ext2_i.i_flags & EXT2_COMPRBLK_FL)
			return -ENOSYS;

		if (datum + inode->i_sb->s_blocksize_bits
		    > EXT2_LOG2_MAX_CLUSTER_BYTES)
			return -EINVAL;

		inode->u.ext2_i.i_log2_clu_nblocks = datum;
		inode->u.ext2_i.i_clu_nblocks = 1 << datum;
		inode->i_ctime = CURRENT_TIME;
		mark_inode_dirty(inode);
		return 0;

	case EXT2_IOC_GETCOMPRRATIO:
		if (!S_ISREG (inode->i_mode)) 
			return -ENOSYS;
		if (inode->u.ext2_i.i_flags & EXT2_ECOMPR_FL)
			return -EPERM;
		if ((long) (datum = ext2_count_blocks (inode)) < 0)
			return datum;
		if ((err = put_user ((long) datum, (long*) arg)))
			return err;
		return put_user ((long) inode->i_blocks, (long*) arg + 1);
#endif

	default:
		return -ENOTTY;
	}
}
